//////////////////////////////////////////////////////////////////////////
// FILE:         vd_diaglog_qualification.cpp
// PROJECT:      all
// SW-COMPONENT: DiagLog
//----------------------------------------------------------------------
//
// DESCRIPTION: DiagLog Report Record 
//              
//----------------------------------------------------------------------
// COPYRIGHT:    (c) 2012 Robert Bosch GmbH, Hildesheim
// HISTORY:      
// Date      | Author                       | Modification
// 25.07.12  | BSOT Plischke                | Initial version
//
//////////////////////////////////////////////////////////////////////////
// first include diaglog settings
#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_main
   #include <common/framework/vd_diaglog_main.h>
#endif

#include <common/framework/vd_diaglog_settings.h>

#include "vd_diaglog_qualification.h"

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_report_record
   #include <common/framework/vd_diaglog_report_record.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_report_memory
   #include <common/framework/vd_diaglog_report_memory.h>
#endif

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DIAGLOG_INFO
#include "trcGenProj/Header/vd_diaglog_qualification.cpp.trc.h"
#endif


class tQualiResultCallBack
{
private:
   tQualiResultCallBack(); //lint !e1704

public:
   tQualiResultCallBack(vdl_teDiaglogReportMsg _state,vdl_tclReportRecord* _pReportRecord,vdl_tclQualificationCallbackInterface* _pCallBackIf ) : state(_state),pReportRecord(_pReportRecord),pCallBackIf(_pCallBackIf) {}

   vdl_teDiaglogReportMsg                 state;
   vdl_tclReportRecord*                   pReportRecord;
   vdl_tclQualificationCallbackInterface* pCallBackIf;
};

typedef std::list<tQualiResultCallBack>                    tQualiResultCallBackList;
typedef std::list<tQualiResultCallBack>::iterator          tQualiResultCallBackListIt;
//typedef std::list<tQualiResultCallBack>::const_iterator    tQualiResultCallBackListCIt;



///////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////
//                          vdl_tclQualificationRecord                           //
///////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclQualificationRecord::vdl_tclQualificationRecord
//
// DESCRIPTION:  constructor
//             
// PARAMETER:    none
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
vdl_tclQualificationRecord::vdl_tclQualificationRecord ():
   m_pReportRecord(NULL),
   m_pCallBackIf(NULL),
   m_s32PassQualiyfingCounter(0), 
   m_s32FailQualiyfingCounter(0)
{
}

   ///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclQualificationRecord::~vdl_tclQualificationRecord
//
// DESCRIPTION:  destructor
//             
// PARAMETER:    none
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
vdl_tclQualificationRecord::~vdl_tclQualificationRecord ()
{
   m_pReportRecord = NULL;
   m_pCallBackIf = NULL;
}

///////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////
//                          vdl_tclQualificationBase                             //
///////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////

OSAL_tSemHandle vdl_tclQualificationBase::m_hQualificationSem = OSAL_C_INVALID_HANDLE;
#define DIAGLOG_QUALIFICATION_SEM_NAME   "DIAGLOG_QUALITYSEM"

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclQualificationBase::vdl_tclQualificationBase
//
// DESCRIPTION:  constructor
//             
// PARAMETER:    none
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
vdl_tclQualificationBase::vdl_tclQualificationBase ()
{

      // Create semaphore to avoid parallel processing
      tS32 s32Result = OSAL_s32SemaphoreCreate(DIAGLOG_QUALIFICATION_SEM_NAME, &m_hQualificationSem, 1); 
      if(OSAL_OK != s32Result)
      {
         tU32 u32ErrorCode= OSAL_u32ErrorCode();
         ETG_TRACE_ERRMEM(("!!! Diaglog: vdl_tclQualificationBase::vdl_tclQualificationBase => OSAL_s32SemaphoreCreate DIAGLOG_QUALIFICATION_SEM_NAME failed: Result=%x, Error=%x",s32Result, static_cast<tUInt>(u32ErrorCode)));
         NORMAL_M_ASSERT_ALWAYS();
         m_hQualificationSem = OSAL_C_INVALID_HANDLE;
      }
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclQualificationBase::~vdl_tclQualificationBase
//
// DESCRIPTION:  destructor
//             
// PARAMETER:    none
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
vdl_tclQualificationBase::~vdl_tclQualificationBase ()
{
   ETG_TRACE_USR1_THR(( "--- vdl_tclQualificationBase::~vdl_tclQualificationBase => Close the Semaphores"));

      tS32 s32Result = OSAL_s32SemaphoreClose(m_hQualificationSem);   
      if(OSAL_OK != s32Result)
      {
         tU32 u32ErrorCode= OSAL_u32ErrorCode();
         et_vErrmemStringNormal( TR_COMP_DIAGNOSIS, "Diaglog: vdl_tclQualificationBase::~vdl_tclQualificationBase => OSAL_s32SemaphoreClose failed: Result=%x, Error=%x",
                                  s32Result,
                                  u32ErrorCode);
         NORMAL_M_ASSERT_ALWAYS();
      }
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclQualificationBase::registerCallBack
//
// DESCRIPTION:  MemoryObject call to register for Qualification
//             
// PARAMETER:    vdl_tclQualificationCallbackInterface* const pCallBack => pointer to MemoryObject
//
// RETURNVALUE:  bool failed/succesfull
//
///////////////////////////////////////////////////////////////////////////////////
bool vdl_tclQualificationBase::registerCallBack(vdl_tclQualificationCallbackInterface* const pCallBack)
{
   bool bFound = false;
   bool bReturn = false;

   for(tCallBackListIt iPos = m_poCallBackList.begin(); iPos.operator!=(m_poCallBackList.end()); ++iPos)
   {
      if(*iPos == pCallBack)
      {
         bFound = true;
         bReturn = false;
         ETG_TRACE_COMP_THR(("--- vdl_tclQualificationBase::registerCallBack => Memory:%08x Still registered -> reject",pCallBack));
         break;
      }
   }

   if(bFound == false)
   {
      ETG_TRACE_COMP_THR(("--- vdl_tclQualificationBase::registerCallBack => Memory:%08x added",pCallBack));
      m_poCallBackList.push_back(pCallBack);
      bReturn = true;
   }

   return bReturn;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclQualificationBase::unregisterCallBack
//
// DESCRIPTION:  MemoryObject call to unregister for Qualification
//             
// PARAMETER:    vdl_tclQualificationCallbackInterface* const pCallBack => pointer to MemoryObject
//
// RETURNVALUE:  bool failed/succesfull
//
///////////////////////////////////////////////////////////////////////////////////
//lint -e438
bool vdl_tclQualificationBase::unregisterCallBack(vdl_tclQualificationCallbackInterface* const pCallBack)
{
   bool bReturn = false;

   for(tCallBackListIt iPos = m_poCallBackList.begin(); iPos.operator!=(m_poCallBackList.end()); ++iPos)
   {
      if(*iPos == pCallBack)
      {
         bReturn = true;
         ETG_TRACE_COMP_THR(("--- vdl_tclQualificationBase::unregisterCallBack => Memory:%08x found -> remove",pCallBack));
         clearQualiListByCallBackIf(*iPos);
         //iPos = m_poCallBackList.erase(iPos);  // delete from list
         (void) m_poCallBackList.erase(iPos);  // delete from list
         break;
      }
   }
   return bReturn;
}
//lint +e438
///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclQualificationBase::isCallBackRegistered
//
// DESCRIPTION:  check if Memory is registered
//             
// PARAMETER:    vdl_tclQualificationCallbackInterface* const pCallBack => pointer to MemoryObject
//
// RETURNVALUE:  bool not found / Found
//
///////////////////////////////////////////////////////////////////////////////////
bool vdl_tclQualificationBase::isCallBackRegistered(vdl_tclQualificationCallbackInterface* const pCallBack)
{
   bool bReturn = false;

   for(tCallBackListIt iPos = m_poCallBackList.begin(); iPos.operator!=(m_poCallBackList.end()); ++iPos)
   {
      if(*iPos == pCallBack)
      {
         bReturn = true;
         ETG_TRACE_COMP_THR(("--- vdl_tclQualificationBase::isCallBackregistered => Memory:%08x found",pCallBack));
         break;
      }
   }
   return bReturn;
}


///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclQualificationBase::addToQualificationListPassed
//
// DESCRIPTION:  Add a DTC (Passed) to the qualification List
//             
// PARAMETER:    pReportRecord => Pointer to Element
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
void vdl_tclQualificationBase::addToQualificationListPassed (vdl_tclQualificationCallbackInterface* const pCallBack, vdl_tclReportRecord* pReportRecord)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclQualificationBase::addToQualificationListPassed"));

   if(  (pReportRecord != NULL)
      &&(pCallBack     != NULL))
   {
      vdl_tclQualificationRecord oQualifcationRecord;

      oQualifcationRecord.m_pCallBackIf      = pCallBack;
      oQualifcationRecord.m_pReportRecord    = pReportRecord;
      tU16 u16ITC                            = pReportRecord->u16ITC();
      tU32 u32DTC                            = pReportRecord->u32DTC();
      tU8  u8MemoryId                        = pReportRecord->u8GetMemoryId();

      oQualifcationRecord.m_s32PassQualiyfingCounter   = pReportRecord->s32GetPassValue();
      oQualifcationRecord.m_s32FailQualiyfingCounter   = 0;

      if (isCallBackRegistered(pCallBack) == true)
      {
         LockQualiSemaphore();
         {
            ETG_TRACE_COMP_THR(("--- vdl_tclQualificationBase::addToQualificationListPassed => Memory:%02d ITC:%04x DTC:%06x Counter:%d",u8MemoryId,u16ITC,u32DTC,oQualifcationRecord.m_s32PassQualiyfingCounter));
            m_oQualificationList.push_back(oQualifcationRecord);
         }
         FreeQualiSemaphore();
         checkToStartQualiTimer();
      }
      else
      {
         ETG_TRACE_FATAL_THR(("--- vdl_tclQualificationBase::addToQualificationListPassed => Memory not registered unkown CallBackIF %08x for Memory:%02d ITC:%04x DTC:%06x",pCallBack,u8MemoryId,u16ITC,u32DTC));
      }
   }
   ETG_TRACE_USR3_THR(( "<-- vdl_tclQualificationBase::addToQualificationListPassed"));
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclQualificationBase::addToQualificationListFailed
//
// DESCRIPTION:  Add a DTC (FAILED) to the qualification List
//             
// PARAMETER:    pReportRecord => Pointer to Element
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
void vdl_tclQualificationBase::addToQualificationListFailed (vdl_tclQualificationCallbackInterface* const pCallBack, vdl_tclReportRecord* pReportRecord)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclQualificationBase::addToQualificationListFailed"));

   if(  (pReportRecord != NULL)
      &&(pCallBack     != NULL))
   {
      vdl_tclQualificationRecord oQualifcationRecord;

      oQualifcationRecord.m_pCallBackIf      = pCallBack;
      oQualifcationRecord.m_pReportRecord    = pReportRecord;
      tU16 u16ITC                            = pReportRecord->u16ITC();
      tU32 u32DTC                            = pReportRecord->u32DTC();
      tU8  u8MemoryId                        = pReportRecord->u8GetMemoryId();

      oQualifcationRecord.m_s32FailQualiyfingCounter   = pReportRecord->s32GetFailValue();
      oQualifcationRecord.m_s32PassQualiyfingCounter   = 0;


      if (isCallBackRegistered(pCallBack) == true)
      {
         LockQualiSemaphore();
         {
            ETG_TRACE_COMP_THR(("--- vdl_tclQualificationBase::addToQualificationListFailed => Memory:%02d ITC:%04x DTC:%06x Counter:%d",u8MemoryId,u16ITC,u32DTC,oQualifcationRecord.m_s32FailQualiyfingCounter));
            m_oQualificationList.push_back(oQualifcationRecord);
         }
         FreeQualiSemaphore();
         checkToStartQualiTimer();
      }
      else
      {
         ETG_TRACE_FATAL_THR(("--- vdl_tclQualificationBase::addToQualificationListFailed => Memory not registered CallBackIF %08x for Memory:%02d ITC:%04x DTC:%06x",pCallBack,u8MemoryId,u16ITC,u32DTC));
      }
   }

   ETG_TRACE_USR3_THR(( "<-- vdl_tclQualificationBase::addToQualificationListFailed"));
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclQualificationBase::processQualificationStep
//
// DESCRIPTION:  do the next qualifikation step, called by Event DIAGLOG_EVENT_TIMER_QUALI_MASK
//             
// PARAMETER:    none
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
void vdl_tclQualificationBase::processQualificationStep()
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclQualificationBase::processQualificationStep"));

   tQualiResultCallBackList callBackList;

   LockQualiSemaphore();
   {
      // for each entry in qualifying list
      tQualificationListIt iPos = m_oQualificationList.begin();
      while(iPos.operator!=(m_oQualificationList.end()))
      {
         vdl_tclReportRecord*                   pReportRecord = iPos->m_pReportRecord;
         vdl_tclQualificationCallbackInterface* pCallBackIf   = iPos->m_pCallBackIf;

         if(  (pReportRecord != NULL)
            &&(pCallBackIf   != NULL))
         {
            tU16 u16ITC                   = pReportRecord->u16ITC();
            tU32 u32DTC                   = pReportRecord->u32DTC();
            tU8  u8MemoryId               = pReportRecord->u8GetMemoryId();
            tU32 u32BlockingModeActive = 0;
            if(isBlockingModeActive(pReportRecord,u32BlockingModeActive)== true)
            {
               ETG_TRACE_COMP_CLS(( TR_CLASS_DIAGLOG_SM, ": [%d] Stop Qualification of ITC:%04x DTC:%06x because of BM:%x",u8MemoryId, u16ITC,u32DTC,u32BlockingModeActive));
               // we have to stop qualifining if BlockingMode is active
               iPos = m_oQualificationList.erase(iPos);  // delete from list
               continue; // continue with next item in List
            }

            // no reason to stop -> do qualifing step now!!!
            if(iPos->m_s32PassQualiyfingCounter)
            {
               ETG_TRACE_USR2_THR(( "--- vdl_tclQualificationBase::processQualificationStep => PASS Qualify Memory:%02d ITC:%04x DTC:%06x Counter:%d [ms]",u8MemoryId,u16ITC,u32DTC,iPos->m_s32PassQualiyfingCounter));

               // we are qualifing a PASS condition
               iPos->m_s32PassQualiyfingCounter -= DIAGLOG_QUALIFICATION_STEP_TIME;
               if(iPos->m_s32PassQualiyfingCounter <= 0)
               {
                  // qualification done
                  // add to callback list, do not call here because of semaphore
                  ETG_TRACE_USR1_THR(( "--- vdl_tclQualificationBase::processQualificationStep => Add to callback list Memory:%d ITC:%04x DTC:%06x PASSED",u8MemoryId,u16ITC,u32DTC));
                  tQualiResultCallBack callBack(DIAGLOG_REPORT_PASSED,pReportRecord,pCallBackIf);
                  callBackList.push_back(callBack);

                  iPos = m_oQualificationList.erase(iPos);  // delete from list
                  continue; // continue with next item in List
               }
            }// if((*iPos)->m_s32PassQualiyfingCounter)
            else if(iPos->m_s32FailQualiyfingCounter)
            {
               ETG_TRACE_USR2_THR(( "--- vdl_tclQualificationBase::processQualificationStep => FAIL Qualify ITC:%04x DTC:%06x Counter:%d [ms]",u16ITC,u32DTC,iPos->m_s32FailQualiyfingCounter));

               // we are qualifing a FAIL condition
               iPos->m_s32FailQualiyfingCounter -= DIAGLOG_QUALIFICATION_STEP_TIME;
               if(iPos->m_s32FailQualiyfingCounter <= 0)
               {
                  // qualification done
                  // add to callback list, do not call here because of semaphore
                  ETG_TRACE_USR1_THR(( "--- vdl_tclQualificationBase::processQualificationStep => Add to callback list Memory:%d ITC:%04x DTC:%06x FAILED",u8MemoryId,u16ITC,u32DTC));
                  tQualiResultCallBack callBack(DIAGLOG_REPORT_FAILED,pReportRecord,pCallBackIf);
                  callBackList.push_back(callBack);

                  iPos = m_oQualificationList.erase(iPos);  // delete from list
                  continue; // continue with next item in List
               }
            }// if((*iPos)->m_s32FailQualiyfingCounter)
         }

         ++iPos;  // next entry
      }// while(iPos.operator!=(m_oQualificationList.end()))
   }
   FreeQualiSemaphore();

   // now forward info to memory class
   tQualiResultCallBackListIt iter = callBackList.begin();
   while(iter.operator!=(callBackList.end()))
   {
      vdl_teDiaglogReportMsg                 state          = iter->state;
      vdl_tclReportRecord*                   pReportRecord  = iter->pReportRecord;
      vdl_tclQualificationCallbackInterface* pCallBackIf    = iter->pCallBackIf;

      if(  (pReportRecord != NULL)
         &&(pCallBackIf != NULL))
      {
         tU16 u16ITC                   = pReportRecord->u16ITC();
         tU32 u32DTC                   = pReportRecord->u32DTC();
         tU8  u8MemoryId               = pReportRecord->u8GetMemoryId();
         if(state == DIAGLOG_REPORT_PASSED)
         {
            ETG_TRACE_COMP_CLS(( TR_CLASS_DIAGLOG_SM, ": [%d] QUALIFIED ITC:%04x DTC:%06x PASSED",u8MemoryId,u16ITC,u32DTC));
            pCallBackIf->vPassQualified(pReportRecord);
            pCallBackIf->vQualificationStepFinshed();
         }
         else if(state == DIAGLOG_REPORT_FAILED)
         {
            ETG_TRACE_COMP_CLS(( TR_CLASS_DIAGLOG_SM, ": [%d] QUALIFIED ITC:%04x DTC:%06x FAILED",u8MemoryId,u16ITC,u32DTC));
            pCallBackIf->vFailQualified(pReportRecord);
            pCallBackIf->vQualificationStepFinshed();
         }
      }
      ++iter;
   }// while(iter.operator!=(callBackList.end()))


   checkToStopQualiTimer();

   ETG_TRACE_USR3_THR(( "<-- vdl_tclQualificationBase::processQualificationStep"));
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclQualificationBase::clearQualiList
//
// DESCRIPTION:  clear the complete qualifikation List
//             
// PARAMETER:    none
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
void vdl_tclQualificationBase::clearQualiList()
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclQualificationBase::clearQualiList"));

   LockQualiSemaphore();
   {
      m_oQualificationList.clear();
   }
   FreeQualiSemaphore();
   checkToStopQualiTimer();

   ETG_TRACE_USR3_THR(( "<-- vdl_tclQualificationBase::clearQualiList"));
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclQualificationBase::clearQualiList
//
// DESCRIPTION:  clear the complete qualifikation List
//             
// PARAMETER:    none
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
void vdl_tclQualificationBase::clearQualiListByDtc(tU32 u32DtcMask)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclQualificationBase::clearQualiList"));

   LockQualiSemaphore();
   {
      // for each entry in qualifying list
      tQualificationListIt iPos = m_oQualificationList.begin();
      while(iPos.operator!=(m_oQualificationList.end()))
      {
         vdl_tclReportRecord* pReportRecord = iPos->m_pReportRecord;
         if(pReportRecord != NULL)
         {
            tU32 u32DTC = pReportRecord->u32DTC();
            if(u32DtcMask == u32DTC)
            {
               tU16 u16ITC          = pReportRecord->u16ITC();
               tU8  u8MemoryId      = pReportRecord->u8GetMemoryId();
               ETG_TRACE_COMP_CLS(( TR_CLASS_DIAGLOG_SM, ": [%d] Stop Qualification of ITC:%04x DTC:%06x because of clearQualiListByDtc",u8MemoryId,u16ITC,u32DTC));
               (void) m_oQualificationList.erase(iPos);  // delete from list
               break;
            }
         }// if(pReportRecord != NULL)
         ++iPos; //lint !e845 
      }// while(iPos.operator!=(m_oQualificationList.end()))
   }
   FreeQualiSemaphore();
   checkToStopQualiTimer();

   ETG_TRACE_USR3_THR(( "<-- vdl_tclQualificationBase::clearQualiList"));
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclQualificationBase::clearQualiListByMemoryId
//
// DESCRIPTION:  the Memory needs to remove all elements from qualification list
//             
// PARAMETER:    tU8 memoryId
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
void vdl_tclQualificationBase::clearQualiListByMemoryId (tU8 memoryId)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclQualificationBase::clearQualiListByMemoryId"));
   LockQualiSemaphore();
   {
      // for each entry in qualifying list
      tQualificationListIt iPos = m_oQualificationList.begin();
      while(iPos.operator!=(m_oQualificationList.end()))
      {
         vdl_tclReportRecord* pReportRecord = iPos->m_pReportRecord;
         if(pReportRecord != NULL)
         {
            if(pReportRecord->u8GetMemoryId() == memoryId)
            {
               tU16 u16ITC       = pReportRecord->u16ITC();
               tU32 u32DTC       = pReportRecord->u32DTC();
               tU8  u8MemoryId   = pReportRecord->u8GetMemoryId();
               ETG_TRACE_COMP_CLS(( TR_CLASS_DIAGLOG_SM, ": [%d] Stop Qualification of ITC:%04x DTC:%06x because of clearQualiListByMemoryId",u8MemoryId,u16ITC,u32DTC));
               // we have to stop qualifining if BlockingMode is active
               iPos = m_oQualificationList.erase(iPos);  // delete from list
               continue; // continue with next item in List
            }
         }// if(pReportRecord != NULL)
         ++iPos;
      }// while(iPos.operator!=(m_oQualificationList.end()))
   }
   FreeQualiSemaphore();
   ETG_TRACE_USR3_THR(( "<-- vdl_tclQualificationBase::clearQualiListByMemoryId"));
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclQualificationBase::clearQualiListByCallBackIf
//
// DESCRIPTION:  the Memory needs to remove all elements from qualification list
//             
// PARAMETER:    pCallBack: Memory Callback IF
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
void vdl_tclQualificationBase::clearQualiListByCallBackIf(const vdl_tclQualificationCallbackInterface* const pCallBack)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclQualificationBase::clearQualiListByCallBackIf"));
   LockQualiSemaphore();
   {
      // for each entry in qualifying list
      tQualificationListIt iPos = m_oQualificationList.begin();
      while(iPos.operator!=(m_oQualificationList.end()))
      {
         if(iPos->m_pCallBackIf == pCallBack)
         {      
            vdl_tclReportRecord* pReportRecord = iPos->m_pReportRecord;
            if(pReportRecord != NULL)
            {
               tU16 u16ITC       = pReportRecord->u16ITC();
               tU32 u32DTC       = pReportRecord->u32DTC();
               tU8  u8MemoryId   = pReportRecord->u8GetMemoryId();
               ETG_TRACE_COMP_CLS(( TR_CLASS_DIAGLOG_SM, ": [%d] Stop Qualification of ITC:%04x DTC:%06x because of clearQualiListByCallBackIf",u8MemoryId,u16ITC,u32DTC));
               // we have to stop qualifining if BlockingMode is active
               iPos = m_oQualificationList.erase(iPos);  // delete from list
               continue; // continue with next item in List
            }// if(pReportRecord != NULL)
         }// if(iPos->m_pCallBackIf == pCallBack)
         ++iPos;
      }// while(iPos.operator!=(m_oQualificationList.end()))
   }
   FreeQualiSemaphore();
   ETG_TRACE_USR3_THR(( "<-- vdl_tclQualificationBase::clearQualiListByCallBackIf"));
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclQualificationBase::clearQualiListByRecord
//
// DESCRIPTION:  the Memory needs to remove all elements from qualification list
//             
// PARAMETER:    pReportRecord: pointer to Record
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
void vdl_tclQualificationBase::clearQualiListByRecord(const vdl_tclReportRecord* const pReportRecord)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclQualificationBase::clearQualiListByRecord"));
   LockQualiSemaphore();
   {
      // for each entry in qualifying list
      tQualificationListIt iPos = m_oQualificationList.begin();
      while(iPos.operator!=(m_oQualificationList.end()))
      {
         if(iPos->m_pReportRecord == pReportRecord)
         {      
            if(pReportRecord != NULL)
            {
               tU16 u16ITC       = pReportRecord->u16ITC();
               tU32 u32DTC       = pReportRecord->u32DTC();
               tU8  u8MemoryId   = pReportRecord->u8GetMemoryId();
               ETG_TRACE_COMP_CLS(( TR_CLASS_DIAGLOG_SM, ": [%d] Stop Qualification of ITC:%04x DTC:%06x because of clearQualiListByRecord",u8MemoryId,u16ITC,u32DTC));
               // we have to stop qualifining if BlockingMode is active
               iPos = m_oQualificationList.erase(iPos);  // delete from list
               continue; // continue with next item in List
            }// if(pReportRecord != NULL)
         }// if(iPos->m_pCallBackIf == pCallBack)
         ++iPos;
      }// while(iPos.operator!=(m_oQualificationList.end()))
   }
   FreeQualiSemaphore();
   ETG_TRACE_USR3_THR(( "<-- vdl_tclQualificationBase::clearQualiListByRecord"));
}


///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclQualificationBase::isQualifiying
//
// DESCRIPTION:  checks if this DTC is still in qualifikation list
//             
// PARAMETER:    u16ITC => Id of ITC to search
//
// RETURNVALUE:  teQualificationState Failed,Passer, or not Running
//
///////////////////////////////////////////////////////////////////////////////////
teQualificationState vdl_tclQualificationBase::isQualifiying(const vdl_tclReportRecord* pReportRecord) 
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclQualificationBase::isQualifiying"));
   teQualificationState eResult = NONE_QUALIFICATION_RUNNING;

   LockQualiSemaphore();
   {
      tQualificationListIt iPos = m_oQualificationList.begin();
      while(iPos.operator!=(m_oQualificationList.end()))
      {
         if( pReportRecord == iPos->m_pReportRecord)
         {
            tU16 u16ITC       = pReportRecord->u16ITC();
            tU32 u32DTC       = pReportRecord->u32DTC();
            tU8  u8MemoryId   = pReportRecord->u8GetMemoryId();

            ETG_TRACE_USR2_THR(( "--- vdl_tclQualificationBase::isQualifiying => Memory:%02d ITC:%04x DTC:%06x found in qualification List",u8MemoryId,u16ITC,u32DTC));
            if(iPos->m_s32FailQualiyfingCounter)
            {
               eResult = FAIL_QUALIFICATION_RUNNING;
            }
            else
            {
               eResult = PASS_QUALIFICATION_RUNNING;
            }
            break;
         }
         ++iPos;
      }// while(iPos.operator!=(m_oQualificationList.end()))
   }
   FreeQualiSemaphore();
   ETG_TRACE_USR3_THR(( "<-- vdl_tclQualificationBase::isQualifiying"));
   return eResult;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclQualificationBase::stopQualifiying
//
// DESCRIPTION:  stop qualifing this ITC
//             
// PARAMETER:    vdl_tclReportRecord* pReportRecord => pointer to record
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
void vdl_tclQualificationBase::stopQualifiying(const vdl_tclReportRecord* pReportRecord)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclQualificationBase::stopQualifiying"));

   LockQualiSemaphore();
   {
      tQualificationListIt iPos = m_oQualificationList.begin();
      while(iPos.operator!=(m_oQualificationList.end()))
      {
         if(pReportRecord == iPos->m_pReportRecord)
         {
            tU16 u16ITC       = pReportRecord->u16ITC();
            tU32 u32DTC       = pReportRecord->u32DTC();
            tU8  u8MemoryId   = pReportRecord->u8GetMemoryId();
            ETG_TRACE_USR2_THR(( "--- vdl_tclQualificationBase::stopQualifiying => Memory:%02d ITC:%04x DTC:%06x found in qualification List now erasing",u8MemoryId,u16ITC,u32DTC));
            (tVoid) m_oQualificationList.erase(iPos);  // delete from list
            break;
         }
         ++iPos;
      }// while(iPos.operator!=(m_oQualificationList.end()))
   }
   FreeQualiSemaphore();
   checkToStopQualiTimer();
   
   ETG_TRACE_USR3_THR(( "<-- vdl_tclQualificationBase::stopQualifiying"));
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclQualificationBase::checkToStartQualiTimer
//
// DESCRIPTION:  checks if the Qualitimer has to start
//             
// PARAMETER:    none
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
void vdl_tclQualificationBase::checkToStartQualiTimer() const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclQualificationBase::checkToStartQualiTimer"));

   // check if still running

   size_t sSize = 0; 
   LockQualiSemaphore();
   {
      sSize = m_oQualificationList.size();
   }
   FreeQualiSemaphore();

   ETG_TRACE_USR2_THR(( "--- vdl_tclQualificationBase::checkToStartQualiTimer => Size=%d",sSize));

   if(sSize == 1)
   {
      // this is the first one in the List, we have to start the Timer now
      tS32 s32Result;
      ETG_TRACE_USR2_THR(( "---  vdl_tclQualificationBase::checkToStartQualiTimer => start m_hQualificationTimer")); 
      s32Result = OSAL_s32TimerSetTime (vdDiagLog_tclApp::m_hQualificationTimer, 
                                        DIAGLOG_QUALIFICATION_STEP_TIME, 
                                        DIAGLOG_QUALIFICATION_STEP_TIME);

      if(OSAL_ERROR == s32Result)
      {
         tU32 u32_OsalError = OSAL_u32ErrorCode();
         ETG_TRACE_ERRMEM(("!!! vdl_tclQualificationBase::checkToStartQualiTimer => ERROR: OSAL_s32TimerSetTime m_hQualificationTimer FAILED Time:%x ErrorCode =%x", DIAGLOG_QUALIFICATION_STEP_TIME, static_cast<tUInt>(u32_OsalError)));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }

   ETG_TRACE_USR3_THR(( "<-- vdl_tclQualificationBase::checkToStartQualiTimer"));
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclQualificationBase::checkToStopQualiTimer
//
// DESCRIPTION:  checks if the Qualitimer has to stop
//             
// PARAMETER:    none
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
void vdl_tclQualificationBase::checkToStopQualiTimer() const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclQualificationBase::checkToStopQualiTimer"));

   // check if still running
   if(bHasTimeQualifiers()== 0)
   {
      // this is the last one so stop now
      tS32 s32Result;
      ETG_TRACE_USR2_THR(( "---  vdl_tclQualificationBase::checkToStopQualiTimer => stop m_hQualificationTimer")); 
      s32Result = OSAL_s32TimerSetTime (vdDiagLog_tclApp::m_hQualificationTimer, 
                                        0, 
                                        0);

      if(OSAL_ERROR == s32Result)
      {
         tU32 u32_OsalError = OSAL_u32ErrorCode();
         ETG_TRACE_ERR_THR(("!!! vdl_tclQualificationBase::checkToStopQualiTimer => ERROR: OSAL_s32TimerSetTime m_hQualificationTimer FAILED Handle:%x Time:0 ErrorCode =%x",vdDiagLog_tclApp::m_hQualificationTimer, u32_OsalError));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }

   ETG_TRACE_USR3_THR(( "<-- vdl_tclQualificationBase::checkToStopQualiTimer"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:   tVoid vdDiagLog_BlockingMode::isBlockingModeActive
//
// DESCRIPTION: forward to main
//
// PARAMETER:  pReportRecord     : pointer to caller record
//             blockingModeMask  : referenz for Block Mask to trace it out by caller
//
// RETURNVALUE: true = Blocked
// 
///////////////////////////////////////////////////////////////////////////////////
bool vdl_tclQualificationBase::isBlockingModeActive(vdl_tclReportRecord const* pReportRecord,tU32& blockingModeMask) const
{
   if(NULL != vdDiagLog_tclApp::m_poInstance)
   {
      return (vdDiagLog_tclApp::m_poInstance->isBlockingModeActive(pReportRecord,blockingModeMask));
   }
   return false;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclQualificationBase::checkToStopQualiTimer
//
// DESCRIPTION:  checks if the Qualitimer has to stop
//             
// PARAMETER:    none
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
inline void vdl_tclQualificationBase::LockQualiSemaphore() const
{
   tS32 s32Ret = OSAL_s32SemaphoreWait(vdl_tclQualificationBase::m_hQualificationSem, OSAL_C_U32_INFINITE);
   if(OSAL_OK != s32Ret)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      et_vErrmemStringNormal( TR_COMP_DIAGNOSIS, "Diaglog: LockQualiSemaphore() => OSAL_s32SemaphoreWait failed: Result=%x, Error=%x",
                               s32Ret,
                               u32ErrorCode);
      NORMAL_M_ASSERT_ALWAYS();
   }
}

inline void vdl_tclQualificationBase::FreeQualiSemaphore() const
{
   tS32 s32Ret = OSAL_s32SemaphorePost(vdl_tclQualificationBase::m_hQualificationSem);
   if(OSAL_OK != s32Ret)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      et_vErrmemStringNormal( TR_COMP_DIAGNOSIS, "Diaglog: FreeQualiSemaphore() => OSAL_s32SemaphorePost failed: Result=%x, Error=%x",
                               s32Ret,
                               u32ErrorCode);
      NORMAL_M_ASSERT_ALWAYS();
   }
}

