//////////////////////////////////////////////////////////////////////////
// FILE:         vd_diaglog_protocol_converter.cpp
// PROJECT:      common
// SW-COMPONENT: DiagLog
//----------------------------------------------------------------------
//
// DESCRIPTION: format converter
//              
//----------------------------------------------------------------------
// COPYRIGHT:    (c) 2015 Robert Bosch GmbH, Hildesheim
// HISTORY:      
// Date      | Author                       | Modification
// 13.04.15  | BSOT Plischke                 | Initial version
//
//////////////////////////////////////////////////////////////////////////

// first include diaglog settings
#include <common/framework/vd_diaglog_settings.h>

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_protocol_converter
   #include <common/framework/vd_diaglog_protocol_converter.h>
#endif

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DIAGLOG_INFO
#include "trcGenProj/Header/vd_diaglog_protocol_converter.cpp.trc.h"
#endif

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: void vdl_tclProtocolConverter::convertStatusIntToExt()
//
// DESCRIPTION: convert from DIAGLOG format to requested Format
//
// PARAMETER:  Status in Diaglog format (tU32)
//
// RETURNVALUE: Status in requested format
//
///////////////////////////////////////////////////////////////////////////////////
//
tU8 vdl_tclProtocolConverter::convertStatusIntToExt(vdl_teDiaglogReportProtocol format, tU32 u32Status) const
{
   tuRecordStatus uRecordStatus;
   uRecordStatus.u32Status = u32Status;
   return convertStatusIntToExt(format,uRecordStatus);
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: void vdl_tclProtocolConverter::convertStatusIntToExt()
//
// DESCRIPTION: convert from DIAGLOG format to requested Format
//
// PARAMETER:  Status in Diaglog format (tuRecordStatus)
//
// RETURNVALUE: Status in requested format
//
///////////////////////////////////////////////////////////////////////////////////
//
tU8 vdl_tclProtocolConverter::convertStatusIntToExt(vdl_teDiaglogReportProtocol format, const tuRecordStatus& recordStatus) const
{
   tU8 status = 0;
   switch(format)
   {
      case DIAGLOG_REPORT_GMLAN:
      {
         status = convertINTERNtoGMLAN(recordStatus) ;
         break;
      }
      case DIAGLOG_REPORT_UDS:
      {
         status = convertINTERNtoUDS(recordStatus) ;
         break;
      }
      default:
      {
         NORMAL_M_ASSERT_ALWAYS();
         break;
      }
   }// switch(format)

   return status;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: void vdl_tclProtocolConverter::convertStatusExtToInt()
//
// DESCRIPTION: convert from GMLAN format to DIAGLOG
//
// PARAMETER:  Status in given format
//
// RETURNVALUE: Status in DIAGLOG format
//
///////////////////////////////////////////////////////////////////////////////////
//
tuRecordStatus vdl_tclProtocolConverter::convertStatusExtToInt(vdl_teDiaglogReportProtocol format, tU8 recordStatus) const
{
   tuRecordStatus uStatus;
   switch(format)
   {
      case DIAGLOG_REPORT_GMLAN:
      {
         uStatus = convertGMLANtoINTERN(recordStatus) ;
         break;
      }
      case DIAGLOG_REPORT_UDS:
      {
         uStatus = convertUDStoINTERN(recordStatus) ;
         break;
      }
      default:
      {
         uStatus.u32Status = 0;
         NORMAL_M_ASSERT_ALWAYS();
         break;
      }
   }// switch(format)

   return uStatus;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: void vdl_tclProtocolConverter::convertINTERNtoGMLAN()
//
// DESCRIPTION: convert from DIAGLOG format to GMLAN
//
// PARAMETER:  Status in Diaglog format
//
// RETURNVALUE: Status in GMLAN format
//
///////////////////////////////////////////////////////////////////////////////////
//
tU8 vdl_tclProtocolConverter::convertINTERNtoGMLAN(const tuRecordStatus& uRecordStatus) const
{
   tuGMLanStatus uStatus;
   uStatus.u8Status = 0;


  if(uRecordStatus.sStatus.m_bIsCalibrationDTCEnabled)           { ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertINTERNtoGMLAN => m_bIsCalibrationDTCEnabled        -> uGMLanStatus.m_bDTCSupportedByCalibration = 1"));        uStatus.sStatus.m_bDTCSupportedByCalibration = 1 ;}
  if(uRecordStatus.sStatus.m_bIsActive)                          { ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertINTERNtoGMLAN => m_bIsActive                       -> uGMLanStatus.m_bIsActive = 1"));                         uStatus.sStatus.m_bCurrentDTC = 1 ;}
  if(uRecordStatus.sStatus.m_bIsNotPassedSinceDTCCleared)        { ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertINTERNtoGMLAN => m_bIsNotPassedSinceDTCCleared     -> uGMLanStatus.m_bIsNotPassedSinceDTCCleared = 1"));       uStatus.sStatus.m_bTestNotPassedSinceDTCCleared = 1 ;}
  if(uRecordStatus.sStatus.m_bIsHistory)                         { ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertINTERNtoGMLAN => m_bIsHistory                      -> uGMLanStatus.m_bIsHistory = 1"));                        uStatus.sStatus.m_bTestFailedSinceDTCCleared = 1 ;}
  if(uRecordStatus.sStatus.m_bIsHistory)                         { ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertINTERNtoGMLAN => m_bIsHistory                      -> uGMLanStatus.m_bIsHistory = 1"));                        uStatus.sStatus.m_bHistoryDTC = 1 ;}
  if(uRecordStatus.sStatus.m_bIsNotPassedSinceCurrentPowerUp)    { ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertINTERNtoGMLAN => m_bIsNotPassedSinceCurrentPowerUp -> uGMLanStatus.m_bIsNotPassedSinceCurrentPowerUp = 1"));   uStatus.sStatus.m_bTestNotPassedSinceCurrentPowerUp = 1 ;}
  if(uRecordStatus.sStatus.m_bIsFailedInCurrentPowerCycle)       { ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertINTERNtoGMLAN => m_bIsFailedInCurrentPowerCycle    -> uGMLanStatus.m_bIsCurrentInCurrentPowerCycle = 1"));     uStatus.sStatus.m_bCurrentDTCSincePowerUp = 1 ;}
  //if(uRecordStatus.sStatus.m_bIsHistory)                         { ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertINTERNtoGMLAN => m_bIsHistory                      -> uGMLanStatus.m_bWarningIndicatorRequested = 1"));        uStatus.sStatus.m_bWarningIndicatorRequested = 1 ;}

  ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertINTERNtoGMLAN => uRecordStatus:%08x -> uGMLanStatus:%02x",uRecordStatus.u32Status,uStatus.u8Status));


  return uStatus.u8Status;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: void vdl_tclProtocolConverter::convertINTERNtoGMLAN()
//
// DESCRIPTION: convert from GMLAN format to DIAGLOG
//
// PARAMETER:  Status in GMLAN format
//
// RETURNVALUE: Status in DIAGLOG format
//
///////////////////////////////////////////////////////////////////////////////////
//
tuRecordStatus vdl_tclProtocolConverter::convertGMLANtoINTERN(tU8 aStatus) const
{
   tuGMLanStatus uGMLanStatus;
   uGMLanStatus.u8Status = aStatus;
   tuRecordStatus uStatus;
   uStatus.u32Status = 0;

   if(uGMLanStatus.sStatus.m_bDTCSupportedByCalibration)         { ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertGMLANtoINTERN => m_bDTCSupportedByCalibration         -> uRecordStatus.m_bIsCalibrationDTCEnabled = 1"));         uStatus.sStatus.m_bIsCalibrationDTCEnabled = 1 ;}
   if(uGMLanStatus.sStatus.m_bCurrentDTC)                        { ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertGMLANtoINTERN => m_bCurrentDTC                        -> uRecordStatus.m_bIsActive = 1"));                        uStatus.sStatus.m_bIsActive = 1 ;}
   if(uGMLanStatus.sStatus.m_bTestNotPassedSinceDTCCleared)      { ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertGMLANtoINTERN => m_bTestNotPassedSinceDTCCleared      -> uRecordStatus.m_bIsNotPassedSinceDTCCleared = 1"));      uStatus.sStatus.m_bIsNotPassedSinceDTCCleared = 1 ;}
   if(uGMLanStatus.sStatus.m_bTestFailedSinceDTCCleared)         { ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertGMLANtoINTERN => m_bTestFailedSinceDTCCleared         -> uRecordStatus.m_bIsHistory = 1"));                       uStatus.sStatus.m_bIsHistory = 1 ;}
   if(uGMLanStatus.sStatus.m_bHistoryDTC)                        { ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertGMLANtoINTERN => m_bHistoryDTC                        -> uRecordStatus.m_bIsHistory = 1"));                       uStatus.sStatus.m_bIsHistory = 1 ;}
   if(uGMLanStatus.sStatus.m_bTestNotPassedSinceCurrentPowerUp)  { ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertGMLANtoINTERN => m_bTestNotPassedSinceCurrentPowerUp  -> uRecordStatus.m_bIsNotPassedSinceCurrentPowerUp = 1"));  uStatus.sStatus.m_bIsNotPassedSinceCurrentPowerUp = 1 ;}
   if(uGMLanStatus.sStatus.m_bCurrentDTCSincePowerUp)            { ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertGMLANtoINTERN => m_bCurrentDTCSincePowerUp            -> uRecordStatus.m_bIsFailedInCurrentPowerCycle = 1"));     uStatus.sStatus.m_bIsFailedInCurrentPowerCycle = 1 ;}
//   if(uGMLanStatus.sStatus.m_bWarningIndicatorRequested)         { ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertGMLANtoINTERN => m_bHistoryDTC                        -> uRecordStatus.m_bIsHistory = 1"));                       uStatus.sStatus.m_bIsHistory = 1 ;}


   ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertGMLANtoINTERN => uGMLanStatus:%02x -> uRecordStatus:%08x",uGMLanStatus.u8Status,uStatus.u32Status));

   return uStatus;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: void vdl_tclProtocolConverter::convertINTERNtoUDS()
//
// DESCRIPTION: convert from DIAGLOG format to UDS
//
// PARAMETER:  Status in Diaglog format
//
// RETURNVALUE: Status in UDS format
//
///////////////////////////////////////////////////////////////////////////////////
//

tU8 vdl_tclProtocolConverter::convertINTERNtoUDS(const tuRecordStatus& uRecordStatus) const
{
   tuUDSStatus uStatus;
   uStatus.u8Status = 0;


  if(uRecordStatus.sStatus.m_bIsActive)                          { ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertINTERNtoUDS => m_bIsActive                       -> uUDSStatus.m_bTestFailed = 1"));                       uStatus.sStatus.m_bTestFailed = 1 ;}
  if(uRecordStatus.sStatus.m_bIsFailedInCurrentPowerCycle)       { ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertINTERNtoUDS => m_bIsFailedInCurrentPowerCycle    -> uUDSStatus.m_bTestFailedThisOperationCycle = 1"));     uStatus.sStatus.m_bTestFailedThisOperationCycle = 1 ;}
  if(uRecordStatus.sStatus.m_bIsHistory)                         { ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertINTERNtoUDS => m_bIsHistory                      -> uUDSStatus.m_bPendingDTC = 1"));                       uStatus.sStatus.m_bPendingDTC = 1 ;}
  if(uRecordStatus.sStatus.m_bIsNotCompleteSinceLastClear)       { ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertINTERNtoUDS => m_bTestNotCompleteSinceLastClear  -> uUDSStatus.m_bTestNotCompleteSinceLastClear = 1"));    uStatus.sStatus.m_bTestNotCompleteSinceLastClear = 1 ;}
  if(uRecordStatus.sStatus.m_bIsHistory)                         { ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertINTERNtoUDS => m_bIsHistory                      -> uUDSStatus.m_bConfirmedDTC = 1"));                     uStatus.sStatus.m_bConfirmedDTC = 1 ;}
  if(uRecordStatus.sStatus.m_bIsFailedSinceDTCCleared)           { ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertINTERNtoUDS => m_bIsFailedSinceDTCCleared        -> uUDSStatus.m_bTestFailedSinceLastClear = 1"));         uStatus.sStatus.m_bTestFailedSinceLastClear = 1 ;}
  if(uRecordStatus.sStatus.m_bIsNotCompleteThisOperationCycle)   { ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertINTERNtoUDS => m_bIsNotCompleteThisOperationCycle-> uUDSStatus.m_bNotCompletedThisCycle = 1"));            uStatus.sStatus.m_bNotCompletedThisCycle = 1 ;}
  // reserve warning indicator

  ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertINTERNtoUDS => uRecordStatus:%08x -> uUDSStatus:%02x",uRecordStatus.u32Status,uStatus.u8Status));

  return uStatus.u8Status;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: void vdl_tclProtocolConverter::convertUDStoINTERN()
//
// DESCRIPTION: convert from UDS format to DIAGLOG
//
// PARAMETER:  Status in UDS format
//
// RETURNVALUE: Status in DIAGLOG format
//
///////////////////////////////////////////////////////////////////////////////////
//
tuRecordStatus vdl_tclProtocolConverter::convertUDStoINTERN(tU8 aStatus) const
{
   tuUDSStatus uUDSStatus;
   uUDSStatus.u8Status = aStatus;
   tuRecordStatus uStatus;
   uStatus.u32Status = 0;

   if(uUDSStatus.sStatus.m_bTestFailed)                        { ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertUDStoINTERN => m_bTestFailed                        -> uRecordStatus.m_bIsActive = 1"));                        uStatus.sStatus.m_bIsActive = 1 ;}
   if(uUDSStatus.sStatus.m_bTestFailedThisOperationCycle)      { ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertUDStoINTERN => m_bTestFailedThisOperationCycle      -> uRecordStatus.m_bIsFailedInCurrentPowerCycle = 1"));     uStatus.sStatus.m_bIsFailedInCurrentPowerCycle = 1 ;}
   // reserve pending
   if(uUDSStatus.sStatus.m_bTestNotCompleteSinceLastClear)     { ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertUDStoINTERN => m_bTestNotCompleteSinceLastClear     -> uRecordStatus.m_bIsNotCompleteSinceLastClear = 1"));     uStatus.sStatus.m_bIsNotCompleteSinceLastClear = 1 ;}
   if(uUDSStatus.sStatus.m_bConfirmedDTC)                      { ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertUDStoINTERN => m_bConfirmedDTC                      -> uRecordStatus.m_bIsHistory = 1"));                       uStatus.sStatus.m_bIsHistory = 1 ;}
   if(uUDSStatus.sStatus.m_bTestFailedSinceLastClear)          { ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertUDStoINTERN => m_bTestFailedSinceLastClear          -> uRecordStatus.m_bIsFailedSinceDTCCleared = 1"));         uStatus.sStatus.m_bIsFailedSinceDTCCleared = 1 ;}
   if(uUDSStatus.sStatus.m_bNotCompletedThisCycle)             { ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertUDStoINTERN => m_bNotCompletedThisCycle             -> uRecordStatus.m_bIsNotCompleteThisOperationCycle = 1")); uStatus.sStatus.m_bIsNotCompleteThisOperationCycle = 1 ;}
   // reserve warning indicator

   ETG_TRACE_USR2_THR(( "--- vdl_tclProtocolConverter::convertUDStoINTERN => uUDSStatus:%02x -> uRecordStatus:%08x",uUDSStatus.u8Status,uStatus.u32Status));

   return uStatus;
}

