/*!
* \file       vd_diaglog_object_manager.cpp
*
* \brief      Used to persistently save the Dialog object (log) files, makes it transaction safe.
*
* \details    Used to persistently save the Dialog object (log) files, makes it transaction safe.
*
* \component  Diaglog
*
* \ingroup    DiaglogFramework
*
* \copyright  (c) 2018 Robert Bosch GmbH
*
* The reproduction, distribution and utilization of this file as
* well as the communication of its contents to others without express
* authorization is prohibited. Offenders will be held liable for the
* payment of damages. All rights reserved in the event of the grant
* of a patent, utility model or design.
*/

// first include diaglog settings
#include <common/framework/vd_diaglog_settings.h>

#ifndef __VD_DIAGLOG_INCLUDEGUARD_OBJECT_MANAGER__
#include "common/framework/vd_diaglog_object_manager.h"
#endif

#ifndef __VD_DIAGLOG_INCLUDEGUARD_CRC_CALCULATOR__
#include "common/framework/utils/vd_diaglog_CRCCalculator.h"
#endif

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DIAGLOG_INFO
#include "trcGenProj/Header/vd_diaglog_object_manager.cpp.trc.h"
#endif

#include <common/framework/utils/vd_diaglog_filedir.h>

#include <errno.h>
#include <stdio.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <fstream>
#include <iterator>

//-----------------------------------------------------------------------------

vdDiaglog_ObjManager::vdDiaglog_ObjManager(tCString logObjectName)
{
    ETG_TRACE_USR3_THR(("--> vdDiaglog_ObjManager::vdDiaglog_ObjManager(tCString )"));

    std::string fullPath(logObjectName);
    size_t pos = fullPath.find_last_of("/");
    
    path = (fullPath.substr(0, pos+1));
    filename = (fullPath.substr(pos + 1));

    logDirOld = path + filename + ".1";
    logDirNew = path + filename + ".2";
    linkDirOld = path + filename + ".oldLink";
    linkDirNew = path + filename + ".newlink";

    ETG_TRACE_USR3_THR(("<-- vdDiaglog_ObjManager::vdDiaglog_ObjManager(tCString )"));
}

//-----------------------------------------------------------------------------

vdDiaglog_ObjManager::~vdDiaglog_ObjManager(void)
{
    ETG_TRACE_USR3_THR(("--> vdDiaglog_ObjManager::~vdDiaglog_ObjManager()"));
    ETG_TRACE_USR3_THR(("<-- vdDiaglog_ObjManager::~vdDiaglog_ObjManager()"));
}

//-----------------------------------------------------------------------------

bool
vdDiaglog_ObjManager::updateData(const tStream& iData)
{
    ETG_TRACE_USR3_THR(("--> vdDiaglog_ObjManager::updateData(const tStream& )"));

    bool res = false;
    struct stat statbuff;

    int old_exists = !lstat(linkDirOld.c_str(), &statbuff);
    int new_exists = !lstat(linkDirNew.c_str(), &statbuff);

    std::string currDir, newDir, otherDir;

    if (!old_exists && !new_exists)
    {
        ETG_TRACE_USR3_THR(("--- vdDiaglog_ObjManager::updateData - the logs files do not exist."));
        newDir = logDirOld;
    }
    else if (old_exists && new_exists)
    {
        ETG_TRACE_USR3_THR(("--- vdDiaglog_ObjManager::updateData - both new and old logs files exist"));

        // check where the old link points to and then check the other directory.
        res = validateLink(linkDirOld, currDir, otherDir);
        if (res != false)
        {
            if(validateData(otherDir) != false)
            {
                //update the newlink 
                if(rename(linkDirNew.c_str(), linkDirOld.c_str()) !=0)
                {
                    ETG_TRACE_ERR_THR(("--- vdDiaglog_ObjManager::updateData - rename link - FAILED."));
                }
                newDir = currDir;
            }
            else
            {
                ETG_TRACE_ERR_THR(("--- vdDiaglog_ObjManager::updateData - data validation unsuccessful, remove the link."));
                if (remove(linkDirNew.c_str()) != 0)
                {
                    ETG_TRACE_ERR_THR(("--- vdDiaglog_ObjManager::updateData - remove link - FAILED."));
                }
                newDir = otherDir;
            }
        }
    }
    else if (!old_exists && new_exists)
    {
        ETG_TRACE_USR3_THR(("--- vdDiaglog_ObjManager::updateData - the old logs files do not exist, but there are new files"));

        res = validateLink(linkDirNew, currDir, otherDir);
        if (res != false)
        {
            if (validateData(currDir) != false)
            {
                //update the newlink 
                if (rename(linkDirNew.c_str(), linkDirOld.c_str()) != 0)
                {
                    ETG_TRACE_ERR_THR(("--- vdDiaglog_ObjManager::updateData - rename link - FAILED."));
                }
                newDir = otherDir;
            }
            else
            {
                ETG_TRACE_ERR_THR(("--- vdDiaglog_ObjManager::updateData - data validation unsuccessful, remove the link."));
                if(remove(linkDirNew.c_str()) != 0)
                {
                    ETG_TRACE_ERR_THR(("--- vdDiaglog_ObjManager::updateData - remove link - FAILED."));
                }
                newDir = logDirOld;
            }
        }
    }
    else //if (old_exists && !new_exists)
    {
        ETG_TRACE_USR3_THR(("--- vdDiaglog_ObjManager::updateData - old log files exists, write data"));

        res = validateLink(linkDirOld, currDir, otherDir);
        if (res == true)
        {
            newDir = otherDir;
        }
        else
        {
            ETG_TRACE_ERR_THR(("--- vdDiaglog_ObjManager::updateData - validate link failed, remove files and recreate"));
            newDir = logDirOld;
        }
    }

    res = writeData(iData, newDir);
    if (res == true)
    {
        // writing into log file was successful, update the checksum.
        std::string currChkFile = newDir + "/" + filename + ".chk";
        res = calcCRCUpdateChecksumFile(iData, currChkFile);
    }

    if (res == true)
    {
        //create new symlink
        if (symlink(newDir.c_str(), linkDirNew.c_str()) != 0)
        {
            ETG_TRACE_ERR_THR(("--- vdDiaglog_ObjManager::updateData - symlink creation - FAILED."));
        }

        //rename old symlink to point to new files.
        if(rename(linkDirNew.c_str(), linkDirOld.c_str()) != 0)
        {
            ETG_TRACE_ERR_THR(("--- vdDiaglog_ObjManager::updateData - rename link - FAILED."));
        }
    }
    else
    {
        ETG_TRACE_ERR_THR(("--- vdDiaglog_ObjManager::updateData - writing into file checksum file FAILED."));
    }

    ETG_TRACE_USR3_THR(("<-- vdDiaglog_ObjManager::updateData(const tStream& )"));
    
    return res;
}

//-----------------------------------------------------------------------------

bool
vdDiaglog_ObjManager::loadData(tStream& oData)
{
    ETG_TRACE_USR3_THR(("--> vdDiaglog_ObjManager::loadData(tStream& )"));
    bool res = false;
    bool bRecreate = false;
    std::string currDir, inputDir, otherDir;

    struct stat statbuff;

    int old_exists = !lstat(linkDirOld.c_str(), &statbuff);

    if (!old_exists) //&& !new_exists)
    {
        ETG_TRACE_USR3_THR(("--- vdDiaglog_ObjManager::loadData - the logs files do not exist"));
        bRecreate = true;
    }
    else //if (old_exists)
    {
        ETG_TRACE_USR3_THR(("--- vdDiaglog_ObjManager::loadData - old log files exists."));
        if (validateLink(linkDirOld, currDir, otherDir) == true)
        {
            inputDir = currDir;
        }
        else
        {
            ETG_TRACE_ERR_THR(("--- vdDiaglog_ObjManager::loadData - validation of link - FAILED."));
            bRecreate = true;
        }
    }

    
    if (bRecreate == false)
    {
        ETG_TRACE_USR3_THR(("--- vdDiaglog_ObjManager::loadData - validate data in the current directory"));

        //read the data from log file
        std::string logfile = inputDir + "/" + filename + ".dat";

        if (readData(oData, logfile) != false)
        {
            if (validateData(inputDir, oData) != false)
            {
                ETG_TRACE_USR3_THR(("--- vdDiaglog_ObjManager::loadData - validate successful"));
                res = true;
            }
            else
            {
                ETG_TRACE_ERR_THR(("--- vdDiaglog_ObjManager::loadData - data validation FAILED!"));
                res = false;
            }
        }
        else
        {
            ETG_TRACE_ERR_THR(("--- vdDiaglog_ObjManager::loadData - data could not be read from %s.", logfile.c_str()));
        }
    }
    else
    {
        res = false;
    }

    ETG_TRACE_USR3_THR(("<-- vdDiaglog_ObjManager::loadData(tStream& )"));
    
    return res;
}

//-----------------------------------------------------------------------------

bool
vdDiaglog_ObjManager::bReCreateStorageArea(std::string newDir)
{
    ETG_TRACE_USR3_THR(("--> vdDiaglog_ObjManager::bReCreateStorageArea(tCString )"));

     //create log directory with permissions
    mode_t mode = (S_IRWXU | S_IRWXG);
    vdDiaglog_FileDir diaglogDir(newDir.c_str(), mode);


    if (!(diaglogDir.createDirIfNotExist()))
    {
        ETG_TRACE_ERR_THR(("--- vdDiaglog_ObjManager::bReCreateStorageArea - ERROR creating log directory!!"));
        ETG_TRACE_ERRMEM(("--- vdDiaglog_ObjManager::bReCreateStorageArea - ERROR creating log directory!!"));
        return false;
    }

    // create the logFile
    std::string logfile = newDir + "/" + filename + ".dat";

    int fd = open(logfile.c_str(), O_WRONLY | O_CREAT, S_IRUSR | S_IWUSR | S_IRGRP);
    if (fd == -1)
    {
        ETG_TRACE_ERR_THR(("--- vdDiaglog_ObjManager::bReCreateStorageArea - %s could not be created.", logfile.c_str()));
        return false;
    }
    (void)fsync(fd);
    (void) close(fd);

    //create the checksum file
    std::string chksumfile = newDir + "/" + filename + ".chk";

    fd = open(chksumfile.c_str(), O_WRONLY | O_CREAT, S_IRUSR | S_IWUSR | S_IRGRP);
    if (fd == -1)
    {
        ETG_TRACE_ERR_THR(("--- vdDiaglog_ObjManager::bReCreateStorageArea - %s could not be created.", chksumfile.c_str()));
        return false;
    }
    (void)fsync(fd);
    (void) close(fd);

    ETG_TRACE_USR3_THR(("<-- vdDiaglog_ObjManager::bReCreateStorageArea(tCString )"));

    return true;
}

//-----------------------------------------------------------------------------

bool
vdDiaglog_ObjManager::validateLink(std::string link, std::string& currDir, std::string& newDir)
{
    ETG_TRACE_USR3_THR(("--> vdDiaglog_ObjManager::validateLink(string, string& , string&)"));
    
    bool res = false;
    tU32 buffSize = 0;
    struct stat statbuff;

    int ret = lstat(link.c_str(), &statbuff);
    if (ret == -1)
    {
        ETG_TRACE_ERR_THR(("--- vdDiaglog_ObjManager::validateLink - lstat- FAILED."));
        return false;
    }
    buffSize = (tU32)(statbuff.st_size + 1);

    char pathbuff[buffSize];

    // Check if the new link links to one of the known data directories.
    ssize_t read_bytes = readlink(link.c_str(), pathbuff, sizeof(pathbuff));
    if (read_bytes == -1)
    {
        ETG_TRACE_ERR_THR(("--- vdDiaglog_ObjManager::r - readlink- FAILED."));
        return false;
    }
    pathbuff[buffSize-1] = '\0';

    if (strcmp(pathbuff, logDirOld.c_str()) == 0)
    {
        ETG_TRACE_USR3_THR(("--- vdDiaglog_ObjManager::validateLink - link points to %s", logDirOld.c_str()));
        currDir = logDirOld;
        newDir = logDirNew;
        res = true;
    }
    else if (strcmp(pathbuff, logDirNew.c_str()) == 0)
    {
        ETG_TRACE_USR3_THR(("--- vdDiaglog_ObjManager::validateLink - link points to %s", logDirNew.c_str()));
        currDir = logDirNew;
        newDir = logDirOld;
        res = true;
    }
    else
    {
        ETG_TRACE_ERR_THR(("--- vdDiaglog_ObjManager::validateLink - link is invalid: %s",link.c_str()));
    }

    ETG_TRACE_USR3_THR(("<-- vdDiaglog_ObjManager::validateLink(string, string& ,string& )"));
    
    return res;
}

//-----------------------------------------------------------------------------
bool
vdDiaglog_ObjManager::validateData(std::string otherDir)
{
    ETG_TRACE_USR3_THR(("--> vdDiaglog_ObjManager::validateData(tCString )"));

    std::string logfile = otherDir + "/" + filename + ".dat";

    //read the data from log file
    tStream Data;
    if (readData(Data, logfile) == false)
    {
        ETG_TRACE_ERR_THR(("--- vdDiaglog_ObjManager::validateData - data could not be read from %s.", logfile.c_str()));
        return false;
    }

    //Read the checksum
    std::string chksumfile = otherDir + "/" + filename + ".chk";

    tStream checksum;
    if (readData(checksum, chksumfile) == false)
    {
        ETG_TRACE_ERR_THR(("--- vdDiaglog_ObjManager::validateData - checksum could not be read from %s.", chksumfile.c_str()));
        return false;
    }

    //compare the calculated checksum and stored checksum.
    if (validateChecksum(checksum, Data) == false)
    {
        ETG_TRACE_ERR_THR(("--- vdDiaglog_ObjManager::validateData - checksums do not match!!!."));
        return false;
    }

    ETG_TRACE_USR3_THR(("<-- vdDiaglog_ObjManager::validateData(tCString )"));

    return true;
}

//-----------------------------------------------------------------------------
bool
vdDiaglog_ObjManager::validateData(std::string otherDir, tStream rData)
{
    ETG_TRACE_USR3_THR(("--> vdDiaglog_ObjManager::validateData(tCString )"));

    //Read the checksum
    std::string chksumfile = otherDir + "/" + filename + ".chk";

    tStream checksum;
    if (readData(checksum, chksumfile) == false)
    {
        ETG_TRACE_USR3_THR(("--- vdDiaglog_ObjManager::validateData - checksum could not be read from %s.", chksumfile.c_str()));
        return false;
    }

    //compare the calculated checksum and stored checksum.
    if (validateChecksum(checksum, rData) == false)
    {
        ETG_TRACE_USR3_THR(("--- vdDiaglog_ObjManager::validateData - checksum do not match!!!."));
        return false;
    }

    ETG_TRACE_USR3_THR(("<-- vdDiaglog_ObjManager::validateData(tCString )"));

    return true;
}

//-----------------------------------------------------------------------------
bool
vdDiaglog_ObjManager::validateChecksum(tStream& checksum, tStream& rData)
{
    ETG_TRACE_USR3_THR(("--> vdDiaglog_ObjManager::validateChecksum(tStream& ,tStream& )"));

    bool res = false;
    tU32 Datalen = (tU32)rData.size();
    tS8 buff[Datalen];

    tStreamCIterator it = rData.begin();

    for (tU32 i = 0; (it != rData.end()) && (i < Datalen); it++, i++)
    {
        buff[i] = *it;
    }

    tU16 crc = 0;

    tStreamCIterator itCS = checksum.begin();
    for (; itCS != checksum.end(); itCS++)
    {
        crc = (tU16) (crc * 10 + (*itCS - '0'));
    }

    ETG_TRACE_USR3_THR(("--- vdDiaglog_ObjManager::validateData - stored crc = %d.", crc));

    vdDiaglog_CRCCalculator* pCRCCalc = getInstanceOfCRCCalculator();
    if (pCRCCalc)
    {
        if (pCRCCalc->calcAndCompareCrcCCITT(&buff[0], Datalen, crc) == true)
        {
            res = true;
        }
        else
        {
            ETG_TRACE_ERR_THR(("--- vdDiaglog_ObjManager::validateChecksum - checsums DO NOT match."));
        }
    }
    else
    {
        ETG_TRACE_ERR_THR(("--- vdDiaglog_ObjManager::validateChecksum - pCRCCalc is NULL!"));
    }

    ETG_TRACE_USR3_THR(("<-- vdDiaglog_ObjManager::validateChecksum(tStream& ,tStream& )"));

    return res;
}

//-----------------------------------------------------------------------------

bool
vdDiaglog_ObjManager::readData(tStream& oData, std::string iFile)
{
    ETG_TRACE_USR3_THR(("--> vdDiaglog_ObjManager::readData(tStream& )"));
    bool res = false;

    std::ifstream logFile(iFile.c_str());
    logFile >> std::noskipws;

    if (logFile.is_open())
    {
        ETG_TRACE_USR3_THR(("--- vdDiaglog_ObjManager::readData - %s opened.", iFile.c_str()));
        copy(std::istream_iterator<tS8>(logFile), std::istream_iterator<tS8>(), std::back_inserter(oData));
        logFile.close();
        res = true;
    }
    else
    {
        ETG_TRACE_ERR_THR(("--- vdDiaglog_ObjManager::readData - %s CANNOT OPEN.", iFile.c_str()));
    }

    ETG_TRACE_USR3_THR(("<-- vdDiaglog_ObjManager::readData(tStream& )"));

    return res;
}

bool
vdDiaglog_ObjManager::writeData(const tStream& iData, std::string newDir)
{
    ETG_TRACE_USR3_THR(("--> vdDiaglog_ObjManager::writeData(tStream& , std::string )"));

    bool res = false;

    res = bReCreateStorageArea(newDir);

    if (res == true)
    {
        std::string iFile = newDir + "/" + filename + ".dat";
        std::ofstream logFile(iFile.c_str());

        if (logFile.is_open())
        {
            ETG_TRACE_USR3_THR(("--- vdDiaglog_ObjManager::writeData - %s opened.", iFile.c_str()));
            std::ostream_iterator<tS8> FileIter(logFile, "");
            copy(iData.begin(), iData.end(), FileIter);
            logFile.flush();
            sync();
            logFile.close();
            res = true;
        }
        else
        {
            ETG_TRACE_ERR_THR(("--- vdDiaglog_ObjManager::writeData - %s CANNOT OPEN.", iFile.c_str()));
        }
    }
    
    ETG_TRACE_USR3_THR(("<-- vdDiaglog_ObjManager::writeData(tStream& , std::string )"));
    
    return res;
}

bool 
vdDiaglog_ObjManager::calcCRCUpdateChecksumFile(const tStream& iData, std::string chkFile)
{
    ETG_TRACE_USR3_THR(("--> vdDiaglog_ObjManager::calcCRCUpdateChecksumFile(tStream& , std::string )"));

    bool res = false;
    tU32 len = (tU32)iData.size();
    tS8 buff[len];
    tStreamCIterator it = iData.begin();

    tU16 crc = 0;

    for (tU32 i=0; (it != iData.end()) && (i < len); it++, i++)
    {
        buff[i] = *it;
    }

    ETG_TRACE_USR3_THR(("--- vdDiaglog_ObjManager::calcCRCUpdateChecksumFile - calculate Checksum"));
    vdDiaglog_CRCCalculator* pCRCCalc = getInstanceOfCRCCalculator();
    if (pCRCCalc)
    {
        crc = pCRCCalc->calcCrCCITT(&buff[0], len);

        ETG_TRACE_USR3_THR(("--- vdDiaglog_ObjManager::calcCRCUpdateChecksumFile - crc = %d",crc));
        ETG_TRACE_USR3_THR(("--- vdDiaglog_ObjManager::calcCRCUpdateChecksumFile - update checksum file %s", chkFile.c_str()));

        std::ofstream checksumFile(chkFile.c_str());
        if (checksumFile.is_open())
        {
            checksumFile << crc;
            checksumFile.flush();
            sync();
            checksumFile.close();
            res = true;
        }
        else
        {
            ETG_TRACE_ERR_THR(("--- vdDiaglog_ObjManager::calcCRCUpdateChecksumFile - %s CANNOT OPEN.", chkFile.c_str()));
        }
    }
    else
    {
        ETG_TRACE_ERR_THR(("--- vdDiaglog_ObjManager::calcCRCUpdateChecksumFile - pCRCCalc is NULL!"));
    }

    ETG_TRACE_USR3_THR(("<-- vdDiaglog_ObjManager::calcCRCUpdateChecksumFile(tStream& , std::string )"));

    return res;
}
