//////////////////////////////////////////////////////////////////////////
// FILE:         vd_diaglog_memory_master.cpp
// PROJECT:      common
// SW-COMPONENT: DiagLog
//----------------------------------------------------------------------
//
// DESCRIPTION: this class takes control over all report_memory objects
//
//----------------------------------------------------------------------
// COPYRIGHT:    (c) 2013 Robert Bosch GmbH, Hildesheim
// HISTORY:
// Date      | Author                       | Modification
// 17.09.13  | BSOT Plischke                | Initial version
//
//////////////////////////////////////////////////////////////////////////
// first include diaglog settings
#include <common/framework/vd_diaglog_settings.h>

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_memory_master
   #include "vd_diaglog_memory_master.h"
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_main
   #include <common/framework/vd_diaglog_main.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_report_memory
   #include <common/framework/vd_diaglog_report_memory.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_report_record_factory
   #include <project/framework/vd_diaglog_report_record_factory.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_inc_comm
   #include <common/interfaces/inc/vd_diaglog_inc_comm.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_service
   #include <project/interfaces/vd_diaglog_service.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_system_variables
   #include <common/framework/vd_diaglog_system_variables.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_plugin_if
   #include <common/plugin/vd_diaglog_plugin_if.h>
#endif



#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_snapshotdata
   #include <common/framework/snapshotData/vd_diaglog_snapshotdata.h>
#endif

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DIAGLOG_INFO
#include "trcGenProj/Header/vd_diaglog_memory_master.cpp.trc.h"
#endif

using namespace VDD;

namespace VDD
{
struct memoryHasId
{
   memoryHasId();
   memoryHasId(tU8 aMemoryId) : memoryId(aMemoryId) {}
   bool operator() (vdl_tclReportMemory const* p) const {return p->u8GetMemoryId() == this->memoryId;}
private:
   tU8 memoryId;
};

}// namespace VDD

///////////////////////////////////////////////////////////////////////////////////
// constructor
vdl_tclMemoryMaster::vdl_tclMemoryMaster()
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::vdl_tclMemoryMaster"));
   ETG_TRACE_USR3_THR(( "<-- vdl_tclMemoryMaster::vdl_tclMemoryMaster"));
}


// destructor
vdl_tclMemoryMaster::~vdl_tclMemoryMaster()
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::~vdl_tclMemoryMaster"));

   _BP_TRY_BEGIN
   {
      vDestroyMemorys();
   }

   _BP_CATCH_ALL
   {
      ETG_TRACE_ERRMEM(( "!!! vdl_tclMemoryMaster::~vdl_tclReportMemory => FAILED"));
      NORMAL_M_ASSERT_ALWAYS();
   }
   _BP_CATCH_END


   ETG_TRACE_USR3_THR(( "<-- vdl_tclMemoryMaster::~vdl_tclMemoryMaster"));
}


///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclMemoryMaster::vProcessOnInit
//
// DESCRIPTION: first call to e.g. Load Data, calibrate device
//              while this, it is forbidden to get DATA from other parts
//              do this in vProcessOperatingCycleStart
//
// PARAMETER:  void
//
// RETURNVALUE: tVoid
//
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclMemoryMaster::vProcessOnInit()
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::vProcessOnInit"));

   // for each memory
   for(tMemoryArrayCIt itMemory =  m_oMemorys.begin();
       itMemory != m_oMemorys.end();
       ++itMemory)
   {
      vdl_tclReportMemory* pMemory = (*itMemory); // get memory
      // process this test result
      if(NULL != pMemory)
      {
         pMemory->vProcessOnInit();
      }// if(NULL != pMemory)
      else
      {
         ETG_TRACE_ERRMEM(( "!!! vdl_tclMemoryMaster::vProcessOnInit => NULL == pMemory"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }// for(tMemoryArrayCIt itMemory =  m_oMemorys.begin() ...

   // do not save while onInit
   (tVoid) bEventStep(EVENT_ON_INIT);

   ETG_TRACE_USR3_THR(( "<-- vdl_tclMemoryMaster::vProcessOnInit"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclMemoryMaster::processSystemEvent
// 
// DESCRIPTION: Process SPM/LCM Events 
//
// PARAMETER:  systemEvent: Event to handle
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclMemoryMaster::processSystemEvent(vdl_teDiaglogSystemEvent systemEvent)
{
   // forward to all memorys
   for(tMemoryArrayCIt itMemory =  m_oMemorys.begin();
       itMemory != m_oMemorys.end();
       ++itMemory)
   {
      vdl_tclReportMemory* pMemory = (*itMemory); // get memory
      // process this test result
      if(NULL != pMemory)
      {
         pMemory->processSystemEvent(systemEvent);
      }// if(NULL != pMemory)
      else
      {
         ETG_TRACE_ERRMEM(( "!!! vdl_tclMemoryMaster::vProcessOperatingCycleStart => NULL == pMemory"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }// for(tMemoryArrayCIt itMemory =  m_oMemorys.begin() ...

   // forward to all Plugin
   switch(systemEvent)
   {
      case DIAGLOG_EVENT_IGNITION_CYCLE_START:
      {
         (tVoid) bEventStep(EVENT_IGNITION_CYCLE_START);
         break;
      }
      case DIAGLOG_EVENT_IGNITION_CYCLE_END:
      {
         (tVoid) bEventStep(EVENT_IGNITION_CYCLE_END);
         break;
      }
      case DIAGLOG_EVENT_OPERATION_CYCLE_START:
      {
         (tVoid) bEventStep(EVENT_OPERATION_CYCLE_START);
         break;
      }
      case DIAGLOG_EVENT_OPERATION_CYCLE_END:
      {
         (tVoid) bEventStep(EVENT_OPERATION_CYCLE_END);
         break;
      }
      default:
      {
         // other ones are not supported by PLUGIN
         ETG_TRACE_USR4_THR(( "--- vdl_tclMemoryMaster::processSystemEvent => systemEvent:%d not supported for PLUGIN",systemEvent));
         break;
      }
   }// switch(systemEvent)
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclMemoryMaster::vProcessAgeingEvent
//
// DESCRIPTION: Process AgeingEvent
//
// PARAMETER:  void
//
// RETURNVALUE: tVoid
//
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclMemoryMaster::vProcessExternalAgeingCounter(tU8 AgeingCounter)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::vProcessExternalAgeingCounter"));
   // for each memory
   for(tMemoryArrayCIt itMemory =  m_oMemorys.begin();
       itMemory != m_oMemorys.end();
       ++itMemory)
   {
      vdl_tclReportMemory* pMemory = (*itMemory); // get memory
      // process this test result
      if(NULL != pMemory)
      {
         pMemory->vOnExternalAgeingCounter(AgeingCounter);
      }// if(NULL != pMemory)
      else
      {
         ETG_TRACE_ERRMEM(( "!!! vdl_tclMemoryMaster::vProcessExternalAgeingCounter => NULL == pMemory"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }// for(tMemoryArrayCIt itMemory =  m_oMemorys.begin() ...

   ETG_TRACE_USR3_THR(( "<-- vdl_tclMemoryMaster::vProcessExternalAgeingCounter"));
}


///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:tVoid vdl_tclMemoryMaster::processClearRequest
//
// DESCRIPTION: process clear request
//
// PARAMETER:
// RETURNVALUE: void
//
///////////////////////////////////////////////////////////////////////////////////
//
tBool vdl_tclMemoryMaster::processClearRequest  (tU8  u8MemoryId, tU32 u32DtcMask)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::processClearRequest"));
   bool bReturn = false;
   bool bFound = false;
   tITCVector ITCs;

   // for each memory
   for(tMemoryArrayCIt itMemory =  m_oMemorys.begin();
       itMemory != m_oMemorys.end();
       ++itMemory)
   {
      vdl_tclReportMemory* pMemory = (*itMemory); // get memory
      // process this test result
      if(NULL != pMemory)
      {
         ETG_TRACE_USR3_THR(( "--- vdl_tclMemoryMaster::processClearRequest Id:%d ReqId:%d",pMemory->u8GetMemoryId(), u8MemoryId));
         // check if MemoryId-bit is set
         if(u8MemoryId & pMemory->u8GetMemoryId())
         {
            bFound = true; // we found one
            tBool bResult = pMemory->processClearRequest(u32DtcMask, ITCs);
            if(bResult == true)
            {
               bReturn = true;
            }
            // continue this is a bit mask
         }// if(u8MemoryId & pMemory->u8GetMemoryId())
      }// if(NULL != pMemory)
      else
      {
         ETG_TRACE_ERRMEM(( "!!! vdl_tclMemoryMaster::processClearRequest => NULL == pMemory"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }// for(tMemoryArrayCIt itMemory =  m_oMemorys.begin() ...

   if(bFound == false)
   {
      ETG_TRACE_ERR_THR(( "!!! vdl_tclMemoryMaster::processClearRequest => Memory ID:%x not found",u8MemoryId));
   }

   // do the follwings only if we found this DTC
   if(bReturn)
   {
      (void) bEventStep(EVENT_CLEAR); // forward info to the PlugIns
      vSendUpdateStatusToAll(ITCs);
   }

   ETG_TRACE_USR3_THR(( "<-- vdl_tclMemoryMaster::processClearRequest"));

   return bReturn;
}

tS8 vdl_tclMemoryMaster::vProcessReadDTCPriority(tReadReportListDM1& DM1List)
{
	tS8 l_u8Ret = -1;
	for(tMemoryArrayCIt itMemory =  m_oMemorys.begin();
			itMemory != m_oMemorys.end();
			++itMemory)
	{
		vdl_tclReportMemory* pMemory = (*itMemory); // get memory
		if(NULL != pMemory)
		{
			ETG_TRACE_COMP_THR (("vdl_tclMemoryMaster::SEv  %X",pMemory->u8GetMemoryId()));
			if(0x01 == pMemory->u8GetMemoryId())
			{
				l_u8Ret =  pMemory->vProcessReadPriority( DM1List );
			}
		}
		else{
			ETG_TRACE_COMP_THR (("!!! vdl_tclMemoryMaster::vProcessReadDTCPriority => NULL"));
		}
	}
	return l_u8Ret;
}
///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:tVoid vdl_tclMemoryMaster::vProcessReadRequest
//
// DESCRIPTION: process read request
//
// PARAMETER:  tU8 const& u8StatusMask - the status maskread request
//             midw_fi_tcl_CustomerReadResultList& roReadResult - request result goes in here
//
// RETURNVALUE: void
//
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclMemoryMaster::vProcessReadRequest(
                                                tU8&                         u8StatusAvailabilityMask,
                                                tReadReportListArray&        roReadResult,
                                                tU8                          u8MemoryId,
                                                tU8 const&                   u8StatusMask,
                                                tU32 const&                  u32DTCMask,
                                                vdl_teDiaglogOptions const&  eOption
                                              )
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::vProcessReadRequest"));
   // for each memory
   for(tMemoryArrayCIt itMemory =  m_oMemorys.begin();
       itMemory != m_oMemorys.end();
       ++itMemory)
   {
      vdl_tclReportMemory* pMemory = (*itMemory); // get memory
      // process this test result
      if(NULL != pMemory)
      {
         if(u8MemoryId == pMemory->u8GetMemoryId())
         {
            tU8 Pos = bGetMemoryPosition(u8MemoryId);
            // get AvailabilityMask from configuration
            u8StatusAvailabilityMask = g_oMemoryFactory[Pos].u8StatusAvailabilityMask;
            // get Protocol from configuration
            vdl_teDiaglogReportProtocol statusProtocol = g_oMemoryFactory[Pos].eReportProtcol;

            // first set the avalibilty mask over the requested one to reduce work
            tU8 statusMask = u8StatusAvailabilityMask & u8StatusMask;

            // convert Status to DIAGLOG format
            tuRecordStatus uRecordStatus = m_oProtocolConverter.convertStatusExtToInt(statusProtocol, statusMask);

            // forward to memory
            pMemory->vProcessReadRequest(roReadResult, uRecordStatus.u32Status, u32DTCMask, eOption);

            // convert DIAGLOG format to Protocol format over all elements
            tReadReportListArrayIt iPos = roReadResult.begin();
            while(iPos.operator!=(roReadResult.end()))
            {
               tuRecordStatus uRecordStatus2;
               uRecordStatus2.u32Status = (*iPos).u32Status;

               tU8 u8Status = m_oProtocolConverter.convertStatusIntToExt(statusProtocol, uRecordStatus2);

               // overwrite status with the correct format
               (*iPos).u32Status = u8Status & u8StatusAvailabilityMask;
               ++iPos;
            }

            // only check in one memory, so we found it
            break;
         }
      }// if(NULL != pMemory)
      else
      {
         ETG_TRACE_ERRMEM(( "!!! vdl_tclMemoryMaster::vProcessReadRequest => NULL == pMemory"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }// for(tMemoryArrayCIt itMemory =  m_oMemorys.begin() ...
   ETG_TRACE_USR3_THR(( "<-- vdl_tclMemoryMaster::vProcessReadRequest"));
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:tVoid vdl_tclMemoryMaster::vProcessReadRequestNumberOfDTC
//
// DESCRIPTION: process read request
//
// PARAMETER:  tU8&          u8StatusAvailabilityMask : Availbilty Mask for this Memory
//             tU32&         u32NumberOfDTC           : Data to Fill
//             tU8 const     u8MemoryId               : Memory ID to forward to
//             tU8 const     u8StatusMask             : status mask to handle
//
// RETURNVALUE: void
//
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclMemoryMaster::vProcessReadNumberOfDtcByStatusMask(  tU8&          u8StatusAvailabilityMask,
                                                                 tU32&         u32NumberOfDTC,
                                                                 tU8 const     u8MemoryId,
                                                                 tU8 const     u8StatusMask)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::vProcessReadNumberOfDtcByStatusMask"));

   // for each memory
   for(tMemoryArrayCIt itMemory =  m_oMemorys.begin();
       itMemory != m_oMemorys.end();
       ++itMemory)
   {
      vdl_tclReportMemory* pMemory = (*itMemory); // get memory
      if(NULL != pMemory)
      {
         if(u8MemoryId == pMemory->u8GetMemoryId())
         {
            tU8 Pos = bGetMemoryPosition(u8MemoryId);
            u8StatusAvailabilityMask = g_oMemoryFactory[Pos].u8StatusAvailabilityMask;
            tU8 Mask = u8StatusMask & u8StatusAvailabilityMask;

            vdl_teDiaglogReportProtocol statusProtocol   = g_oMemoryFactory[Pos].eReportProtcol;

            tuRecordStatus uRecordStatus = m_oProtocolConverter.convertStatusExtToInt(statusProtocol, Mask);
            pMemory->vProcessReadNumberOfDtcByStatusMask(u32NumberOfDTC, uRecordStatus.u32Status);
         }
      }// if(NULL != pMemory)
      else
      {
         ETG_TRACE_ERRMEM(( "!!! vdl_tclMemoryMaster::vProcessReadNumberOfDtcByStatusMask => NULL == pMemory"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }// for(tMemoryArrayCIt itMemory =  m_oMemorys.begin() ...
   ETG_TRACE_USR3_THR(( "<-- vdl_tclMemoryMaster::vProcessReadNumberOfDtcByStatusMask"));
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:tVoid vdl_tclMemoryMaster::vProcessReadLastConfirmedDTC
//
// DESCRIPTION: process read last confiremd DTC
//
// PARAMETER:  tU8&                       u8StatusAvailabilityMask   : Availbilty Mask for this Memory
//             tReadReportListArray&      roReadResult               : Data to fill
//             tU8 const                  u8MemoryId                 : Memory ID to forward to
//
// RETURNVALUE: void
//
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclMemoryMaster::vProcessReadLastConfirmedDTC(tU8&                       u8StatusAvailabilityMask,
                                                        tReadReportListArray&      roReadResult,
                                                        tU8 const                  u8MemoryId)const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::vProcessReadLastConfirmedDTC"));

   // for each memory
   for(tMemoryArrayCIt itMemory =  m_oMemorys.begin();
       itMemory != m_oMemorys.end();
       ++itMemory)
   {
      vdl_tclReportMemory* pMemory = (*itMemory); // get memory
      if(NULL != pMemory)
      {
         if(u8MemoryId == pMemory->u8GetMemoryId())
         {
            tU8 Pos = bGetMemoryPosition(u8MemoryId);
            u8StatusAvailabilityMask = g_oMemoryFactory[Pos].u8StatusAvailabilityMask;
            vdl_teDiaglogReportProtocol statusProtocol   = g_oMemoryFactory[Pos].eReportProtcol;

            pMemory->vProcessReadLastConfirmedDTC(roReadResult);
            // convert status to UDS
            tReadReportListArrayIt iPos = roReadResult.begin();
            while(iPos.operator!=(roReadResult.end()))
            {
               // convert status back to UDS
               tuRecordStatus uRecordStatus;
               uRecordStatus.u32Status = (*iPos).u32Status;

               tU8  u8Status = m_oProtocolConverter.convertStatusIntToExt(statusProtocol, uRecordStatus);
               (*iPos).u32Status = u8Status & u8StatusAvailabilityMask ;
               ++iPos;
            }

            // only check in one memory, so we found it
            break;
         }
      }// if(NULL != pMemory)
      else
      {
         ETG_TRACE_ERRMEM(( "!!! vdl_tclMemoryMaster::vProcessReadLastConfirmedDTC => NULL == pMemory"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }// for(tMemoryArrayCIt itMemory =  m_oMemorys.begin() ...
   ETG_TRACE_USR3_THR(( "<-- vdl_tclMemoryMaster::vProcessReadLastConfirmedDTC"));
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:tVoid vdl_tclMemoryMaster::vProcessReadSupportedDTCs
//
// DESCRIPTION: process read all Supported DTC's
//
// PARAMETER:  tU8&                       u8StatusAvailabilityMask   : Availbilty Mask for this Memory
//             tReadReportListArray&      roReadResult               : Data to fill
//             tU8 const                  u8MemoryId                 : Memory ID to forward to
//
// RETURNVALUE: void
//
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclMemoryMaster::vProcessReadSupportedDTCs(tU8&                       u8StatusAvailabilityMask,
                                                        tReadReportListArray&      roReadResult,
                                                        tU8 const                  u8MemoryId)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::vProcessReadSupportedDTCs"));

   // for each memory
   for(tMemoryArrayCIt itMemory =  m_oMemorys.begin();
       itMemory != m_oMemorys.end();
       ++itMemory)
   {
      vdl_tclReportMemory* pMemory = (*itMemory); // get memory
      if(NULL != pMemory)
      {
         if(u8MemoryId == pMemory->u8GetMemoryId())
         {
            tU8 Pos = bGetMemoryPosition(u8MemoryId);
            u8StatusAvailabilityMask = g_oMemoryFactory[Pos].u8StatusAvailabilityMask;
            vdl_teDiaglogReportProtocol statusProtocol = g_oMemoryFactory[Pos].eReportProtcol;
            pMemory->vProcessReadSupportedDTCs(roReadResult);
            // convert status to UDS
            tReadReportListArrayIt iPos = roReadResult.begin();
            while(iPos.operator!=(roReadResult.end()))
            {
               // convert status back to UDS
               tuRecordStatus uRecordStatus;
               uRecordStatus.u32Status = (*iPos).u32Status;
               tU8  u8Status = m_oProtocolConverter.convertStatusIntToExt(statusProtocol, uRecordStatus);
               (*iPos).u32Status = u8Status & u8StatusAvailabilityMask ;
               ++iPos;
            }

            // only check in one memory, so we found it
            break;
         }
      }// if(NULL != pMemory)
      else
      {
         ETG_TRACE_ERRMEM(( "!!! vdl_tclMemoryMaster::vProcessReadSupportedDTCs => NULL == pMemory"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }// for(tMemoryArrayCIt itMemory =  m_oMemorys.begin() ...
   ETG_TRACE_USR3_THR(( "<-- vdl_tclMemoryMaster::vProcessReadSupportedDTCs"));
}



///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:tVoid vdl_tclMemoryMaster::bProcessTestReports
//
// DESCRIPTION: process test reports in worklist
//
// PARAMETER:  tU16 u16ITC                             : unique ITC Identifier
//             vdl_teDiaglogReportMsg eReportMsgStatus : ITC Status (e.g. Passed/Failed)
//
// RETURNVALUE: bool - true if data was sucessfully process
//
///////////////////////////////////////////////////////////////////////////////////
//
bool vdl_tclMemoryMaster::bProcessTestReport(tU16 u16ITC, vdl_teDiaglogReportMsg eReportMsgStatus)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::bProcessTestReport"));

   bool bReturn = false;
   bool bResult = false;
   bool dtcEnabled = true;
   
   /* before processing any request first check if the DTC is enabled */
   /* DTC Configuration Feature - CM-C19  start */
   
   if( vdDiagLog_tclApp::m_poInstance != NULL )
   {
	   dtcEnabled = (vdDiagLog_tclApp::m_poInstance)->m_poDtcCfgChk->isDTCEnabled( u16ITC );
	   
	   if(!dtcEnabled)
	   {
		   // DTC is not enabled; we will not process it
		   return false;
	   }
   }
   
   /* DTC Configuration Feature - CM-C19  end */

   // for each memory
   for(tMemoryArrayCIt itMemory =  m_oMemorys.begin();
       itMemory != m_oMemorys.end();
       ++itMemory)
   {
      vdl_tclReportMemory* pMemory = (*itMemory); // get memory
      // process this test result
      if(NULL != pMemory)
      {
         bReturn = pMemory->bProcessTestReport(u16ITC,eReportMsgStatus);

         if(bReturn == true)
         {
            bResult = true;
         }
      }// if(NULL != pMemory)
      else
      {
         ETG_TRACE_ERRMEM(( "!!! vdl_tclMemoryMaster::bProcessTestReport => NULL == pMemory"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }// for(tMemoryArrayCIt itMemory =  m_oMemorys.begin() ...

   if(bResult == false)
   {
      ETG_TRACE_COMP_CLS((TR_CLASS_DIAGLOG_SM,"  => INVALID ITC (0x%04x) received",u16ITC));
   }

   ETG_TRACE_USR3_THR(( "<-- vdl_tclMemoryMaster::bProcessTestReport"));
   return bResult;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:tVoid vdl_tclMemoryMaster::bProcessTestReports
//
// DESCRIPTION: process test reports in worklist
//
// PARAMETER:  vdl_tsDiaglogTestResult oTestResult : Diaglog Object with ITC, Status and Ext Data
//
// RETURNVALUE: bool - true if data was sucessfully process
//
///////////////////////////////////////////////////////////////////////////////////
//
bool vdl_tclMemoryMaster::bProcessTestReport(vdl_tsDiaglogTestResult oTestResult)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::bProcessTestReport"));

   bool bReturn = false;
   bool bResult = false;
   bool dtcEnabled = true;
   
   /* before processing any request first check if the DTC is enabled */
   /* DTC Configuration Feature - CM-C19  start */
   
   if( vdDiagLog_tclApp::m_poInstance != NULL )
   {
	   dtcEnabled = (vdDiagLog_tclApp::m_poInstance)->m_poDtcCfgChk->isDTCEnabled( tU16 (oTestResult.u32ITC ));
	   
	   if(!dtcEnabled)
	   {
		   // DTC is not enabled; we will not process it
		   return false;
	   }
   }
   
   /* DTC Configuration Feature - CM-C19  end */

   // for each memory
   for(tMemoryArrayCIt itMemory =  m_oMemorys.begin();
       itMemory != m_oMemorys.end();
       ++itMemory)
   {
      vdl_tclReportMemory* pMemory = (*itMemory); // get memory
      // process this test result
      if(NULL != pMemory)
      {
         bReturn = pMemory->bProcessTestReport(oTestResult);

         if(bReturn == true)
         {
            bResult = true;
         }
      }// if(NULL != pMemory)
      else
      {
         ETG_TRACE_ERRMEM(( "!!! vdl_tclMemoryMaster::bProcessTestReport => NULL == pMemory"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }// for(tMemoryArrayCIt itMemory =  m_oMemorys.begin() ...

   if(bResult == false)
   {
      ETG_TRACE_COMP_CLS((TR_CLASS_DIAGLOG_SM,"  => INVALID ITC (0x%04x) received",oTestResult.u32ITC));
   }

   ETG_TRACE_USR3_THR(( "<-- vdl_tclMemoryMaster::bProcessTestReport"));
   return bResult;
}


///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclMemoryMaster::vProcessSetToDefault
//
// DESCRIPTION: Process SetToDefault
//
// PARAMETER:  void
//
// RETURNVALUE: tVoid
//
///////////////////////////////////////////////////////////////////////////////////
//
tBool vdl_tclMemoryMaster::bProcessSetToDefault()
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::vProcessSetToDefault"));
   tBool bReturn;

   // for each memory
   for(tMemoryArrayCIt itMemory =  m_oMemorys.begin();
       itMemory != m_oMemorys.end();
       ++itMemory)
   {
      vdl_tclReportMemory* pMemory = (*itMemory); // get memory
      // process this test result
      if(NULL != pMemory)
      {
         pMemory->bProcessSetToDefault();
      }// if(NULL != pMemory)
      else
      {
         ETG_TRACE_ERRMEM(( "!!! vdl_tclMemoryMaster::vProcessSetToDefault => NULL == pMemory"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }// for(tMemoryArrayCIt itMemory =  m_oMemorys.begin() ...


   bReturn = bEventStep(EVENT_SET_TO_DEFAULT);

   ETG_TRACE_USR3_THR(( "<-- vdl_tclMemoryMaster::vProcessSetToDefault"));
   return bReturn;
}



///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: bool vdl_tclMemoryMaster::vDestroyMemory
//
// DESCRIPTION: check load status, load and merge results
//
// PARAMETER:  none
//
// RETURNVALUE: bool - true = loaded, false = not loaded
//
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclMemoryMaster::vDestroyMemorys()
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::vDestroyMemorys "));

   for(tMemoryArrayIt itMemory =  m_oMemorys.begin(); itMemory != m_oMemorys.end(); ++itMemory)
   {
      vdl_tclReportMemory* pMemory = (*itMemory); // get memory
      if(NULL != pMemory)
      {
         delete pMemory;
      }// if(NULL != pMemory)
      else
      {
         ETG_TRACE_ERRMEM(( "!!! vdl_tclMemoryMaster::vDestroyMemorys => NULL == pMemory"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }// for(tMemoryArrayCIt itMemory =  m_oMemorys.begin() ...

   ETG_TRACE_USR3_THR(( "<-- vdl_tclMemoryMaster::vDestroyMemorys "));
}


///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclMemoryMaster::vAddMemorys
//
// DESCRIPTION: create a emory object and put it to the memory master
//
// PARAMETER:  none
//
// RETURNVALUE: none
//
///////////////////////////////////////////////////////////////////////////////////
//


void vdl_tclMemoryMaster::vCreateMemorys()
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::vCreateMemorys"));

   int iMemorySize = g_iMemoryFactoryTableSize();

   ETG_TRACE_USR1_THR(( "--- vdl_tclMemoryMaster::vCreateMemorys => MemorySize = %d",g_iMemoryFactoryTableSize()));

   /*lint -save -e429 custodial pointer is freed in function vDestroyMemorys */
   for(int i = 0; i < iMemorySize; ++i)
   {
      vdl_tclReportMemory* pMemory;
      pMemory = new vdl_tclReportMemory(g_oMemoryFactory[i].u8MemoryId,
                                        g_oMemoryFactory[i].acMemoryFileName,
                                        g_oMemoryFactory[i].iSizeOfRecordDataArray,
                                        g_oMemoryFactory[i].pConstRecordDataArray);

      if(pMemory != NULL)
      {
         pMemory->initialise();
         m_oMemorys.push_back(pMemory);
      }
      else
      {
         ETG_TRACE_ERRMEM(( "!!! vdl_tclMemoryMaster::vCreateMemorys => NULL == pMemory"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }// for(int i = 0; i < size; ++i)
   /*lint -restore */

   ETG_TRACE_USR3_THR(( "<-- vdl_tclMemoryMaster::vCreateMemorys"));
}
///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclReportMemory::copyRecordToPlugin
//
// DESCRIPTION:  copy the struct to plugin
//
// PARAMETER:    repordRecord:   referenz to destination buffer
//               pReportRecord:  pointer to Data
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
void vdl_tclMemoryMaster::copyRecordToPlugin(vdl_tsDiaglogPlugInReport& repordRecord, const vdl_tclReportRecord* const pReportRecord) const
{
   repordRecord.u32DTC = pReportRecord->u32DTC();
   repordRecord.u16ITC = pReportRecord->u16ITC ();
   repordRecord.u32Status = pReportRecord->u32GetStatus();
   repordRecord.u8MemoryId = pReportRecord->u8GetMemoryId();
   repordRecord.u8Severity = pReportRecord->u8GetSeverity();
   repordRecord.u8OccCnt = pReportRecord->u8GetFailOccurenceCounter();
}


///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclReportMemory::bEventStep
//
// DESCRIPTION:  forward the EventStep to the PlugIns
//
// PARAMETER:    vdl_teDiaglogEventStep enEventStep: Kind of EventStep
//               vdl_tclReportRecord* pReportRecord: pointer to Data
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
tBool vdl_tclMemoryMaster::bEventStep(vdl_teDiaglogEventStep enEventStep, vdl_tclReportRecord* pReportRecord)const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::bEventStep"));
   tBool bReturn = true;
   ETG_TRACE_COMP_THR(( "--- vdl_tclMemoryMaster::bEventStep => Step=%d EventsInList=%d",ETG_ENUM(DIAGLOG_EVENT ,enEventStep), m_oEventRegList.size()));

   for(tEventRegistrationListCIt iPos = m_oEventRegList.begin(); iPos != m_oEventRegList.end(); ++iPos)
   {
      ETG_TRACE_USR2_THR(( "--- vdl_tclMemoryMaster::bEventStep => pCallbackClass=%x",iPos->pCallbackClass));
      if(iPos->pCallbackClass != NULL)
      {
         switch(enEventStep)
         {
            case EVENT_ON_INIT:
            {
               iPos->pCallbackClass->vOnInit();
               break;
            }
            case EVENT_OPERATION_CYCLE_START:
            {
               iPos->pCallbackClass->vOperationCycleStart();
               break;
            }
            case EVENT_OPERATION_CYCLE_END:
            {
               iPos->pCallbackClass->vOperationCycleEnd();
               break;
            }
            case EVENT_IGNITION_CYCLE_START:
            {
               iPos->pCallbackClass->vIgnitionCycleStart();
               break;
            }
            case EVENT_IGNITION_CYCLE_END:
            {
               iPos->pCallbackClass->vIgnitionCycleEnd();
               break;
            }
            case EVENT_CLEAR:
            {
               iPos->pCallbackClass->vClear();
               break;
            }
            case EVENT_SET_TO_DEFAULT:
            {
               bReturn = iPos->pCallbackClass->bSetToDefault();
               break;
            }
            case EVENT_FAIL_QUALIFIED:
            {
               if(pReportRecord != NULL)
               {
                  vdl_tsDiaglogPlugInReport repordRecord;
                  copyRecordToPlugin(repordRecord,pReportRecord);

                  iPos->pCallbackClass->vFailQualified(pReportRecord);
                  iPos->pCallbackClass->vFailQualified(repordRecord);
               }
               else
               {
                  ETG_TRACE_ERRMEM(( "!!! vdl_tclMemoryMaster::bEventStep => EVENT_FAIL_QUALIFIED => NULL == pReportRecord"));
                  NORMAL_M_ASSERT_ALWAYS();
               }
               break;
            }
            case EVENT_PASS_QUALIFIED:
            {
               if(pReportRecord != NULL)
               {
                  vdl_tsDiaglogPlugInReport repordRecord;
                  copyRecordToPlugin(repordRecord,pReportRecord);

                  iPos->pCallbackClass->vPassQualified(pReportRecord);
                  iPos->pCallbackClass->vPassQualified(repordRecord);
               }
               else
               {
                  ETG_TRACE_ERRMEM(( "!!! vdl_tclMemoryMaster::bEventStep => EVENT_PASS_QUALIFIED => NULL == pReportRecord"));
                  NORMAL_M_ASSERT_ALWAYS();
               }
               break;
            }
            case EVENT_AGEING_COMPLETED:
            {
               if(pReportRecord != NULL)
               {
                  vdl_tsDiaglogPlugInReport repordRecord;
                  copyRecordToPlugin(repordRecord,pReportRecord);

                  iPos->pCallbackClass->vAgeingCompleted(pReportRecord);
                  iPos->pCallbackClass->vAgeingCompleted(repordRecord);
               }
               else
               {
                  ETG_TRACE_ERRMEM(( "!!! vdl_tclMemoryMaster::bEventStep => EVENT_AGEING_COMPLETED => NULL == pReportRecord"));
                  NORMAL_M_ASSERT_ALWAYS();
               }
               break;
            }

            case EVENT_BLOCKING_MODE_UPDATE:
            {
               iPos->pCallbackClass->vUpdateAfterBlockingMode();
               break;
            }

            case EVENT_CLEAR_BY_MASK:
            {
               if(pReportRecord != NULL)
               {
                  vdl_tsDiaglogPlugInReport repordRecord;
                  copyRecordToPlugin(repordRecord,pReportRecord);
                  iPos->pCallbackClass->vClearByMask(repordRecord);
               }
               else
               {
                  ETG_TRACE_ERRMEM(( "!!! vdl_tclMemoryMaster::bEventStep => EVENT_CLEAR_BY_MASK => NULL == pReportRecord"));
                  NORMAL_M_ASSERT_ALWAYS();
               }
               break;
            }

            default:
            {
               ETG_TRACE_ERRMEM(( "!!! vdl_tclMemoryMaster::bEventStep => switch(enEventStep:%x) UNKNOWN",enEventStep));
               NORMAL_M_ASSERT_ALWAYS();
               break;
            }
         }// switch(enEventStep)
      }// if(iPos->pCallbackClass == pCallbackClass)
   }//for(tEventRegistrationListIt iPos = m_oEventRegList.begin(); iPos != m_oEventRegList.end(); iPos++)

   ETG_TRACE_USR3_THR(( "<-- vdl_tclMemoryMaster::bEventStep"));
   return bReturn;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclMemoryMaster::bRegisterForEvent
//
// DESCRIPTION:  bRegisterForEvent
//
// PARAMETER:    vdl_tclPluginBaseClass* pCallbackClass: pointer to class to register
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
bool vdl_tclMemoryMaster::bRegisterForEvent(vdl_tclPluginBaseClass* pCallbackClass)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::bRegisterForEvent"));
   bool bReturn = true;

   for(tEventRegistrationListIt iPos = m_oEventRegList.begin(); iPos.operator!=(m_oEventRegList.end()); ++iPos)
   {
      if(iPos->pCallbackClass == pCallbackClass)
      {
         ETG_TRACE_COMP_THR(( "--- vdl_tclMemoryMaster::bRegisterForEvent => still registered %x",pCallbackClass));
         bReturn = false;
         break;
      }
   }// for(tEventRegistrationListIt iPos = m_oEventRegList.begin(); iPos.operator!=(m_oEventRegList.end()); ++iPos)

   if(bReturn)
   {
      vdl_tsDiaglogEventRegistration oNewEvent;
      oNewEvent.pCallbackClass = pCallbackClass;

      m_oEventRegList.push_back(oNewEvent);
      ETG_TRACE_COMP_THR(( "--- vdl_tclMemoryMaster::bRegisterForEvent => Register %x",pCallbackClass));
   }// if(bReturn)

   ETG_TRACE_USR3_THR(( "<-- vdl_tclMemoryMaster::bRegisterForEvent"));
   return bReturn;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclMemoryMaster::bUnRegisterForEvent
//
// DESCRIPTION:  bUnRegisterForEvent
//
// PARAMETER:    vdl_tclPluginBaseClass* pCallbackClass: pointer to Class to Unregister
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
bool vdl_tclMemoryMaster::bUnRegisterForEvent(vdl_tclPluginBaseClass* pCallbackClass)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::bUnRegisterForEvent"));
   bool bFound = false;

   for(tEventRegistrationListIt iPos = m_oEventRegList.begin(); iPos.operator!=(m_oEventRegList.end()); ++iPos)
   {
      if(iPos->pCallbackClass == pCallbackClass)
      {
         ETG_TRACE_COMP_THR(( "--- vdl_tclMemoryMaster::bUnRegisterForEvent => remove %x",pCallbackClass));
         (tVoid) m_oEventRegList.erase(iPos);  // delete from list
         bFound = true;
         break;
      }
   }// for(tEventRegistrationListIt iPos = m_oEventRegList.begin(); iPos.operator!=(m_oEventRegList.end()); ++iPos)

   ETG_TRACE_USR3_THR(( "<-- vdl_tclMemoryMaster::bUnRegisterForEvent"));
   return bFound;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:tVoid vdl_tclMemoryMaster::vAgeingCompleted
//
// DESCRIPTION:
//
// PARAMETER:  vdl_tclReportRecord* pReportRecord: Pointer to Record
//
// RETURNVALUE: none
//
///////////////////////////////////////////////////////////////////////////////////
void vdl_tclMemoryMaster::vAgeingCompleted(vdl_tclReportRecord* pReportRecord)const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::vAgeingCompleted"));
   if(pReportRecord != NULL)
   {
      (tVoid) bEventStep(EVENT_AGEING_COMPLETED,pReportRecord); // forward info to the PlugIns
   }
   ETG_TRACE_USR3_THR(( "<-- vdl_tclMemoryMaster::vAgeingCompleted"));
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclReportMemory::vTraceReports
//
// DESCRIPTION: trace each report
//
// PARAMETER:  void
//
// RETURNVALUE: tVoid
//
///////////////////////////////////////////////////////////////////////////////////
//
void vdl_tclMemoryMaster::vTraceReports(tU8 u8Memory)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::vTraceReports"));
   // for each memory
   for(tMemoryArrayCIt itMemory =  m_oMemorys.begin();
       itMemory != m_oMemorys.end();
       ++itMemory)
   {
      vdl_tclReportMemory* pMemory = (*itMemory); // get memory
      // process this test result
      if(NULL != pMemory)
      {
         if(  (u8Memory == pMemory->u8GetMemoryId())
            ||(u8Memory == 0xFF))
         {
            pMemory->vTraceReports();
         }
      }// if(NULL != pMemory)
      else
      {
         ETG_TRACE_ERRMEM(( "!!! vdl_tclMemoryMaster::vTraceReports => NULL == pMemory"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }// for(tMemoryArrayCIt itMemory =  m_oMemorys.begin() ...
   ETG_TRACE_USR3_THR(( "<-- vdl_tclMemoryMaster::vTraceReports"));
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: bool vdl_tclMemoryMaster::bCheckLoadStatus
//
// DESCRIPTION: check load status, load and merge results
//
// PARAMETER:  midw_fi_tcl_TestResultList const& roTestReport - test report(s)
//
// RETURNVALUE: bool - true = loaded, false = not loaded
//
///////////////////////////////////////////////////////////////////////////////////
//
bool vdl_tclMemoryMaster::bCheckLoadStatus()
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::bCheckLoadStatus"));
   bool bReturn = true;

   // for each memory
   for(tMemoryArrayCIt itMemory =  m_oMemorys.begin();
       itMemory != m_oMemorys.end();
       ++itMemory)
   {
      vdl_tclReportMemory* pMemory = (*itMemory); // get memory
      // process this test result
      if(NULL != pMemory)
      {
         bReturn = pMemory->bCheckLoadStatus();

         if(bReturn == false)
         {
            break;
         }
      }// if(NULL != pMemory)
      else
      {
         ETG_TRACE_ERRMEM(( "!!! vdl_tclMemoryMaster::bCheckLoadStatus => NULL == pMemory"));
         NORMAL_M_ASSERT_ALWAYS();
         bReturn = false;
         break;
      }
   }// for(tMemoryArrayCIt itMemory =  m_oMemorys.begin() ...

   ETG_TRACE_USR3_THR(( "<-- vdl_tclMemoryMaster::bCheckLoadStatus"));
   return bReturn;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: bool vdl_tclMemoryMaster::vSendUpdateStatusToAll
//
// DESCRIPTION: send update to components
//
// PARAMETER:  none
//
// RETURNVALUE: none
//
///////////////////////////////////////////////////////////////////////////////////
//
void vdl_tclMemoryMaster::vSendUpdateStatusToAll(tITCVector& ITCs) const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::vSendUpdateStatusToAll"));

   // update V850
   if(  ((vdDiagLog_tclApp::m_poInstance) != NULL)
      &&((vdDiagLog_tclApp::m_poInstance)->m_poIncInterface != NULL))
   {
      (vdDiagLog_tclApp::m_poInstance)->m_poIncInterface->SendClearRequest();
   }
   else
   {
      ETG_TRACE_ERR_THR(( "!!! vdl_tclMemoryMaster::vSendUpdateStatusToAll => Can't send to V850 components"));
   }


   // update CCA
   if(  ((vdDiagLog_tclApp::m_poInstance) != NULL)
      &&((vdDiagLog_tclApp::m_poInstance)->m_poService != NULL))
   {
      (vdDiagLog_tclApp::m_poInstance)->m_poService->vSendUpdateStatus(ITCs);
   }
   else
   {
      ETG_TRACE_ERR_THR(( "!!! vdl_tclMemoryMaster::vSendUpdateStatusToAll => Can't send to CCA components"));
   }

   // update PlugIn
   bEventStep(EVENT_BLOCKING_MODE_UPDATE,NULL);

   ETG_TRACE_USR3_THR(( "<-- vdl_tclMemoryMaster::vSendUpdateStatusToAll"));
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: bool vdl_tclMemoryMaster::vSendUpdateStatusToRequested
//
// DESCRIPTION: send update to components
//
// PARAMETER:  none
//
// RETURNVALUE: none
//
///////////////////////////////////////////////////////////////////////////////////
//
void vdl_tclMemoryMaster::vSendUpdateStatusToRequested(tServiceUpdateMap updateMap) const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::vSendUpdateStatusToRequested"));

   bool retriggerCCA    = false;
   bool retriggerV850   = false;
   bool retriggerPlugIn = false;

   for (tServiceUpdateMapCIt iterUpdateMap = updateMap.begin(); iterUpdateMap != updateMap.end(); ++iterUpdateMap)
   {
      tU32 AppId = iterUpdateMap->first;
      if(AppId & DIAGLOG_RETRIGGER_ID_V850)
      {
         retriggerV850 = true;
      }
      else if(AppId & DIAGLOG_RETRIGGER_ID_CCA)
      {
         retriggerCCA = true;
      }
      else if(AppId & DIAGLOG_RETRIGGER_ID_PLUGIN)
      {
         retriggerPlugIn = true;
      }
   }

   // update CCA
   if( retriggerCCA)
   {
      if(  ((vdDiagLog_tclApp::m_poInstance) != NULL)
         &&((vdDiagLog_tclApp::m_poInstance)->m_poService != NULL))
      {
         ETG_TRACE_USR2_THR(( "--- vdl_tclMemoryMaster::vSendUpdateStatusToRequested => update CCA"));
         (vdDiagLog_tclApp::m_poInstance)->m_poService->vUpdateAfterBlockingMode(updateMap);
      }
      else
      {
         ETG_TRACE_ERR_THR(( "!!! vdl_tclMemoryMaster::vSendUpdateStatusToRequested => Can't send to CCA components"));
      }
   }

   // update V850
   if( retriggerV850)
   {
      if(  ((vdDiagLog_tclApp::m_poInstance) != NULL)
         &&((vdDiagLog_tclApp::m_poInstance)->m_poIncInterface != NULL))
      {
         ETG_TRACE_USR2_THR(( "--- vdl_tclMemoryMaster::vSendUpdateStatusToRequested => update V850"));
         (vdDiagLog_tclApp::m_poInstance)->m_poIncInterface->SendClearRequest();
      }
      else
      {
         ETG_TRACE_ERR_THR(( "!!! vdl_tclMemoryMaster::vSendUpdateStatusToRequested => Can't send to V850 components"));
      }
   }

   // update PlugIn
   if( retriggerPlugIn)
   {
      ETG_TRACE_USR2_THR(( "--- vdl_tclMemoryMaster::vSendUpdateStatusToRequested => update PLUGIN"));
      bEventStep(EVENT_BLOCKING_MODE_UPDATE,NULL);
   }

   ETG_TRACE_USR3_THR(( "<-- vdl_tclMemoryMaster::vSendUpdateStatusToRequested"));
}


///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclReportRecord* pGetReportRecord
//
// DESCRIPTION: Get ReportRecorod by given Index and ITC
//
// PARAMETER:  u8Index: Memoyr ID
//             u16ITC:  unique ITC Identification
//
// RETURNVALUE: none
//
///////////////////////////////////////////////////////////////////////////////////
//
vdl_tclReportRecord* vdl_tclMemoryMaster::pGetReportRecord(tU8 u8MemoryId, tU16 u16ITC)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::pGetReportRecord"));
   vdl_tclReportRecord* pRecord = NULL;

   // for each memory
   for(tMemoryArrayCIt itMemory =  m_oMemorys.begin();
       itMemory != m_oMemorys.end();
       ++itMemory)
   {
      vdl_tclReportMemory* pMemory = (*itMemory); // get memory
      // process this test result
      if(NULL != pMemory)
      {
         if(u8MemoryId == pMemory->u8GetMemoryId())
         {
            pRecord = pMemory->pGetReportRecord(u16ITC);
            break;
         }
      }// if(NULL != pMemory)
      else
      {
         ETG_TRACE_ERRMEM(( "!!! vdl_tclMemoryMaster::pGetReportRecord => NULL == pMemory"));
         NORMAL_M_ASSERT_ALWAYS();
         break;
      }
   }// for(tMemoryArrayCIt itMemory =  m_oMemorys.begin() ...

   ETG_TRACE_USR3_THR(( "<-- vdl_tclMemoryMaster::pGetReportRecord"));
   return pRecord;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:tVoid vdl_tclMemoryMaster::bProcessControlCommand
//
// DESCRIPTION: process control comannd in worklist
//
// PARAMETER:  vdl_teDiaglogControlMsg eControlCmd     : command
//             tDTCListArray oDTCList                  : List of DTC (not ITC) valid for this command
//
// RETURNVALUE: bool - true if data was sucessfully process
//
///////////////////////////////////////////////////////////////////////////////////
//
bool vdl_tclMemoryMaster::bProcessControlCommand(vdl_teDiaglogControlMsg eControlCmd, tDTCListArray const& oDTCList)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::bProcessControlCommand"));

   bool bResult = false;

   switch(eControlCmd)
   {
      case DIAGLOG_CONTROL_READ_DISABLE_BLOCKING_ENABLE:
      {
         ETG_TRACE_USR1_THR(( "--- vdl_tclMemoryMaster::bProcessControlCommand => DIAGLOG_CONTROL_READ_DISABLE_BLOCKING_ENABLE"));
         vdl_tclGlobalSystemVariables::vReSetIgnoreCalibration();
         break;
      }// case DIAGLOG_CONTROL_READ_DISABLE_BLOCKING_ENABLE:

      case DIAGLOG_CONTROL_READ_DISABLE_BLOCKING_DISABLE:
      {
         ETG_TRACE_USR1_THR(( "--- vdl_tclMemoryMaster::bProcessControlCommand => DIAGLOG_CONTROL_READ_DISABLE_BLOCKING_DISABLE"));
         vdl_tclGlobalSystemVariables::vSetIgnoreCalibration();
         break;
      }// case DIAGLOG_CONTROL_READ_DISABLE_BLOCKING_DISABLE:

      case DIAGLOG_CONTROL_CTRL_LOGGING_TEMP_ENABLE:
      {
         ETG_TRACE_USR1_THR(( "--- vdl_tclMemoryMaster::bProcessControlCommand => DIAGLOG_CONTROL_CTRL_LOGGING_TEMP_ENABLE"));
         if(NULL != vdDiagLog_tclApp::m_poInstance)
         {
            (vdDiagLog_tclApp::m_poInstance)->vResetBlockingMode(DIAGLOG_BLOCKING_MODE_DIAGNOSIS);
         }
         break;
      }// case DIAGLOG_CONTROL_CTRL_LOGGING_TEMP_ENABLE:

      case DIAGLOG_CONTROL_CTRL_LOGGING_TEMP_DISABLE:
      {
         ETG_TRACE_USR1_THR(( "--- vdl_tclMemoryMaster::bProcessControlCommand => DIAGLOG_CONTROL_CTRL_LOGGING_TEMP_DISABLE"));
         if(NULL != vdDiagLog_tclApp::m_poInstance)
         {
            (vdDiagLog_tclApp::m_poInstance)->vSetBlockingMode(DIAGLOG_BLOCKING_MODE_DIAGNOSIS);
         }
         break;
      }// case DIAGLOG_CONTROL_CTRL_LOGGING_TEMP_DISABLE:


      // forward the following to the single Memory
      case DIAGLOG_CONTROL_READ_DISABLE:
      case DIAGLOG_CONTROL_READ_ENABLE:
      case DIAGLOG_CONTROL_CALIB_DISABLE:
      case DIAGLOG_CONTROL_CALIB_ENABLE:
      case DIAGLOG_CONTROL_CTRL_LOGGING_PERSIST_ENABLE:
      case DIAGLOG_CONTROL_CTRL_LOGGING_PERSIST_DISABLE:
      {
         ETG_TRACE_USR1_THR(( "--- vdl_tclMemoryMaster::bProcessControlCommand => DIAGLOG_CONTROL... forward to Memory Object"));
         bResult = processControlCommand(eControlCmd, oDTCList);
         break;
      }// case DIAGLOG_CONTROL_CTRL_ENABLE: // case DIAGLOG_CONTROL_CTRL_DISABLE:

      default:
      {
         ETG_TRACE_ERRMEM(( "!!! vdl_tclMemoryMaster::bProcessControlCommand => switch(eControlCmd:%x) UNKNOWN",eControlCmd));
         NORMAL_M_ASSERT_ALWAYS();
         break;
      }// default
   }//  switch(eControlCmd)

   ETG_TRACE_USR3_THR(( "<-- vdl_tclMemoryMaster::bProcessControlCommand"));
   return bResult;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:tVoid vdl_tclMemoryMaster::processControlCommand
//
// DESCRIPTION: forward the control command to each Memory
//
// PARAMETER:  vdl_teDiaglogControlMsg eControlCmd    Control Command
//             tDTCListArray const& oDTCList          :List of DTC's
//
// RETURNVALUE: TRUE if succesfully
//
///////////////////////////////////////////////////////////////////////////////////
//
bool vdl_tclMemoryMaster::processControlCommand(vdl_teDiaglogControlMsg eControlCmd, tDTCListArray const& oDTCList)
{
   // for each memory
   bool bResult = false;
   for(tMemoryArrayCIt itMemory =  m_oMemorys.begin();
       itMemory != m_oMemorys.end();
       ++itMemory)
   {
      vdl_tclReportMemory* pMemory = (*itMemory); // get memory
      // process this test result
      if(NULL != pMemory)
      {
         if(pMemory->bProcessControlCommand(eControlCmd, oDTCList))
         {
            bResult = true;
         }
      }// if(NULL != pMemory)
      else
      {
         ETG_TRACE_ERRMEM(( "!!! vdl_tclMemoryMaster::processControlCommand => NULL == pMemory"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }// for(tMemoryArrayCIt itMemory =  m_oMemorys.begin() ...
   return bResult;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:tVoid vdl_tclMemoryMaster::bGetMemoryPosition
//
// DESCRIPTION: search memory ID in g_oMemoryFactory
//
// PARAMETER:  MemoryId    : Memory Id to search for
//
// RETURNVALUE: tU8 - Memory Pos
//
///////////////////////////////////////////////////////////////////////////////////
//
tU8 vdl_tclMemoryMaster::bGetMemoryPosition(tU8 MemoryId) const
{

   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::bGetMemoryPosition"));
   int iPos = 0xFFFF;
   int iMemorySize = g_iMemoryFactoryTableSize();

   for(int i = 0; i < iMemorySize; ++i)
   {
      if(g_oMemoryFactory[i].u8MemoryId == MemoryId)
      {
         iPos =i;
         break;
      }
   }// for(int i = 0; i < size; ++i)

   if(iPos >= 0xFF)
   {
       ETG_TRACE_ERRMEM(( "!!! vdl_tclMemoryMaster::bGetMemoryPosition => INVALID iPos:%x",iPos));
       NORMAL_M_ASSERT_ALWAYS();
       iPos = 0;
   }
   ETG_TRACE_USR3_THR(( "<-- vdl_tclMemoryMaster::bGetMemoryPosition"));
   return (tU8)iPos;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: ExtendedData const* vdl_tclMemoryMaster::getExtendedData
//
// DESCRIPTION: Get extended Data of a requested DTC
//
// PARAMETER:  tU8 group         : Memory Id to search in
//             tU32 dtc          : requested DTC
//             tU8 recordNumber  :
//
// RETURNVALUE: ExtendedData     : Extended Data Object
//
///////////////////////////////////////////////////////////////////////////////////
//
ExtendedData const* vdl_tclMemoryMaster::getExtendedData(tU8 group, tU32 dtc, tU8 recordNumber) const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::getExtendedData"));

   ExtendedData const* ptrExtData = NULL;

   vdl_tclReportMemory* const reportMemory = this->getReportMemory(group);

   if(reportMemory != NULL)
   {
      return reportMemory->getExtendedData(dtc,recordNumber);
   }

   ETG_TRACE_USR3_THR(( "<-- vdl_tclMemoryMaster::getExtendedData"));
   return ptrExtData;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: ExtendedData const* vdl_tclMemoryMaster::getSnapShotData
//
// DESCRIPTION: Get extended Data of a requested DTC
//
// PARAMETER:  tU8 memoryId      : Memory Id to search in
//             tU32 dtc          : requested DTC
//             tU8 recordNumber  :
//
// RETURNVALUE: ExtendedData     : Extended Data Object
//
///////////////////////////////////////////////////////////////////////////////////
//
void vdl_tclMemoryMaster::getSnapShotData(tU8 memoryId, tU32 dtc, tU8 recordNumber, tU8& status, tSnapShotDataList& SnapShotDataList) const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::getSnapShotData"));

   vdl_tclReportMemory* const reportMemory = this->getReportMemory(memoryId);
   if(reportMemory != NULL)
   {
      /* 1. convert given status to DIAGLOG format */
      tU8 Pos = bGetMemoryPosition(memoryId);
      // get AvailabilityMask from configuration
      tU8 u8StatusAvailabilityMask = g_oMemoryFactory[Pos].u8StatusAvailabilityMask;
      // get Protocol from configuration
      vdl_teDiaglogReportProtocol statusProtocol = g_oMemoryFactory[Pos].eReportProtcol;
      // creat RecordStatus for correct response
      tuRecordStatus uRecordStatus;
      uRecordStatus.u32Status = 0;

      /* 2. send now to memory */
      reportMemory->getSnapShotData(dtc, recordNumber, uRecordStatus.u32Status, SnapShotDataList);

      /* 3. convert back status to protocol format */
      status = m_oProtocolConverter.convertStatusIntToExt(statusProtocol, uRecordStatus) & u8StatusAvailabilityMask;
   }
   ETG_TRACE_USR3_THR(( "<-- vdl_tclMemoryMaster::getSnapShotData"));
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: ExtendedData const* vdl_tclMemoryMaster::getSnapShotData
//
// DESCRIPTION: Get extended Data of a requested DTC
//
// PARAMETER:  tU8 group         : Memory Id to search in
//             tU32 dtc          : requested DTC
//             tU8 recordNumber  :
//
// RETURNVALUE: ExtendedData     : Extended Data Object
//
///////////////////////////////////////////////////////////////////////////////////
//
//lint -e1764 // Reference parameter 'SnapShotIdList' could be declared const ref
//lint -e1762 // Member function 'vdl_tclMemoryMaster::getSnapShotId(unsigned char, std::list<vdl_tsSnapShotIdStruct> &)' could be made const
void  vdl_tclMemoryMaster::getSnapShotId(tU8 memoryId, tSnapShotIdList& SnapShotIdList) 
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::getSnapShotId"));

   #ifdef DIAGLOG_USE_SNAPSHOT_DATA 
   if(vdDiagLog_tclApp::m_poInstance != NULL)
   {
      if(vdDiagLog_tclApp::m_poInstance->m_poSnapShotData != NULL)
      {
        (void) vdDiagLog_tclApp::m_poInstance->m_poSnapShotData->getSnapShotIdList(memoryId,SnapShotIdList);
      }
      else
      {
         // should never happen if DIAGLOG_USE_SNAPSHOT_DATA is defined
         ETG_TRACE_ERRMEM(( "!!! vdl_tclMemoryMaster::getSnapShotId => NULL == m_poSnapShotData"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }//if(vdDiagLog_tclApp::m_poInstance != NULL)
   else
   {
      ETG_TRACE_ERRMEM(( "!!! vdl_tclMemoryMaster::getSnapShotId => NULL == m_poInstance"));
      NORMAL_M_ASSERT_ALWAYS();
   }// else // if(vdDiagLog_tclApp::m_poInstance != NULL)
   #else
      DIA_PARAMETER_INTENTIONALLY_UNUSED( memoryId );
      DIA_PARAMETER_INTENTIONALLY_UNUSED( SnapShotIdList );
      ETG_TRACE_ERR_THR(( "!!! vdl_tclMemoryMaster::getSnapShotId => no snap shot data defined for ALL DTC'S, check define DIAGLOG_USE_SNAPSHOT_DATA"));
   #endif

   ETG_TRACE_USR3_THR(( "<-- vdl_tclMemoryMaster::getSnapShotId"));
}
//lint +e1762
//lint +e1764
///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclReportMemory const& vdl_tclMemoryMaster::getReportMemory(tU8 memoryId) const
//
// DESCRIPTION: Get Memory Object of requested ID
//
// PARAMETER:  tU8 memoryId          : Memory Id of Memory to get
//
// RETURNVALUE: vdl_tclReportMemory  : Memory Object
//
///////////////////////////////////////////////////////////////////////////////////
//
vdl_tclReportMemory* vdl_tclMemoryMaster::getReportMemory(tU8 memoryId) const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::getReportMemory"));

	tMemoryArrayCIt itMemory = std::find_if(m_oMemorys.begin(), m_oMemorys.end(), VDD::memoryHasId(memoryId));

   ETG_TRACE_USR3_THR(( "<-- vdl_tclMemoryMaster::getReportMemory"));
   if(itMemory == m_oMemorys.end())
    {
       return NULL;
    }
    else
    {
      return (*itMemory);
    }
}


///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: tU8 vdl_tclMemoryMaster::getDtcStatus(tU8 aGroup, tU32 aDTC) const
//
// DESCRIPTION: Get Status of a requested DTC
//
// PARAMETER:  tU8 aGroup            : Memory Id to search in
//             tU32 aDTC             : DTC to search for
//
// RETURNVALUE: vdl_tclReportMemory  : Status of requested DTC and Memory
//
///////////////////////////////////////////////////////////////////////////////////
//

tU8 vdl_tclMemoryMaster::getDtcStatus(tU8 u8MemoryId, tU32 aDTC) const
{
   // get Memory 
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::getDtcStatus"));
   vdl_tclReportMemory const* reportMemory = this->getReportMemory(u8MemoryId);
   if( reportMemory == NULL )
   {
	   return 0;
   }
   // get DTC Record
   vdl_tclReportRecord const* reportRecord = reportMemory->getReportRecordForDTC(aDTC);
   if(reportRecord == NULL )
   {
	   return 0;
   }
   // convert status to correct Protocol before returning

   tU8 Pos = bGetMemoryPosition(u8MemoryId);
   // get AvailabilityMask from configuration
   tU8 u8StatusAvailabilityMask = g_oMemoryFactory[Pos].u8StatusAvailabilityMask;
   // get Protocol from configuration
   vdl_teDiaglogReportProtocol statusProtocol = g_oMemoryFactory[Pos].eReportProtcol;

   // get status from record
   tuRecordStatus uRecordStatus = reportRecord->getRecordStatus();
   // convert to correct protocol
   tU8 status = m_oProtocolConverter.convertStatusIntToExt(statusProtocol, uRecordStatus) & u8StatusAvailabilityMask;

   ETG_TRACE_USR3_THR(( "<-- vdl_tclMemoryMaster::getDtcStatus"));
   return status;
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: ExtendedData const* vdl_tclReportMemory::getSnapShotData
// 
// DESCRIPTION: Get Snap Shot Data of a requested DTC
//
// PARAMETER:  tU32 dtc          : requested DTC
//             tU8 recordNumber  : 
//
// RETURNVALUE: none
// 
///////////////////////////////////////////////////////////////////////////////////
//
vdl_tclReportRecord const* vdl_tclMemoryMaster::getMostRecentReportRecordByDTC(tU8 MemoryId, tU32 u32DTC)const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::getMostRecentReportRecordByDTC"));
   // for each memory
   for(tMemoryArrayCIt itMemory =  m_oMemorys.begin();
       itMemory != m_oMemorys.end();
       ++itMemory)
   {
      vdl_tclReportMemory* pMemory = (*itMemory); // get memory
      // process this test result
      if(NULL != pMemory)
      {
         if(MemoryId & pMemory->u8GetMemoryId())
         {
            return (pMemory->getMostRecentReportRecordByDTC(u32DTC));
         }
      }// if(NULL != pMemory)
      else
      {
         ETG_TRACE_ERRMEM(( "!!! vdl_tclMemoryMaster::getMostRecentReportRecordByDTC => NULL == pMemory"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }// for(tMemoryArrayCIt itMemory =  m_oMemorys.begin() ...
   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::getMostRecentReportRecordByDTC"));
   return NULL;
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: ExtendedData const* vdl_tclReportMemory::getSnapShotData
// 
// DESCRIPTION: Get Snap Shot Data of a requested DTC
//
// PARAMETER:  tU32 dtc          : requested DTC
//             tU8 recordNumber  : 
//
// RETURNVALUE: none
// 
///////////////////////////////////////////////////////////////////////////////////
//
void vdl_tclMemoryMaster::getItcInformation(tU8 MemoryId, tU16 ITC, vdl_tsDiaglogItcInformation& itcInformation)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::getItcInformation"));
   // for each memory
   for(tMemoryArrayCIt itMemory =  m_oMemorys.begin();
       itMemory != m_oMemorys.end();
       ++itMemory)
   {
      vdl_tclReportMemory* pMemory = (*itMemory); // get memory
      // process this test result
      if(NULL != pMemory)
      {
         if(MemoryId & pMemory->u8GetMemoryId())
         {
            pMemory->getItcInformation(ITC,itcInformation);

            // convert status to correct Protocol before returning
            if(itcInformation.u32Status != 0)
            {
               tU8 Pos = bGetMemoryPosition(MemoryId & pMemory->u8GetMemoryId());
               // get AvailabilityMask from configuration
               tU8 u8StatusAvailabilityMask = g_oMemoryFactory[Pos].u8StatusAvailabilityMask;
               // get Protocol from configuration
               vdl_teDiaglogReportProtocol statusProtocol = g_oMemoryFactory[Pos].eReportProtcol;

               // convert to correct protocol
               itcInformation.u8Status = m_oProtocolConverter.convertStatusIntToExt(statusProtocol, itcInformation.u32Status) & u8StatusAvailabilityMask;
            }
         }
      }// if(NULL != pMemory)
      else
      {
         ETG_TRACE_ERRMEM(( "!!! vdl_tclMemoryMaster::getItcInformation => NULL == pMemory"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }// for(tMemoryArrayCIt itMemory =  m_oMemorys.begin() ...
   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::getItcInformation"));
}


///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: void vdl_tclMemoryMaster::LockMemorySemaphore()
//
// DESCRIPTION: lock the Memory Semaphore
//
// PARAMETER:  none
//
// RETURNVALUE: none
//
///////////////////////////////////////////////////////////////////////////////////
//
void vdl_tclMemoryMaster::LockMemorySemaphore() const
{
   tS32 s32Ret = OSAL_s32SemaphoreWait(vdDiagLog_tclApp::m_hReportMemSem, OSAL_C_U32_INFINITE);
   if(OSAL_OK != s32Ret)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      et_vErrmemStringNormal( TR_COMP_DIAGNOSIS, "Diaglog: LockMemorySemaphore() => OSAL_s32SemaphoreWait failed: Result=%x, Error=%x",
                               s32Ret,
                               u32ErrorCode);
      NORMAL_M_ASSERT_ALWAYS();
   }

}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: void vdl_tclMemoryMaster::FreeMemorySemaphore()
//
// DESCRIPTION: free the Memory Semaphore
//
// PARAMETER:  none
//
// RETURNVALUE: none
//
///////////////////////////////////////////////////////////////////////////////////
//
void vdl_tclMemoryMaster::FreeMemorySemaphore() const
{
   tS32 s32Ret = OSAL_s32SemaphorePost(vdDiagLog_tclApp::m_hReportMemSem);
   if(OSAL_OK != s32Ret)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      et_vErrmemStringNormal( TR_COMP_DIAGNOSIS, "Diaglog: FreeMemorySemaphore() => OSAL_s32SemaphorePost failed: Result=%x, Error=%x",
                               s32Ret,
                               u32ErrorCode);
      NORMAL_M_ASSERT_ALWAYS();
   }
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: ExtendedData const* vdl_tclReportMemory::getOccuranceCounterByDTC
//
// DESCRIPTION: search for DTC and accumulate the occurance counter for different ITC's
//              1. active
//              2. history
//              3. inactive
//
// PARAMETER:  tU32 dtc          : requested DTC
//             tU8 MemoryId  :
//
// RETURNVALUE: none
//
///////////////////////////////////////////////////////////////////////////////////
//
tU8 vdl_tclMemoryMaster::getOccuranceCounterByDTC(tU8 MemoryId, tU32 u32DTC)const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclMemoryMaster::getOccuranceCounterByDTC"));

   tU8 occuranceCounter = 0;
   // for each memory
   for(tMemoryArrayCIt itMemory =  m_oMemorys.begin();
       itMemory != m_oMemorys.end();
       ++itMemory)
   {
      const vdl_tclReportMemory* pMemory = (*itMemory); // get memory
      // process this test result
      if(NULL != pMemory)
      {
         if(MemoryId & pMemory->u8GetMemoryId())
         {
            occuranceCounter = pMemory->getOccuranceCounterByDTC(u32DTC);
            break;
         }
      }// if(NULL != pMemory)
      else
      {
         ETG_TRACE_ERRMEM(( "!!! vdl_tclMemoryMaster::getOccuranceCounterByDTC => NULL == pMemory"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }// for(tMemoryArrayCIt itMemory =  m_oMemorys.begin() ...
   ETG_TRACE_USR3_THR(( "<-- vdl_tclReportMemory::getOccuranceCounterByDTC"));
   return occuranceCounter;
}
