/************************************************************************
* FILE:         vd_diaglog_main.cpp
* PROJECT:      common
* SW-COMPONENT: Diaglog
*----------------------------------------------------------------------
*
* DESCRIPTION: DiagLog main application class
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2005 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 17.10.05  | CM-DI/ESA2 Barber  | initial version
* 23.07.10  | TMS Daoud          | GMG2 Updates
* 05.11.13  | BSOT Plischke      | new diaglog
*************************************************************************/
// first include diaglog settings
#include <common/framework/vd_diaglog_settings.h>

#define VD_DIAGLOG_S_IMPORT_INTERFACE_MSG
#include <vd_diaglog_if.h>

//-----------------------------------------------------------------------------
// includes
//-----------------------------------------------------------------------------
#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h"

#include "vd_diaglog_main.h"


#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_service
   #include <project/interfaces/vd_diaglog_service.h>
#endif
#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_clienthandlerspm
   #include <project/interfaces/vd_diaglog_clienthandlerspm.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_memory_master
   #include <common/framework/vd_diaglog_memory_master.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_inc_comm
   #include <common/interfaces/inc/vd_diaglog_inc_comm.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_plugin_if
   #include <common/plugin/vd_diaglog_plugin_if.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_snapshotDataPrj
   #include <project/framework/snapshotData/vd_diaglog_snapshotDataPrj.h>
#endif


#define SCD_S_IMPORT_INTERFACE_GENERIC
#include "scd_if.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DIAGLOG_INFO
#include "trcGenProj/Header/vd_diaglog_main.cpp.trc.h"
#endif

// limit of PluginTimer
#define DIAGLOG_PLUGIN_MAX_NUMBER_OF_TIMER 100

OSAL_tSemHandle vdDiagLog_tclApp::m_hNotTableSem = OSAL_C_INVALID_HANDLE;
OSAL_tSemHandle vdDiagLog_tclApp::m_hReportMemSem = OSAL_C_INVALID_HANDLE;
OSAL_tSemHandle vdDiagLog_tclApp::m_hASFSem = OSAL_C_INVALID_HANDLE;

// static handles


OSAL_tTimerHandle vdDiagLog_tclApp::m_hQualificationTimer = OSAL_C_INVALID_HANDLE;
OSAL_tTimerHandle vdDiagLog_tclApp::m_hPlugInTimer = OSAL_C_INVALID_HANDLE;
OSAL_tTimerHandle vdDiagLog_tclApp::m_hEventHandle = OSAL_C_INVALID_HANDLE;
// instance pointer for trace
vdDiagLog_tclApp*   vdDiagLog_tclApp::m_poInstance = NULL;
vdDiagLog_tclApp*   vdDiagLog_tclApp::_theServer = NULL;

bool vdDiagLog_tclApp::m_fTerminated = false;
bool vdDiagLog_tclApp::m_isPluginTimerRunning = false;

tS32                                         vdDiagLog_tclApp::m_s32TimerId = 0;
vdDiagLog_tclApp::tTimerRegistrationList     vdDiagLog_tclApp::m_oTimerRegList;


void
releaseInstanceOfVdDiagLog_tclApp (void )
{
  return vdDiagLog_tclApp::deleteInstance();
}

void
vdDiagLog_tclApp::deleteInstance ( void )
{
   if ( _theServer )
   {
      delete _theServer;
      _theServer = 0;
   }
}

/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     vdDiagLog_tclApp::theServer
 *
 *DESCRIPTION:  create new server object
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  vdDiagLog_tclApp* _theServer
 *
 *HISTORY:
 *27.01.07 kol2hi
 *         Initial version
 *
 ******************************************************************-FUNCHEADER*/
vdDiagLog_tclApp*
vdDiagLog_tclApp::theServer()
{
   ETG_TRACE_USR3(( "--> vdDiagLog_tclApp::theServer"));
   if (_theServer == NULL)
   {
      _theServer = new vdDiagLog_tclApp();
   }
   ETG_TRACE_USR3(( "<-- vdDiagLog_tclApp::theServer"));
   return _theServer;
}


/*************************************************************************
*
* FUNCTION: vdDiagLog_tclApp::vdDiagLog_tclApp()
*
* DESCRIPTION:
*
*  Constructor: - Initialize AMT, create Sem, create notification table
*
*
* PARAMETER: void
*
* RETURNVALUE: none
*
*************************************************************************/
vdDiagLog_tclApp::vdDiagLog_tclApp() :
   m_fdTrace(OSAL_ERROR),
   m_poTraceHandler(NULL),
   m_threadID(OSAL_ERROR),
   m_poNotTable(NULL),
   m_poService(NULL),
   m_poSpmClientHandler(NULL),
   m_poIncInterface(NULL),
   m_poSnapShotData(NULL)
{

   ETG_TRACE_USR3_THR(( "--> vdDiagLog_tclApp::vdDiagLog_tclApp"));

   // initialise CCA notification arrays
   (tVoid)OSAL_pvMemorySet(m_aFuncArray, 0, sizeof(ahl_tFunction) * DIAGLOG_FUNCARRAY_SIZE);
   (tVoid)OSAL_pvMemorySet(m_aNotArray, 0, sizeof(ahl_tNotification) * DIAGLOG_NOTARRAY_SIZE);
   ETG_TRACE_USR3_THR(( "<-- vdDiagLog_tclApp::vdDiagLog_tclApp"));
}

/*************************************************************************
*
* FUNCTION: vdDiagLog_tclApp::~vdDiagLog_tclApp()
*
* DESCRIPTION: destructor, frees all ressources
*             ( if not done by vOnApplicationClose before )
*
* PARAMETER: void
*
* RETURNVALUE: none
*
*************************************************************************/
vdDiagLog_tclApp::~vdDiagLog_tclApp()
{
   ETG_TRACE_USR3_THR(( "--> vdDiagLog_tclApp::~vdDiagLog_tclApp"));
   m_poTraceHandler = NULL;
   m_poService = NULL;
   m_poSpmClientHandler = NULL;
   m_poSnapShotData = NULL;
   m_poNotTable = NULL;
   m_poIncInterface = 0;
   ETG_TRACE_USR3_THR(( "<--  vdDiagLog_tclApp::~vdDiagLog_tclApp"));
}

/*************************************************************************
*
* FUNCTION: tBool vdDiagLog_tclApp::bOnInit ()
*
* DESCRIPTION: is called after the initialisation of the framework is finished
*
* PARAMETER: void
*
* RETURNVALUE: TRUE
*
*************************************************************************/
tBool vdDiagLog_tclApp::bOnInit()
{
   ETG_TRACE_USR3_THR(( "-->  vdDiagLog_tclApp::bOnInit()"));

   /**************************************************************************/
   // STEP 1
   // On Init start do initalising now
   /**************************************************************************/
   tS32 s32Result = OSAL_s32SemaphoreCreate(DIAGLOG_ASF_SEM_NAME, &m_hASFSem, 1);
   if(OSAL_OK != s32Result)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      ETG_TRACE_ERRMEM(("!!! Diaglog: vdDiagLog_tclApp::bOnInit => OSAL_s32SemaphoreCreate DIAGLOG_ASF_SEM_NAME failed: Result=%x, Error=%x",s32Result, static_cast<tUInt>(u32ErrorCode)));
      NORMAL_M_ASSERT_ALWAYS();
      m_hASFSem = OSAL_C_INVALID_HANDLE;
   }
   ScopedLockSemaphore lock(m_hASFSem);

   //initialize the AMT/CCA framework for this component
   (tVoid) amt_bInit();

   m_poInstance = this;

   ETG_TRACE_USR1_THR(( "--- vdDiagLog_tclApp::bOnInit => Create Semaphore m_hReportMemSem"));
   // Create semaphore to avoid parallel processing
   s32Result = OSAL_s32SemaphoreCreate(DIAGLOG_REPORTMEM_SEM_NAME, &m_hReportMemSem, 1);
   if(OSAL_OK != s32Result)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      ETG_TRACE_ERRMEM(("!!! Diaglog: vdDiagLog_tclApp::bOnInit => OSAL_s32SemaphoreCreate DIAGLOG_REPORTMEM_SEM_NAME failed: Result=%x, Error=%x",s32Result, static_cast<tUInt>(u32ErrorCode)));
      NORMAL_M_ASSERT_ALWAYS();
      m_hReportMemSem = OSAL_C_INVALID_HANDLE;
   }



   ETG_TRACE_USR1_THR(( "--- vdDiagLog_tclApp::bOnInit => Create Semaphore m_hNotTableSem"));
   //create semaphore for protecting NotificationTable
   s32Result = OSAL_s32SemaphoreCreate(DIAGLOG_NOTTABLE_SEM_NAME, &m_hNotTableSem, 1);
   if(OSAL_OK != s32Result)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      ETG_TRACE_ERRMEM(("!!! Diaglog: vdDiagLog_tclApp::bOnInit => OSAL_s32SemaphoreCreate DIAGLOG_NOTTABLE_SEM_NAME failed: Result=%x, Error=%x",s32Result, static_cast<tUInt>(u32ErrorCode)));
      NORMAL_M_ASSERT_ALWAYS();
      m_hNotTableSem = OSAL_C_INVALID_HANDLE;
   }

   ETG_TRACE_USR1_THR(( "--- vdDiagLog_tclApp::bOnInit => Create NotificationTable"));
   //create NotificationTable.
   //Here we store information about all registered clients
   m_poNotTable = OSAL_NEW ahl_tclNotificationTable(
                                                   m_aFuncArray,
                                                   DIAGLOG_FUNCARRAY_SIZE,
                                                   m_aNotArray,
                                                   DIAGLOG_NOTARRAY_SIZE
                                                   );
   if(OSAL_NULL == m_poNotTable)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      ETG_TRACE_ERRMEM(("!!! Diaglog: vdDiagLog_tclApp::bOnInit => OSAL_NEW ahl_tclNotificationTable failed: Error=%x", static_cast<tUInt>(u32ErrorCode)));
      NORMAL_M_ASSERT_ALWAYS();
   }

   // create Timer
   // QUALIFICATION create timer
   ETG_TRACE_USR1_THR(( "--- vdDiagLog_tclApp::bOnInit => Create QUALIFICATION-TIMER m_hQualificationTimer"));
   s32Result = OSAL_s32TimerCreate(vQualificationTimerHandler, (tPVoid)this, &m_hQualificationTimer);
   if(OSAL_ERROR == s32Result)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      ETG_TRACE_ERRMEM(("!!! vdDiagLog_tclApp::bOnInit => ERROR: OSAL_s32TimerCreate m_hQualificationTimer FAILED: Error=%x", static_cast<tUInt>(u32ErrorCode)));
      NORMAL_M_ASSERT_ALWAYS();
      return FALSE;
   }

   // PLUGIN create timer
   ETG_TRACE_USR1_THR(( "--- vdDiagLog_tclApp::bOnInit => Create PLUGIN-TIMER m_hPlugInTimer"));
   s32Result = OSAL_s32TimerCreate(vTriggerTimerHandler, (tPVoid)this, &m_hPlugInTimer);
   if(OSAL_ERROR == s32Result)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      ETG_TRACE_ERRMEM(("!!! vdDiagLog_tclApp::bOnInit => ERROR: OSAL_s32TimerCreate m_hPlugInTimer FAILED: Error=%x", static_cast<tUInt>(u32ErrorCode)));
      NORMAL_M_ASSERT_ALWAYS();
   }

   ETG_TRACE_USR1_THR(( "--- vdDiagLog_tclApp::bOnInit => Create Thread and Events"));
   // create thread for timer quali
   if(false == bThreadSetup())
   {
      ETG_TRACE_ERRMEM(("!!! vdDiagLog_tclApp::bOnInit => ERROR: bThreadSetup"));
      NORMAL_M_ASSERT_ALWAYS();
   }

   m_oBlockingModeHandler.bOnInit();

   ETG_TRACE_USR1_THR(( "--- vdDiagLog_tclApp::bOnInit => Create vdl_nsReportMemoryFactory"));
   _BP_TRY_BEGIN
   {
      // create report memory object
       vCreateMemory();
   }
   _BP_CATCH_ALL
   {
      ETG_TRACE_ERRMEM(("!!! vdDiagLog_tclApp::bOnInit => ERROR: _BP_CATCH_ALL"));
      NORMAL_M_ASSERT_ALWAYS();
   }
   _BP_CATCH_END

   ETG_TRACE_USR1_THR(( "--- vdDiagLog_tclApp::bOnInit => Create Prj Configuration"));
   if(!bCreatePrjSpecificConfiguration(this))
   {
      ETG_TRACE_ERRMEM(("!!! vdDiagLog_tclApp::bOnInit => ERROR: bCreatePrjSpecificConfiguration"));
      NORMAL_M_ASSERT_ALWAYS();
      return FALSE;
   }

   ETG_TRACE_USR1_THR(( "--- vdDiagLog_tclApp::bOnInit => vProcessOnInit"));
   m_oMemoryMaster.vProcessOnInit();


   ETG_TRACE_USR1_THR(( "--- vdDiagLog_tclApp::bOnInit => Create INTERFACE vdDiaglog_tclclienthandlerspm"));
   m_poSpmClientHandler = OSAL_NEW vdDiaglog_tclclienthandlerspm(this);
   if(NULL == m_poSpmClientHandler)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      ETG_TRACE_ERRMEM(("!!! vdDiagLog_tclApp::bOnInit => ERROR: OSAL_NEW vdDiaglog_tclclienthandlerspm FAILED Error=%x", static_cast<tUInt>(u32ErrorCode)));
      NORMAL_M_ASSERT_ALWAYS();
   }

   ETG_TRACE_USR1_THR(( "--- vdDiagLog_tclApp::bOnInit => Create INTERFACE vdDiagLog_tclService"));
   //create the cca service handlers.
   m_poService = OSAL_NEW vdDiagLog_tclService(this);
   if(NULL == m_poService)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      ETG_TRACE_ERRMEM(("!!! vdDiagLog_tclApp::bOnInit => ERROR: OSAL_NEW vdDiagLog_tclService FAILED Error=%x", static_cast<tUInt>(u32ErrorCode)));
      NORMAL_M_ASSERT_ALWAYS();
   }
   //create the spm cca service handlers.


   ETG_TRACE_USR1_THR(( "--- vdDiagLog_tclApp::bOnInit => Create INTERFACE vdl_tclIncCommunication"));
   m_poIncInterface = OSAL_NEW vdl_tclIncCommunication(this);
   if(NULL == m_poIncInterface)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      ETG_TRACE_ERRMEM(("!!! vdDiagLog_tclApp::bOnInit => ERROR: OSAL_NEW vdl_tclIncCommunication FAILED Error=%x", static_cast<tUInt>(u32ErrorCode)));
      NORMAL_M_ASSERT_ALWAYS();
   }
   else
   {
      if(false == m_poIncInterface->create())
      {
         ETG_TRACE_ERRMEM(("DIAGLOG => INC-Interface not created, no V850 DTC's will be available"));
         NORMAL_M_ASSERT_ALWAYS();
      }
      else if(false == m_poIncInterface->connect())
      {
         ETG_TRACE_ERRMEM(("DIAGLOG => INC-Interface not connect, no V850 DTC's will be available"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }

   ETG_TRACE_USR1_THR(( "--- vdDiagLog_tclApp::bOnInit => Create  vdl_tclTrace"));
   m_poTraceHandler = new vdl_tclTrace(m_poInstance);
   if(NULL == m_poTraceHandler)
   {
      ETG_TRACE_ERRMEM(( "!!! vdDiagLog_tclApp::bOnInit =>  m_poTraceHandler FAILED"));
      NORMAL_M_ASSERT_ALWAYS();
   }
   vRegisterTraceService();

   ETG_TRACE_USR1_THR(( "--- vdDiagLog_tclApp::bOnInit => Create SnapShotData"));
   #ifdef DIAGLOG_USE_SNAPSHOT_DATA
      m_poSnapShotData = new vdl_tclSnapshotDataPrj;
   #endif

   /**************************************************************************/
   // STEP 2
   // On Init finshed (everthing load and initalized)
   // beginn now with Operation Cycle Start
   /**************************************************************************/
   ETG_TRACE_USR1_THR(( "--- vdDiagLog_tclApp::bOnInit => vProcessOperatingCycleStart"));

   m_oMemoryMaster.processSystemEvent(DIAGLOG_EVENT_OPERATION_CYCLE_START);

  		 // DM1 Message Cycle Start
  		 {
  			 vStartDm1MessageOperatingCycle();
  		 }

   ETG_TRACE_USR3_THR(("<--- vdDiagLog_tclApp::bOnInit()"));
   return  TRUE;
}


tVoid vdDiagLog_tclApp::vCreateMemory()
{
   ETG_TRACE_USR3_THR(( "--> vdDiagLog_tclApp::poCreateMemory "));

   m_oMemoryMaster.vCreateMemorys();

   ETG_TRACE_USR3_THR(( "<-- vdDiagLog_tclApp::poCreateMemory "));
}

tVoid vdDiagLog_tclApp::vDestroyMemory()
{
   ETG_TRACE_USR3_THR(( "--> vdDiagLog_tclApp::vDestroyMemory "));

   m_oMemoryMaster.vDestroyMemorys();

   ETG_TRACE_USR3_THR(( "<-- vdDiagLog_tclApp::vDestroyMemory "));
}

/*************************************************************************
*
* FUNCTION: tVoid vdDiagLog_tclApp::vOnServiceState
*
* DESCRIPTION: ???
*
* PARAMETER:
*
* RETURNVALUE:
*
*************************************************************************/
tVoid vdDiagLog_tclApp::vOnServiceState(
                                          tU16 u16ServiceId,
                                          tU16 u16ServerId,
                                          tU16 u16RegisterId,
                                          tU8  u8ServiceState,
                                          tU16 u16SubId
                                       )
{

   ETG_TRACE_USR3_THR(("--> vdDiagLog_tclApp::vOnServiceState"));

   switch ( u16ServiceId )
   {
      case CCA_C_U16_SRV_SPM:
      {
         if(m_poSpmClientHandler != OSAL_NULL)
         {
              m_poSpmClientHandler->vOnServiceState(  u16ServiceId,
                                                      u16ServerId,
                                                      u16RegisterId,
                                                      u8ServiceState,
                                                      u16SubId);
         }
         break;
      }

      default:
      {
         ETG_TRACE_ERRMEM(( "!!! vdDiagLog_tclApp::vOnServiceState =>  u16ServiceId:%x UNKNOWN",u16ServiceId));
         NORMAL_M_ASSERT_ALWAYS();
         break;
      }
   }

   ail_tclAppInterface::vOnServiceState(u16ServiceId, u16ServerId, u16RegisterId, u8ServiceState, u16SubId);
   ETG_TRACE_USR3_THR(("<-- vdDiagLog_tclApp::vOnServiceState"));
}



/*************************************************************************
*
* FUNCTION: tVoid vdDiagLog_tclApp::vOnNewAppState (tU32 u32OldAppState, tU32 u32AppState)
*
* DESCRIPTION: * is call from SPM to inform this application about new power states
*              * changes the state of the application
*              * sends ServiceAvailability state to all notified clients
*              * changes state of downloadservice(-object)
*
* PARAMETER: tU32 u32OldAppState (actual state),
*            tU32 u32AppState (state to change to)
*
* RETURNVALUE: void
*
you have to implement the behavior concerning power states here
*************************************************************************/
tVoid vdDiagLog_tclApp::vOnNewAppState (tU32 u32OldAppState, tU32 u32AppState)
{

   ETG_TRACE_USR3_THR(( "-->  vdDiagLog_tclApp::vOnNewAppState. Old state: %u, New State: %u", u32OldAppState, u32AppState));


   /* ---
   analyse the state we have to go to
   --- */
   switch(u32AppState)
   {
      case AMT_C_U32_STATE_NORMAL:
      {
         /* ---
         the server has already entered this state, so it has nothing to do
         --- */
         if(u32OldAppState == AMT_C_U32_STATE_NORMAL)
         {
            break; //switching to same state
         }

         /* +++
         do everything you have to do to enter the state NORMAL

         the state NORMAL is enter on system start and when the system leaves the states PAUSE and DIAG
         so if you want to do something only once, don't do it here!!!!

         +++ */

         /* ---
         inform the framework that the we are ready to receive and send messages
         --- */
         vServiceAvailabilityChanged ( CCA_C_U16_SRV_DIAGLOG, AMT_C_U8_SVCSTATE_AVAILABLE);
         break;
      }

      case AMT_C_U32_STATE_DIAGNOSIS:
      {
      /* ---
         the server has already entered this state, so it has nothing to do
         --- */
         if (u32OldAppState == AMT_C_U32_STATE_DIAGNOSIS)
         {
            break; //switching to same state
         }

         /* +++
         do everything you have to do to enter the state DIAGNOSIS
         +++ */

         /* +++
         inform the framework if you are ready to receive and send messages
         in this state
         +++ */
         vServiceAvailabilityChanged ( CCA_C_U16_SRV_DIAGLOG, AMT_C_U8_SVCSTATE_AVAILABLE);

         break;
      }

      case AMT_C_U32_STATE_PAUSE:
      {
         /* ---
         the server has already entered this state, so it has nothing to do
         --- */
         if (u32OldAppState == AMT_C_U32_STATE_PAUSE)
         {
            break; //switching to same state
         }

         /* +++
         do everything you have to do to enter the state PAUSE
         +++ */

         /* +++
         inform the framework if you are ready to receive and send messages
         in this state
         +++ */
         vServiceAvailabilityChanged ( CCA_C_U16_SRV_DIAGLOG, AMT_C_U8_SVCSTATE_NOT_AVAILABLE);

         break;
      }

      case AMT_C_U32_STATE_OFF:
      {
         ETG_TRACE_USR4(( "------ AMT_C_U32_STATE_OFF received - start"));
         /* ---
         the server has already entered this state, so it has nothing to do
         --- */
         if (u32OldAppState == AMT_C_U32_STATE_OFF)
         {
            break; //switching to same state
         }

         ETG_TRACE_USR4(( "------ AMT_C_U32_STATE_OFF received - before vProcessOperatingCycleEnd"));
         // if cycle end call needed and report memory is allocated
         m_oMemoryMaster.processSystemEvent(DIAGLOG_EVENT_OPERATION_CYCLE_END);

         ETG_TRACE_USR4(( "------ AMT_C_U32_STATE_OFF received - After vProcessOperatingCycleEnd"));

         /* ---
         inform the framework that we are not ready to received or send messages
         --- */
         vServiceAvailabilityChanged ( CCA_C_U16_SRV_DIAGLOG, AMT_C_U8_SVCSTATE_NOT_AVAILABLE);

         break;
      }

      default:
      {
         /* +++
         this should never happen, you can call a trace or assert here
         +++ */
      }
   }// switch(u32AppState)

   /* ---
   inform the message handlers about the new state
   --- */
   if(m_poService != OSAL_NULL)
   {
      m_poService->vOnNewAppState(u32OldAppState, u32AppState);
   }
   if(m_poSpmClientHandler != OSAL_NULL)
   {
      m_poSpmClientHandler->vOnNewAppState(u32OldAppState, u32AppState);
   }

   /* ---
   create and send an acknowledge message so the SPM knows you have reached the new state
   --- */
   vAppStateChanged (u32AppState, 0);

   ETG_TRACE_USR3_THR(( "<--  vdDiagLog_tclApp::vOnNewAppState"));
}


/*************************************************************************
*
* FUNCTION: tVoid vdDiagLog_tclApp::vOnNewMessage (amt_tclBaseMessage* poMessage)
*
* DESCRIPTION: * casts Message to amt_tclServiceData (if legal)
*              * gives amt_tclServiceData - Message to service for analyzing
*              * ignores all unexpected messsages
*
* PARAMETER: amt_tclBaseMessage* poMessage ( Message to analyse)
*
* RETURNVALUE: void
*
*************************************************************************/
tVoid vdDiagLog_tclApp::vOnNewMessage (amt_tclBaseMessage* poMessage)
{

   ETG_TRACE_USR3_THR(("--> vdDiagLog_tclApp::vOnNewMessage"));

   ETG_TRACE_USR2_THR(("--- vdDiagLog_tclApp::vOnNewMessage => Type:%d",poMessage->u8GetType()));

   NORMAL_M_ASSERT(m_poService != NULL);

   if(NULL != poMessage)
   {
      /* ---
      check if the received message is a AMT Service data message
      --- */
      if(poMessage->u8GetType() == AMT_C_U8_CCAMSGTYPE_SVCDATA)
      {
         /* ---
         cast the received message to a AMT Service data message so we
         can access the standard members
         --- */
         amt_tclServiceData oServiceData(poMessage);

         tU16 u16ServiceID = oServiceData.u16GetServiceID();

        ETG_TRACE_USR4(("--- vdDiagLog_tclApp::vOnNewMessage SourceAppID = %d, Fid = %d, Opcode = %d", oServiceData.u16GetSourceAppID(), oServiceData.u16GetFunctionID(), oServiceData.u8GetOpCode() ));

         /* ---
         one server can offer more than one service. So we analyse the ServiceID
         to know which service handler we have to call.
         99% of all Servers only offer one service ( with a different function ID for each feature ).
         --- */
         switch(u16ServiceID)
         {
            case CCA_C_U16_SRV_DIAGLOG:
            {
               /* ---
               give the message to your cca service handler for analysing.
               vDispatchMessage is a framework function. it will compare
               the function ID with your message map an call the corresponding
               function
               ---*/

               if(m_poService)
               {
                  m_poService->vDispatchMessage(&oServiceData);
               }
               break;
            }

            case CCA_C_U16_SRV_SPM:
            {
               /* ---
               give the message to your cca service handler for analysing.
               vDispatchMessage is a framework function. it will compare
               the function ID with your message map an call the corresponding
               function
               ---*/

               if(m_poSpmClientHandler)
               {
                  m_poSpmClientHandler->vDispatchMessage(&oServiceData);
               }
               break;
            }

            default:
            {
               /* +++
               we have received a message with an unknown service ID.
               this is an error so make a trace or assert here
               +++ */
               break;
            }
         }
      }
      else
      {
         /* +++
         we have received a message with an unknown format.
         this is an error so make a trace or assert here
         +++ */

         /* ---
         we can't do anything with this message so free the ressources
         --- */
         (tVoid) poMessage->bDelete();
      }
   }// if(NULL != poMessage)
   ETG_TRACE_USR3_THR(("<-- vdDiagLog_tclApp::vOnNewMessage"));
}

/*************************************************************************
*
* FUNCTION: tBool vdDiagLog_tclApp::bGetServiceVersion (tU16 u16ServiceID, tU16& rfu16MajorVersion, tU16& rfu16MinorVersion, tU16& rfu16PatchVersion)
*
* DESCRIPTION: stores data about the version of this service (magics)
*
* PARAMETER: * u16ServiceID: the service we should describe
*            * rfu16MajorVersion, rfu16MinorVersion, rfu16PatchVersion: targets for storing the data
*
* RETURNVALUE: success: true / false
*
*************************************************************************/
tBool vdDiagLog_tclApp::bGetServiceVersion (tU16 u16ServiceID, tU16& rfu16MajorVersion, tU16& rfu16MinorVersion, tU16& rfu16PatchVersion)
{
   ETG_TRACE_USR3_THR(( "-->  vdDiagLog_tclApp::bGetServiceVersion"));
   tBool bReturn = false;

   if(u16ServiceID == CCA_C_U16_SRV_DIAGLOG)
   {
      /* +++
      Enter the version of this service you offer
      +++ */

      rfu16MajorVersion = MIDW_DIAGLOGFI_C_U16_SERVICE_MAJORVERSION;
      rfu16MinorVersion = MIDW_DIAGLOGFI_C_U16_SERVICE_MINORVERSION;
      rfu16PatchVersion = 0;

      bReturn = TRUE;
   }
   else
   {
      /* +++
      we have received a request with an unknown service ID.
      this is an error so make a trace or assert here
      +++ */
   }
   ETG_TRACE_USR3_THR(( "<--  vdDiagLog_tclApp::bGetServiceVersion"));

   return bReturn;
}

/************************************************************************
*
* FUNCTION: tBool vdDiagLog_tclApp::bOnWatchdog()
*
* DESCRIPTION: ????
*
* PARAMETER: void
*
* RETURNVALUE: true
*
*************************************************************************/
tBool vdDiagLog_tclApp::bOnWatchdog()
{
   return TRUE;
}

/*************************************************************************
*
* FUNCTION: tVoid vdDiagLog_tclApp::vClose()
*
* DESCRIPTION: releases all resources and sends close - message
*
* PARAMETER: void
*
* RETURNVALUE: void
*
*************************************************************************/
tVoid vdDiagLog_tclApp::vClose()
{
   ETG_TRACE_USR3_THR(( "-->  vdDiagLog_tclApp::vClose"));

   LockMemorySemaphore();
   {

	   // DM1 Message Cycle End
	   		 {
	   			 vEndDm1MessageOperatingCycle();
	   		 }
      vStopPluginTimer();
      // kill timers
      ETG_TRACE_USR1_THR(( "--- vdDiagLog_tclApp::vClose => Close QUALIFICATION-TIMER m_hQualificationTimer =%x ",m_hQualificationTimer));
      tS32 s32Result = OSAL_s32TimerDelete(m_hQualificationTimer);
      if(OSAL_ERROR == s32Result)
      {
         tU32 u32ErrorCode= OSAL_u32ErrorCode();
         ETG_TRACE_ERRMEM(("!!! vdDiagLog_tclApp::vClose => ERROR: OSAL_s32TimerDelete m_hQualificationTimer FAILED: Error=%x", static_cast<tUInt>(u32ErrorCode)));
         NORMAL_M_ASSERT_ALWAYS();
      }

      if(m_poIncInterface != NULL)
      {
         if(false == m_poIncInterface->disconnect())
         {
            ETG_TRACE_ERRMEM(( "!!! vdDiagLog_tclApp::vClose =>  m_poIncInterface.disconnect FAILED"));
            NORMAL_M_ASSERT_ALWAYS();
         }
      }

      // Delete the timer thread
      // end thread
      m_fTerminated = true;
      if(m_hEventHandle != OSAL_C_INVALID_HANDLE)
      {
         s32Result = OSAL_s32EventPost(m_hEventHandle, DIAGLOG_EVENT_TERMINATE_MASK, OSAL_EN_EVENTMASK_OR);
         if(OSAL_ERROR == s32Result)
         {
            tU32 u32ErrorCode= OSAL_u32ErrorCode();
            ETG_TRACE_ERRMEM(("!!! vdDiagLog_tclApp::vClose => ERROR: OSAL_s32EventPost DIAGLOG_EVENT_TERMINATE_MASK FAILED: Error=%x", static_cast<tUInt>(u32ErrorCode)));
            NORMAL_M_ASSERT_ALWAYS();
         }
      }

      ETG_TRACE_USR1_THR(( "--- vdDiagLog_tclApp::vClose => Close the Events"));
      // Delete the created events
      if(OSAL_C_INVALID_HANDLE != m_hEventHandle)
      {
         s32Result = OSAL_s32EventClose(m_hEventHandle);
         if(OSAL_ERROR == s32Result)
         {
            tU32 u32ErrorCode= OSAL_u32ErrorCode();
            ETG_TRACE_ERRMEM(("!!! vdDiagLog_tclApp::vClose => ERROR: OSAL_s32EventClose DIAGLOG_EVENT_NAME FAILED: Error=%x", static_cast<tUInt>(u32ErrorCode)));
            NORMAL_M_ASSERT_ALWAYS();
         }

         m_hEventHandle = OSAL_C_INVALID_HANDLE;
         s32Result = OSAL_s32EventDelete(DIAGLOG_EVENT_NAME);
         if(OSAL_ERROR == s32Result)
         {
            tU32 u32ErrorCode= OSAL_u32ErrorCode();
            ETG_TRACE_ERRMEM(("!!! vdDiagLog_tclApp::vClose => ERROR: OSAL_s32EventDelete DIAGLOG_EVENT_NAME FAILED: Error=%x", static_cast<tUInt>(u32ErrorCode)));
            NORMAL_M_ASSERT_ALWAYS();
         }
      }

      // delete report memory object
      vDestroyPrjSpecificConfiguration();
      vDestroyMemory();
   }
   FreeMemorySemaphore();

   ETG_TRACE_USR1_THR(( "--- vdDiagLog_tclApp::vClose => Close the Semaphores"));
   tS32 s32Result = OSAL_s32SemaphoreClose(m_hReportMemSem);
   if(OSAL_OK != s32Result)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      et_vErrmemStringNormal( TR_COMP_DIAGNOSIS, "Diaglog: vdDiagLog_tclApp::vClose => OSAL_s32SemaphoreClose failed: Result=%x, Error=%x",
                               s32Result,
                               u32ErrorCode);
      NORMAL_M_ASSERT_ALWAYS();
   }

   s32Result = OSAL_s32SemaphoreDelete(DIAGLOG_REPORTMEM_SEM_NAME);
   if(OSAL_OK != s32Result)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      et_vErrmemStringNormal( TR_COMP_DIAGNOSIS, "Diaglog: vdDiagLog_tclApp::vClose => OSAL_s32SemaphoreDelete failed: Result=%x, Error=%x",
                               s32Result,
                               u32ErrorCode);
      NORMAL_M_ASSERT_ALWAYS();
   }



   // close and delete notification semaphore
   s32Result = OSAL_s32SemaphoreClose(m_hNotTableSem);
   if(OSAL_OK != s32Result)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      et_vErrmemStringNormal( TR_COMP_DIAGNOSIS, "Diaglog: vdDiagLog_tclApp::vClose => OSAL_s32SemaphoreDelete failed: Result=%x, Error=%x",
                               s32Result,
                               u32ErrorCode);
      NORMAL_M_ASSERT_ALWAYS();
   }

   s32Result = OSAL_s32SemaphoreDelete(DIAGLOG_NOTTABLE_SEM_NAME);
   if(OSAL_OK != s32Result)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      et_vErrmemStringNormal( TR_COMP_DIAGNOSIS, "Diaglog: vdDiagLog_tclApp::vClose => OSAL_s32SemaphoreDelete failed: Result=%x, Error=%x",
                               s32Result,
                               u32ErrorCode);
      NORMAL_M_ASSERT_ALWAYS();
   }

   s32Result = OSAL_s32SemaphoreClose(m_hASFSem);
   if(OSAL_OK != s32Result)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      et_vErrmemStringNormal( TR_COMP_DIAGNOSIS, "Diaglog: vdDiagLog_tclApp::vClose => OSAL_s32SemaphoreDelete failed: Result=%x, Error=%x",
                               s32Result,
                               u32ErrorCode);
      NORMAL_M_ASSERT_ALWAYS();
   }
   s32Result = OSAL_s32SemaphoreDelete(DIAGLOG_ASF_SEM_NAME);
   if(OSAL_OK != s32Result)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      et_vErrmemStringNormal( TR_COMP_DIAGNOSIS, "Diaglog: vdDiagLog_tclApp::vClose => OSAL_s32SemaphoreDelete failed: Result=%x, Error=%x",
                               s32Result,
                               u32ErrorCode);
      NORMAL_M_ASSERT_ALWAYS();
   }
   vUnregisterTraceService();

   // free resources
   // free notification table
   if(OSAL_NULL != m_poNotTable)
   {
      OSAL_DELETE m_poNotTable;
      m_poNotTable = OSAL_NULL;
   }

   OSAL_DELETE m_poService;
   m_poService = NULL;

   OSAL_DELETE m_poSpmClientHandler;
   m_poSpmClientHandler = NULL;

   OSAL_DELETE m_poTraceHandler;
   m_poTraceHandler = NULL;

   OSAL_DELETE m_poIncInterface;
   m_poIncInterface = 0;

   if (m_poSnapShotData != NULL)
   {
      delete m_poSnapShotData;
      m_poSnapShotData = NULL;
   }

   //inform SPM about successful shut down
   vApplicationCloseAcknowledge(AIL_C_U8_APP_END_SUCCESSFUL);

   ETG_TRACE_USR3_THR(( "<--  vdDiagLog_tclApp::vClose"));
}
/*************************************************************************
*
* FUNCTION: tVoid vdDiagLog_tclApp::vOnApplicationClose()
*
* DESCRIPTION: releases all resources and sends close - message
*
* PARAMETER: void
*
* RETURNVALUE: void
*
*************************************************************************/
tVoid vdDiagLog_tclApp::vOnApplicationClose()
{
   ETG_TRACE_USR3_THR(( "--> vdDiagLog_tclApp::vOnApplicationClose"));
   vClose();
   ETG_TRACE_USR3_THR(( "<-- vdDiagLog_tclApp::vOnApplicationClose"));
}


///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: OSAL_tpfCallback vdDiagLog_tclApp::vTimerHandler
//
// DESCRIPTION: timer callback handler (called every one sec for Qualifiation)
//
// PARAMETER: tVoid
//
// RETURNVALUE: void
//
///////////////////////////////////////////////////////////////////////////////////
//
void vdDiagLog_tclApp::vQualificationTimerHandler(tPVoid /*pvArg*/)
{
   if(m_hEventHandle != OSAL_C_INVALID_HANDLE)
   {
      tS32 s32Result = OSAL_s32EventPost(m_hEventHandle, DIAGLOG_EVENT_TIMER_QUALI_MASK, OSAL_EN_EVENTMASK_OR);
      if(OSAL_ERROR == s32Result)
      {
         tU32 u32ErrorCode= OSAL_u32ErrorCode();
         ETG_TRACE_ERRMEM(("!!! vdDiagLog_tclApp::vQualificationTimerHandler => ERROR: OSAL_s32EventPost DIAGLOG_EVENT_TIMER_QUALI_MASK FAILED: Error=%x", static_cast<tUInt>(u32ErrorCode)));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: OSAL_tpfCallback vdDiagLog_tclApp::vTriggerTimerHandler
//
// DESCRIPTION: timer callback handler for Trigger Timer (general Timer for special Timings)
//
// PARAMETER: tVoid
//
// RETURNVALUE: void
//
///////////////////////////////////////////////////////////////////////////////////
//
void vdDiagLog_tclApp::vTriggerTimerHandler(tPVoid /*pvArg*/)
{

   if(m_hEventHandle != OSAL_C_INVALID_HANDLE)
   {
      tS32 s32Result = OSAL_s32EventPost(m_hEventHandle, DIAGLOG_EVENT_TIMER_PLUGIN_STEP_MASK, OSAL_EN_EVENTMASK_OR);
      if(OSAL_ERROR == s32Result)
      {
         tU32 u32ErrorCode= OSAL_u32ErrorCode();
         ETG_TRACE_ERRMEM(("!!! vdDiagLog_tclApp::vTriggerTimerHandler => ERROR: OSAL_s32EventPost DIAGLOG_EVENT_TIMER_PLUGIN_STEP_MASK FAILED: Error=%x", static_cast<tUInt>(u32ErrorCode)));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdDiagLog_tclApp::vRegisterTraceService
//
// DESCRIPTION:  Registers trace channel to send commands from TTFis
//
// PARAMETER:    None
//
// RETURNVALUE:  None
//
///////////////////////////////////////////////////////////////////////////////////

tVoid vdDiagLog_tclApp::vRegisterTraceService()
{
   ETG_TRACE_USR3_THR(( "--> vdDiagLog_tclApp::vRegisterTraceService"));
   tU32                        u32Error;
   OSAL_trIOCtrlLaunchChannel  oTraceChannel;

   m_fdTrace = OSAL_IOOpen(OSAL_C_STRING_DEVICE_TRACE, OSAL_EN_READWRITE);

   oTraceChannel.enTraceChannel = TR_TTFIS_DIAGNOSIS;
   oTraceChannel.pCallback = (OSAL_tpfCallback)vChannelMgr;

   u32Error = (tU32)OSAL_s32IOControl(m_fdTrace, OSAL_C_S32_IOCTRL_CALLBACK_REG, (intptr_t)&oTraceChannel);
   if(OSAL_OK != u32Error)
   {
      ETG_TRACE_ERR_THR(( "!!! vdDiagLog_tclApp::vRegisterTraceService => OSAL_s32IOControl FAILED"));
   }
   DIA_PARAMETER_INTENTIONALLY_UNUSED( oTraceChannel ); // LINT G3G don't accept this, so do this dummy
   ETG_TRACE_USR3_THR(( "<-- vdDiagLog_tclApp::vRegisterTraceService"));
}

///////////////////////////////////////////////////////////////////////////////////
//
//FUNCTION:     vdDiagLog_tclApp::vChannelMgr
//
//DESCRIPTION:  React on TTFis commands
//
//PARAMETER:    None
//
//RETURNVALUE:  None
//
//
/////////////////////////////////////////////////////////////////////////////////////
tVoid vdDiagLog_tclApp::vChannelMgr( tU8 const* puchData )
{
   m_poInstance->m_poTraceHandler->vTraceRx(puchData);
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdDiagLog_tclApp::vUnregisterTraceService
//
// DESCRIPTION:  Unregisters trace channel
//
// PARAMETER:    None
//
// RETURNVALUE:  None
//
///////////////////////////////////////////////////////////////////////////////////
tVoid vdDiagLog_tclApp::vUnregisterTraceService( )
{
   tU32 u32Error;
   OSAL_trIOCtrlLaunchChannel oTraceChannel;

   oTraceChannel.enTraceChannel = TR_TTFIS_DIAGNOSIS;
   oTraceChannel.pCallback      = (OSAL_tpfCallback)vChannelMgr;

   u32Error = (tU32)OSAL_s32IOControl(m_fdTrace, OSAL_C_S32_IOCTRL_CALLBACK_UNREG, (intptr_t)&oTraceChannel);
   if( OSAL_OK != u32Error )
   {
      ETG_TRACE_ERR_THR(( "!!! vdDiagLog_tclApp::vRegisterTraceService => OSAL_s32IOControl FAILED"));
      return;
   }
   m_fdTrace = OSAL_s32IOClose(m_fdTrace);

   DIA_PARAMETER_INTENTIONALLY_UNUSED( oTraceChannel ); // LINT G3G don't accept this, so do this dummy
}
///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdDiagLog_tclApp::bThreadSetup
//
// DESCRIPTION:  creates and starts timer thread
//
// PARAMETER:    None
//
// RETURNVALUE:  true - success
//
///////////////////////////////////////////////////////////////////////////////////
//

bool vdDiagLog_tclApp::bThreadSetup()
{
   bool result = true;
   OSAL_trThreadAttribute  rThAttr;

   rThAttr.szName        = const_cast<tString>("DIALOG_R");
   rThAttr.pfEntry       = (OSAL_tpfThreadEntry)vTimerThread;
   rThAttr.pvArg         = (void*)this;
   rThAttr.u32Priority   = DIAGLOG_REG_THREAD_DEFAULT_PRIO;
   rThAttr.s32StackSize  = DIAGLOG_REG_THREAD_DEFAULT_STACKSIZE;

   tU32 u32Value ;

   if ( !scd_bGetAppConfigurationValue(CCA_C_U16_APP_DIAGLOG, DIAGLOG_REG_THREAD_PATH, DIAGLOG_REG_THREAD_STACK_SIZE, &u32Value))
   {
      ETG_TRACE_FATAL_THR(( "!!! vdDiagLog_tclApp::bThreadSetup => DIAGLOG_REG_THREAD_STACK_SIZE FAILED"));
   }
   else
   {
      rThAttr.s32StackSize = (tS32)u32Value;
   }

   if ( !scd_bGetAppConfigurationValue(CCA_C_U16_APP_DIAGLOG, DIAGLOG_REG_THREAD_PATH, DIAGLOG_REG_THREAD_PRIO, &u32Value))
   {
      ETG_TRACE_FATAL_THR(( "!!! vdDiagLog_tclApp::bThreadSetup => DIAGLOG_REG_THREAD_PRIO FAILED"));
   }
   else
   {
      rThAttr.u32Priority = u32Value;
   }

   //create timer event
   if(OSAL_s32EventCreate(DIAGLOG_EVENT_NAME, &m_hEventHandle) == OSAL_ERROR)
   {
       result = false;
   }

   m_fTerminated = false;
    // And spawn it from the OS
   m_threadID = OSAL_ThreadSpawn(&rThAttr);

   if(OSAL_ERROR == m_threadID)
   {
      ETG_TRACE_FATAL_THR(( "!!! vdDiagLog_tclApp::bThreadSetup => m_threadID FAILED Size:%d Prio:%d",rThAttr.s32StackSize,rThAttr.u32Priority));
      result = false;
   }

   return result;
}



///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdDiagLog_tclApp::vTimerThread
//
// DESCRIPTION:  worker thread
//
// PARAMETER:    void*
//
// RETURNVALUE:  void
//
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdDiagLog_tclApp::vTimerThread(tVoid *pVoid)
{
   OSAL_tEventMask hEvRequest = 0;
   tU32 u32Mask = DIAGLOG_EVENT_TIMER_QUALI_MASK |
                  DIAGLOG_EVENT_TERMINATE_MASK |
                  DIAGLOG_EVENT_TRIGGER_BLOCKINGMODE_MASK |
                  DIAGLOG_EVENT_TIMER_PLUGIN_STEP_MASK |
                  DIAGLOG_EVENT_TIMER_STORAGE_MASK |
                  DIAGLOG_EVENT_TRIGGER_UPDATE_AFTER_BM;
   while (!m_fTerminated)
   {
      if(vdDiagLog_tclApp::m_hEventHandle != OSAL_C_INVALID_HANDLE)
      {
         if( OSAL_OK == OSAL_s32EventWait (vdDiagLog_tclApp::m_hEventHandle,
                                           u32Mask,
                                           OSAL_EN_EVENTMASK_OR,
                                           OSAL_C_TIMEOUT_FOREVER,
                                           &hEvRequest))
         {
            if(vdDiagLog_tclApp::m_hEventHandle != OSAL_C_INVALID_HANDLE)
            {
               // clear events
               if(OSAL_s32EventPost(vdDiagLog_tclApp::m_hEventHandle, ~hEvRequest, OSAL_EN_EVENTMASK_AND) != OSAL_OK)
               {
                  tU32 u32ErrorCode= OSAL_u32ErrorCode();
                  ETG_TRACE_ERRMEM(("!!! Diaglog: vdDiagLog_tclApp::vTimerThread => OSAL_s32EventPost m_hEventHandle FAILED: Error=%x", static_cast<tUInt>(u32ErrorCode)));
                  NORMAL_M_ASSERT_ALWAYS();
               }

               if((hEvRequest & DIAGLOG_EVENT_TERMINATE_MASK)  == DIAGLOG_EVENT_TERMINATE_MASK)
               {
                  break;
               }

               if((hEvRequest & DIAGLOG_EVENT_TRIGGER_UPDATE_AFTER_BM)  == DIAGLOG_EVENT_TRIGGER_UPDATE_AFTER_BM)
               {
                  ETG_TRACE_USR2_THR(( "--- vdDiagLog_tclApp::vTimerThread => DIAGLOG_EVENT_TRIGGER_UPDATE_AFTER_BM"));
                  if(m_poInstance != NULL)
                  {
                     m_poInstance->vSetResetBlockingModeEventUpdate();
                  }
               }

               // Storage Timer (every DIAGLOG_STORAGE_STEP_TIME) permanently
               if((hEvRequest & DIAGLOG_EVENT_TIMER_STORAGE_MASK)  == DIAGLOG_EVENT_TIMER_STORAGE_MASK)
               {
                  ETG_TRACE_USR2_THR(( "--- vdDiagLog_tclApp::vTimerThread => DIAGLOG_EVENT_TIMER_STORAGE_MASK"));
                  LockMemorySemaphore();
                  {
                     vdDiagLog_tclApp* pApp = static_cast<vdDiagLog_tclApp*>(pVoid);
                     if(pApp != NULL)
                     {
                        pApp->m_oStorageHandler.updateAfterStorageTimer();
                     }// if(  (pApp != NULL) ...
                  }
                  FreeMemorySemaphore();
               }// if((hEvRequest & DIAGLOG_EVENT_TIMER_STORAGE_MASK)  == DIAGLOG_EVENT_TIMER_STORAGE_MASK)

               //Qualification Timer (every DIAGLOG_QUALIFICATION_STEP_TIME) only on demand
               if((hEvRequest & DIAGLOG_EVENT_TIMER_QUALI_MASK)  == DIAGLOG_EVENT_TIMER_QUALI_MASK)
               {
                  // on entry, get the semaphore
                  ETG_TRACE_USR2_THR(( "--- vdDiagLog_tclApp::vTimerThread => DIAGLOG_EVENT_TIMER_QUALI_MASK"));
                  LockMemorySemaphore();
                  {
                     vdDiagLog_tclApp* pApp = static_cast<vdDiagLog_tclApp*>(pVoid);
                     if(pApp != NULL)
                     {
                        // do time qualification
                        pApp->m_oMemoryMaster.m_oQualificationList.processQualificationStep();
                     }// if(  (pApp != NULL) ...
                  }
                  FreeMemorySemaphore();


               }// if((hEvRequest & DIAGLOG_EVENT_TIMER_QUALI_MASK)  == DIAGLOG_EVENT_TIMER_QUALI_MASK)

               //Trigger Timer (every DIAGLOG_PLUGIN_STEP_TIME) permanently
               if((hEvRequest & DIAGLOG_EVENT_TIMER_PLUGIN_STEP_MASK)  == DIAGLOG_EVENT_TIMER_PLUGIN_STEP_MASK)
               {
                  //ETG_TRACE_USR4_THR(( "--- vdDiagLog_tclApp::vTimerThread => DIAGLOG_EVENT_TIMER_PLUGIN_STEP_MASK"));
                  bTimerStep();
               }// if((hEvRequest & DIAGLOG_EVENT_TIMER_PLUGIN_STEP_MASK)  == DIAGLOG_EVENT_TIMER_PLUGIN_STEP_MASK)
            }// if(m_hEventHandle != OSAL_C_INVALID_HANDLE)
            else
            {
               // handel is invalid so we can stop now
               break;
            }
         }//if( OSAL_OK == OSAL_s32EventWait (vdDiagLog_tclApp::m_hEventHandle,
         else
         {
            // should not happen, but can be
            (tVoid) OSAL_s32ThreadWait(500);
         }
      }// if(m_hEventHandle != OSAL_C_INVALID_HANDLE)
      else
      {
         // handel is invalid so we can stop now
         break;
      }
   }// while ( TRUE)
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdDiagLog_tclApp::vOnUnregister
//
// DESCRIPTION:  Unregisters a Service
//
// PARAMETER:    None
//
// RETURNVALUE:  None
//
///////////////////////////////////////////////////////////////////////////////////
tVoid vdDiagLog_tclApp::vOnUnregister(tU16 u16ServiceId, tU16 u16RegisterId)
{
   ETG_TRACE_USR3_THR (("--> vdDiagLog_tclApp::vOnUnregister"));
   ail_tclAppInterface::vOnUnregister(u16ServiceId, u16RegisterId);

   if ( u16ServiceId == CCA_C_U16_SRV_DIAGLOG )
   {
      /* +++
      for safety make sure the notification table is cleared
      +++ */
      (tVoid) ahl_bEnterCritical( m_hNotTableSem );

       // check if pointer is valid
      if ( OSAL_NULL == m_poNotTable )
      {
         ETG_TRACE_ERRMEM(("!!! Diaglog: vdDiagLog_tclApp::vOnUnregister =>  OSAL_NULL == m_poNotTable"));
         NORMAL_M_ASSERT_ALWAYS();
      }
      else
      {
         (tVoid) m_poNotTable->bRemoveAllEntriesWithRegID(u16RegisterId);
      }

      (tVoid) ahl_bReleaseCritical( m_hNotTableSem );
   }
   ETG_TRACE_USR3_THR (("<-- vdDiagLog_tclApp::vOnUnregister"));
}


///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdDiagLog_tclApp::s32CreateTimer
//
// DESCRIPTION:  vStartTriggerTimer
//
// PARAMETER:    none
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
tS32 vdDiagLog_tclApp::s32CreateTimer(vdl_tclPluginBaseClass* pCallbackClass) const
{
   tS32 s32Return = -1;
   if(m_s32TimerId < DIAGLOG_PLUGIN_MAX_NUMBER_OF_TIMER)
   {
      if(pCallbackClass == NULL)
      {
         // invalid handle
         ETG_TRACE_ERRMEM(("!!! Diaglog: vdDiagLog_tclApp::s32CreateTimer =>  OSAL_NULL == pCallbackClass"));
         NORMAL_M_ASSERT_ALWAYS();
      }
      else
      {
         vdl_tsDiaglogTimerRegistration oNewTimer;
         oNewTimer.pCallbackClass      = pCallbackClass;
         oNewTimer.s32TimerId          = m_s32TimerId++; // increase global timer also
         oNewTimer.s32IntervalTime     = 0;
         oNewTimer.s32CurrTimeCounter  = 0;
         m_oTimerRegList.push_back(oNewTimer);
         ETG_TRACE_COMP_THR(( "--- vdDiagLog_tclApp::s32CreateTimer => Add TimerId:%d PlugInPtr:$%08x",oNewTimer.s32TimerId, pCallbackClass));
         s32Return = oNewTimer.s32TimerId;
      }
   }
   else
   {
      // too many Timers
      ETG_TRACE_ERRMEM(("!!! Diaglog: vdDiagLog_tclApp::s32CreateTimer =>  too many timer requested"));
      NORMAL_M_ASSERT_ALWAYS();
   }
   return s32Return;
}
///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdDiagLog_tclApp::bUnRegisterForTimer
//
// DESCRIPTION:  vStartTriggerTimer
//
// PARAMETER:    none
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
tBool vdDiagLog_tclApp::bDeleteTimer(vdl_tclPluginBaseClass* pCallbackClass) const
{
   tBool bReturn = FALSE;
   bool isOneTimerStillRunning = false;  // addional variable to check if we can stop timer completely

   if(m_isPluginTimerRunning)
   {
      for(tTimerRegistrationListIt iPos = m_oTimerRegList.begin(); iPos.operator!=(m_oTimerRegList.end()); ++iPos)
      {
         if(iPos->pCallbackClass == pCallbackClass)
         {
            ETG_TRACE_COMP_THR(( "--- vdDiagLog_tclApp::bUnRegisterForTimer => remove %x",pCallbackClass));
            iPos = m_oTimerRegList.erase(iPos);  // delete from list
            bReturn =  TRUE;
            continue;
            // do not break, there could be more then one Timer
         }

         // check if Timer is running
         if(iPos->s32CurrTimeCounter != 0)
         {
            isOneTimerStillRunning = true;
         }
      }

      if(isOneTimerStillRunning == false)
      {
         // no Timer running, can stop now
         vStopPluginTimer();
      }
   }
   return bReturn;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdDiagLog_tclApp::u32RegisterForTimer
//
// DESCRIPTION:  vStartTriggerTimer
//
// PARAMETER:    s32TimerId   : unique Timer ID
//               s32Time      : Time to set
//
// RETURNVALUE:  true: if timer found
//
///////////////////////////////////////////////////////////////////////////////////
tBool vdDiagLog_tclApp::bSetTimer(tS32 s32TimerId, tS32 s32Time) const
{
   tBool bReturn = false;

   if(  (s32Time < 0)     // do not allow negative values
      ||(s32TimerId < 0)) // do not allow negative TimerIDs
   {
      ETG_TRACE_ERRMEM(("!!! vdDiagLog_tclApp::bSetTimer => ERROR: Timer FAILED Id:%d Time:%d",s32TimerId,s32Time));
      NORMAL_M_ASSERT_ALWAYS();
   }
   else if(s32Time == 0) // Stop Timer
   {
      bool isOneTimerStillRunning = false;  // addional variable to check if we can stop timer completely

      // over all Timer
      for(tTimerRegistrationListIt iPos = m_oTimerRegList.begin(); iPos.operator!=(m_oTimerRegList.end()); ++iPos)
      {
         ETG_TRACE_USR1_THR(( "--- vdDiagLog_tclApp::bSetTimer => Check Timer $%04d",iPos->s32TimerId));
         // check Timer ID
         if (iPos->s32TimerId == s32TimerId)
         {
            // Timer found!!!
            // stop this Timer
            iPos->s32IntervalTime    = 0;
            iPos->s32CurrTimeCounter = 0;
            bReturn = true;
            // continune check for other running Timer
         }
         if(iPos->s32CurrTimeCounter != 0)
         {
            isOneTimerStillRunning = true;
         }
      }// for(tTimerRegistrationListIt iPos = m_oTimerRegList.begin(); iPos.operator!=(m_oTimerRegList.end()); ++iPos)

      if(isOneTimerStillRunning == false)
      {
         // no Timer running, can stop now
         vStopPluginTimer();
      }
   }// else if(s32Time == 0)
   else // Set Timer
   {
      // over all Timer
      for(tTimerRegistrationListIt iPos = m_oTimerRegList.begin(); iPos.operator!=(m_oTimerRegList.end()); ++iPos)
      {
         ETG_TRACE_USR1_THR(( "--- vdDiagLog_tclApp::bSetTimer => Check Timer $%04d",iPos->s32TimerId));
         // check Timer ID
         if (iPos->s32TimerId == s32TimerId)
         {
            // Timer found!!!
            if(m_isPluginTimerRunning == false)
            {
               if(m_hPlugInTimer != OSAL_C_INVALID_HANDLE)
               {
                  vStartPluginTimer(DIAGLOG_PLUGIN_STEP_TIME);
               }
               else
               {
                  ETG_TRACE_ERRMEM(("!!! vdDiagLog_tclApp::bSetTimer => ERROR: can't start Timer m_hPlugInTimer == OSAL_C_INVALID_HANDLE"));
                  NORMAL_M_ASSERT_ALWAYS();
               }
            }// if(m_isPluginTimerRunning == false)

            ETG_TRACE_COMP_THR(( "--- vdDiagLog_tclApp::bSetTimer => SetTimer $%04d to %d[ms]",s32TimerId,s32Time));
            iPos->s32IntervalTime    = s32Time;
            iPos->s32CurrTimeCounter = s32Time;
            // we found the ID and can stop now
            bReturn = true;
            break;
         }// if (iPos->s32TimerId == s32TimerId)
      }// for(tTimerRegistrationListIt iPos = m_oTimerRegList.begin(); iPos.operator!=(m_oTimerRegList.end()); ++iPos)
   }// else // if(  (s32Time < 0)     // do not allow negative values

   return bReturn;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdDiagLog_tclApp::u32RegisterForTimer
//
// DESCRIPTION:  vStartTriggerTimer
//
// PARAMETER:    none
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
tVoid vdDiagLog_tclApp::bTimerStep()
{
   ETG_TRACE_USR4_THR(("--- vdDiagLog_tclApp::bTimerStep"));

   if(m_isPluginTimerRunning)
   {
      // over all Timer
      for(tTimerRegistrationListIt iPos = m_oTimerRegList.begin(); iPos.operator!=(m_oTimerRegList.end()); ++iPos)
      {
         // decrease counter by Step Time
         (iPos->s32CurrTimeCounter)-= DIAGLOG_PLUGIN_STEP_TIME;

         // this Timer running?
         if(iPos->s32IntervalTime != 0)
         {
            // check if Timer run out
            if(iPos->s32CurrTimeCounter <= 0)
            {
               if(iPos->pCallbackClass != NULL)
               {
                  // forward trigger to PlugIn
                  iPos->pCallbackClass->vTimer(iPos->s32TimerId);

                  // reset Time
                  iPos->s32CurrTimeCounter = iPos->s32IntervalTime;
               }
            }// if(iPos->s32CurrTime <= 0)
         }// if(iPos->s32CurrTime != 0)
      }// for(tTimerRegistrationListIt iPos = m_oTimerRegList.begin(); iPos != m_oTimerRegList.end(); iPos++)
   }
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdDiagLog_tclApp::bRegisterForEvent
//
// DESCRIPTION:  bRegisterForEvent
//
// PARAMETER:    none
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
tBool vdDiagLog_tclApp::bRegisterForEvent(vdl_tclPluginBaseClass* pCallbackClass)
{
   bool bReturn = m_oMemoryMaster.bRegisterForEvent(pCallbackClass);
   return bReturn;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdDiagLog_tclApp::bUnRegisterForEvent
//
// DESCRIPTION:  bUnRegisterForEvent
//
// PARAMETER:    none
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
tBool vdDiagLog_tclApp::bUnRegisterForEvent(vdl_tclPluginBaseClass* pCallbackClass)
{
   tBool bReturn = m_oMemoryMaster.bUnRegisterForEvent(pCallbackClass);
   return bReturn;
}

/*static*/ inline void vdDiagLog_tclApp::LockMemorySemaphore()
{
   tS32 s32Ret = OSAL_s32SemaphoreWait(vdDiagLog_tclApp::m_hReportMemSem, OSAL_C_U32_INFINITE);
   if(OSAL_OK != s32Ret)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      et_vErrmemStringNormal( TR_COMP_DIAGNOSIS, "Diaglog: LockMemorySemaphore() => OSAL_s32SemaphoreWait failed: Result=%x, Error=%x",
                               s32Ret,
                               u32ErrorCode);
      NORMAL_M_ASSERT_ALWAYS();
   }

}

/*static*/ inline void vdDiagLog_tclApp::FreeMemorySemaphore()
{
   tS32 s32Ret = OSAL_s32SemaphorePost(vdDiagLog_tclApp::m_hReportMemSem);
   if(OSAL_OK != s32Ret)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      et_vErrmemStringNormal( TR_COMP_DIAGNOSIS, "Diaglog: FreeMemorySemaphore() => OSAL_s32SemaphorePost failed: Result=%x, Error=%x",
                               s32Ret,
                               u32ErrorCode);
      NORMAL_M_ASSERT_ALWAYS();
   }
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdDiagLog_tclApp::vPostEvent
//
// DESCRIPTION:  post an event
//
// PARAMETER:    tU32 Event: Event to post
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
/*static*/ tVoid vdDiagLog_tclApp::vPostEvent(tU32 Event)
{
   if(m_hEventHandle != OSAL_C_INVALID_HANDLE)
   {
      tS32 s32Result = OSAL_s32EventPost(vdDiagLog_tclApp::m_hEventHandle, Event, OSAL_EN_EVENTMASK_OR);
      if(OSAL_ERROR == s32Result)
      {
         tU32 u32ErrorCode= OSAL_u32ErrorCode();
         ETG_TRACE_ERRMEM(("!!! vdDiagLog_tclApp::vPostEvent => ERROR: OSAL_s32EventPost FAILED: Error=%x", static_cast<tUInt>(u32ErrorCode)));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }
   else
   {
      ETG_TRACE_ERRMEM(("!!! vdDiagLog_tclApp::vPostEvent => ERROR: OSAL_C_INVALID_HANDLE"));
      NORMAL_M_ASSERT_ALWAYS();
   }
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdDiagLog_tclApp::vStartPluginTimer
//
// DESCRIPTION:  start the PluginTimer
//
// PARAMETER:    tU32 Time: Plugin Step Time
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
/*static*/ tVoid vdDiagLog_tclApp::vStartPluginTimer(tU32 Timer)
{
   ETG_TRACE_USR1_THR(( "--- vdDiagLog_tclApp::vStartPluginTimer => Start PLUGIN-TIMER m_hPlugInTimer"));
   tS32 s32Result = OSAL_s32TimerSetTime (m_hPlugInTimer, Timer, Timer);
   if(OSAL_ERROR == s32Result)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      ETG_TRACE_ERRMEM(("!!! vdDiagLog_tclApp::vStartPluginTimer => ERROR: OSAL_s32TimerSetTime FAILED: Error=%x", static_cast<tUInt>(u32ErrorCode)));
      NORMAL_M_ASSERT_ALWAYS();
   }
   else
   {
      m_isPluginTimerRunning = true;
   }
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdDiagLog_tclApp::vStopPluginTimer
//
// DESCRIPTION:  stop the pluginTimer
//
// PARAMETER:    none
//
// RETURNVALUE:  none
//
///////////////////////////////////////////////////////////////////////////////////
/*static*/ tVoid vdDiagLog_tclApp::vStopPluginTimer()
{
   ETG_TRACE_USR1_THR(( "--- vdDiagLog_tclApp::vStopPluginTimer => Start PLUGIN-TIMER m_hPlugInTimer"));

   // stop only if running
   if(m_isPluginTimerRunning == true)
   {
      tS32 s32Result = OSAL_s32TimerSetTime (m_hPlugInTimer, 0, 0);
      if(OSAL_ERROR == s32Result)
      {
         tU32 u32ErrorCode= OSAL_u32ErrorCode();
         ETG_TRACE_ERRMEM(("!!! vdDiagLog_tclApp::vStopPluginTimer => ERROR: OSAL_s32TimerSetTime FAILED: Error=%x", static_cast<tUInt>(u32ErrorCode)));
         NORMAL_M_ASSERT_ALWAYS();
      }
      else
      {
         m_isPluginTimerRunning = false;
      }
   }// if(m_PluginTimerRunning == true)
}


///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: tVoid vdDiagLog_tclApp::vOperatingCycleStart()
//
// DESCRIPTION:   on OperationCycle start
//
// PARAMETER:  void
//
// RETURNVALUE: void
//
///////////////////////////////////////////////////////////////////////////////////

tVoid vdDiagLog_tclApp::vStartDm1MessageOperatingCycle()
{
	ETG_TRACE_USR1_THR(( "--> vdDiagLog_tclApp::vStartDm1MessageOperatingCycle"));

   if((vdDiagLog_tclApp::m_poInstance) != NULL)
   {
	   vdDiagLog_tclApp::m_poInstance->m_poReportDtcDM1TriggerPlugIn->startDm1MsgTimer();
   }

   ETG_TRACE_USR1_THR(( "<-- vdDiagLog_tclApp::vStartDm1MessageOperatingCycle"));
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: tVoid vdDiagLog_tclApp::vOperatingCycleEnd()
//
// DESCRIPTION:   on OperationCycle end
//
// PARAMETER:  void
//
// RETURNVALUE: void
//
///////////////////////////////////////////////////////////////////////////////////


tVoid vdDiagLog_tclApp::vEndDm1MessageOperatingCycle()
{
	ETG_TRACE_USR1_THR(( "--> vdDiagLog_tclApp::vEndDm1MessageOperatingCycle"));

	if((vdDiagLog_tclApp::m_poInstance) != NULL)
    {
	   vdDiagLog_tclApp::m_poInstance->m_poReportDtcDM1TriggerPlugIn->stopDm1MsgTimer();
    }

	ETG_TRACE_USR1_THR(( "<-- vdDiagLog_tclApp::vEndDm1MessageOperatingCycle"));
}


