/************************************************************************
* FILE:         dia_tclUDS_WIFI_GetMacAddress.cpp
* PROJECT:      DIAG Core
* SW-COMPONENT: Diagnostic application
*----------------------------------------------------------------------
*
* DESCRIPTION: Class for WIFI MAC Address
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 26.11.15  | nce5kor	         | initial version
*
*************************************************************************/

#include "dia_tclUDS_WIFI_GetMacAddress.h"

#ifndef __INCLUDED_DIA_DEFINES_UDS__
#include <common/framework/protocols/uds/dia_defsUds.h>
#endif

#define DATA_START            (1/*sizeof(SID)*/ + mDIDLen)
#define DATA_LENGTH           (17U)    //string length "AA:BB:CC:DD:EE:FF"
#define DIA_MAC_ADDRESS_LEN   (6U)

#define CONVERT_TABLE_SIZE    16
#define RESPONSE_BYTE_0 0
#define RESPONSE_BYTE_1 1
#define RESPONSE_BYTE_2 2
#define RESP_ELEMENT_BIT_SIZE 4
#define RESPONSE_ELEMENT_SIZE 3
#define MAC_ADDR_ELEMENT_SIZE 1
#define HEX_F ((tU8)0x0F)

//-------------------------------------------------------------------------------

dia_tclUDS_WIFI_GetMacAddress::dia_tclUDS_WIFI_GetMacAddress(tCString name, tU8 sid, tU16 did):
      dia_ServiceHandlerUDS(name,sid,(tU16) did)
{
   dia_tclFnctTrace trc("dia_tclUDS_WIFI_GetMacAddress::dia_tclUDS_WIFI_GetMacAddress");
}

//-------------------------------------------------------------------------------

dia_tclUDS_WIFI_GetMacAddress::~dia_tclUDS_WIFI_GetMacAddress()
{
   _BP_TRY_BEGIN
   {
     (void) unsetSysAdapterListener<dia_IBluetoothListener>(this);
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("dia_tclUDS_WIFI_GetMacAddress::~dia_tclUDS_WIFI_GetMacAddress - Exception caught!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-------------------------------------------------------------------------------

void
dia_tclUDS_WIFI_GetMacAddress::vProcessRequest ( const std::vector<tArgsType>& /*vecArgs*/ )
{
   dia_tclFnctTrace trc("dia_tclUDS_WIFI_GetMacAddress::vProcessRequest");

   tBool errorDetected = TRUE;

   dia_IBluetooth* pIBluetooth = 0;
   tDiaResult sysAdapterResult = querySysAdapterInterface<dia_IBluetooth>(&pIBluetooth);

   if ((DIA_SUCCESS==sysAdapterResult) && pIBluetooth)
   {
      (void) setSysAdapterListener<dia_IBluetoothListener>(this);
      if ( pIBluetooth->vGetWifiMacAddress() != DIA_SUCCESS )
      {
         DIA_TR_ERR("!!! dia_tclUDS_WIFI_GetMacAddress::vProcessRequest => ERROR: vGetWifiMacAddress != DIA_SUCCESS");
      }
      else
      {
         errorDetected = FALSE;
         DIA_TR_INF("dia_tclUDS_WIFI_GetMacAddress::vProcessRequest => Send Successfully to Server");
      }
   }
   else
   {
      DIA_TR_INF("!!! dia_tclUDS_WIFI_GetMacAddress::vProcessRequest ERROR: sysAdapterResult=0x%X pIBluetooth=%p", sysAdapterResult, pIBluetooth);
   }

   if ( errorDetected )
   {
      DIA_TR_INF("dia_tclUDS_WIFI_GetMacAddress::vProcessRequest => ERROR DETECTED");
      (void) unsetSysAdapterListener<dia_IBluetoothListener>(this);
      oDiagMsgBuffer().vSetNegResp(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
      vResReadyAndQuit();
   }
}

//-------------------------------------------------------------------------------

void
dia_tclUDS_WIFI_GetMacAddress::vOnReadWifiMacAddressResult(const tU8* const pu8Result)
{
   dia_tclFnctTrace trc("dia_tclUDS_WIFI_GetMacAddress::vOnReadWifiMacAddressResult");

   (void) unsetSysAdapterListener<dia_IBluetoothListener>(this);

   oDiagMsgBuffer().vSetPosResp();
   oDiagMsgBuffer().vSetDataLength(DATA_START + DATA_LENGTH);
   const tU8 u8ConvertTable[CONVERT_TABLE_SIZE] =
   {
      '0', '1', '2', '3',  '4', '5', '6', '7',
      '8', '9', 'A', 'B',  'C', 'D', 'E', 'F'
   };

   for(tU8 i=0; i < DIA_MAC_ADDRESS_LEN; i++)
   {
      oDiagMsgBuffer().vSetDataU8(DATA_START + (RESPONSE_ELEMENT_SIZE*i + RESPONSE_BYTE_0), u8ConvertTable[((pu8Result[i] >> RESP_ELEMENT_BIT_SIZE) & HEX_F)]);
      oDiagMsgBuffer().vSetDataU8(DATA_START + (RESPONSE_ELEMENT_SIZE*i + RESPONSE_BYTE_1), u8ConvertTable[((pu8Result[i]     ) & HEX_F)]);

      if (i < (DIA_MAC_ADDRESS_LEN-MAC_ADDR_ELEMENT_SIZE))
      {
         oDiagMsgBuffer().vSetDataU8(DATA_START + (RESPONSE_ELEMENT_SIZE*i + RESPONSE_BYTE_2), ':');
      }
   }

   vResReadyAndQuit();
   DIA_TR_INF("dia_tclUDS_WIFI_GetMacAddress::vOnReadWifiMacAddressResult MAC %02X:%02X:%02X:%02X:%02X:%02X",
               pu8Result[0], pu8Result[1], pu8Result[2], pu8Result[3], pu8Result[4], pu8Result[5]);
}

//-------------------------------------------------------------------------------

void
dia_tclUDS_WIFI_GetMacAddress::vOnReadWifiMacAddressError (const tU32 errCode )
{
   dia_tclFnctTrace trc("dia_tclUDS_WIFI_GetMacAddress::vOnReadWifiMacAddressError");

   (void) unsetSysAdapterListener<dia_IBluetoothListener>(this);

   DIA_TR_INF( "dia_tclUDS_WIFI_GetMacAddress::vOnReadWifiMacAddressError=> errCode=%08X",errCode);
   oDiagMsgBuffer().vSetNegResp(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
   vResReadyAndQuit();
}
