/**
 * @file   : dia_SrvHandler_AuxIn_Get.cpp
 * @author : bsu9kor
 * @date   : Aug 17, 2016
 * @brief  : Read the status of the Aux-In Detect Plug and the Detect Plug Diag GPIO.
 * @copyright (c) 2016 Robert Bosch Car Multimedia Gmbh
 * @addgroup : Diagnosis
 * @{
 */
#ifndef DIA_SRVHANDLER_AUXIN_GET_H_
#include "dia_SrvHandler_AuxIn_Get.h"
#endif

#ifndef __INCLUDED_DIA_DEFINES_UDS__
#include <common/framework/protocols/uds/dia_defsUds.h>
#endif

#ifndef __INCLUDED_DIA_COMMON_CONFIG__
#include "common/framework/config/dia_common_config.h"
#endif

#define DATA_START 					3
#define DATA_LENGTH 				2

#define DIA_PROP_GPIO_SIZE          0x01U
#define DIA_PROP_GPIO_VALUE_MAX     0x01U

#define DATA_ZERO					0x00
#define DATA_ONE					0x01

//#define INDEX						0


//-----------------------------------------------------------------------------
/**
 * \brief      class constructor
 *             class constructor with name, service identifier and 2 byte DID
 *
 * \param[in]  name     name used to distinguish between multiple instances
 * \param[in]  sid      diagnostic service identifier (e.g. 0x22)
 * \param[in]  did      diagnostic data identifier (2 byte)
 */
dia_SrvHandler_AuxIn_Get::dia_SrvHandler_AuxIn_Get( void )
	: dia_ServiceHandlerUDS("dia_SrvHandler_AuxIn_Get", DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_AIVI_AUXIN_DETECT /*DID*/)
{
	 mPropData = DATA_ZERO;
	 dia_tclFnctTrace trc("dia_SrvHandler_AuxIn_Get::dia_SrvHandler_AuxIn_Get");
}
//-------------------------------------------------------------------------------
/**
  * \brief    	handle the received request
  *				Function receives the message via the oDiagMsgBuffer.
  *
  * \param[in]	void*     -
  */

void dia_SrvHandler_AuxIn_Get::vProcessRequest ( const std::vector<void*>& /*vecArgs*/ )
{
   dia_tclFnctTrace trc("dia_SrvHandler_AuxIn_Get::vProcessRequest");
   tU8  detectPlugdata[DIA_PROP_GPIO_SIZE] = {0};
   tU8  detectPlugDiagdata[DIA_PROP_GPIO_SIZE] = {0};
   //Read DetectPLUG GPIO
   if(!getPropertyDetails(DIA_PROP_GPIO_DETECT_PLUG))
   {
	   detectPlugdata[0] = mPropData;
	   DIA_TR_INF("detectPlugdata = %d", detectPlugdata[0]);
   }
   else
   {
	   DIA_TR_INF("dia_SrvHandler_AuxIn_Get::vProcessRequest FAILED to read DetectPLUG GPIO");
	   vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
	   vResReadyAndQuit();
	   return;
   }
   //Read DetectPLUGDiag GPIO
   if(!getPropertyDetails(DIA_PROP_GPIO_DETECT_PLUG_DIAG))
   {
	   detectPlugDiagdata[0] = mPropData;
	   DIA_TR_INF("detectPlugDiagdata = %d", detectPlugDiagdata[0]);
   }
   else
   {
	   DIA_TR_INF("dia_SrvHandler_AuxIn_Get::vProcessRequest FAILED to read DetectPLUGDiag GPIO");
	   vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
	   vResReadyAndQuit();
	   return;
   }

   // Assemble the reponse
   if ((detectPlugdata[0] > DIA_PROP_GPIO_VALUE_MAX ) || (detectPlugDiagdata[0] > DIA_PROP_GPIO_VALUE_MAX))
   {
      // Sending back the Negative answer
      vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
   }
   else
   {
      DIA_TR_INF("dia_SrvHandler_AuxIn_Get::vProcessRequest SUCCESS detectPlugdata[0]=0x%02X", detectPlugdata[0]);
      DIA_TR_INF("dia_SrvHandler_AuxIn_Get::vProcessRequest SUCCESS detectPlugDiagdata[0]=0x%02X", detectPlugDiagdata[0]);

      oDiagMsgBuffer().vSetPosResp();
      oDiagMsgBuffer().vSetDataLength(DATA_START + DATA_LENGTH);
      //Detect Plug Data
      (void) oDiagMsgBuffer().vSetDataU8(DATA_START + DATA_ZERO, detectPlugdata[0]);
      //Detect Plug Diag Data
      (void) oDiagMsgBuffer().vSetDataU8(DATA_START + DATA_ONE, detectPlugDiagdata[0]);
   }
   vResReadyAndQuit();
   return;
}

//--------------------------------------------------------------------------------------
/**
 *\brief      class destructor
 */
dia_SrvHandler_AuxIn_Get::~dia_SrvHandler_AuxIn_Get()
{
}
//---------------------------------------------------------------------------------------
/**
  * \brief    	Get Property Details
  *				Function retrieves the data based on Property ID.
  *
  * \param[in]	tU32     - Property ID
  */
tDiaResult dia_SrvHandler_AuxIn_Get::getPropertyDetails( tU32 PropID)
{
	dia_tclFnctTrace trc("dia_SrvHandler_AuxIn_Get::getPropertyDetails");
	// data memory
	std::vector<tU8> PropDataVec;

	// retrieve size of property
	tU16 Proplength = dia_getPropertySize(PropID); //DIA_PROP_LENGTH_MAX;
	PropDataVec.reserve(Proplength);
	PropDataVec.resize(Proplength);
	tDiaResult retCode = dia_getProperty(PropID,PropDataVec);
	if ( retCode == DIA_SUCCESS )
	{
		for (tU16 i = 0; i < PropDataVec.size(); ++i)
		{
			mPropData = PropDataVec[i];
		}
	}
	else
	{
		DIA_TR_INF("dia_SrvHandler_AuxIn_Get::getPropertyDetails Get property details failed for Property %d",PropID);
	}
	return retCode;
}
//-----------------------------------------------------------------------------------------------------------------------------
