
/*!
 * \file       dia_SrvHandler_ReadCalibrationXtraData.cpp
 *
 * \brief      RDBI service to read first 10 bytes out of logical block 0. (AIVI-43057, AIVI-57691)
 *
 * \details    This RDBI service returns following values depending on result of CheckProgrammingDependency (see DIA_PROP_AIVI_RESULT_CHECK_PROG_DEP)
 *             1/ 10 bytes with value 0x30 in case of failure (or DIA_PROP_AIVI_RESULT_CHECK_PROG_DEP is N/A)
 *             2/ First 10 bytes of the Logical Block $00 (in session $02) in case of success
 *             3/ NRC 0x22 if CheckProgrammingDependency was successful, but Logical Block $00 cannot be read.
 *
 * \component  Diagnosis
 *
 * \ingroup
 *
 * \copyright  (c) 2017 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_DEFS_CONFIG__
#include "common/framework/config/dia_defsConfig.h"
#endif

#ifndef __INCLUDED_DIA_DEFINES_UDS__
#include <common/framework/protocols/uds/dia_defsUds.h>
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include "common/framework/config/dia_ConfigManager.h"
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG_PROJECT__
#include "project/framework/config/dia_defsProjectConfig.h"
#endif

#ifndef __INCLUDED_DIA_FACTORY__
#include "common/framework/application/dia_Factory.h"
#endif

#ifndef DIA_SRVHANDLER_READCALIBRATIONXTRADATA_H_
#include "dia_SrvHandler_ReadCalibrationXtraData.h"
#endif


#define DIA_U8_DEFAULT_VALUE                    ((tU8) (0x30))
#define DIA_U16_READDATA_BY_ID_REQUEST_LENGTH   ((tU16) (1 + mDIDLen))
#define DIA_U16_DATA_LENGTH                     ((tU16) (10))

dia_SrvHandler_ReadCalibrationXtraData::dia_SrvHandler_ReadCalibrationXtraData ( void )
   : dia_ServiceHandlerUDS("dia_SrvHandler_ReadCalibrationXtraData", DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, DIA_C_U16_DID_AIVI_CALIBRATION_XTRA_DATA_REF),
     mRoutineDid(DIA_C_U16_ID_AIVI_RTCTRL_CHECK_PROGRAMMING_DEPENDENCIES)
{
   dia_tclFnctTrace trc("dia_SrvHandler_ReadCalibrationXtraData::dia_SrvHandler_ReadCalibrationXtraData");
}
//-----------------------------------------------------------------------------

dia_SrvHandler_ReadCalibrationXtraData::~dia_SrvHandler_ReadCalibrationXtraData ( void )
{
}
//-----------------------------------------------------------------------------

void
dia_SrvHandler_ReadCalibrationXtraData::vProcessRequest( const std::vector<void*>& /*vecArgs*/ )
{
   dia_tclFnctTrace trc("dia_SrvHandler_ReadCalibrationXtraData::vProcessRequest");

   dia_IRoutineControl* pRtCtrlInterface = 0;
   tDiaResult retCode = querySysAdapterInterface<dia_IRoutineControl>(&pRtCtrlInterface);
   if ( (retCode != DIA_SUCCESS) || !pRtCtrlInterface )
   {
      DIA_TR_ERR("querySysAdapterInterface for dia_IRoutineControl failed");
      DIA_ASSERT_ALWAYS();
   }
   else
   {
      dia_Routine* pRoutine = 0;
      retCode = pRtCtrlInterface->queryRoutineByDID(mRoutineDid, &pRoutine);
      if ((retCode != DIA_SUCCESS) || !pRoutine)
      {
         DIA_TR_ERR("queryRoutineByDID for DID=0x%x failed", mRoutineDid);
         DIA_ASSERT_ALWAYS();
      }
      else
      {
         std::vector<tU8> params;

         (tVoid) setSysAdapterListener<dia_IRoutineControlListener>(this);

         retCode = pRoutine->start(params, DIA_C_U8_UDS_RTCTRL_TIMER_VALUE_INFINITE);
      }
   }

   if (retCode != DIA_SUCCESS)
   {
      vSendNegativeResponse(getInstanceOfFactory()->makeNRC(DIA_E_CONDITIONS_NOT_CORRECT));
   }
}

//------------------------------------------------------------------------------

void dia_SrvHandler_ReadCalibrationXtraData::vOnRoutineUpdate ( dia_Routine& routine )
{
   dia_tclFnctTrace trc("dia_SrvHandler_ReadCalibrationXtraData::vOnRoutineUpdate");

   if ( (routine.getDID() == mRoutineDid) && routine.bIsResultReady() )
   {
      (void) unsetSysAdapterListener<dia_IRoutineControlListener>(this);

      std::vector<tU8> dV;
      std::vector<tU8> results;
      tDiaResult retCode = routine.requestResult(results);

      DIA_TR_INF("dia_SrvHandler_ReadCalibrationXtraData::vOnRoutineUpdate Routine=0x%04X result=0x%08X.", mRoutineDid, retCode);

      if ( retCode == DIA_SUCCESS )
      {
         retCode = dia_getProperty(DIA_PROP_AIVI_BLOCK_FD00_REFERENCE_CHECKSUM, dV);
         DIA_TR_INF("dia_SrvHandler_ReadCalibrationXtraData::vOnRoutineUpdate read LB00 result=0x%08X.", retCode);
      }
      else if ( retCode == DIA_E_FAILURE_PREVENTS_EXECUTION_OF_REQUESTED_ACTION )
      {
         //set 10 bytes with value 0x30
         dV.reserve(DIA_U16_DATA_LENGTH); //set capacity (avoid triggering of reallocation)
         dV.resize(DIA_U16_DATA_LENGTH);  //set size
         std::fill(dV.begin(), dV.end(), DIA_U8_DEFAULT_VALUE);
         retCode = DIA_SUCCESS;
      }
      else
      {
         DIA_ASSERT_ALWAYS();
      }

      if ( (retCode == DIA_SUCCESS) && (dV.size()>=DIA_U16_DATA_LENGTH))
      {
         oDiagMsgBuffer().vSetPosResp();
         oDiagMsgBuffer().vSetDataLength(DIA_U16_READDATA_BY_ID_REQUEST_LENGTH + DIA_U16_DATA_LENGTH);

         for ( tU16 i = 0; i < DIA_U16_DATA_LENGTH; ++i )
         {
            (void) oDiagMsgBuffer().vSetDataU8(DIA_U16_READDATA_BY_ID_REQUEST_LENGTH+i, dV[i]);
         }
         vResReadyAndQuit();
      }
      else
      {
         if (retCode != DIA_SUCCESS)          DIA_TR_ERR("### dia_SrvHandler_ReadCalibrationXtraData::vOnRoutineUpdate UNABLE TO READ ITEM 0x%08X !!### ", retCode);
         if (dV.size()<DIA_U16_DATA_LENGTH)   DIA_TR_ERR("### dia_SrvHandler_ReadCalibrationXtraData::vOnRoutineUpdate exp size is %d, act size is %d !!### ", DIA_U16_DATA_LENGTH, dV.size());
         vSendNegativeResponse(getInstanceOfFactory()->makeNRC(DIA_E_CONDITIONS_NOT_CORRECT));
      }

   }
}


