/*!
* \file       dia_SrvHandler_GPSReceptionLevel.cpp
*
* \brief      Get the GPS Reception Level
*
* \created on 02.09.2016
*
* \author	  bma9kor
*
* \details    Class dia_SrvHandler_GPSReceptionLevel provides the diagnosis tester the ability to read the GPS Reception Level [ Scope 2.1 Lead ]
* 				Bit 7 : reception status (=0 no satellite,=1 at least one satellite)
*				Bit 6 to 4 : reserved
*				Bit 3 to 0 : number of satellites
*				0 to F
*				Range : 0 to 15
* 				Resolution : 1
*				Offset : 0
*
* \component  Diagnosis
*
* \ingroup    diaCoreSecurity
*
* \copyright  (c) 2015-2016 Robert Bosch GmbH
*
* The reproduction, distribution and utilization of this file as
* well as the communication of its contents to others without express
* authorization is prohibited. Offenders will be held liable for the
* payment of damages. All rights reserved in the event of the grant
* of a patent, utility model or design.
*/

// TTFis: DIA_REQ UDS 04 22 30 04


#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#ifndef __INCLUDED_DIA_COMMON_PROTOCOLS_UDS__
#include "common/framework/protocols/uds/dia_common_uds.h"
#endif

#ifndef __INCLUDED_DIA_DEFS_PROJECT__
#include "project/framework/application/dia_defsProject.h"
#endif

#ifndef DIA_SRVHANDLER_GPSRECEPTIONLEVEL_H_
#include "dia_SrvHandler_GPSReceptionLevel.h"
#endif

#ifndef __INCLUDED_DIA_INTERFACE_GNSS_LISTENER__
#include "common/interfaces/dia_IGNSSListener.h"
#endif

#define DATA_START 3
#define DATA_LENGTH 1

#define ZERO 0
#define FIX_TWO 0x02
#define FIX_FOUR 0x04
#define ZERO_HEX 0x00
#define SAT_USED_MASK 0x0F
#define REC_STS_MASK 0x80
#define NUMBER_OF_SAT 33


//-----------------------------------------------------------------------------

dia_SrvHandler_GPSReceptionLevel::dia_SrvHandler_GPSReceptionLevel ( tVoid )
    : dia_ServiceHandlerUDS("dia_SrvHandler_GPS_ReceptionLevel",DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, DIA_C_U16_DID_AIVI_GPS_RECEPTION_LEVEL /*DID*/)

{
   dia_tclFnctTrace trc("dia_SrvHandler_GPSReceptionLevel::dia_SrvHandler_GPSReceptionLevel(tVoid)");
   m_u8GPSReceptionLevel = ZERO_HEX;
}

//-----------------------------------------------------------------------------

dia_SrvHandler_GPSReceptionLevel::~dia_SrvHandler_GPSReceptionLevel( tVoid )
{
    _BP_TRY_BEGIN
    {
       (tVoid) unsetSysAdapterListener<dia_IGNSSListener>(this);
    }
    _BP_CATCH_ALL
    {
        DIA_TR_ERR("EXCEPTION CAUGHT: dia_SrvHandler_GPSReceptionLevel::~dia_SrvHandler_GPSReceptionLevel !!!");
        NORMAL_M_ASSERT_ALWAYS();
    }
    _BP_CATCH_END
}

//-------------------------------------------------------------------------------

tVoid
dia_SrvHandler_GPSReceptionLevel::vProcessRequest ( const std::vector<tVoid*>& /*vecArgs*/ )
{
   dia_tclFnctTrace trc("dia_SrvHandler_GPSReceptionLevel::vProcessRequest");

   tBool errorDetected = TRUE;

   dia_IGNSS* pGNSS = ZERO;
   if ( querySysAdapterInterface<dia_IGNSS>(&pGNSS) == DIA_SUCCESS )
   {
      if ( pGNSS )
      {
         (tVoid) setSysAdapterListener<dia_IGNSSListener>(this);
         if ( pGNSS->getGNSSData() == DIA_SUCCESS )
         {
            errorDetected = FALSE;
         }
      }
   }

   if ( errorDetected )
   {
      (tVoid) unsetSysAdapterListener<dia_IGNSSListener>(this);
      DIA_TR_ERR(" ---  SEND TO GNSS_Sensor Component failed!!!!");
      oDiagMsgBuffer().vSetNegResp(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
      vResReadyAndQuit();
   }
}

//-----------------------------------------------------------------------------

tVoid
dia_SrvHandler_GPSReceptionLevel::vOnGPSReceptionLevel ( const dia_tGPSReceptionLevel& info )
{
   dia_tclFnctTrace trc("dia_SrvHandler_GPS_ReceptionLevel::vOnGPS_ReceptionLevel");

   (void) unsetSysAdapterListener<dia_IGNSSListener>(this);

   if( info.SatellitesUsed < NUMBER_OF_SAT )
   {
      DIA_TR_INF("FixStatus          = %d / 0x%02x .", info.FixStatus, info.FixStatus);	//1 byte
      DIA_TR_INF("SatellitesUsed     = %d / 0x%02x .", info.SatellitesUsed, info.SatellitesUsed);		//1 byte

	  bool flag = FALSE;
	  if(info.FixStatus == FIX_TWO || info.FixStatus == FIX_FOUR)
		  flag = TRUE;

	  //num of satellites: 0 to 15 stored in bits 3-0
	  m_u8GPSReceptionLevel = (info.SatellitesUsed & SAT_USED_MASK);

	  if(flag)
	  {
		  m_u8GPSReceptionLevel = (m_u8GPSReceptionLevel | REC_STS_MASK);
	  }
      oDiagMsgBuffer().vSetPosResp();
      oDiagMsgBuffer().vSetDataLength(DATA_START+ DATA_LENGTH);

      (void) oDiagMsgBuffer().vSetDataU8(DATA_START+ZERO,(tU8) m_u8GPSReceptionLevel);

       vResReadyAndQuit();
       DIA_TR_INF("dia_SrvHandler_GPS_ReceptionLevel::vOnGPS_ReceptionLevel SUCCESSFUL");
   }
   else
   {
      DIA_TR_INF("dia_SrvHandler_GPS_ReceptionLevel::vOnGPS_ReceptionLevel FAIL");
      // Sending back the Negative answer
      vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
   }
}

