 /*!
 * \file       dia_SrvHandlerInstallationContainerSwVersion.cpp
 *
 * \brief      Service handler to read the Software Version of CIS Installation Container
 *
 * \details    ...
 *
 * \component  Diagnostics
 *
 * \ingroup    diag services
 *
 * \author     Arjun Manjunath Sanu (RBEI/ECA2)
 *
 * \date       29.04.2020
 *
 * \copyright  (c) 2020 Robert Bosch Engineering & Business Solutions Ltd.
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __DIA_SRV_HANDLER_CIS_INSTALLATION_CONTAINER_SOFTWARE_VERSION_H__
#include "project/services/customer/dia_SrvHandler_CISInstallationContainerSwVersion.h"
#endif

#ifndef __INCLUDED_DIA_DEFINES_UDS__
#include <common/framework/protocols/uds/dia_defsUds.h>
#endif

#ifndef __INCLUDED_DIA_MESSAGE_BUFFER__
#include "common/framework/engine/dia_MessageBuffer.h"
#endif

#ifndef __INCLUDED_DIA_FACTORY__
#include "common/framework/application/dia_Factory.h"
#endif

#include <sstream>
#include <common/framework/xml/tinyxml2.h>

using namespace tinyxml2;

namespace dia {

#define DIA_U16_READ_DATA_BY_ID_REQUEST_LENGTH  ((tU16) (1 + mDIDLen))                                             
#define DIA_U16_SOFTWARE_VERSION_LENGTH         ((tU16) (3)) 
#define DIA_U16_SOFTWARE_VERSION_YEAR_INDEX_0   ((tU16) (0)) 
#define DIA_U16_SOFTWARE_VERSION_YEAR_INDEX_1   ((tU16) (1)) 
#define DIA_U16_SOFTWARE_VERSION_CW_INDEX_0     ((tU16) (2)) 
#define DIA_U16_SOFTWARE_VERSION_CW_INDEX_1     ((tU16) (3)) 
#define DIA_U16_SOFTWARE_VERSION_PATCH_INDEX_0  ((tU16) (4)) 
#define DIA_U16_SOFTWARE_VERSION_PATCH_INDEX_1  ((tU16) (5)) 
#define DIA_C_BASE_FORMAT_DECIMAL               (int)10                                            

static const std::string targetXMLpath  ("/var/opt/bosch/persistent/avdecc/cisinstallation/CIS_network.xml");
static const std::string targetNodeName ("CfgVersion");
static const std::string defaultSoftwareVersion ("000000");

//------------------------------------------------------------------------------

dia_SrvHandlerInstallationContainerSwVersion::dia_SrvHandlerInstallationContainerSwVersion ( tCString name, tU8 sid, tU16 did )
   : dia_ServiceHandlerUDS(name, sid, did),
     __CISInstallationContainerSoftwareVersion(defaultSoftwareVersion),
     __CISInstallationContainerSoftwareVersionFound(FALSE)
{
   dia_tclFnctTrace trc("dia_SrvHandlerInstallationContainerSwVersion::dia_SrvHandlerInstallationContainerSwVersion(tCString,tU8,tU16)");
}

//------------------------------------------------------------------------------

dia_SrvHandlerInstallationContainerSwVersion::~dia_SrvHandlerInstallationContainerSwVersion ( void )
{}

//------------------------------------------------------------------------------

void
dia_SrvHandlerInstallationContainerSwVersion::vProcessRequest ( const std::vector<void*>& vecArgs )
{
    dia_tclFnctTrace trc("dia_SrvHandlerInstallationContainerSwVersion::vProcessRequest(...)");

    __CISInstallationContainerSoftwareVersion = defaultSoftwareVersion;
    __CISInstallationContainerSoftwareVersionFound = FALSE;
	
	getInstallationContainerSoftwareVersion();
	
    // Sending back the positive answer
    oDiagMsgBuffer().vSetPosResp();
    oDiagMsgBuffer().vSetDataLength(static_cast<tU16>(DIA_U16_READ_DATA_BY_ID_REQUEST_LENGTH + DIA_U16_SOFTWARE_VERSION_LENGTH));

    // Fill Response Buffer
    for(int j = 0; j < DIA_U16_SOFTWARE_VERSION_LENGTH; j++)
    {
       (void) oDiagMsgBuffer().vSetDataU8(static_cast<tU16>(DIA_U16_READ_DATA_BY_ID_REQUEST_LENGTH + j), mSwVersion[j]);
    }

    vResReadyAndQuit();
}
/**
*
* \brief      method called to retrive the software version
*             of the CIS Installation Container
*             
* \param[in]  none
*
* \return     none
*/
void
dia_SrvHandlerInstallationContainerSwVersion::getInstallationContainerSoftwareVersion ( void )
{
	ScopeTrace oTrace("dia_SrvHandlerInstallationContainerSwVersion::getInstallationContainerSoftwareVersion()");
	
	std::string strYear, strCW, strPatch;
	unsigned long year, cw, patch;
		
	parseXML(targetXMLpath);
	
	// Parse the year
	strYear.push_back(__CISInstallationContainerSoftwareVersion[DIA_U16_SOFTWARE_VERSION_YEAR_INDEX_0]);
	strYear.push_back(__CISInstallationContainerSoftwareVersion[DIA_U16_SOFTWARE_VERSION_YEAR_INDEX_1]);
	
	const char* cpYear = strYear.c_str();
	year = strtoul(cpYear, 0, DIA_C_BASE_FORMAT_DECIMAL);
	
	// Parse the CW
	strCW.push_back(__CISInstallationContainerSoftwareVersion[DIA_U16_SOFTWARE_VERSION_CW_INDEX_0]);
	strCW.push_back(__CISInstallationContainerSoftwareVersion[DIA_U16_SOFTWARE_VERSION_CW_INDEX_1]);
	
	const char* cpCW = strCW.c_str();
	cw = strtoul(cpCW, 0, DIA_C_BASE_FORMAT_DECIMAL);
	
	// Parse the patch level
	strPatch.push_back(__CISInstallationContainerSoftwareVersion[DIA_U16_SOFTWARE_VERSION_PATCH_INDEX_0]);
	strPatch.push_back(__CISInstallationContainerSoftwareVersion[DIA_U16_SOFTWARE_VERSION_PATCH_INDEX_1]);
	
	const char* cpPatch = strPatch.c_str();
	patch = strtoul(cpPatch, 0, DIA_C_BASE_FORMAT_DECIMAL);
	
    DIA_TR_INF("#### Version Info -> Year : %d | Calendar Week : %d | Patch Level : %d ####", year, cw, patch);

	mSwVersion.clear();
	mSwVersion.push_back(year);
	mSwVersion.push_back(cw);
	mSwVersion.push_back(patch);
	
}
/**
*
* \brief      method called to parse xml
*
*             This method is used to parse the xml file.
*             
* \param[in]  file      xml file path
*
* \return     none
*/
void
dia_SrvHandlerInstallationContainerSwVersion::parseXML ( const std::string& file )
{
   ScopeTrace oTrace("dia::dia_SrvHandlerInstallationContainerSwVersion::parseXML(...)");
   
   XMLDocument xmlDoc;
   XMLError retCode = xmlDoc.LoadFile(file.c_str());

   if(XML_SUCCESS == retCode)
   {
   		size_t found = file.find_last_of("/\\");
      	std::string FileName = file.substr(found+1);
      	std::string FilePath = file.substr(0,found);

      	DIA_TR_INF("+------------------------------+");
      	DIA_TR_INF("| START PARSING OF XML CONTENT |");
      	DIA_TR_INF("+------------------------------+");
      	DIA_TR_INF("");
      	DIA_TR_INF("+ PATH: \"%s\"",FilePath.c_str());
      	DIA_TR_INF("+ FILE: \"%s\"",FileName.c_str());

      	XMLElement* root = xmlDoc.RootElement();
	
		for( XMLNode* node = root->FirstChild(); node; node = node->NextSibling() )
		{
            if(!__CISInstallationContainerSoftwareVersionFound)
            {
			     std::string Name = node->Value();
			     DIA_TR_INF("dia_SrvHandlerInstallationContainerSwVersion:  Node Name: \"%s\"", Name.c_str());
			     recursiveParse(node);  	
            }
		}
   }

}
/**
*
* \brief      recursive method called to parse the node
*
*             This method is used to parse the xml node and search the desired target value. 
*             Once the target is available, the value is extraced and stored in member variable.
*             
* \param[in]  node      xml node pointer
*
* \return     none
*/
void 
dia_SrvHandlerInstallationContainerSwVersion::recursiveParse(XMLNode* node)
{
    // putting below trace makes my trace ugly & un-readable as this is a recursive function :(
    // it simply floods the trace
	//ScopeTrace oTrace("dia::dia_SrvHandlerInstallationContainerSwVersion::recursiveElementParse(...)");	
	
	if(node)
	{
		XMLElement* InstallationContainerElement = node->ToElement();
		if(InstallationContainerElement)
		{
			std::string InstallationContainerElementName = InstallationContainerElement->Value();

			if(targetNodeName == InstallationContainerElementName)
			{
				if(InstallationContainerElement->GetText() != NULL)
				{
					__CISInstallationContainerSoftwareVersion = InstallationContainerElement->GetText();
					__CISInstallationContainerSoftwareVersionFound = TRUE;
					DIA_TR_INF("Element Name: \"%s\"",InstallationContainerElementName.c_str());
					DIA_TR_INF("Software Version : \"%s\"",__CISInstallationContainerSoftwareVersion.c_str());
					return;
				}
				else
				{
					DIA_TR_INF("dia_SrvHandlerInstallationContainerSwVersion:  Element Name: CfgVersion=Not found"	);
					return;
				}
			}
		}

		XMLNode* n = node->FirstChild();
		if(!n)
		{
			XMLElement* e = node->ToElement();
								
			if(e)
			{
				std::string eName = e->Value();
				DIA_TR_INF("dia_SrvHandlerInstallationContainerSwVersion:  Element Name: \"%s\"",eName.c_str());
			}
		}
		else
		{
			std::string nName = node->Value();
			DIA_TR_INF("dia_SrvHandlerInstallationContainerSwVersion:  Node Name: \"%s\"", nName.c_str());
		
			while(n)
			{
				recursiveParse(n);

                if(!__CISInstallationContainerSoftwareVersionFound)
                {
				    n  = n->NextSibling();
                }
                else
                {
                    break;
                }
			}
		}
	}
} /* dia_SrvHandlerInstallationContainerSwVersion::recursiveElementParse(XMLNode* node) */

} /* namespace dia */
