/*!
 * \file       dia_SrvHandlerGenericReadFaultMemoryCMC.cpp
 *
 * \brief      Special handling for CMC project in Programming Session (NRC 0x12 instead of NRC 0x7E according to UDS)
 *
 * \details
 *
 * \component  Diagnosis
 *
 * \ingroup    Service Handler for Session Switch
 *
 * \copyright  (c) 2017 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_ENGINE_MANAGER__
#include "common/framework/engine/dia_EngineManager.h"
#endif

#ifndef __INCLUDED_DIA_ENGINE_SERVER__
#include "common/framework/engine/dia_EngineServer.h"
#endif

#ifndef __INCLUDED_DIA_SESSION__
#include "common/framework/engine/dia_Session.h"
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include "common/framework/config/dia_ConfigManager.h"
#endif

#ifndef __INCLUDED_DIA_CANSIGNALHANDLER__
#include "project/framework/vin/dia_CANSignalHandler.h"
#endif

#ifndef __INCLUDED_DIA_INTERFACE_VEHICLECLIENT_LISTENER__
#include "project/interfaces/dia_IVehicleClientListener.h"
#endif

#ifndef __INCLUDED_DIA_INTERFACE_VEHICLECLIENT__
#include "project/interfaces/dia_IVehicleClient.h"
#endif
 
#ifndef DIA_SRV_HANDLER_GENERIC_READ_FAULT_MEMORY_CMC_H_
#include "dia_SrvHandlerGenericReadFaultMemoryCMC.h"
#endif

#define DIA_DID_POS_IN_BUFFER                  ((tU8)2)

//------------------------------------------------------------------------------

dia_SrvHandlerGenericReadFaultMemoryCMC::dia_SrvHandlerGenericReadFaultMemoryCMC(tU8 u8DID)
   : dia_SrvHandlerGenericReadFaultMemory(u8DID)
{
   dia_tclFnctTrace oTrace("dia_SrvHandlerGenericReadFaultMemoryCMC::dia_SrvHandlerGenericReadFaultMemoryCMC(tU8)");
}

//------------------------------------------------------------------------------

dia_SrvHandlerGenericReadFaultMemoryCMC::~dia_SrvHandlerGenericReadFaultMemoryCMC ( void )
{
	_BP_TRY_BEGIN
    {
       (void) unsetSysAdapterListener<dia_IVehicleClientListener>(this);
    }
    _BP_CATCH_ALL
    {
        DIA_TR_ERR("EXCEPTION CAUGHT: dia_SrvHandlerGenericReadFaultMemoryCMC::~dia_SrvHandlerGenericReadFaultMemoryCMC !!!");
        DIA_ASSERT_ALWAYS();
    }
    _BP_CATCH_END
}

//------------------------------------------------------------------------------

void
dia_SrvHandlerGenericReadFaultMemoryCMC::vProcessRequest ( const std::vector<tVoid*>& vecArgs )
{
   dia_tclFnctTrace oTrace("dia_SrvHandlerGenericReadFaultMemoryCMC::vProcessRequest()");

   tU8 u8SubId = oDiagMsgBuffer().u8GetData(DIA_DID_POS_IN_BUFFER);
   tU8 activeSesssionID = getActiveSessionID();

   if (DIA_C_U8_UDS_SESSION_PROGRAMMING==activeSesssionID)
   {

      if ((UDS_REPORT_DTC_SNAPSHOT_RECORD_BY_DTC_NUMBER==u8SubId) || (UDS_REPORT_DTC_EXTENDED_DATARECORD_BY_DTC_NUMBER==u8SubId))
      {
         DIA_TR_INF("dia_SrvHandlerGenericReadFaultMemoryCMC::vProcessRequest Programming session.");
         vSendNegativeResponse(DIA_E_U8_UDS_SUBFUNCTION_NOT_SUPPORTED);
         return;
      }
      else
      {
         DIA_TR_INF("dia_SrvHandlerGenericReadFaultMemoryCMC::vProcessRequest u8SubId=0x%02X.", u8SubId);
      }
   }
   else
   {
      DIA_TR_INF("dia_SrvHandlerGenericReadFaultMemoryCMC::vProcessRequest sessionID = 0x%02X.", activeSesssionID);
   }

   /* CMC-19 Proj specific Changes */
   if(UDS_REPORT_NUMBER_OF_DTC_BY_STATUS_MASK==u8SubId || UDS_REPORT_DTC_BY_STATUS_MASK==u8SubId || UDS_REPORT_DTC_EXTENDED_DATARECORD_BY_DTC_NUMBER == u8SubId)
   {
	   /*Update Historical Interrogation Record DPRS 1723*/
	   vUpdateHistoricalInterrogationRecord();
   }
   else
   {
	   /*Do Nothing*/
   }

   return dia_SrvHandlerGenericReadFaultMemory::vProcessRequest(vecArgs);
}

//------------------------------------------------------------------------------

tU8
dia_SrvHandlerGenericReadFaultMemoryCMC::getActiveSessionID(void)
{
   dia_tclFnctTrace oTrace("dia_SrvHandlerGenericReadFaultMemoryCMC::getActiveSessionID()");

   tU8 SesID = 0x00; //invalid session ID

   dia_EngineServer* pEngine = 0;
   if ( getInstanceOfEngineManager()->queryEngineServer(DIA_UID_ENGINE_CUSTOMER_UDS,&pEngine) != DIA_SUCCESS )
   {
      DIA_TR_ERR("dia_SrvHandlerGenericReadFaultMemoryCMC::getActiveSessionID FAILED TO GET ACTIVE DIAGNOSTIC SESSION !!!!");
   }
   else
   {
      if (NULL!=pEngine)
      {
         SesID = pEngine->getActiveSession()->getID();
      }
      else
      {
         DIA_TR_ERR("dia_SrvHandlerGenericReadFaultMemoryCMC::getActiveSessionID pEngine is NULL.");
      }
   }

   DIA_TR_INF("dia_SrvHandlerGenericReadFaultMemoryCMC::getActiveSessionID returned 0x%02X.", SesID);

   return SesID;
}

void
dia_SrvHandlerGenericReadFaultMemoryCMC::vUpdateHistoricalInterrogationRecord(void)
{
	dia_tclFnctTrace oTrace("dia_SrvHandlerGenericReadFaultMemoryCMC::vUpdateHistoricalInterrogationRecord()");
	// data memory
	
	dataVec.reserve(DIA_HIST_INT_REC_LEN);
	dataVec.resize(DIA_HIST_INT_REC_LEN);

    tDiaResult retCode = dia_getProperty(DIA_PROP_CENTER_CMC_19_HISTORICAL_INTERROGATON_RECORD,dataVec);
	if ( retCode == DIA_SUCCESS )
	{
		DIA_TR_INF("dia_SrvHandlerGenericReadFaultMemoryCMC::vUpdateHistoricalInterrogationRecord DIA_SUCCESS ");
	}
	else
	{
		DIA_TR_ERR("dia_SrvHandlerGenericReadFaultMemoryCMC::vUpdateHistoricalInterrogationRecord UNABLE TO READ ITEM (READ OPERATION FAILED) !!");
		// return ; we have to update the data anyway!
	}

	/*Update DTC Read Counter*/
	vUpdateDTCReadCounter(dataVec);
	/*Update Odometer Value*/
	vUpdateOdometer();

    return ;
}

void
dia_SrvHandlerGenericReadFaultMemoryCMC::vUpdateDTCReadCounter(std::vector<tU8> &dataVec)
{
	dia_tclFnctTrace oTrace("dia_SrvHandlerGenericReadFaultMemoryCMC::vUpdateDTCReadCounter()");
	if(dataVec[0] != DIA_MAX_READ_DTC_COUNTER)
	{
		dataVec[0] += 1;
		DIA_TR_INF("dia_SrvHandlerGenericReadFaultMemoryCMC::vUpdateDTCReadCounter()  Read  DTC Counter =0x%02X ",dataVec[0]  );
	}
	else
	{
		/*Read DTC Counter will be latched at 0xFE once value reaches 0xFE DPRS 1925 */
		DIA_TR_ERR("dia_SrvHandlerGenericReadFaultMemoryCMC::vUpdateDTCReadCounter()  Read  DTC Counter Latched at FE ");
	}
    return ;
}

void
dia_SrvHandlerGenericReadFaultMemoryCMC::vUpdateOdometer()
{
	dia_tclFnctTrace oTrace("dia_SrvHandlerGenericReadFaultMemoryCMC::vUpdateOdometer()");

	tU8 u8CANSign_Odo_Data[4] = {0};
	tU32 u32RawOdometerVal, u32PhyOdometerVal;
	tU16 u16OdometerVal_16km_bit;

	tBool errorDetected = TRUE;

	dia_IVehicleClient* pVehClient = 0;

	if ( querySysAdapterInterface<dia_IVehicleClient>(&pVehClient) == DIA_SUCCESS )
	{
		if ( pVehClient )
		{
			(void) setSysAdapterListener<dia_IVehicleClientListener>(this);
			if ( pVehClient->getTotalVehDistDiag() == DIA_SUCCESS )
			{
				errorDetected = FALSE;
			}
		}
	}
	if ( errorDetected )
	{
		(void) unsetSysAdapterListener<dia_IVehicleClientListener>(this);
		DIA_TR_ERR("dia_SrvHandlerGenericReadFaultMemoryCMC ---  SEND TO vehicle Component failed!!!!");
		oDiagMsgBuffer().vSetNegResp(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
		vResReadyAndQuit();
	}

	return;
}


//! reply on getting total distance info from CAN
void dia_SrvHandlerGenericReadFaultMemoryCMC::vOnTotalVehDistDiag ( tU32 u32RawOdometerVal )
{
	tU32 u32PhyOdometerVal;
	tU16 u16OdometerVal_16km_bit;
	
	DIA_TR_INF("dia_SrvHandlerGenericReadFaultMemoryCMC::vOnTotalVehDistDiag() Raw Odometer=0x%08X ", u32RawOdometerVal);

	u32PhyOdometerVal = DIA_M_MAKE_ODOMETER_VALUE_PHY(u32RawOdometerVal);
	DIA_TR_INF("dia_SrvHandlerGenericReadFaultMemoryCMC::vOnTotalVehDistDiag() Physical Odometer=0x%08X ", u32PhyOdometerVal);

	u16OdometerVal_16km_bit = tU16(u32PhyOdometerVal/DIA_ODOMETER_FACTOR);
	DIA_TR_INF("dia_SrvHandlerGenericReadFaultMemoryCMC::vOnTotalVehDistDiag() Current Odometer=0x%04X ", u16OdometerVal_16km_bit);

	if(u16OdometerVal_16km_bit < DIA_ODOMETER_MAX_VALID)
	{
		dataVec[1] = tU8((0xFF00 & u16OdometerVal_16km_bit )>>8);
		dataVec[2] = tU8(0x00FF & u16OdometerVal_16km_bit) ;

		DIA_TR_INF("dia_SrvHandlerGenericReadFaultMemoryCMC::vOnTotalVehDistDiag() Final value to save in Memory Odometer=0x%04X ",u16OdometerVal_16km_bit);
	}
	else
	{
		dataVec[1] = tU8((0xFF00 & DIA_ODOMETER_MAX_VALID )>>8);
		dataVec[2] = tU8(0x00FF & DIA_ODOMETER_MAX_VALID) ;
		DIA_TR_INF("dia_SrvHandlerGenericReadFaultMemoryCMC::vOnTotalVehDistDiag()  Odometer exceeds maximum value 0xFFFD and latched at 0xFFFD ");
	}

	dataVec.resize(DIA_HIST_INT_REC_LEN);
	tDiaResult tmpResult = dia_setProperty(DIA_PROP_CENTER_CMC_19_HISTORICAL_INTERROGATON_RECORD, dataVec);
	if ( tmpResult == DIA_SUCCESS )
	{
		DIA_TR_INF("dia_SrvHandlerGenericReadFaultMemoryCMC::vOnTotalVehDistDiag");
	}
	else
	{
		/*Data writing to kDS Failed*/
		DIA_TR_ERR("dia_SrvHandlerGenericReadFaultMemoryCMC::vOnTotalVehDistDiag UNABLE TO WRITE ITEM (WRITE TO KDS OPERATION FAILED) !!");
	}
	return;
}
