/*!
 * \file       dia_SrvHandlerCVBSConnectionStatus.cpp
 *
 * \brief      Service handler that will read the current CVBS Connection Status
 *
 * \details    ...
 *
 * \component  Diagnostics
 *
 * \ingroup    customer diagnosis
 *
 * \author     Arjun Manjunath Sanu (RBEI/ECA2)
 *
 * \date       11.06.2020
 *
 * \copyright  (c) 2020 Robert Bosch Engineering & Business Solutions Ltd.
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */
 
#ifndef __INCLUDED_DIA_COMMON_SYSTEM_ADAPTERS__
#include "common/framework/sysadapters/dia_common_system_adapters.h"
#endif

#ifndef __INCLUDED_DIA_SRV_HANDLER_RDID_CVBS_CONNECTION_STATUS__
#include "project/services/customer/dia_SrvHandlerCVBSConnectionStatus.h"
#endif

#define DATA_START 3
#define DATA_LENGTH 1
#define CVBS_CONNECTED  	(tU8(0x01))
#define CVBS_NOT_CONNECTED  (tU8(0x00))

//-----------------------------------------------------------------------------

dia_SrvHandlerCVBSConnectionStatus::dia_SrvHandlerCVBSConnectionStatus ( tCString name, tU8 sid, tU16 did )
    : dia_ServiceHandlerUDS(name,sid,did)
{
	dia_tclFnctTrace trc("dia_SrvHandlerCVBSConnectionStatus::dia_SrvHandlerCVBSConnectionStatus");
}

//-----------------------------------------------------------------------------

dia_SrvHandlerCVBSConnectionStatus::~dia_SrvHandlerCVBSConnectionStatus ( void )
{
    _BP_TRY_BEGIN
    {
       (void) unsetSysAdapterListener<dia_ICVBSConnectionListener>(this);
    }
    _BP_CATCH_ALL
    {
        DIA_TR_ERR("EXCEPTION CAUGHT: dia_SrvHandlerCVBSConnectionStatus::~dia_SrvHandlerCVBSConnectionStatus !!!");
        DIA_ASSERT_ALWAYS();
    }
    _BP_CATCH_END
}

//-------------------------------------------------------------------------------

void
dia_SrvHandlerCVBSConnectionStatus::vProcessRequest ( const std::vector<void*>& /*vecArgs*/ )
{
   dia_tclFnctTrace trc("dia_SrvHandlerCVBSConnectionStatus::vProcessRequest");

   tBool errorDetected = TRUE;

   dia_ICVBSConnection* pInterface = NULL;
   if ((querySysAdapterInterface<dia_ICVBSConnection>(&pInterface) == DIA_SUCCESS) && pInterface)
   {
      (void) setSysAdapterListener<dia_ICVBSConnectionListener>(this);

      if (pInterface->getCVBSConnectionStatus() == DIA_SUCCESS)
      {
         errorDetected = FALSE;
      }
      else
      {
         DIA_TR_INF("dia_SrvHandlerCVBSConnectionStatus::vProcessRequest - getCVBSConnectionStatus() failed.");
      }
   }
   else
   {
      DIA_TR_ERR("dia_SrvHandlerCVBSConnectionStatus::vProcessRequest - querySysAdapterInterface NOK !");
   }

   if (errorDetected)
   {
      (void) unsetSysAdapterListener<dia_ICVBSConnectionListener>(this);
      DIA_TR_ERR("dia_SrvHandlerCVBSConnectionStatus ---  SEND TO INTERFACE FAILED !!!");
      vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
   }
}

//-----------------------------------------------------------------------------

void
dia_SrvHandlerCVBSConnectionStatus::vOnCVBSConnectionStatus (dia_eCVBSConnectionInfo status)
{
   dia_tclFnctTrace trc("dia_SrvHandlerCVBSConnectionStatus::vOnCVBSConnectionStatus");

   DIA_TR_INF("dia_SrvHandlerCVBSConnectionStatus:: CVBS Connection Status Received: %s", ((status == 0x01) ? "Connected" : "Not Connected"));

    tU8 CvbsConStatus = CVBS_NOT_CONNECTED;
   (void) unsetSysAdapterListener<dia_ICVBSConnectionListener>(this);

   oDiagMsgBuffer().vSetPosResp();
   oDiagMsgBuffer().vSetDataLength(DATA_START + DATA_LENGTH);
   
   if(status == DIA_EN_CVBS_CONNECTION_INFO_CONNECTED)
   {
	   CvbsConStatus = CVBS_CONNECTED;
   }

   (void) oDiagMsgBuffer().vSetDataU8((tS16) DATA_START, CvbsConStatus);

   vResReadyAndQuit();
}