/*!
 * \file       dia_RtCtrlSelfDiagnosis.cpp
 *
 * \brief      service handler for routine control Self Diagnosis
 *
 * \details    this service sends trigger to external ECUs to start SelfDiagnosis
 *
 * \component  Diagnosis
 *
 * \ingroup    diaServicesProject
 *
 * \copyright  (c) 2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include <common/framework/sysadapters/dia_SystemAdapterFacade.h>
#endif

#ifndef __INCLUDED_DIA_COMMON_UDS_RTCTRL__
#include <common/framework/protocols/uds/rtctrl/dia_common_uds_rtctrl.h>
#endif

#ifndef __INCLUDED_DIA_ROUTINE_CTRL_SELF_DIAGNOSIS__
#include "dia_RtCtrlSelfDiagnosis.h"
#endif

#ifndef __INCLUDED_DIA_INTERFACE_SYSTEMSETTINGS__
#include <common/interfaces/dia_ISystemSettings.h>
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include "common/framework/config/dia_ConfigManager.h"
#endif

#define routineStatusRecord 0x10

static tU8 sResultMapping[DIA_EN_RTCTRL_STATUS_COUNT] = {
      0xFF, // DIA_EN_RTCTRL_STATUS_UNKNOWN
      0xFF, // DIA_EN_RTCTRL_STATUS_IDLE
      0xFF, // DIA_EN_RTCTRL_STATUS_IN_PROGRESS
      0x00, // DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK
      0x01, // DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK
      0xFF, // DIA_EN_RTCTRL_STATUS_ABORTED
      0xFF  // DIA_EN_RTCTRL_STATUS_TIMED_OUT
};

//------------------------------------------------------------------------------

dia_RtCtrlSelfDiagnosis::dia_RtCtrlSelfDiagnosis ( void )
   : dia_Routine("dia_RtCtrlSelfDiagnosis", DIA_C_U16_DID_RTCTRL_SELF_DIAGNOSIS)
{
   dia_tclFnctTrace oTrace("dia_RtCtrlSelfDiagnosis::dia_RtCtrlSelfDiagnosis()");
}

//------------------------------------------------------------------------------

dia_RtCtrlSelfDiagnosis::~dia_RtCtrlSelfDiagnosis ( void )
{
}

//------------------------------------------------------------------------------

tDiaResult
dia_RtCtrlSelfDiagnosis::start ( std::vector<tU8>& params, tU8 /*timerValue*/ )
{
   dia_tclFnctTrace oTrace("dia_RtCtrlSelfDiagnosis::start()");

   // prepare processing of the routine
   vInitialize();

   if ( params.size() != 0 ) {
      DIA_TR_INF("dia_RtCtrlSelfDiagnosis::start INVALID PARAMETERS");
      return DIA_FAILED;
   }

   dia_ISystemSettings* pInterface = 0;
   tDiaResult retCode = queryInterface<dia_ISystemSettings>(&pInterface);
   if ((retCode == DIA_SUCCESS) && pInterface) {
      eSetStatus(DIA_EN_RTCTRL_STATUS_IN_PROGRESS);
      if (DIA_SUCCESS == pInterface->processSystemSetting(DIA_C_UID_SYSTEM_SETTING_TYPE_SELF_DIAGNOSIS, *this, this)) {
         DIA_TR_INF("dia_RtCtrlSelfDiagnosis::start processSystemSetting is success.");
         retCode = DIA_SUCCESS;
      } else {
         DIA_TR_INF("dia_RtCtrlSelfDiagnosis::start no flush done!!!");
         mResults.push_back(sResultMapping[DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK]);
         mIsResultReady = TRUE;
         eSetStatus(DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK);
      }
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_RtCtrlSelfDiagnosis::requestResult ( std::vector<tU8>& results )
{
   dia_tclFnctTrace oTrace("dia_RtCtrlSelfDiagnosis::requestResult()");

   tDiaResult retCode = DIA_E_SEQUENCE_ERROR;

   DIA_TR_INF("dia_RtCtrlSelfDiagnosis::requestResult: mStatus = %d", mStatus);

   results.clear();
   if ( !(mResults.empty()) )
   {
      DIA_TR_INF("requestResult --- 1");
      std::vector<tU8>::iterator iter = mResults.begin();
      for ( ; iter != mResults.end(); iter++ )
      {
         results.push_back(*iter);
      }
      mResults.clear();
      retCode = DIA_SUCCESS;
   }
   else
   {
      DIA_TR_INF("requestResult --- 2");
      switch ( mStatus )
      {
      case DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK:
      case DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK:
         {
            DIA_TR_INF("requestResult --- 3 (Status %d) --> return value 0x%02x", mStatus, sResultMapping[mStatus]);
            // constant value 10
            results.push_back(routineStatusRecord);
            // collect results
            results.push_back(sResultMapping[mStatus]);
            mIsResultReady = FALSE; //prevent doubled positive response in case of synchronous processing
            retCode = DIA_SUCCESS;
         }
         break;

      default:
         DIA_TR_INF("requestResult --- 4");
         break;
      }
   }

   return retCode;
}

//------------------------------------------------------------------------------

void
dia_RtCtrlSelfDiagnosis::vOnServiceTimeout ( void )
{
   dia_tclFnctTrace oTrace("dia_RtCtrlSelfDiagnosis::vOnServiceTimeout()");
}

//------------------------------------------------------------------------------

void
dia_RtCtrlSelfDiagnosis::onSystemSettingProcessed( dia_UID /*typeID*/, const dia_SystemSettingExtendedData& /*extData*/, tDiaResult resultCode, void* cookie )
{
   dia_tclFnctTrace oTrace("dia_RtCtrlSelfDiagnosis::onSystemSettingProcessed()");

   if ( cookie != ((void*) this) ) return;

   if(resultCode==DIA_FAILED){
      DIA_TR_INF("dia_RtCtrlSelfDiagnosis::onSystemSettingProcessed result DIA_FAILED.");
      eSetStatus(DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK);
   } else {
      DIA_TR_INF("dia_RtCtrlSelfDiagnosis::onSystemSettingProcessed result DIA_SUCCESS.");
      eSetStatus(DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK);
   }

   mIsResultReady = TRUE;
   dia_RoutineCtrlManager::getInstance()->vOnRoutineUpdate(*this);
}

