/*!
* \file       dia_RtCtrlGetBT_SigQ_RSSI.cpp
*
* \brief      service handler for routine control Bluetooth RSSI and Link Quality
*
* \details    this service sends trigger to Bluetooth component to read its Signal Info
*
* \component  Diagnosis
*
* \ingroup    diaServicesProject
*
* \copyright  (c) 2016 Robert Bosch GmbH
*
* The reproduction, distribution and utilization of this file as
* well as the communication of its contents to others without express
* authorization is prohibited. Offenders will be held liable for the
* payment of damages. All rights reserved in the event of the grant
* of a patent, utility model or design.
*/


#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#ifndef DIA_RTCTRLGETBT_SIGQ_RSSI_H_
#include "dia_RtCtrlGetBT_SigQ_RSSI.h"
#endif

enum
{
   DIA_RTCTRL_COMPLETED_OK = 0,
   DIA_RTCTRL_COMPLETED_NOK
};

//------------------------------------------------------------------------------

dia_RtCtrlGetBT_SigQ_RSSI::dia_RtCtrlGetBT_SigQ_RSSI()
         : dia_Routine("dia_RtCtrlGetBT_SigQ_RSSI", DIA_C_U16_ID_RTCTRL_GET_BT_SIGQ_RSSI,
               DIA_EN_RTCTRL_TYPE_SHORT_TERM)
{
   dia_tclFnctTrace oTrace("dia_RtCtrlGetBT_SigQ_RSSI::dia_RtCtrlGetBT_SigQ_RSSI()");
}

//------------------------------------------------------------------------------

dia_RtCtrlGetBT_SigQ_RSSI::~dia_RtCtrlGetBT_SigQ_RSSI()
{
   _BP_TRY_BEGIN
   {
      (void) unsetSysAdapterListener<dia_IBluetoothListener>(this);
   }
   _BP_CATCH_ALL
   {
       DIA_TR_ERR("EXCEPTION CAUGHT: dia_RtCtrlGetBT_SigQ_RSSI::~dia_RtCtrlGetBT_SigQ_RSSI !!!");
       DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//------------------------------------------------------------------------------

tDiaResult
dia_RtCtrlGetBT_SigQ_RSSI::start ( std::vector<tU8>& params, tU8 /*timerValue*/ )
{
   dia_tclFnctTrace trc("dia_RtCtrlGetBT_SigQ_RSSI::start");

   tDiaResult retCode = DIA_FAILED;

   // prepare processing of the routine
   vInitialize();

   if (params.size() < DIA_SIZE_OF_MAC_ADDRESS)
   {
      DIA_TR_INF("dia_RtCtrlGetBT_SigQ_RSSI::start ERROR Size of MAC address incorrect.");

      tU8 i = 0;
      for (std::vector<tU8>::iterator it=params.begin(); it!=params.end(); ++it, ++i)
      {
         DIA_TR_INF("dia_RtCtrlGetBT_SigQ_RSSI::start params[%d] = 0x%02X", i, (*it));
      }
	  retCode = DIA_E_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT;
      return retCode;
   }

   dia_IBluetooth* pIBluetooth = 0;
   if ( querySysAdapterInterface<dia_IBluetooth>(&pIBluetooth) == DIA_SUCCESS )
   {
      if ( pIBluetooth )
      {
         (void) setSysAdapterListener<dia_IBluetoothListener>(this);

         //tU8 au8BDAddress[DIA_SIZE_OF_MAC_ADDRESS] = { 0 };

         if (params.size() >= DIA_SIZE_OF_MAC_ADDRESS)
         {
            for (tU8 i=0; i<DIA_SIZE_OF_MAC_ADDRESS; i++)
            {
               //au8BDAddress[i] = params.at(i);
               macAddr[i] = params.at(i);
            }
         }

         DIA_TR_INF("dia_RtCtrlGetBT_SigQ_RSSI::start => MAC=%02x:%02x:%02x:%02x:%02x:%02x",
               macAddr[0],macAddr[1],macAddr[2],macAddr[3],macAddr[4],macAddr[5]);

         if ( pIBluetooth->vGetBluetoothFieldstrength(&macAddr[0]) != DIA_SUCCESS)
         {
            DIA_TR_INF("!!! dia_RtCtrlGetBT_SigQ_RSSI::start => ERROR: vGetBluetoothFieldstrength != DIA_SUCCESS");
         }
         else
         {
            DIA_TR_INF("--- dia_RtCtrlGetBT_SigQ_RSSI::start => Send Successfully to Server");
            eSetStatus(DIA_EN_RTCTRL_STATUS_IN_PROGRESS);
            retCode = DIA_SUCCESS;
         }
      }
      else
      {
         DIA_TR_INF("dia_RtCtrlGetBT_SigQ_RSSI::start pInterface is NULL");
      }
   }
   else
   {
      DIA_TR_ERR("dia_RtCtrlGetBT_SigQ_RSSI::start querySysAdapterInterface FAILED");
   }

   if ( retCode != DIA_SUCCESS )
   {
      (void) unsetSysAdapterListener<dia_IBluetoothListener>(this);
   }

   return retCode;
}

//------------------------------------------------------------------------------

void
dia_RtCtrlGetBT_SigQ_RSSI::vOnBT_SigQ_RSSI ( tU8 u8SigQ, tU8 u8RSSI )
{
   dia_tclFnctTrace trc("dia_RtCtrlGetBT_SigQ_RSSI::vOnBT_SigQ_RSSI");

   DIA_TR_INF("dia_RtCtrlGetBT_SigQ_RSSI::vOnBT_SigQ_RSSI SigQ=0x%02X, RSSI=0x%02X", u8SigQ , u8RSSI);

   (void) unsetSysAdapterListener<dia_IBluetoothListener>(this);

   eSetStatus(DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK);

   mResults.push_back(RT_STATUS_RECORD);        // Routine Status Record
   mResults.push_back(DIA_RTCTRL_COMPLETED_OK); // Status

   for (tU8 i=0; i<DIA_SIZE_OF_MAC_ADDRESS; i++)
   {
	   mResults.push_back(macAddr[i]);          // MacAddress of BT Device
   }

   mResults.push_back(u8RSSI);                  // RSSI

   mResults.push_back(u8SigQ);                  // Signal Link Quality

   mIsResultReady = TRUE;

#ifndef __DIA_UNIT_TESTING__
   dia_RoutineCtrlManager* pRTCtrlMgr = getInstanceOfRoutineControlManager();
   if ( pRTCtrlMgr )
   {
      pRTCtrlMgr->vOnRoutineUpdate(*this);
   }
#endif
}

//----------------------------------------------------------------------------------------

void
dia_RtCtrlGetBT_SigQ_RSSI::vOnBT_SigQ_RSSIError (const tU32 errCode)
{
   dia_tclFnctTrace trc("dia_RtCtrlGetBT_SigQ_RSSI::vOnBT_SigQ_RSSIError");

      DIA_TR_INF( " dia_RtCtrlGetBT_SigQ_RSSI::vOnBT_SigQ_RSSIError => errCode= %d",errCode);

      (tVoid) unsetSysAdapterListener<dia_IBluetoothListener>(this);

      eSetStatus(DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK);
      mResults.push_back(DIA_RTCTRL_COMPLETED_NOK); // status

      mIsResultReady = TRUE;

#ifndef __DIA_UNIT_TESTING__
         dia_RoutineCtrlManager* pRTCtrlMgr = getInstanceOfRoutineControlManager();
         if ( pRTCtrlMgr )
         {
            pRTCtrlMgr->vOnRoutineUpdate(*this);
         }
#endif

}

//------------------------------------------------------------------------------

tDiaResult
dia_RtCtrlGetBT_SigQ_RSSI::requestResult ( std::vector<tU8>& results )
{
   dia_tclFnctTrace oTrace("dia_RtCtrlGetBT_SigQ_RSSI::requestResult()");

   tDiaResult retCode = DIA_E_SEQUENCE_ERROR;

   results.clear();
   if ( !(mResults.empty()) )
   {
      DIA_TR_INF("requestResult --- 1");
      std::vector<tU8>::iterator iter = mResults.begin();
      for ( ; iter != mResults.end(); ++iter )
      {
         results.push_back(*iter);
      }
      mResults.clear();
      retCode = DIA_SUCCESS;
   }
   else
   {
      DIA_TR_INF("requestResult --- 2");
      switch ( mStatus )
      {
         case DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK:
         case DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK:
         {
            DIA_TR_ERR("dia_RtCtrlGetBT_SigQ_RSSI::requestResult => Status NOK");
            break;
         }

      default:
         break;
      }
   }

   return retCode;
}

//------------------------------------------------------------------------------

void
dia_RtCtrlGetBT_SigQ_RSSI::vOnServiceTimeout ( void )
{
   dia_tclFnctTrace oTrace("dia_RtCtrlGetBT_SigQ_RSSI::vOnServiceTimeout");

   (void) unsetSysAdapterListener<dia_IBluetoothListener>(this);
}
