/*
 * dia_RoutineCtrlSystemInstallation.cpp
 * brief:Implementation of Routine service to execute SystemInstallation.
 * Created on: May 15, 2019
 * Author: urm7kor
 * copyright  (c) 2017 Robert Bosch GmbH
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_COMMON_UDS_RTCTRL__
#include <common/framework/protocols/uds/rtctrl/dia_common_uds_rtctrl.h>
#endif

#ifndef __INCLUDED_DIA_DEFINES_UDS__
#include <common/framework/protocols/uds/dia_defsUds.h>
#endif

#ifndef __INCLUDED_DIA_ROUTINE_CONTROL_MANAGER__
#include "common/framework/protocols/uds/rtctrl/dia_RoutineCtrlManager.h"
#endif

#ifndef __INCLUDED_DIA_UTILITIES__
#include "common/framework/utils/dia_utilities.h"
#endif
#ifndef __INCLUDED_DIA_DEFS_CONFIG_PROJECT__
#include "project/framework/config/dia_defsProjectConfig.h"
#endif
#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include "common/framework/config/dia_ConfigManager.h"
#endif

#ifndef __INCLUDED_DIA_FILE__
#include <common/framework/application/dia_File.h>
#endif
#include <sys/wait.h>
#ifndef __INCLUDED_DIA_ROUTINE_CTRL_SYSTEM_INSTALLATION_H__
#include "dia_RoutineCtrlSystemInstallation.h"
#endif

#include <sstream>
#include <common/framework/xml/tinyxml2.h>

using namespace tinyxml2;

static const std::string fullCmdName("/bin/bash");
static const std::string scriptFilePathName("/opt/bosch/base/bin/diag_doipcmcinstall.sh");
static const std::string scriptFilePathName2("/var/opt/bosch/persistent/avdecc/cisinstallation/doipcmcinstall.py");

static const std::string targetXMLFilepath  ("/var/opt/bosch/persistent/avdecc/doipcmcinstallresult.xml");
static const std::string targetXMLNodeName ("result");
static const std::string defaultCISSytemInstallationStatusResult ("NoResultFound");

#define DIA_C_U8_ROUTINE_RESULT_COMPLETED_OK       ( (tU8) 0x00)
#define DIA_C_U8_ROUTINE_RESULT_COMPLETED_NOK      ( (tU8) 0x01)
static const char*
convertString2ConstCharPtr ( const std::string& str )
{
   return str.c_str();
}
namespace dia
{
//-----------------------------------------------------------------------------------
dia_RoutineCtrlSystemInstallation::dia_RoutineCtrlSystemInstallation( void )
      : RoutineCtrlExecuteSystemCommand("dia_RoutineCtrlSystemInstallation", DIA_C_U16_DID_CENTER_CMC_19_SYSTEM_INSTALLATION, fullCmdName),
        InStartRoutine(FALSE),
		CISSytemInstallationStatusResult(defaultCISSytemInstallationStatusResult)
{
    dia_tclFnctTrace oTrace("dia_RoutineCtrlSystemInstallation::dia_RoutineCtrlSystemInstallation");
	mType=DIA_EN_RTCTRL_TYPE_LONG_TERM;
}
//--------------------------------------------------------------------------------------
dia_RoutineCtrlSystemInstallation::~dia_RoutineCtrlSystemInstallation() {
	DIA_TR_INF("dia_RoutineCtrlSystemInstallation::~dia_RoutineCtrlSystemInstallation");
	 _BP_TRY_BEGIN
   {
      (void) dia_RoutineCtrlSystemInstallation::tearDown();
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("dia::dia_RoutineCtrlSystemInstallation::~dia_RoutineCtrlSystemInstallation - Exception caught!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}
//---------------------------------------------------------------------------------------------
tDiaResult
dia_RoutineCtrlSystemInstallation::getCommandArguments ( std::vector<std::string>& argv )
{
   dia_tclFnctTrace oTrace("dia::dia_RoutineCtrlSystemInstallation::getCommandArguments");

   argv.push_back(scriptFilePathName);

   return DIA_SUCCESS;
}

bool
dia_RoutineCtrlSystemInstallation::isExisting ( void )
{
   dia_tclFnctTrace oTrace("dia::dia_RoutineCtrlSystemInstallation::isExisting");

   bool retCode = true;

   if ( ::access(scriptFilePathName.c_str(), F_OK) == -1 )
   {
      // the proxy script is not existing
      DIA_TR_ERR("##### PROXY SCRIPT NOT FOUND (\"%s\") #####", scriptFilePathName.c_str());
      // do not continue otherwise execv will crash
      mErrorCode = DIA_E_GENERAL_REJECT;
      mResponseStatus = ROUTINE_NOK;
      retCode = false;
   }
   else if ( ::access(scriptFilePathName2.c_str(), F_OK) == -1 )
   {
      // the final script is not existing
      DIA_TR_ERR("##### FINAL SCRIPT NOT FOUND (\"%s\") #####", scriptFilePathName2.c_str());
      mErrorCode = DIA_E_GENERAL_REJECT;
      mResponseStatus = ROUTINE_NOK;
      retCode = false;
   }

   if (retCode)
   {
      DIA_TR_INF("dia::dia_RoutineCtrlSystemInstallation::isExisting - Check interpreter.");

      //check script interpreter - execute base method
      retCode = RoutineCtrlExecuteSystemCommand::isExisting();
   }

   DIA_TR_INF("##### dia::dia_RoutineCtrlSystemInstallation::isExisting => RETURNS \"%s\" #####", (retCode) ? "TRUE" : "FALSE");

   return retCode;
}

//----------------------------------------------------------------------------------------------
void
dia_RoutineCtrlSystemInstallation::startSystemCommand ( void )
{
	dia_tclFnctTrace oTrace("dia_RoutineCtrlSystemInstallation::startSystemCommand()");
	InStartRoutine = TRUE;
	DIA_TR_INF("dia_RoutineCtrlSystemInstallation::startSystemCommand => fork now !!");
	mChildProcessID = ::fork();
	if ( mChildProcessID >= 0 )
	{
		// child process forked successfully

		if ( mChildProcessID == 0 )
		{
			// at this point we are running in the child process ( pid == 0 )

			std::vector<const char*> cargv(mSystemCommandArgs.size()+1);
			std::transform(mSystemCommandArgs.begin(), mSystemCommandArgs.end(), cargv.begin(), convertString2ConstCharPtr);  //lint !e864 Info : Expression involving variable possibly depends on order of evaluation
			cargv[mSystemCommandArgs.size()] = 0;
			// Create new process group with child as leader.
			if ( ::setpgid(0,0) < 0 ) exit(EXIT_FAILURE);
			(void) ::execv(mFullCommandName.c_str(),const_cast<char**>(cargv.data()));

			// otherwise exit failed
			::exit(EXIT_FAILURE);
		}
		else
		{
			// at this point we are running in the parent process ( pid > 0 )

			// check status of child process, but do not wait (WNOHANG)!!!
			int status = -1;
			pid_t cpid = ::waitpid(mChildProcessID, &status, WNOHANG);

			switch ( cpid )
			{
			case 0: // FOUND
			{
				// we can't say at this moment we really run succesfully or not, but still trace info
				DIA_TR_INF("dia::RoutineCtrlExecuteSystemCommand::start => Child Running Id: %d", mChildProcessID);
				DIA_TR_INF("dia::RoutineCtrlExecuteSystemCommand::start => Child WIFEXITED code: %d", WIFEXITED(status));
				DIA_TR_INF("dia::RoutineCtrlExecuteSystemCommand::start => Child WEXITSTATUS code: %d", WEXITSTATUS(status));
				mIsRunning = true;
				mResponseStatus = ROUTINE_RUNNING;
				mIsResultReady  = TRUE;

				onSystemCommandStarted();
			}
			break;

			case -1:// ERROR
			{
				DIA_TR_ERR("dia::RoutineCtrlExecuteSystemCommand::start => waitpid FAILED - Child Id:%d Errno:%x", mChildProcessID,errno);
				DIA_ASSERT_ALWAYS();
				mIsRunning = false;
				mResponseStatus = ROUTINE_NOK;
				mIsResultReady  = TRUE;

				onSystemCommandStartupFailed();
			}
			break;

			default:
			{
				// from The Open Group Base Specifications Issue 7; IEEE Std 1003.1, 2013 Edition
				//
				// WIFEXITED(stat_val)
				//   Evaluates to a non-zero value if status was returned for a child process that terminated normally.
				//
				// WEXITSTATUS(stat_val)
				//   If value of WIFEXITED(stat_val) is non-zero, this macro evaluates to the low-order 8 bits
				//   of the status argument that the child process passed to _exit() or exit(), or the value
				//   the child process returned from main().

				mResponseStatus = ROUTINE_NOK;

				// this is the first request after exit, so data is valid
				int response = WIFEXITED (status);
				DIA_TR_INF("dia::RoutineCtrlExecuteSystemCommand::start::requestResult => Child WIFEXITED code: %d", response);
				if ( response != 0 )
				{
					// terminated normally
					response = WEXITSTATUS (status);
					DIA_TR_INF("dia::RoutineCtrlExecuteSystemCommand::start::requestResult => Child WEXITSTATUS code: %d",response);
					if ( response == 0 )
					{
						mResponseStatus = ROUTINE_OK;
					}
					(void) evaluateCommandResponse(response);
				}

				mChildProcessID = 0;
				mIsRunning = false;
				mIsResultReady  = TRUE;
			}
			break;
			}
		}
	}
	else
	{
		DIA_TR_ERR("dia::RoutineCtrlExecuteSystemCommand::start => fork failed");
		mResponseStatus = ROUTINE_NOK;
	}
}
////-----------------------------------------------------------------------------------------
tDiaResult
dia_RoutineCtrlSystemInstallation::requestResult ( std::vector<tU8>& results )
{
	dia_tclFnctTrace oTrace("dia_RoutineCtrlSystemInstallation::requestResult()");
	results.clear();

	DIA_TR_INF("mErrorCode is 0x%08X, mResponseStatus is 0x%08X", mErrorCode, mResponseStatus);

	//bug fix for double start
	if ( (DIA_E_SEQUENCE_ERROR==mErrorCode) && (ROUTINE_NOK==mResponseStatus) )
	{
		if (mIsRunning)
		{
			executeRequestSystemCommandResults();

			if (!mIsRunning || (ROUTINE_RUNNING==mResponseStatus))
			{
				DIA_TR_INF("mErrorCode is 0x%08X, mResponseStatus is 0x%08X. Set mErrorCode to DIA_SUCCESS.", mErrorCode, mResponseStatus);
				mErrorCode = DIA_SUCCESS;
			}
		}
	}

	if (DIA_SUCCESS==mErrorCode)
	{
		DIA_TR_INF("mErrorCode is DIA_SUCCESS");

		if ( mIsRunning )
		{
			executeRequestSystemCommandResults();
		}
		else
		{
			// process is still terminate, so we have to use last data we get return last mResponseStatus
			DIA_TR_INF("RoutineCtrlExecuteSystemCommand::requestResult => Child is not running");
		}
		DIA_TR_INF("RoutineCtrlExecuteSystemCommand::requestResult => mResponseStatus:%d", mResponseStatus);
		if(InStartRoutine)
		{
			DIA_TR_INF("dia_RoutineCtrlSystemInstallation::request for start_Routine only!!!");
			results.push_back(mResponseStatus);
		}
		else
		{
			DIA_TR_INF("RoutineCtrlExecuteSystemCommand::request for result_Routine!!!");
			if(mResponseStatus == ROUTINE_OK)
			{
				results.push_back(mResponseStatus);
				results.push_back(DIA_C_U8_ROUTINE_RESULT_COMPLETED_OK);
			}
			else if(mResponseStatus == ROUTINE_RUNNING)
			{
				results.push_back(mResponseStatus);
				results.push_back(DIA_C_U8_ROUTINE_RESULT_COMPLETED_OK);
			}
			else{
				results.push_back(DIA_C_U8_ROUTINE_RESULT_COMPLETED_NOK);
				results.push_back(mResultStatus);
			}
		}

		if(!InStartRoutine)
		{
			CISSytemInstallationStatusResult = defaultCISSytemInstallationStatusResult;

			parseXML(targetXMLFilepath);

			std::vector<tU8> vCISSytemInstallationStatusResult(CISSytemInstallationStatusResult.begin(), CISSytemInstallationStatusResult.end());

			std::vector<tU8>::iterator iter = vCISSytemInstallationStatusResult.begin();
			for ( ; iter != vCISSytemInstallationStatusResult.end(); iter++ )
			{
			results.push_back(*iter);
			}
		}
		else
		{
			InStartRoutine = FALSE;
		}

		// data was filled before by start or stop, so use this
		results.insert(results.end(),mResults.begin(),mResults.end());

		for ( tU16 i=0; i<mResults.size(); i++ )
		{
			DIA_TR_INF("mResults[%02d] = 0x%02x",i,mResults[i]);
		}
	}
	mIsResultReady = TRUE;
	return mErrorCode;
}

/**
*
* \brief      method called to parse xml
*
*             This method is used to parse the xml file.
*
* \param[in]  file      xml file path
*
* \return     none
*/
void
dia_RoutineCtrlSystemInstallation::parseXML ( const std::string& file )
{
   ScopeTrace oTrace("dia::dia_RoutineCtrlSystemInstallation::parseXML(...)");

   XMLDocument xmlDoc;
   XMLError retCode = xmlDoc.LoadFile(file.c_str());

   if(XML_SUCCESS == retCode)
   {
   		size_t found = file.find_last_of("/\\");
      	std::string FileName = file.substr(found+1);
      	std::string FilePath = file.substr(0,found);

      	DIA_TR_INF("+------------------------------+");
      	DIA_TR_INF("| START PARSING OF XML CONTENT |");
      	DIA_TR_INF("+------------------------------+");
      	DIA_TR_INF("");
      	DIA_TR_INF("+ PATH: \"%s\"",FilePath.c_str());
      	DIA_TR_INF("+ FILE: \"%s\"",FileName.c_str());

      	XMLElement* pNode = xmlDoc.RootElement();
		if ( pNode )
		  {
				std::string nodeName = pNode->Value();
				DIA_TR_INF("XML-Node: %s", nodeName.c_str());

				/*current xml file has first element as root*/
				if(nodeName == targetXMLNodeName)
				{
					/*extracting the data of result status*/
					 const char* Attrdata =  pNode->Attribute("status");
					 if(Attrdata)
					 {
						 CISSytemInstallationStatusResult = std::string(Attrdata);
						 DIA_TR_INF("dia_RoutineCtrlSystemInstallation: Result Status : \"%s\"",CISSytemInstallationStatusResult.c_str());
					 }

				}
				else
				 {
					DIA_TR_INF("dia_RoutineCtrlSystemInstallation: Result Status not available in xml file");
				 }
		  }


   }

}

}//namespace dia
