/*
 * \file       dia_IOCtrlTunerSettings.cpp
 *
 *  \author    shw2abt 05.11.2015, kaa1hi 08.05.2017.
 *
 *  \details   Customer Service.
 *             Attention! Valid FM frequency is from 87.7 MHz to 107.9 MHz with step 0.2 MHz.
 *             Attention! Audio source could be changed when System Remote Control is active.
 *             Attention! Request "Return control to ECU" shall reply with NRC 0x22.
 *             Set tuner settings, perform following actions:
 *             1. Enable System Remote Control without HMI and wait for a confirmation (see method vOnRemoteControlModeChanged)
 *             2. Set test mode (see Tuner Master FI SRCCHG_TESTMODE) for a tuner (e.g. DIA_EN_TUNERMASTER_1) and wait for tuner response.
 *             3. Set tuner source (e.g. DIA_EN_AUDIO_SOURCE_TUNER_FM) and wait for tuner response.
 *             4. Set frequency and wait for the tuner response.
 *             5. Wait 20 seconds.
 *             6. Disable System Remote Control without HMI.
 *
 *             ==============================================
 *             | INPUTS   | AM/FM           |  DAB          |
 *             ==============================================
 *             | byte[0]  | tuner type      | tuner type    |
 *             | byte[1]  | (don't care)    | 'C'           |
 *             | byte[2]  | freq. byte#0    | 'N'           |
 *             | byte[3]  | freq. byte#1    | freq. byte#0  |
 *             | byte[4]  | freq. byte#2    | freq. byte#1  |
 *             | byte[5]  | freq. byte#3    | freq. byte#2  |
 *             ==============================================
 *
 * \copyright  Robert Bosch Car Multimedia 2017
 */
//TTFis:> DIA_REQ UDS 02 2F 90 01 03 21 43 4E 31 32 44 (Set DAB with 12D )
//TTFis:> DIA_REQ UDS 02 2F 90 01 03 02 00 06 26 B5 00 (Set FM2 with 103.2MHz (in Hz))
//TTFis:> DIA_REQ UDS 02 2F 90 01 03 02 00 05 F4 5A 60 (Set FM2 with 99.9MHz (in Hz))
//TTFis:> DIA_REQ UDS 02 2F 90 01 03 11 00 00 09 EE F8 (Set AM  with 650kHz (in Hz))

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#ifndef __INCLUDED_DIA_UTILITIES__
#include "common/framework/utils/dia_utilities.h"
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG_PROJECT__
#include "project/framework/config/dia_defsProjectConfig.h"
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include "common/framework/config/dia_ConfigManager.h"
#endif

#ifndef __INCLUDED_DIA_AUDIO_MANAGER__
#include <common/framework/platform/cmd/dia_AudioManager.h>
#endif

#include "project/services/customer/dia_IOCtrlTunerSettings.h"
#include "common/services/uds/generic/dia_SrvHandlerGenericIOCtrlByIdentifier.h"

static const tU32 DIA_PROP_AIVI_SYSTEM_INFO_COMMON_LEN               = 64;   // according definition in dia_defsProjectConfig.h
static const tU32 SYSTEM_INFO_COMMON_TUNER_DAB_BIT_FLAG              = 0x01;
static const tU32 SYSTEM_INFO_COMMON_TUNER_DAB_BYTE_OFFSET           = 30;
static const tU32 IOCTRL_TIMER_VALUE_FOR_TUNER_SETTINGS_IN_SECONDS   = 60 /*seconds */;
static const tU32 DURATION_SYSTEM_REMOTE_CONTROL_IN_MILLI_SECONDS    = 20 /*seconds */ * 1000;
static const tU32 NUMBER_OF_BYTES_INPUT = 6;

//------------------------------------------------------------------------------

dia_IOCtrlTunerSettings::dia_IOCtrlTunerSettings(void) :
   dia_IOCtrlSignal(
         DIA_C_U16_DID_AIVI_TUNER_SETTINGS,
         DIA_EN_IOCTRL_SIGTYPE_TUNER_SETTINGS,
         6 /* Payload Length */
      ),
      mFreqAMFM(0),
      mFreqDAB(0),
      mRequestedSourceID(DIA_EN_AUDIO_SOURCE_UNKNOWN),
      mTunerType(TUNER_TYPE_UNKNOWN),
      mIsCN(false),
      mIsSystemRemoteActiveWithoutHMI(false),
      mDurationSystemRemoteControlInMSeconds(DURATION_SYSTEM_REMOTE_CONTROL_IN_MILLI_SECONDS),
      mOrigSourceID(DIA_EN_AUDIO_SOURCE_UNKNOWN),
      mIsSourceRestoring(false)
{
   mTimer.s32Create();
}

//------------------------------------------------------------------------------

dia_IOCtrlTunerSettings::~dia_IOCtrlTunerSettings(void)
{
   _BP_TRY_BEGIN
   {
      (void) unsetSysAdapterListener<dia_ITunerAMFMListener>(this);

      mTimer.removeTimerListener(this);
      mTimer.s32Delete();
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia_IOCtrlTunerSettings::~dia_IOCtrlTunerSettings !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//------------------------------------------------------------------------------

tDiaResult
dia_IOCtrlTunerSettings::handleRequest(tU8 /*timerValue*/, std::vector<tU8>* ctrlValue)
{
   dia_tclFnctTrace oTrace("dia_IOCtrlTunerSettings::handleRequest(tU8,vector<tU8>*)");
   tDiaResult retCode = DIA_FAILED;

   for(tU8 i=0; i<ctrlValue->size(); i++)
   {
      DIA_TR_INF("dia_IOCtrlTunerSettings::handleRequest Input Vector [%02d]=0x%02X.", i, ctrlValue->at(i));
   }

   if (ctrlValue->size() != NUMBER_OF_BYTES_INPUT)
   {
      DIA_TR_ERR("### Number of input bytes (%d) is different than exp (%d).", ctrlValue->size(), NUMBER_OF_BYTES_INPUT);
      return retCode;
   }

   mTunerType = (TunerType)ctrlValue->at(0);
   mFreqAMFM = U8_2_U32(ctrlValue->at(2),ctrlValue->at(3),ctrlValue->at(4),ctrlValue->at(5));
   mFreqDAB =  U8_2_U32(0x00,ctrlValue->at(3),ctrlValue->at(4),ctrlValue->at(5));  //first byte is zero
   mIsCN = (ctrlValue->at(1)=='C' && ctrlValue->at(2)=='N')? true : false;

   DIA_TR_INF("dia_IOCtrlTunerSettings::handleRequest mTunerType=0x%02X.", mTunerType);
   DIA_TR_INF("dia_IOCtrlTunerSettings::handleRequest mFreqAMFM=0x%08X (dec %d).",  mFreqAMFM, mFreqAMFM);
   DIA_TR_INF("dia_IOCtrlTunerSettings::handleRequest mFreqDAB=0x%08X (dec %d).", mFreqDAB, mFreqDAB);
   DIA_TR_INF("dia_IOCtrlTunerSettings::handleRequest mIsCN=%s.", (mIsCN ? "true" : "false"));

   mOrigSourceID = DIA_EN_AUDIO_SOURCE_UNKNOWN;

   //check DAB condition
   if ((TUNER_TYPE_DAB==mTunerType) && (!isDabActive()))
   {
      DIA_TR_ERR("### dia_IOCtrlTunerSettings::handleRequest DAB requested, but DAB is inactive.");
      return retCode;
   }

   retCode = getInstanceOfAudioManager()->getAudioSource(*this);    //response: onAudioSourceGetResult

   // now initialize the counter for the timer
   vSetTimer(IOCTRL_TIMER_VALUE_FOR_TUNER_SETTINGS_IN_SECONDS);

   mIsResultReady = FALSE;

   if (DIA_SUCCESS==retCode)
   {
      DIA_TR_INF("dia_IOCtrlTunerSettings::handleRequest timer start.");
      startTimer();
   }

   if (DIA_SUCCESS!=retCode)
   {
      DIA_TR_ERR("### dia_IOCtrlTunerSettings::handleRequest retCode=0x%08X.", retCode);
   }

   // NO_ERROR allows the service handler to return immediately
   return retCode;
}

//-------------------------------------------------------------------------

void
dia_IOCtrlTunerSettings::onAudioSourceGetResult ( dia_eAudioSource sourceID  )
{
   dia_tclFnctTrace oTrace("dia_IOCtrlTunerSettings::onAudioSourceGetResult(dia_eAudioSource)");

   mOrigSourceID = sourceID;

   DIA_TR_INF("dia_IOCtrlTunerSettings::onAudioSourceGetResult mOrigSourceID=%d.", mOrigSourceID);

   //Enable System Remote Control without HMI
   if (DIA_SUCCESS!=activateRemoteControl())
   {
      DIA_TR_ERR("### activateRemoteControl failed.");
      return;
   }

   if ( mIsSystemRemoteActiveWithoutHMI )
   {
      DIA_TR_INF("SYSTEM REMOTE CONTROL enabled.");

      //start communication with tuners
      if (DIA_SUCCESS!=setTuner())
      {
         DIA_TR_ERR("### dia_IOCtrlTunerSettings::handleRequest setTuner() failed.");
      }
      else
      {
         /* do nothing */
      }
   }
   else
   {
      DIA_TR_ERR("### dia_IOCtrlTunerSettings::handleRequest() wait for enable for Remote Control...");
   }
}

//-------------------------------------------------------------------------

void
dia_IOCtrlTunerSettings::vOnTimerElapsed ( dia_TimerID /*id*/ )
{
   dia_tclFnctTrace oTrace("dia_IOCtrlTunerSettings::vOnTimerElapsed(dia_TimerID)");

   tDiaResult retCode = DIA_FAILED;

   mIsSourceRestoring = true;

   DIA_TR_INF("### dia_IOCtrlTunerSettings::vOnTimerElapsed ###");

   if (mRequestedSourceID!=mOrigSourceID)
   {
      DIA_TR_INF("dia_IOCtrlTunerSettings::vOnTimerElapsed Restore Source ID.");
      DIA_TR_INF("dia_IOCtrlTunerSettings::vOnTimerElapsed mRequestedSourceID=%d, mOrigSourceID=%d.", mRequestedSourceID, mOrigSourceID);
      retCode = getInstanceOfAudioManager()->setAudioSource(mOrigSourceID,*this);    //response: onAudioSourceSetResult
   }
   else
   {
      DIA_TR_INF("dia_IOCtrlTunerSettings::vOnTimerElapsed Source ID not changed.");
      retCode = deactivateRemoteControl();
   }

   if (DIA_SUCCESS!=retCode)
   {
      DIA_TR_ERR("### dia_IOCtrlTunerSettings::vOnTimerElapsed retCode=0x%08X.", retCode);
   }
}

//------------------------------------------------------------------------------

tDiaResult
dia_IOCtrlTunerSettings::setTunerSource(tU32 /*dia_eAudioSource*/ newSource)
{
   dia_tclFnctTrace oTrace("dia_IOCtrlTunerSettings::setTunerSource()");
   tDiaResult retCode = DIA_FAILED;

   mRequestedSourceID = (dia_eAudioSource)newSource;

   switch(mRequestedSourceID)
   {
      case DIA_EN_AUDIO_SOURCE_TUNER_AM:     DIA_TR_INF("dia_IOCtrlTunerSettings::setTunerSource DIA_EN_AUDIO_SOURCE_TUNER_AM (%d).",   mRequestedSourceID);   break;
      case DIA_EN_AUDIO_SOURCE_TUNER_DAB:    DIA_TR_INF("dia_IOCtrlTunerSettings::setTunerSource DIA_EN_AUDIO_SOURCE_TUNER_DAB (%d).",  mRequestedSourceID);   break;
      case DIA_EN_AUDIO_SOURCE_TUNER_FM:     DIA_TR_INF("dia_IOCtrlTunerSettings::setTunerSource DIA_EN_AUDIO_SOURCE_TUNER_FM (%d).",   mRequestedSourceID);   break;
      default:                               DIA_TR_INF("dia_IOCtrlTunerSettings::setTunerSource with mRequestedSourceID=%d.", mRequestedSourceID); break;
   }

   retCode = getInstanceOfAudioManager()->setAudioSource(mRequestedSourceID,*this);    //response: onAudioSourceSetResult

   if (DIA_SUCCESS!=retCode)
   {
      DIA_TR_ERR("dia_IOCtrlTunerSettings::setTunerSource setAudioSource failed retCode=0x%08X.", retCode);
   }

   if (DIA_SUCCESS!=retCode)
   {
      DIA_TR_ERR("### dia_IOCtrlTunerSettings::setTunerSource Error.");
      stopTimer();
      deactivateRemoteControl();
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_IOCtrlTunerSettings::setTunerToActive(tU32 /*dia_eTuner*/ newTuner)
{
   dia_tclFnctTrace oTrace("dia_IOCtrlTunerSettings::setTunerToActive()");
   tDiaResult retCode = DIA_FAILED;

   dia_ITunerMaster* pInterface = 0;
   if (querySysAdapterInterface<dia_ITunerMaster>(&pInterface) == DIA_SUCCESS)
   {
      if (pInterface)
      {
         (void) setSysAdapterListener<dia_ITunerMasterListener>(this);
         DIA_TR_INF("dia_IOCtrlTunerSettings::setTunerToActive setSysAdapterListener for dia_ITunerMasterListener.");
         if (pInterface->tunerAMFM_ActiveTuner_Set((dia_eTuner)newTuner) == DIA_SUCCESS)
         {
            DIA_TR_INF("dia_IOCtrlTunerSettings::setTunerToActive tunerAMFM_ActiveTuner_Set for %d SUCCESS.", newTuner);
            retCode = DIA_SUCCESS;
         }
         else
         {
            (void) unsetSysAdapterListener<dia_ITunerMasterListener>(this);
            DIA_TR_ERR("dia_IOCtrlTunerSettings::setTunerToActive tunerAMFM_ActiveTuner_Set FAILED with newTuner=%d.", newTuner);
            DIA_TR_INF("dia_IOCtrlTunerSettings::setTunerToActive unsetSysAdapterListener for dia_ITunerMasterListener.");
         }
      }
   }

   if (DIA_SUCCESS!=retCode)
   {
      DIA_TR_ERR("### dia_IOCtrlTunerSettings::setTunerToActive Error.");
      stopTimer();
      deactivateRemoteControl();
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_IOCtrlTunerSettings::setTunerFrequencyAMFM(tU32 frequencyHz)
{
   dia_tclFnctTrace oTrace("dia_IOCtrlTunerSettings::setTunerFrequencyAMFM()");
   tDiaResult retCode = DIA_FAILED;

   dia_ITunerAMFM* pInterface = 0;
   if (querySysAdapterInterface<dia_ITunerAMFM>(&pInterface) == DIA_SUCCESS)
   {
      if (pInterface)
      {
         DIA_TR_INF("dia_IOCtrlTunerSettings::setTunerFrequencyAMFM received frequency = %d (Hz).", frequencyHz);

         // convert: diagnosis message in Hz <-> FI expects kHz
         tU32 frequencyKHz = frequencyHz/1000;
         DIA_TR_INF("dia_IOCtrlTunerSettings::setTunerFrequencyAMFM received frequency = %d (kHz).", frequencyKHz);

         (void) setSysAdapterListener<dia_ITunerAMFMListener>(this);
         DIA_TR_INF("dia_IOCtrlTunerSettings::setTunerFrequencyAMFM setSysAdapterListener for dia_ITunerAMFMListener.");
         if (DIA_SUCCESS == pInterface->tunerAMFM_Frequency_Set(frequencyKHz))
         {
            retCode = DIA_SUCCESS;
         }
         else
         {
            DIA_TR_ERR("dia_IOCtrlTunerSettings::setTunerFrequencyAMFM tunerAMFM_Frequency_Set FAILED frequencyKHz=0x%08X.", frequencyKHz);
         }
      }
   }

   if (DIA_SUCCESS!=retCode)
   {
      (void) unsetSysAdapterListener<dia_ITunerAMFMListener>(this);
      DIA_TR_INF("dia_IOCtrlTunerSettings::setTunerFrequencyAMFM unsetSysAdapterListener for dia_ITunerAMFMListener.");
      DIA_TR_ERR("dia_IOCtrlTunerSettings::setTunerFrequencyAMFM --- SEND TO TUNER DAB CONTROL FAILED !!!");
   }

   if (DIA_SUCCESS!=retCode)
   {
      DIA_TR_ERR("### dia_IOCtrlTunerSettings::setTunerFrequencyAMFM Error.");
      stopTimer();
      deactivateRemoteControl();
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_IOCtrlTunerSettings::setTunerFrequencyDAB(tU32 serviceID)
{
   dia_tclFnctTrace oTrace("dia_IOCtrlTunerSettings::setTunerFrequencyDAB()");
   tDiaResult retCode = DIA_FAILED;

   if (isDabActive())
   {
      DIA_TR_INF("dia_IOCtrlTunerSettings::setTunerFrequencyDAB serviceID = 0x%04x", serviceID);

      dia_ITunerDABControl* pInterface = 0;
      if (querySysAdapterInterface<dia_ITunerDABControl>(&pInterface) == DIA_SUCCESS)
      {
         if (pInterface)
         {
            (void) setSysAdapterListener<dia_ITunerDABControlListener>(this);
            DIA_TR_INF("dia_IOCtrlTunerSettings::setTunerFrequencyDAB setSysAdapterListener for dia_ITunerDABControlListener.");
            if (pInterface->tunerDAB_EnsembleFrequency_Set(serviceID) == DIA_SUCCESS)
            {
               DIA_TR_INF("dia_IOCtrlTunerSettings::setTunerFrequencyDAB tunerDAB_EnsembleFrequency_Set SUCCESS.");
               retCode = DIA_SUCCESS;
            }
            else
            {
               DIA_TR_ERR("dia_IOCtrlTunerSettings::setTunerFrequencyDAB tunerDAB_EnsembleFrequency_Set FAILED with serviceID=0x%08X.", serviceID);
            }

//            // DAB with name, e.g. 12D
//            if (pInterface->tunerDAB_ServiceID_Set(serviceID) == DIA_SUCCESS) {
//               DIA_TR_INF("dia_IOCtrlTunerSettings::setTunerFrequencyDAB tunerDAB_ServiceID_Set SUCCESS.");
//               retCode = DIA_SUCCESS;
//            } else {
//               DIA_TR_ERR("dia_IOCtrlTunerSettings::setTunerFrequencyDAB tunerDAB_ServiceID_Set FAILED.");
//            }
         }
      }

      if (DIA_FAILED == retCode)
      {
         (void) unsetSysAdapterListener<dia_ITunerDABControlListener>(this);
         DIA_TR_INF("dia_IOCtrlTunerSettings::setTunerFrequencyDAB unsetSysAdapterListener for dia_ITunerDABControlListener.");
         DIA_TR_ERR("dia_IOCtrlTunerSettings::setTunerFrequencyDAB --- SEND TO TUNER DAB CONTROL FAILED!!!!");
      }
   }
   else
   {
      DIA_TR_ERR("### dia_IOCtrlTunerSettings::setTunerFrequencyDAB DAB is inactive.");
   }

   if (DIA_SUCCESS!=retCode)
   {
      DIA_TR_ERR("### dia_IOCtrlTunerSettings::setTunerFrequencyDAB Error.");
      stopTimer();
      deactivateRemoteControl();
   }

   return retCode;
}

//------------------------------------------------------------------------------

bool
dia_IOCtrlTunerSettings::isDabActive() const
{
   dia_tclFnctTrace oTrace("dia_IOCtrlTunerSettings::isDabActive()");
   bool retCode = false;

   tU16 propLen = dia_getPropertySize(DIA_PROP_AIVI_SYSTEM_INFO_COMMON);
   if (DIA_PROP_AIVI_SYSTEM_INFO_COMMON_LEN==propLen)
   {
      tU8 systemInformationCommon[DIA_PROP_AIVI_SYSTEM_INFO_COMMON_LEN] = { 0 };
      tDiaResult getPropRes = dia_getProperty(DIA_PROP_AIVI_SYSTEM_INFO_COMMON, systemInformationCommon, propLen);

      if (DIA_SUCCESS == getPropRes)
      {
         DIA_TR_INF("dia_IOCtrlTunerSettings::isDabActive Tuner Byte = 0x%02X.", systemInformationCommon[SYSTEM_INFO_COMMON_TUNER_DAB_BYTE_OFFSET]);

         //true if DAB is active
         bool isDabActive = (systemInformationCommon[SYSTEM_INFO_COMMON_TUNER_DAB_BYTE_OFFSET] & SYSTEM_INFO_COMMON_TUNER_DAB_BIT_FLAG)>0 ? true : false;
         DIA_TR_INF("dia_IOCtrlTunerSettings::isDabActive dab flag = 0x%02X.", (systemInformationCommon[SYSTEM_INFO_COMMON_TUNER_DAB_BYTE_OFFSET] & SYSTEM_INFO_COMMON_TUNER_DAB_BIT_FLAG));

         if (isDabActive)
         {
            DIA_TR_INF("dia_IOCtrlTunerSettings::isDabActive DAB active.");
            retCode = true;
         }
         else
         {
            DIA_TR_INF("dia_IOCtrlTunerSettings::isDabActive DAB passive.");
         }
      }
      else
      {
         DIA_TR_ERR("dia_IOCtrlTunerSettings::isDabActive Tuner Byte FAILURE IN READING KDS PROPERTY =x%04x !", DIA_PROP_AIVI_SYSTEM_INFO_COMMON);
      }
   }
   else
   {
      DIA_TR_ERR("dia_IOCtrlTunerSettings::isDabActive KDS property length different than expected!");
      DIA_TR_ERR("dia_IOCtrlTunerSettings::isDabActive KDS property IS: %d ; SHOULD: 0x%04x !", propLen, DIA_PROP_AIVI_SYSTEM_INFO_COMMON_LEN);
   }

   return retCode;
}

//------------------------------------------------------------------------------


tDiaResult
dia_IOCtrlTunerSettings::vOnTerminate(dia_eIOCtrlStatus status)      //call after Return Control with status=0
{
   dia_tclFnctTrace oTrace("dia_IOCtrlTunerSettings::vOnTerminate()");

   DIA_TR_INF("dia_IOCtrlTunerSettings::vOnTerminate status=%d.", status);
   DIA_TR_INF("dia_IOCtrlTunerSettings::vOnTerminate returned DIA_FAILED.");

   return DIA_FAILED;
}

//------------------------------------------------------------------------------
//-----------------------------------------------------------------------------

void
dia_IOCtrlTunerSettings::vOnTunerAMFMFrequency_Set ( tDiaResult success  )
{
   dia_tclFnctTrace trc("dia_IOCtrlTunerSettings::vOnTunerAMFMFrequency_Set");

   bool updateSignal = false;

   DIA_TR_INF("dia_IOCtrlTunerSettings::vOnTunerAMFMFrequency_Set unsetSysAdapterListener for dia_ITunerAMFMListener.");
   (void) unsetSysAdapterListener<dia_ITunerAMFMListener>(this);

   mIsResultReady = TRUE;
   if( DIA_SUCCESS==success )
   {
      DIA_TR_INF("dia_IOCtrlTunerSettings::vOnTunerAMFMFrequency_Set SUCCESSFUL .");

      JobReturnCode retVal= doNextJob();
      switch (retVal)
      {
         case JOB_RETURN_CODE_ERROR:
            setErrorInfo(DIA_E_CONDITIONS_NOT_CORRECT);
            updateSignal = true;
         break;

         case JOB_RETURN_CODE_NO_TASK_TO_BE_TRIGGERED:
            DIA_TR_INF("dia_IOCtrlTunerSettings::vOnTunerAMFMFrequency_Set Update set TRUE.");
            updateSignal = true;
         break;

         case JOB_RETURN_CODE_OK_AND_WAIT_FOR_NEXT_TASK:
         case JOB_RETURN_CODE_OK_AND_NO_NEXT_TASKS:
            /* do nothing */
         break;

         default:
            setErrorInfo(DIA_E_CONDITIONS_NOT_CORRECT);
            updateSignal = true;
         break;
      }
   }
   else
   {
      setErrorInfo(DIA_E_CONDITIONS_NOT_CORRECT);
      updateSignal = true;
      DIA_TR_ERR("dia_IOCtrlTunerSettings::vOnTunerAMFMFrequency_Set success=0x%08X", success);
      DIA_TR_ERR("dia_IOCtrlTunerSettings::vOnTunerAMFMFrequency_Set DIA_FAILED !!!");
   }

   if (updateSignal)
   {
      if (DIA_E_NOERROR!=getErrorInfo())
      {
         tDiaResult retCode = DIA_FAILED;

         DIA_TR_ERR("### dia_IOCtrlTunerSettings::vOnTunerAMFMFrequency_Set getErrorInfo()=0x%08X.", getErrorInfo());

         stopTimer();
         mIsSourceRestoring = true;

         if (mRequestedSourceID!=mOrigSourceID)
         {
            DIA_TR_INF("dia_IOCtrlTunerSettings::vOnTunerAMFMFrequency_Set Restore Source ID.");
            DIA_TR_INF("dia_IOCtrlTunerSettings::vOnTunerAMFMFrequency_Set mRequestedSourceID=%d, mOrigSourceID=%d.", mRequestedSourceID, mOrigSourceID);
            retCode = getInstanceOfAudioManager()->setAudioSource(mOrigSourceID,*this);    //response: onAudioSourceSetResult
         }
         else
         {
            DIA_TR_INF("dia_IOCtrlTunerSettings::vOnTunerAMFMFrequency_Set Source ID not changed.");
            retCode = deactivateRemoteControl();
         }

         if (DIA_SUCCESS!=retCode)
         {
            DIA_TR_ERR("### dia_IOCtrlTunerSettings::vOnTunerAMFMFrequency_Set Failed retCode=0x%08X.", retCode);
         }
      }

      dia_SrvHandlerGenericIOCtrlByIdentifier::vOnSignalUpdate(this);
   }
}

//-----------------------------------------------------------------------------

void
dia_IOCtrlTunerSettings::vOnTunerDABServiceID_Set ( tDiaResult success )
{
   dia_tclFnctTrace trc("dia_IOCtrlTunerSettings::vOnTunerDABServiceID_Set");

   bool updateSignal = false;

   DIA_TR_INF("dia_IOCtrlTunerSettings::vOnTunerDABServiceID_Set unsetSysAdapterListener for dia_ITunerDABControlListener.");
   (void) unsetSysAdapterListener<dia_ITunerDABControlListener>(this);
   mIsResultReady = TRUE;
   if (DIA_SUCCESS==success)
   {
      DIA_TR_INF("dia_IOCtrlTunerSettings::vOnTunerDABServiceID_Set SUCCESSFUL .");

      JobReturnCode retVal= doNextJob();
      switch (retVal)
      {
         case JOB_RETURN_CODE_ERROR:
            setErrorInfo(DIA_E_CONDITIONS_NOT_CORRECT);
            updateSignal = true;
         break;

         case JOB_RETURN_CODE_NO_TASK_TO_BE_TRIGGERED:
            DIA_TR_INF("dia_IOCtrlTunerSettings::vOnTunerDABServiceID_Set Update set TRUE.");
            updateSignal = true;
         break;

         case JOB_RETURN_CODE_OK_AND_WAIT_FOR_NEXT_TASK:
         case JOB_RETURN_CODE_OK_AND_NO_NEXT_TASKS:
            /* do nothing */
         break;

         default:
            setErrorInfo(DIA_E_CONDITIONS_NOT_CORRECT);
            updateSignal = true;
         break;
      }
   }
   else
   {
      setErrorInfo(DIA_E_CONDITIONS_NOT_CORRECT);
      updateSignal = true;
      DIA_TR_ERR("dia_IOCtrlTunerSettings::vOnTunerDABServiceID_Set DIA_FAILED !!!");
   }

   if (updateSignal)
   {
      if (DIA_E_NOERROR!=getErrorInfo())
      {
         DIA_TR_ERR("### dia_IOCtrlTunerSettings::vOnTunerDABServiceID_Set getErrorInfo()=%d.", getErrorInfo());

         stopTimer();
         deactivateRemoteControl();
      }

      dia_SrvHandlerGenericIOCtrlByIdentifier::vOnSignalUpdate(this);
   }
}

//-----------------------------------------------------------------------------

void
dia_IOCtrlTunerSettings::vOnTunerAMFMActiveTuner_Set ( tDiaResult success )
{
   dia_tclFnctTrace trc("dia_IOCtrlTunerSettings::vOnTunerAMFMActiveTuner_Set");
   bool updateSignal = false;

   (void) unsetSysAdapterListener<dia_ITunerMasterListener>(this);
   DIA_TR_INF("dia_IOCtrlTunerSettings::vOnTunerAMFMActiveTuner_Set unsetSysAdapterListener for dia_ITunerMasterListener.");
   mIsResultReady = TRUE;
   if( DIA_SUCCESS == success)
   {
      DIA_TR_INF("dia_IOCtrlTunerSettings::vOnTunerAMFMActiveTuner_Set SUCCESSFUL .");

      JobReturnCode retVal= doNextJob();
      switch (retVal)
      {
         case JOB_RETURN_CODE_ERROR:
            setErrorInfo(DIA_E_CONDITIONS_NOT_CORRECT);
            updateSignal = true;
         break;

         case JOB_RETURN_CODE_NO_TASK_TO_BE_TRIGGERED:
            DIA_TR_INF("dia_IOCtrlTunerSettings::vOnTunerAMFMActiveTuner_Set Update set TRUE.");
            updateSignal = true;
         break;

         case JOB_RETURN_CODE_OK_AND_WAIT_FOR_NEXT_TASK:
         case JOB_RETURN_CODE_OK_AND_NO_NEXT_TASKS:
            /* do nothing */
         break;

         default:
            setErrorInfo(DIA_E_CONDITIONS_NOT_CORRECT);
            updateSignal = true;
         break;
      }
   }
   else
   {
      setErrorInfo(DIA_E_CONDITIONS_NOT_CORRECT);
      updateSignal = true;
      DIA_TR_ERR("dia_IOCtrlTunerSettings::vOnTunerAMFMActiveTuner_Set DIA_FAILED !!!");
   }

   if (updateSignal)
   {
      if (DIA_E_NOERROR!=getErrorInfo())
      {
         DIA_TR_ERR("### dia_IOCtrlTunerSettings::vOnTunerAMFMActiveTuner_Set getErrorInfo()=%d.", getErrorInfo());

         stopTimer();
         deactivateRemoteControl();
      }

      dia_SrvHandlerGenericIOCtrlByIdentifier::vOnSignalUpdate(this);
   }
}

//-----------------------------------------------------------------------------

void
dia_IOCtrlTunerSettings::vOnAudioConnectionFailure ( tDiaResult errInfo )
{
   dia_tclFnctTrace trc("dia_IOCtrlTunerSettings::vOnAudioConnectionFailure");

   DIA_TR_INF("dia_IOCtrlTunerSettings::vOnAudioConnectionFailure unsetSysAdapterListener for dia_IAudioSourceListener.");
   (void) unsetSysAdapterListener<dia_IAudioSourceListener>(this);

   DIA_TR_ERR("dia_IOCtrlTunerSettings::vOnAudioConnectionFailure errInfo=0x%08X", errInfo);

   stopTimer();
   deactivateRemoteControl();

   //stop all next tasks
   performActions.clear();

   setErrorInfo(DIA_E_CONDITIONS_NOT_CORRECT);
   dia_SrvHandlerGenericIOCtrlByIdentifier::vOnSignalUpdate(this);
}

//-----------------------------------------------------------------------------

void
dia_IOCtrlTunerSettings::vOnAudioSourceConnectionDone ( tDiaResult res )
{
   dia_tclFnctTrace trc("dia_IOCtrlTunerSettings::vOnAudioSourceConnectionDone");
   bool updateSignal = false;

   DIA_TR_INF("dia_IOCtrlTunerSettings::vOnAudioSourceConnectionDone unsetSysAdapterListener for dia_IAudioSourceListener.");
   (void) unsetSysAdapterListener<dia_IAudioSourceListener>(this);

   DIA_TR_INF("dia_IOCtrlTunerSettings::vOnAudioSourceConnectionDone res=0x%08X", res);

   mIsResultReady = TRUE;
   if( DIA_SUCCESS == res)
   {
      DIA_TR_INF("dia_IOCtrlTunerSettings::vOnAudioSourceConnectionDone SUCCESSFUL.");

      JobReturnCode retVal= doNextJob();
      switch (retVal)
      {
         case JOB_RETURN_CODE_ERROR:
            setErrorInfo(DIA_E_CONDITIONS_NOT_CORRECT);
            updateSignal = true;
         break;

         case JOB_RETURN_CODE_NO_TASK_TO_BE_TRIGGERED:
            DIA_TR_INF("dia_IOCtrlTunerSettings::vOnAudioSourceConnectionDone Update set TRUE.");
            updateSignal = true;
         break;

         case JOB_RETURN_CODE_OK_AND_WAIT_FOR_NEXT_TASK:
         case JOB_RETURN_CODE_OK_AND_NO_NEXT_TASKS:
            /* do nothing */
         break;

         default:
            setErrorInfo(DIA_E_CONDITIONS_NOT_CORRECT);
            updateSignal = true;
         break;
      }
   }
   else
   {
      setErrorInfo(DIA_E_CONDITIONS_NOT_CORRECT);
      updateSignal = true;
      DIA_TR_ERR("dia_IOCtrlTunerSettings::vOnAudioSourceConnectionDone DIA_FAILED !!!");
   }

   if (updateSignal)
   {
      if (DIA_E_NOERROR!=getErrorInfo())
      {
         DIA_TR_ERR("### dia_IOCtrlTunerSettings::vOnAudioSourceConnectionDone getErrorInfo()=%d.", getErrorInfo());

         stopTimer();
         deactivateRemoteControl();
      }

      dia_SrvHandlerGenericIOCtrlByIdentifier::vOnSignalUpdate(this);
   }
}

//-----------------------------------------------------------------------------

void
dia_IOCtrlTunerSettings::handleTimeout(void)
{
   dia_tclFnctTrace trc("dia_IOCtrlTunerSettings::handleTimeout");

   (void) unsetSysAdapterListener<dia_ITunerAMFMListener>(this);
   (void) unsetSysAdapterListener<dia_ITunerMasterListener>(this);
   (void) unsetSysAdapterListener<dia_ITunerDABControlListener>(this);
   (void) unsetSysAdapterListener<dia_IAudioSourceListener>(this);

   DIA_TR_INF("dia_IOCtrlTunerSettings::handleTimeout unsetSysAdapterListener for dia_ITunerAMFMListener.");
   DIA_TR_INF("dia_IOCtrlTunerSettings::handleTimeout unsetSysAdapterListener for dia_ITunerMasterListener.");
   DIA_TR_INF("dia_IOCtrlTunerSettings::handleTimeout unsetSysAdapterListener for dia_ITunerDABControlListener.");
   DIA_TR_INF("dia_IOCtrlTunerSettings::handleTimeout unsetSysAdapterListener for dia_IAudioSourceListener.");

   //stopTimer
   mTimer.s32SetTime(0,0);
   mTimer.removeTimerListener(this);

   tDiaResult retCode = deactivateRemoteControl();
   if (DIA_SUCCESS!=retCode)
   {
      DIA_TR_ERR("### dia_IOCtrlTunerSettings::handleTimeout deactivateRemoteControl failed.");
   }
}

//-----------------------------------------------------------------------------

JobReturnCode
dia_IOCtrlTunerSettings::doNextJob()
{
   dia_tclFnctTrace trc("dia_IOCtrlTunerSettings::doNextJob");

   JobReturnCode retVal = JOB_RETURN_CODE_UNKNOWN;

   DIA_TR_INF("dia_IOCtrlTunerSettings::doNextJob performActions.size()=%d", performActions.size());

   if (!performActions.empty())
   {
      std::pair <FP,tU32> pair = performActions.back();
      performActions.pop_back(); //remove last element

      if (NULL!=pair.first)
      {
         tDiaResult retCode = (this->*pair.first)(pair.second);
         if (DIA_SUCCESS==retCode)
         {
            DIA_TR_INF("dia_IOCtrlTunerSettings::doNextJob next job triggered.");

            if (performActions.size()>0)
            {
               retVal = JOB_RETURN_CODE_OK_AND_WAIT_FOR_NEXT_TASK;
            }
            else
            {
               retVal = JOB_RETURN_CODE_OK_AND_NO_NEXT_TASKS;
            }
         }
         else
         {
            retVal = JOB_RETURN_CODE_ERROR;
            DIA_TR_ERR("### dia_IOCtrlTunerSettings::doNextJob Trigger failed. retCode=0x%08X", retCode);
            DIA_TR_ERR("### dia_IOCtrlTunerSettings::doNextJob performActions.size()=%d.", performActions.size());
         }
      }
      else
      {
         DIA_TR_ERR("### dia_IOCtrlTunerSettings::doNextJob PTR is null. ERROR.");
      }
   }
   else
   {
      retVal = JOB_RETURN_CODE_NO_TASK_TO_BE_TRIGGERED;
      DIA_TR_INF("dia_IOCtrlTunerSettings::doNextJob No job in the queue.");
   }

   if (retVal==JOB_RETURN_CODE_OK_AND_WAIT_FOR_NEXT_TASK) DIA_TR_INF("dia_IOCtrlTunerSettings::doNextJob returned JOB_RETURN_CODE_OK_AND_WAIT_FOR_NEXT_TASK");
   if (retVal==JOB_RETURN_CODE_OK_AND_NO_NEXT_TASKS)      DIA_TR_INF("dia_IOCtrlTunerSettings::doNextJob returned JOB_RETURN_CODE_OK_AND_NO_NEXT_TASKS");
   if (retVal==JOB_RETURN_CODE_NO_TASK_TO_BE_TRIGGERED)   DIA_TR_INF("dia_IOCtrlTunerSettings::doNextJob returned JOB_RETURN_CODE_NO_TASK_TO_BE_TRIGGERED");
   if (retVal==JOB_RETURN_CODE_ERROR)                     DIA_TR_INF("dia_IOCtrlTunerSettings::doNextJob returned JOB_RETURN_CODE_ERROR");

   return retVal;
}

//-----------------------------------------------------------------------------

void
dia_IOCtrlTunerSettings::vOnRemoteControlModeChanged ( dia_eRemoteCtrlMode newMode, dia_eRemoteCtrlMode /*prevMode*/ )
{
   dia_tclFnctTrace trc("dia_IOCtrlTunerSettings::vOnRemoteControlModeChanged");

   tDiaResult retCode = DIA_FAILED;

   (void) unsetSysAdapterListener<dia_IRemoteControlListener>(this);

   switch (newMode)
   {
      case DIA_EN_REMOTE_CONTROL_MODE_ACTIVE:
      {
         DIA_TR_INF("dia_IOCtrlTunerSettings::vOnRemoteControlModeChanged DIA_EN_REMOTE_CONTROL_MODE_ACTIVE.");
         //start communication with tuners, System Remote Control is Enabled without HMI.
         if (DIA_SUCCESS!=setTuner())
         {
            DIA_TR_ERR("### dia_IOCtrlTunerSettings::vOnRemoteControlModeChanged setTuner() failed.");
         }
      }
      break;

      case DIA_EN_REMOTE_CONTROL_MODE_NOT_ACTIVE:
      {
         DIA_TR_INF("dia_IOCtrlTunerSettings::vOnRemoteControlModeChanged DIA_EN_REMOTE_CONTROL_MODE_NOT_ACTIVE.");
         mIsSystemRemoteActiveWithoutHMI = false;

         DIA_TR_INF("dia_IOCtrlTunerSettings::vOnRemoteControlModeChanged mOrigSourceID=%d.", mOrigSourceID);



         /* no confirmation to the tester. */
      }
      break;

      default:
      {
         /* do nothing */
      }
      break;
   }
}

//-----------------------------------------------------------------------------

tDiaResult
dia_IOCtrlTunerSettings::setTuner ( void )
{
   dia_tclFnctTrace trc("dia_IOCtrlTunerSettings::setTuner");

   tDiaResult retCode = DIA_FAILED;

   performActions.clear();

   // Switch according Tuner type
   switch(mTunerType)
   {
      // Set AM/FM frequency
      // For AM/FM the first Byte has not to be used. It's for compatibility issues with DAB
      case TUNER_TYPE_FM1:
      {
         DIA_TR_INF("dia_IOCtrlTunerSettings::setTuner mTunerType=0x%02X - FM1.", mTunerType);

         //first Test mode
         retCode = setTunerToActive(DIA_EN_TUNERMASTER_2);  //according to presentation for A-IVI (Renault_Nissan_A-IVI_Receiver_Concept_20151019.pptx)

         //then set tuner and its frequency
         if (DIA_SUCCESS==retCode)
         {
            std::pair <FP,tU32> func1(&dia_IOCtrlTunerSettings::setTunerSource,        (tU32)DIA_EN_AUDIO_SOURCE_TUNER_FM);
            std::pair <FP,tU32> func2(&dia_IOCtrlTunerSettings::setTunerFrequencyAMFM, mFreqAMFM);
            performActions.push_back( func2 );
            performActions.push_back( func1 );

            DIA_TR_INF("dia_IOCtrlTunerSettings::setTuner Loaded jobs: setTunerSource and setTunerFrequencyAMFM.");
         }
      }
      break;

      case TUNER_TYPE_FM2:
      {
         DIA_TR_INF("dia_IOCtrlTunerSettings::setTuner mTunerType=0x%02X - FM2.", mTunerType);

         //first Test mode
         retCode = setTunerToActive(DIA_EN_TUNERMASTER_1);  //according to presentation for A-IVI (Renault_Nissan_A-IVI_Receiver_Concept_20151019.pptx)

         //then set tuner and its frequency
         if (DIA_SUCCESS==retCode)
         {
            std::pair <FP,tU32> func1(&dia_IOCtrlTunerSettings::setTunerSource,        (tU32)DIA_EN_AUDIO_SOURCE_TUNER_FM);
            std::pair <FP,tU32> func2(&dia_IOCtrlTunerSettings::setTunerFrequencyAMFM, mFreqAMFM );
            performActions.push_back( func2 );
            performActions.push_back( func1 );

            DIA_TR_INF("dia_IOCtrlTunerSettings::setTuner Loaded jobs: setTunerSource and setTunerFrequencyAMFM.");
         }
      }
      break;

      case TUNER_TYPE_AM:
      {
         DIA_TR_INF("dia_IOCtrlTunerSettings::setTuner mTunerType=0x%02x - AM.", mTunerType);

         //first Test mode
         retCode = setTunerToActive(DIA_EN_TUNERMASTER_2);  //according to presentation for A-IVI (Renault_Nissan_A-IVI_Receiver_Concept_20151019.pptx)

         //then set tuner and its frequency
         if (DIA_SUCCESS==retCode)
         {
            std::pair <FP,tU32> func1(&dia_IOCtrlTunerSettings::setTunerSource,        (tU32)DIA_EN_AUDIO_SOURCE_TUNER_AM);
            std::pair <FP,tU32> func2(&dia_IOCtrlTunerSettings::setTunerFrequencyAMFM, mFreqAMFM );
            performActions.push_back( func2 );
            performActions.push_back( func1 );

            DIA_TR_INF("dia_IOCtrlTunerSettings::setTuner Loaded jobs: setTunerSource and setTunerFrequencyAMFM.");
         }
      }
      break;

      case TUNER_TYPE_DAB:
      {
         DIA_TR_INF("dia_IOCtrlTunerSettings::setTuner mTunerType=0x%02X - DAB.", mTunerType);

         if (isDabActive())
         {
            DIA_TR_INF("dia_IOCtrlTunerSettings::setTuner DAB is active.");

            // Received String starts always with "CN.." and has to removed for the component
            if(mIsCN)
            {
               DIA_TR_INF("dia_IOCtrlTunerSettings::setTuner Frequency Pre-Set (\"CN\") OK and removed.");

               //first Test mode
               retCode = setTunerToActive(DIA_EN_TUNERMASTER_4);  //according to presentation for A-IVI (Renault_Nissan_A-IVI_Receiver_Concept_20151019.pptx)

               //then set tuner and its frequency
               if (DIA_SUCCESS==retCode)
               {
                  std::pair <FP,tU32> func1(&dia_IOCtrlTunerSettings::setTunerSource,       DIA_EN_AUDIO_SOURCE_TUNER_DAB );
                  std::pair <FP,tU32> func2(&dia_IOCtrlTunerSettings::setTunerFrequencyDAB, mFreqDAB );
                  performActions.push_back( func2 );
                  performActions.push_back( func1 );

                  DIA_TR_INF("dia_IOCtrlTunerSettings::setTuner Loaded jobs: setTunerSource and setTunerFrequencyDAB.");
               }
            }
            else
            {
               DIA_TR_ERR("dia_IOCtrlTunerSettings::setTuner mIsCN is FALSE.");
            }
         }
         else
         {
            DIA_TR_ERR("### dia_IOCtrlTunerSettings::setTuner DAB inactive.");
         }
      }
      break;

      default:
      {
         DIA_TR_ERR("### dia_IOCtrlTunerSettings::setTuner Unknown current ControlState = 0x%02X.", mTunerType);
      }
      break;
   }

   if (DIA_SUCCESS!=retCode)
   {
      DIA_TR_ERR("### dia_IOCtrlTunerSettings::setTuner ERROR retCode=0x%08X.", retCode);

      stopTimer();
      deactivateRemoteControl();
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_IOCtrlTunerSettings::activateRemoteControl ( void )
{
   dia_tclFnctTrace trc("dia_IOCtrlTunerSettings::activateRemoteControl()");
   tDiaResult retCode = DIA_FAILED;

   dia_IRemoteControl* pInterface = 0;
   tDiaResult queryRes;

   if ( ((queryRes = querySysAdapterInterface<dia_IRemoteControl>(&pInterface)) == DIA_SUCCESS) && pInterface )
   {
      DIA_TR_INF("dia_IOCtrlTunerSettings::activateRemoteControl(): Enable System Remote Control without HMI");
      (void) setSysAdapterListener<dia_IRemoteControlListener>(this);

      dia_eRemoteCtrlMode mode = DIA_EN_REMOTE_CONTROL_MODE_ACTIVE;

      if ( pInterface->getRemoteControlMode() == mode  )
      {
         DIA_TR_INF("dia_IOCtrlTunerSettings::activateRemoteControl System remote control already active.");

         // requested remote control mode is already active
         (void) unsetSysAdapterListener<dia_IRemoteControlListener>(this);

         if ( pInterface->getRemoteControlSettings() & DIA_C_U32_REMOTE_CTRL_SETTINGS_NO_HMI  )
         {
            DIA_TR_INF("dia_IOCtrlTunerSettings::activateRemoteControl System remote control w/o HMI.");
            mIsSystemRemoteActiveWithoutHMI = true;
            retCode = DIA_SUCCESS;
         }
         else
         {
            DIA_TR_ERR("### dia_IOCtrlTunerSettings::activateRemoteControl: System Remote Control active, but with HMI (expected: without HMI).");
         }
      }
      else if ( pInterface->setRemoteControlMode(mode, DIA_C_U32_REMOTE_CTRL_SETTINGS_NO_HMI) == DIA_SUCCESS )
      {
         DIA_TR_INF("dia_IOCtrlTunerSettings::activateRemoteControl setRemoteControlMode(mode, DIA_C_U32_REMOTE_CTRL_SETTINGS_NO_HMI) success.");
         mIsSystemRemoteActiveWithoutHMI = false;  //will be set to true in vOnRemoteControlModeChanged
         retCode = DIA_SUCCESS;
      }
      else
      {
         DIA_TR_ERR("### dia_IOCtrlTunerSettings::activateRemoteControl pInterface->setRemoteControlMode(mode, DIA_C_U32_REMOTE_CTRL_SETTINGS_NO_HMI) error.");
         mIsSystemRemoteActiveWithoutHMI = false;
         (void) unsetSysAdapterListener<dia_IRemoteControlListener>(this);
      }
   }
   else
   {

      DIA_TR_ERR("### dia_IOCtrlTunerSettings::activateRemoteControl FAILED! queryRes=0x%08X pInterface=0x%p.", queryRes, pInterface);
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_IOCtrlTunerSettings::deactivateRemoteControl ( void )
{
   dia_tclFnctTrace trc("dia_IOCtrlTunerSettings::deactivateRemoteControl()");
   tDiaResult retCode = DIA_FAILED;

   dia_IRemoteControl* pInterface = 0;
   tDiaResult queryRes;

   if ( ((queryRes = querySysAdapterInterface<dia_IRemoteControl>(&pInterface)) == DIA_SUCCESS) && pInterface )
   {
      DIA_TR_INF("dia_IOCtrlTunerSettings::deactivateRemoteControl(): Disable System Remote Control without HMI");
      (void) setSysAdapterListener<dia_IRemoteControlListener>(this);

      dia_eRemoteCtrlMode mode = DIA_EN_REMOTE_CONTROL_MODE_NOT_ACTIVE;

      DIA_TR_INF("dia_IOCtrlTunerSettings::deactivateRemoteControl(): getRemoteControlMode()=%d, mode=%d.", pInterface->getRemoteControlMode(), mode);

      if ( pInterface->getRemoteControlMode() == mode  )
      {
         DIA_TR_INF("dia_IOCtrlTunerSettings::deactivateRemoteControl System remote control already inactive.");

         // requested remote control mode is already inactive
         (void) unsetSysAdapterListener<dia_IRemoteControlListener>(this);

         mIsSystemRemoteActiveWithoutHMI = false;
         retCode = DIA_SUCCESS;
      }
      else if ( pInterface->setRemoteControlMode(mode, DIA_C_U32_REMOTE_CTRL_SETTINGS_NO_HMI) == DIA_SUCCESS )
      {
         DIA_TR_INF("setRemoteControlMode(NOT_ACTIVE, DIA_C_U32_REMOTE_CTRL_SETTINGS_NO_HMI) success.");
         retCode = DIA_SUCCESS;
      }
      else
      {
         DIA_TR_ERR("### pInterface->setRemoteControlMode(NOT_ACTIVE, DIA_C_U32_REMOTE_CTRL_SETTINGS_NO_HMI) error.");
         (void) unsetSysAdapterListener<dia_IRemoteControlListener>(this);
      }
   }
   else
   {
      DIA_TR_ERR("### querySysAdapterInterface FAILED! queryRes=0x%08X (decimal %d), pInterface=0x%p", queryRes, queryRes, pInterface);
   }

   //reset AudioManager (switch off Remote Control)
   getInstanceOfAudioManager()->resetClient();

   return retCode;
}

//-----------------------------------------------------------------------------

void
dia_IOCtrlTunerSettings::onAudioSourceSetResult ( dia_eAudioSource sourceID, tDiaResult resultCode )
{
   dia_tclFnctTrace trc("dia_IOCtrlTunerSettings::onAudioSourceSetResult");

   DIA_TR_INF("dia_IOCtrlTunerSettings::onAudioSourceSetResult sourceID=0x%08X resultCode=0x%08X.", sourceID, resultCode);

   if (!mIsSourceRestoring)
   {
      DIA_TR_INF("dia_IOCtrlTunerSettings::onAudioSourceSetResult Change Source ID.");
      if ( sourceID != mRequestedSourceID )
      {
         DIA_TR_ERR("### dia_IOCtrlTunerSettings::onAudioSourceSetResult mRequestedSourceID=%d, sourceID=%d.", mRequestedSourceID, sourceID);
         return;
      }
   }
   else
   {
      DIA_TR_INF("dia_IOCtrlTunerSettings::onAudioSourceSetResult Restoring of Source ID.");
      if ( sourceID != mOrigSourceID )
      {
         DIA_TR_ERR("### dia_IOCtrlTunerSettings::onAudioSourceSetResult mOrigSourceID=%d, sourceID=%d.", mOrigSourceID, sourceID);
         return;
      }
   }

   bool updateSignal = false;

   setErrorInfo(resultCode);
   mIsResultReady = TRUE;

   DIA_TR_INF("dia_IOCtrlTunerSettings::onAudioSourceSetResult (%s) !!!", (mErrorInfo == DIA_SUCCESS) ? "SUCCESSFUL" : "DIA_FAILED");

   if( DIA_SUCCESS == resultCode)
   {
      DIA_TR_INF("dia_IOCtrlTunerSettings::onAudioSourceSetResult SUCCESSFUL .");

      JobReturnCode retVal= doNextJob();
      switch (retVal)
      {
         case JOB_RETURN_CODE_ERROR:
            DIA_TR_INF("dia_IOCtrlTunerSettings::onAudioSourceSetResult JOB_RETURN_CODE_ERROR");
            setErrorInfo(DIA_E_CONDITIONS_NOT_CORRECT);
            updateSignal = true;
         break;

         case JOB_RETURN_CODE_NO_TASK_TO_BE_TRIGGERED:
            DIA_TR_INF("dia_IOCtrlTunerSettings::onAudioSourceSetResult Update set TRUE.");
            updateSignal = true;
         break;

         case JOB_RETURN_CODE_OK_AND_WAIT_FOR_NEXT_TASK:
         case JOB_RETURN_CODE_OK_AND_NO_NEXT_TASKS:
            /* do nothing */
            if (JOB_RETURN_CODE_OK_AND_WAIT_FOR_NEXT_TASK==retVal) DIA_TR_INF("dia_IOCtrlTunerSettings::onAudioSourceSetResult JOB_RETURN_CODE_OK_AND_WAIT_FOR_NEXT_TASK");
            if (JOB_RETURN_CODE_OK_AND_NO_NEXT_TASKS==retVal)      DIA_TR_INF("dia_IOCtrlTunerSettings::onAudioSourceSetResult JOB_RETURN_CODE_OK_AND_NO_NEXT_TASKS");
         break;

         default:
            setErrorInfo(DIA_E_CONDITIONS_NOT_CORRECT);
            updateSignal = true;
            DIA_TR_INF("dia_IOCtrlTunerSettings::onAudioSourceSetResult retVal=%d.", retVal);
         break;
      }
   }
   else
   {
      setErrorInfo(DIA_E_CONDITIONS_NOT_CORRECT);
      updateSignal = true;
      DIA_TR_ERR("dia_IOCtrlTunerSettings::onAudioSourceSetResult DIA_FAILED !!!");
   }

   if (!mIsSourceRestoring && updateSignal)
   {
      DIA_TR_INF("dia_IOCtrlTunerSettings::onAudioSourceSetResult updateSignal.");

      dia_SrvHandlerGenericIOCtrlByIdentifier::vOnSignalUpdate(this);

      if (DIA_E_NOERROR!=getErrorInfo())
      {
         DIA_TR_ERR("### dia_IOCtrlTunerSettings::onAudioSourceSetResult getErrorInfo()=%d.", getErrorInfo());

         stopTimer();
         (void)deactivateRemoteControl();
      }
   }
   else if (mIsSourceRestoring)
   {
      DIA_TR_INF("dia_IOCtrlTunerSettings::onAudioSourceSetResult deactivate Remote Control.");

      (void)deactivateRemoteControl();
   }
   else
   {
      DIA_TR_ERR("### dia_IOCtrlTunerSettings::onAudioSourceSetResult Do nothing.");
   }
}

//-----------------------------------------------------------------------------

void
dia_IOCtrlTunerSettings::startTimer ( void )
{
   dia_tclFnctTrace trc("dia_IOCtrlTunerSettings::startTimer()");

   DIA_TR_INF("dia_IOCtrlTunerSettings::startTimer mDurationSystemRemoteControlInMSeconds=%d.", mDurationSystemRemoteControlInMSeconds);

   mTimer.addTimerListener(this);
   mTimer.s32SetTime(0,0);
   mTimer.s32SetTime(mDurationSystemRemoteControlInMSeconds, 0);

   mIsSourceRestoring = false;
}

//-----------------------------------------------------------------------------

void
dia_IOCtrlTunerSettings::stopTimer ( void )
{
   dia_tclFnctTrace trc("dia_IOCtrlTunerSettings::stopTimer()");

   mTimer.s32SetTime(0,0);
   mTimer.removeTimerListener(this);
}
