/*
 * dia_WlanInfo.cpp
 *
 *  Created on: Mar 10, 2016
 *      Author: jas1hi
 */

#include <libnl3/netlink/netlink.h>
#include <libnl3/netlink/genl/genl.h>
#include <libnl3/netlink/genl/ctrl.h>
#include <libnl3/netlink/socket.h>
#include <libnl3/netlink/attr.h>

//copy this from iw
#include <linux/nl80211.h>

#include "dia_WlanInfo.h"

static struct {
    struct nl_sock *nls;
    tS32 nl80211_id;
} wifi;

static std::vector<WiFiInfoElement> wifiInfoElements;


static tS32 getInterfaceHandler(struct nl_msg *msg, tVoid* /*arg*/)
{
	struct nlmsghdr* ret_hdr = nlmsg_hdr(msg);
	struct nlattr *tb_msg[NL80211_ATTR_MAX + 1];

	struct genlmsghdr *gnlh = (struct genlmsghdr*) nlmsg_data(ret_hdr);
    nla_parse(tb_msg, NL80211_ATTR_MAX, genlmsg_attrdata(gnlh, 0), genlmsg_attrlen(gnlh, 0), NULL);

    if (ret_hdr->nlmsg_type != wifi.nl80211_id)	{
       DIA_TR_ERR("dia_WlanInfo::getInterfaceHandler(): ret_hdr->nlmsg_type: %d != wifi.nl80211_id: %d", ret_hdr->nlmsg_type, wifi.nl80211_id);
	   return NL_STOP;
	}

	if (tb_msg[NL80211_ATTR_IFNAME]) { // e.g. wlan0
		WiFiInfoElement tmpWiFiInfoElement;

		memcpy(tmpWiFiInfoElement.iface, nla_get_string(tb_msg[NL80211_ATTR_IFNAME]), sizeof(tChar)*IFNAMSIZ);
		DIA_TR_INF("dia_WlanInfo::getInterfaceHandler(): Interface: %s", tmpWiFiInfoElement.iface);

		// Extract SSID
		if (tb_msg[NL80211_ATTR_SSID]) {
			memcpy(tmpWiFiInfoElement.essId, nla_data(tb_msg[NL80211_ATTR_SSID]), sizeof(tChar)*MAX_ESSID_STR_LENGTH);
			DIA_TR_INF("dia_WlanInfo::getInterfaceHandler(): SSID: %s", tmpWiFiInfoElement.essId);
			tmpWiFiInfoElement.essIdAvailable = TRUE;
		}

		// Extract Frequency
		if (tb_msg[NL80211_ATTR_WIPHY_FREQ]) {
			tmpWiFiInfoElement.frequency = nla_get_u16(tb_msg[NL80211_ATTR_WIPHY_FREQ]);
			DIA_TR_INF("dia_WlanInfo::getInterfaceHandler(): Freq: %d MHz", tmpWiFiInfoElement.frequency);
		}

		wifiInfoElements.push_back(tmpWiFiInfoElement);
	}
	else {
		DIA_TR_ERR("dia_WlanInfo::getInterfaceHandler(): Failed to get Interface type");
	}

    return NL_SKIP; // Continue with next message
}

static tS32 getStationHandler(struct nl_msg *msg, tVoid *arg)
{
	struct nlmsghdr* ret_hdr = nlmsg_hdr(msg);
	struct nlattr *tb_msg[NL80211_ATTR_MAX + 1];
	struct nlattr *sinfo[NL80211_STA_INFO_MAX + 1]; // for bitrate
	struct nlattr *rinfo[NL80211_RATE_INFO_MAX + 1]; // for bitrate

	static struct nla_policy stats_policy[NL80211_STA_INFO_MAX + 1];
	stats_policy[NL80211_STA_INFO_TX_BITRATE].type = NLA_NESTED;
	static struct nla_policy rate_policy[NL80211_RATE_INFO_MAX + 1];
	rate_policy[NL80211_RATE_INFO_BITRATE].type = NLA_U16;

	tU8* ifIndex = (tU8*)arg;

	struct genlmsghdr *gnlh = (struct genlmsghdr*) nlmsg_data(ret_hdr);
    nla_parse(tb_msg, NL80211_ATTR_MAX, genlmsg_attrdata(gnlh, 0), genlmsg_attrlen(gnlh, 0), NULL);

    if (ret_hdr->nlmsg_type != wifi.nl80211_id) {
       DIA_TR_ERR("dia_WlanInfo::getStationHandler(): ret_hdr->nlmsg_type: %d != wifi.nl80211_id: %d", ret_hdr->nlmsg_type, wifi.nl80211_id);
	   return NL_STOP;
	}

	// Extract bitrate
	if (!tb_msg[NL80211_ATTR_STA_INFO]) {
		DIA_TR_ERR("dia_WlanInfo::getStationHandler(): sta stats missing!");
		return NL_SKIP;
	}
	if (nla_parse_nested(sinfo, NL80211_STA_INFO_MAX, tb_msg[NL80211_ATTR_STA_INFO], stats_policy) < 0) {
		DIA_TR_ERR("dia_WlanInfo::getStationHandler(): Failed to parse nested attributes!");
		return NL_SKIP;
	}
	if (sinfo[NL80211_STA_INFO_TX_BITRATE]) {
		if (nla_parse_nested(rinfo, NL80211_RATE_INFO_MAX, sinfo[NL80211_STA_INFO_TX_BITRATE], rate_policy)) {
			DIA_TR_ERR("dia_WlanInfo::getStationHandler(): Failed to parse nested bitrate attribute!");
			wifiInfoElements[*ifIndex].bitRate = 0;
		}
		else {
			if (rinfo[NL80211_RATE_INFO_BITRATE]) {
				wifiInfoElements[*ifIndex].bitRate = nla_get_u16(rinfo[NL80211_RATE_INFO_BITRATE]);
				DIA_TR_INF("dia_WlanInfo::getStationHandler(): Bitrate: %d.%d MBit/s", wifiInfoElements[*ifIndex].bitRate / 10, wifiInfoElements[*ifIndex].bitRate % 10);
				wifiInfoElements[*ifIndex].bitRate = wifiInfoElements[*ifIndex].bitRate / 10;
			}
		}
	}
	else {
		DIA_TR_ERR("dia_WlanInfo::getStationHandler(): Failed to get bitrate!");
	}

    return NL_SKIP; // Continue with next message
}

static tS32 getScanHandler(struct nl_msg *msg, tVoid *arg)
{
	struct nlmsghdr* ret_hdr = nlmsg_hdr(msg);
	struct nlattr *tb_msg[NL80211_ATTR_MAX + 1];

	struct nlattr *bss[NL80211_BSS_MAX + 1];
	static struct nla_policy bss_policy[NL80211_BSS_MAX + 1];
	bss_policy[NL80211_BSS_FREQUENCY].type = NLA_U32;
	bss_policy[NL80211_BSS_BSSID].type = NLA_UNSPEC;
	bss_policy[NL80211_BSS_STATUS].type = NLA_U32;

	tU8* ifIndex = (tU8*)arg;

	struct genlmsghdr *gnlh = (struct genlmsghdr*) nlmsg_data(ret_hdr);

    nla_parse(tb_msg, NL80211_ATTR_MAX, genlmsg_attrdata(gnlh, 0), genlmsg_attrlen(gnlh, 0), NULL);

    if (ret_hdr->nlmsg_type != wifi.nl80211_id)	{
       DIA_TR_ERR("dia_WlanInfo::getScanHandler(): ret_hdr->nlmsg_type: %d != wifi.nl80211_id: %d", ret_hdr->nlmsg_type, wifi.nl80211_id);
	   return NL_STOP;
	}

    if (!tb_msg[NL80211_ATTR_BSS]) {
    	DIA_TR_ERR("dia_WlanInfo::getScanHandler(): bss info missing!");
		return NL_SKIP;
	}
	if (nla_parse_nested(bss, NL80211_BSS_MAX, tb_msg[NL80211_ATTR_BSS], bss_policy)) {
		DIA_TR_ERR("dia_WlanInfo::getScanHandler(): failed to parse nested attributes!");
		return NL_SKIP;
	}

	if (!bss[NL80211_BSS_BSSID])
		return NL_SKIP;

	if (!bss[NL80211_BSS_STATUS])
		return NL_SKIP;

	switch (nla_get_u32(bss[NL80211_BSS_STATUS])) {
		case NL80211_BSS_STATUS_ASSOCIATED:
			wifiInfoElements[*ifIndex].connected = TRUE;
			DIA_TR_INF("dia_WlanInfo::getScanHandler(): Connected");
			break;

		default:
			return NL_SKIP;
	}

	if (bss[NL80211_BSS_FREQUENCY]) {
		wifiInfoElements[*ifIndex].frequency = (tU16) nla_get_u32(bss[NL80211_BSS_FREQUENCY]);
		DIA_TR_INF("dia_WlanInfo::getScanHandler(): freq: %d", wifiInfoElements[*ifIndex].frequency);
		return NL_STOP; // No need for a further update
	}
	else {
		DIA_TR_ERR("dia_WlanInfo::getScanHandler(): Failed to get frequency");
	}

    return NL_SKIP; // Continue with next message
}

static tS32 getInterfaceFinishHandler(struct nl_msg* /*msg*/, tVoid *arg)
{
	tS32 *ret = (tS32*)arg;
    *ret = 0;

    return NL_SKIP;
}

static tS32 getStationFinishHandler(struct nl_msg* /*msg*/, tVoid *arg)
{
	tS32 *ret = (tS32*)arg;
    *ret = 0;

    return NL_SKIP;
}

static tS32 getScanFinishHandler(struct nl_msg* /*msg*/, tVoid *arg)
{
	tS32 *ret = (tS32*)arg;
    *ret = 0;

    return NL_SKIP;
}

dia_WlanInfo::dia_WlanInfo()
:   m_nlSocketInitialized(FALSE),
    m_wlanInfoAvailable(FALSE)
{

}

dia_WlanInfo::~dia_WlanInfo()
{
	nl_socket_free(wifi.nls);
}

tDiaResult dia_WlanInfo::initNlSocket()
{
	//allocate socket
	wifi.nls = nl_socket_alloc();
	if (!wifi.nls) {
		DIA_TR_ERR("dia_WlanInfo::initNlSocket(): Failed to allocate netlink socket");
		return DIA_FAILED;
	}
//	nl_socket_set_buffer_size(m_sk, 8192, 8192); // the buffer sizes are taken from the iw implementation, default is 32 KByte

	//generic netlink to connect to the nl80211 control channel
	if (genl_connect(wifi.nls)) {
		DIA_TR_ERR("dia_WlanInfo::initNlSocket(): Failed to connect to generic netlink");
		nl_socket_free(wifi.nls);
		return DIA_FAILED;
	}

	//find the nl80211 driver ID
	wifi.nl80211_id = genl_ctrl_resolve(wifi.nls, "nl80211"); // nl80211 is a new userspace <-> kernelspace wireless driver communication transport
	if (wifi.nl80211_id  < 0) {
		DIA_TR_ERR("dia_WlanInfo::initNlSocket(): nl80211 not found");
		nl_socket_free(wifi.nls);
		return DIA_FAILED;
	}

	DIA_TR_INF("dia_WlanInfo::initNlSocket socket creation SUCCESS!");
	m_nlSocketInitialized = TRUE;

	return DIA_SUCCESS;
}

tDiaResult dia_WlanInfo::getInterfaceInfo()
{
	DIA_TR_INF("dia_WlanInfo::getInterfaceInfo()");

	// allocate a callback for which we can register various handlers
	struct nl_cb *cb = nl_cb_alloc(NL_CB_DEFAULT);
	if (!cb) {
		DIA_TR_ERR("dia_WlanInfo::getInterfaceInfo(): Failed to allocate netlink callback");
	    return DIA_FAILED;
	}

	struct nl_msg *msgCmd = nlmsg_alloc();
	if (!msgCmd) {
		DIA_TR_ERR("dia_WlanInfo::getInterfaceInfo(): Failed to allocate netlink message");
		return DIA_FAILED;
	}

	/**** Prepare and send message for the command GET_INTERFACE ***/
	// Create the message headers.
	if (!genlmsg_put(msgCmd, 0, 0, wifi.nl80211_id, 0, NLM_F_DUMP, NL80211_CMD_GET_INTERFACE, 0)) { // The GET_INTERFACE command receives also the SSID as part of the reply
		DIA_TR_ERR("dia_WlanInfo::getInterfaceInfo(): Failed to create genl header for the message NL80211_CMD_GET_INTERFACE");
		return DIA_FAILED;
	}
	// Finalize and transmit Netlink message.
	if (nl_send_auto(wifi.nls, msgCmd) < 0) {
		DIA_TR_ERR("dia_WlanInfo::getInterfaceInfo(): Failed to send nl message NL80211_CMD_GET_INTERFACE");
		return DIA_FAILED;
	}
	nl_cb_set(cb, NL_CB_VALID, NL_CB_CUSTOM, getInterfaceHandler, NULL);

	// dump is finished
	tS32 err = 1;
	nl_cb_set(cb, NL_CB_FINISH, NL_CB_CUSTOM, getInterfaceFinishHandler, &err);

	while (err > 0) {
		err = nl_recvmsgs(wifi.nls, cb); // Stops reading if one of the callbacks returns NL_STOP//Coverity Fix(CID:28447)
		DIA_TR_INF("dia_WlanInfo::getInterfaceInfo(): nl_recvmsgs()");
	}

	nlmsg_free(msgCmd);
	nl_cb_put(cb);

	return DIA_SUCCESS;
}

tDiaResult dia_WlanInfo::getStationInfo(tCString ifname, tU8 ifIndex)
{
	DIA_TR_INF("dia_WlanInfo::getStationInfo()");

	// allocate a callback for which we can register various handlers
	struct nl_cb *cb = nl_cb_alloc(NL_CB_DEFAULT);
	if (!cb) {
		DIA_TR_ERR("dia_WlanInfo::getStationInfo(): Failed to allocate netlink callback");
	    return DIA_FAILED;
	}

	struct nl_msg *msgCmd = nlmsg_alloc();
	if (!msgCmd) {
		DIA_TR_ERR("dia_WlanInfo::getStationInfo(): Failed to allocate netlink message");
		return DIA_FAILED;
	}

	/**** Prepare and send message for the command GET_STATION ***/
	// Create the message headers.
	if (!genlmsg_put(msgCmd, 0, 0, wifi.nl80211_id, 0, NLM_F_DUMP, NL80211_CMD_GET_STATION, 0)) {// The GET_STATION command gives tx bitrate
		DIA_TR_ERR("dia_WlanInfo::getStationInfo(): Failed to create genl header for the message NL80211_CMD_GET_STATION");
		return DIA_FAILED;
	}

	if (nla_put_u32(msgCmd, NL80211_ATTR_IFINDEX, if_nametoindex(ifname)) < 0) {
		DIA_TR_ERR("dia_WlanInfo::getStationInfo(): Failed to put attr NL80211_ATTR_IFINDEX to the message NL80211_CMD_GET_STATION");
		return DIA_FAILED;
	}

	// Finalize and transmit Netlink message.
	if (nl_send_auto(wifi.nls, msgCmd) < 0) {
		DIA_TR_ERR("dia_WlanInfo::getStationInfo(): Failed to send nl message NL80211_CMD_GET_STATION");
		return DIA_FAILED;
	}
	nl_cb_set(cb, NL_CB_VALID, NL_CB_CUSTOM, getStationHandler, &ifIndex);

	// dump is finished
	tS32 err = 1;
	nl_cb_set(cb, NL_CB_FINISH, NL_CB_CUSTOM, getStationFinishHandler, &err);

	while (err > 0) {
		err = nl_recvmsgs(wifi.nls, cb); // Stops reading if one of the callbacks returns NL_STOP//Coverity Fix(CID:28447)
		DIA_TR_INF("dia_WlanInfo::getStationInfo(): nl_recvmsgs()");
	}

	nlmsg_free(msgCmd);
	nl_cb_put(cb);

	return DIA_SUCCESS;
}

tDiaResult dia_WlanInfo::getScanInfo(tCString ifname, tU8 ifIndex)
{
	DIA_TR_INF("dia_WlanInfo::getScanInfo()");

	// allocate a callback for which we can register various handlers
	struct nl_cb *cb = nl_cb_alloc(NL_CB_DEFAULT);
	if (!cb) {
		DIA_TR_ERR("dia_WlanInfo::getScanInfo(): Failed to allocate netlink callback");
	    return DIA_FAILED;
	}

	struct nl_msg *msgCmd = nlmsg_alloc();
	if (!msgCmd) {
		DIA_TR_ERR("dia_WlanInfo::getScanInfo(): Failed to allocate netlink message");
		return DIA_FAILED;
	}

	/**** Prepare and send message for the command GET_INTERFACE ***/
	// Create the message headers.
	if (!genlmsg_put(msgCmd, 0, 0, wifi.nl80211_id, 0, NLM_F_DUMP, NL80211_CMD_GET_SCAN, 0)) { // The NL80211_CMD_GET_SCAN command receives mac_addr and if connected the frequency of the channel
		DIA_TR_ERR("dia_WlanInfo::getScanInfo(): Failed to create genl header for the message NL80211_CMD_GET_SCAN");
		return DIA_FAILED;
	}

	if (nla_put_u32(msgCmd, NL80211_ATTR_IFINDEX, if_nametoindex(ifname)) < 0) {
		DIA_TR_ERR("dia_WlanInfo::getStationInfo(): Failed to put attr NL80211_ATTR_IFINDEX to the message NL80211_CMD_GET_SCAN");
		return DIA_FAILED;
	}

	// Finalize and transmit Netlink message.
	if (nl_send_auto(wifi.nls, msgCmd) < 0) {
		DIA_TR_ERR("dia_WlanInfo::getScanInfo(): Failed to send nl message NL80211_CMD_GET_SCAN");
		return DIA_FAILED;
	}
	nl_cb_set(cb, NL_CB_VALID, NL_CB_CUSTOM, getScanHandler, &ifIndex);

	// dump is finished
	tS32 err = 1;
	nl_cb_set(cb, NL_CB_FINISH, NL_CB_CUSTOM, getScanFinishHandler, &err);

	while (err > 0) {
		err = nl_recvmsgs(wifi.nls, cb); // Stops reading if one of the callbacks returns NL_STOP//Coverity Fix(CID:28447)
		DIA_TR_INF("dia_WlanInfo::getScanInfo(): nl_recvmsgs()");
	}

	nlmsg_free(msgCmd);
	nl_cb_put(cb);

	return DIA_SUCCESS;
}

tDiaResult dia_WlanInfo::updateWiFiInfo()
{
	DIA_TR_INF("dia_WlanInfo::updateWiFiInfo()");

	m_wlanInfoAvailable = FALSE;
	wifiInfoElements.clear(); // will be filled with new content within the callback handler: getInterfaceInfo()

	if (getInterfaceInfo() != DIA_SUCCESS) {
		return DIA_FAILED;
	}

	for (tU8 i=0; i<wifiInfoElements.size(); i++) {
		if (wifiInfoElements[i].essIdAvailable) { // Precede only if SSID is available

			if (getStationInfo(wifiInfoElements[i].iface, i) != DIA_SUCCESS) {
				return DIA_FAILED;
			}

			if (getScanInfo(wifiInfoElements[i].iface, i) != DIA_SUCCESS) {
				return DIA_FAILED;
			}
		}
	}

	m_wlanInfoAvailable = TRUE;

	return DIA_SUCCESS;
}

WiFiInfoElement dia_WlanInfo::getWiFiInfoElement()
{
	WiFiInfoElement wifiInfoElement;

	for (tU8 i=0; i<wifiInfoElements.size(); i++) {
		if (wifiInfoElements[i].connected) {
			wifiInfoElement = wifiInfoElements[i];
			break; // Only one interface (e.g. wlan0) will be in connected state (either as AP or Managed)
		}
	}

	return wifiInfoElement;
}
