/*
 * dia_GPUInfo.cpp
 *
 *  Created on: Nov 27, 2015
 *      Author: jas1hi
 */

#include <fstream>

#ifndef DIA_HELPER_H_
#include "project/services/customer/MonitoringCalibration/helper/dia_Helper.h"
#endif

#include "dia_GPUInfo.h"


#ifndef SYSGALCOREDIR
#define SYSGALCOREDIR "/sys/kernel/debug/gc" // "/sys/bus/platform/drivers/galcore"
#endif

#ifndef MEMINFOFILE
#define MEMINFOFILE SYSGALCOREDIR "/meminfo"
#endif

#ifndef IDLETIMEFILE
#define IDLETIMEFILE SYSGALCOREDIR "/idle" //"/idletime"
#endif

#ifndef DATABASEFILE
#define DATABASEFILE SYSGALCOREDIR "/database"
#endif

#ifndef PIDFILE
#define PIDFILE SYSGALCOREDIR "/clients" //"/pid"
#endif

#ifndef VIDMEMORYFILE
#define VIDMEMORYFILE SYSGALCOREDIR "/vidmem" //"/sys/kernel/debug/gpu/vidmem"
#endif

#define NANO 1000000000L

const char* USED_MEMORY_STR = "Used";
const char* PID_STR = "PID";



dia_GPUInfo::dia_GPUInfo()
:	 mMemInfo()
{

}

dia_GPUInfo::~dia_GPUInfo()
{

}

tDiaResult dia_GPUInfo::scanMemInfo(tULong& usedMem)
{
// Structure of the new meminfo file
/*
VIDEO MEMORY:
    gcvPOOL_SYSTEM:
        Free  :   51699904 B
        Used  :  350953280 B
        Total :  402653184 B
    gcvPOOL_CONTIGUOUS:
        Used  :          0 B
    gcvPOOL_VIRTUAL:
        Used  :          0 B

NON PAGED MEMORY:
    Used  :          0 B
 */

	std::string memInfoFileName;
#ifdef __DIA_UNIT_TESTING__
	memInfoFileName = std::string(getenv("_SWNAVISUBST")) + std::string("/di_middleware_server/components/fc_diagnosis/project/utest/fakes/test_data/galcore/meminfo");
#else
	memInfoFileName = std::string(MEMINFOFILE);
#endif

	std::ifstream infile(memInfoFileName.c_str());
	if (!infile.is_open()) {
		DIA_TR_ERR("dia_GPUInfo::scanMemInfo: Failed to open file %s", memInfoFileName.c_str());
		return DIA_FAILED;
	}

	usedMem = 0;
	std::string memInfoLineStr;
	tSize firstNonWhiteSpace;
	while (std::getline(infile, memInfoLineStr)) {
//		DIA_TR_INF("dia_GPUInfo::scanMemInfo: readLine: '%s'", memInfoLineStr.c_str());
		firstNonWhiteSpace = memInfoLineStr.find_first_not_of(' ');
		if (memInfoLineStr.compare(firstNonWhiteSpace, strlen(USED_MEMORY_STR), USED_MEMORY_STR) == 0) {
			sscanf(memInfoLineStr.c_str(), "%*s  :   %lu B", &mMemInfo.mUsedMem);
			break;
		}
	}
	infile.close();

	usedMem =  mMemInfo.mUsedMem >> 10; // convert Bytes to KBytes

	return DIA_SUCCESS;
}

tDiaResult dia_GPUInfo::updateProcessList()
{
/*
	**************************
	***  PROCESS DB DUMP   ***
	**************************
	PID     NAME
	1152    apphmi_homescre
	96      procvddim_out.o
	1156    apphmi_media-rn
	773     apphmi_navigati
	1142    apphmi_sds-rnai
	502     procmap.out
	1129    apphmi_telemati
	1145    apphmi_vehicle-
	762     apphmi_testmode
	1147    apphmi_sxm-rnai
	1163    apphmi_phone-rn
	412     apphmi_tuner-rn
	765     apphmi_system-r
	109     LayerManagerSer
	365     apphmi_master-r
*/

// Structure of the new clients file (old pid file)
/*
	PID     NAME
	------------------------
	1333    apphmi_testmode
	581     apphmi_homescre
	1207    apphmi_system_o
	791     apphmi_vehicle_
	119     procdispvidctrl
	584     apphmi_tuner_ou
	778     apphmi_telemati
	1194    apphmi_sds_out.
	794     apphmi_media_ou
	571     apphmi_navigati
	1020    apphmi_phone_ou
	125     LayerManagerSer
	589     apphmi_sxm_out.
	558     apphmi_master_o
*/


	std::string pidFileName;
#ifdef __DIA_UNIT_TESTING__
	pidFileName = std::string(getenv("_SWNAVISUBST")) + std::string("/di_middleware_server/components/fc_diagnosis/project/utest/fakes/test_data/galcore/pid");
#else
	pidFileName = std::string(PIDFILE);
#endif

	// Determine the processes that are using GPU
	std::ifstream infilePid(pidFileName.c_str());
	if (!infilePid.is_open()) {
		DIA_TR_ERR("dia_GPUInfo::updateProcessList: Failed to open file %s", pidFileName.c_str());
		return DIA_FAILED;
	}

	mTopProcessSortedViaGPUMemoryUsage.clear();
	tBool foundPidEntry = FALSE;
	std::string pidLineStr;
	while (std::getline(infilePid, pidLineStr)) {
		if (!foundPidEntry && pidLineStr.find(PID_STR) == 0) { // First line after which pids will be listed
			foundPidEntry = TRUE;
			std::getline(infilePid, pidLineStr); // Read one more line containing "------------------------"
			continue;
		}

		if (foundPidEntry) {
			struct GPUProcess gpuProcess;
			tS32 sscanfResult = sscanf(pidLineStr.c_str(), "%d	%s", &gpuProcess.m_pid, &gpuProcess.command[0]);
			 if (sscanfResult == EOF || sscanfResult != 2) {
				 DIA_TR_ERR("dia_GPUInfo::updateProcessList: sscanf() failed with return value: %d", sscanfResult);
				 return DIA_FAILED;
			 }

			 std::string vidMemFileName;
#ifdef __DIA_UNIT_TESTING__
			 vidMemFileName = std::string(getenv("_SWNAVISUBST")) + std::string("/di_middleware_server/components/fc_diagnosis/project/utest/fakes/test_data/galcore/vidmem");
#else
			 vidMemFileName = std::string(VIDMEMORYFILE);
#endif

			// Determine the video memory usage of pid by writing the pid into the file and reading later the content from the same file
			std::ofstream outfileVidMem(vidMemFileName.c_str());
#ifndef __DIA_UNIT_TESTING__ // No need to write into file when doing UNIT testing

			if (!outfileVidMem.is_open()) {
				DIA_TR_ERR("dia_GPUInfo::updateProcessList: Failed to open file %s for writing", vidMemFileName.c_str());
				return DIA_FAILED;
			}

			std::string pidStr = dia_Helper::tU16ToString((tU16)gpuProcess.m_pid);
//			DIA_TR_INF("dia_GPUInfo::updateProcessList: Read pid: %s [%s]", pidStr.c_str(), gpuProcess.command);

			outfileVidMem.write(pidStr.c_str(), (tS32)pidStr.size());
			if (outfileVidMem.bad()) {
				DIA_TR_ERR("dia_GPUInfo::updateProcessList: Failed to write pid(%s) into file %s", pidStr.c_str(), VIDMEMORYFILE);
				outfileVidMem.close();
				return DIA_FAILED;
			}
#endif
			outfileVidMem.close();
/*
			VidMem Usage (Process 1104):
			                All     Index    Vertex   Texture        RT     Depth    Bitmap        TS     Image      Mask   Scissor   HZDepth
			Current    30793056         0        32   3084352   6160384   3080192  18432000     36096         0         0         0         0
			Maximum    30793056         0        32   3084352   6160384   3080192  18432000     36096         0         0         0         0
			Total      33254240         0        32   5150784   6422528   3211264  18432000     37632         0         0         0         0
*/
			// Now read the result from the same file
			std::ifstream infileVidMem(vidMemFileName.c_str());
			if (!infileVidMem.is_open()) {
				DIA_TR_ERR("dia_GPUInfo::updateProcessList: Failed to open file %s", vidMemFileName.c_str());
				return DIA_FAILED;
			}

			std::string vidmemLineStr;
//			tFloat percent_gpumem = 0.0;
			while (std::getline(infileVidMem, vidmemLineStr)) {
//				DIA_TR_INF("dia_GPUInfo::updateProcessList: vidmemLineStr: %s", vidmemLineStr.c_str());
				tULong currentMemKByte = 0;
				switch (vidmemLineStr[0]) {
						case 'C': // Note: Instead of comparing the whole string "Current" we compare only to "C"
							if (vidmemLineStr[1] == 'u') { // make sure that the second letter is 'u'
								sscanf(vidmemLineStr.c_str(), "%*s	%lu", &gpuProcess.m_currentMemByte);
	//							percent_gpumem = (gpuProcess.m_currentMemByte) / (tDouble)(mMemInfo.mTotalMem) * 100.0;
	//							DIA_TR_INF("dia_GPUInfo::updateProcessList: gpuProcess.m_currentMem: %lu", gpuProcess.m_currentMemByte);
							}
							break;

						case 'M': // Note: Instead of comparing the whole string "Maximum" we compare only to "M"
							if (vidmemLineStr[1] == 'a') {  // make sure that the second letter is 'a'
								sscanf(vidmemLineStr.c_str(), "%*s	%lu", &gpuProcess.m_maximumMemKByte); // Read in Bytes from file
								gpuProcess.m_maximumMemKByte = gpuProcess.m_maximumMemKByte >> 10; // Convert to KBytes
								currentMemKByte = gpuProcess.m_currentMemByte >> 10;
	#ifdef __DIA_UNIT_TESTING__
								currentMemKByte++; // Increase it for UNIT testing because we have only one source file for vidmem with the same information
	#endif
								mTopProcessSortedViaGPUMemoryUsage[currentMemKByte] = gpuProcess; // After detecting the maximum memory usage, the structure can be saved into map
							}
							break;

						default:
							break;
				}
			}
			infileVidMem.close();
		}
	}
	infilePid.close();

	return DIA_SUCCESS;
}

std::map<tULong, struct GPUProcess> dia_GPUInfo::getTopProcessSortedViaVideoMemoryUsage(tU16 numberOfProcess)
{
	std::map<tULong, struct GPUProcess> topProcessSortedViaGPUMemoryUsage;

	if (mTopProcessSortedViaGPUMemoryUsage.size() >= numberOfProcess) {
		mTopProcessReverseIterator = mTopProcessSortedViaGPUMemoryUsage.rbegin();
		for(tU16 i=0; i<numberOfProcess; i++) {
			topProcessSortedViaGPUMemoryUsage[mTopProcessReverseIterator->first] = mTopProcessReverseIterator->second;
			mTopProcessReverseIterator++;
		}
	}
	else { // if the size is available elements is smaller than the requested one, return empty map
		DIA_TR_WRN("dia_GPUInfo::getTopProcessSortedViaVideoMemoryUsage: numberOfProcess: %d > size: %d", mTopProcessSortedViaGPUMemoryUsage.size(), numberOfProcess);
	}

	return topProcessSortedViaGPUMemoryUsage;
}

tDiaResult dia_GPUInfo::getGPUIdleTime(tFloat& gpuIdleTimePercent)
{
// GPU idle time since last query: 38582137136 ns

// Structure of the new database file (contains GPU Idle time)
/*
	GPU Idle: 296050802710 ns
	--------------------------------------------------------------------------------
	Process: 1328     apphmi_navigati
*/

// Structure of the new idle file (old idletime file)
/*
	Start:   2006142516929 ns
	End:     2210540531344 ns
	On:      204398014415 ns
	Off:     0 ns
	Idle:    0 ns
	Suspend: 0 ns
*/

	 std::string gpuDatabaseFileName;
	 std::string gpuIdleFileName;
#ifdef __DIA_UNIT_TESTING__
	 gpuDatabaseFileName = std::string(getenv("_SWNAVISUBST")) + std::string("/di_middleware_server/components/fc_diagnosis/project/utest/fakes/test_data/galcore/idletime");
	 gpuIdleFileName = std::string(getenv("_SWNAVISUBST")) + std::string("/di_middleware_server/components/fc_diagnosis/project/utest/fakes/test_data/galcore/idle");
#else
	 gpuDatabaseFileName = std::string(DATABASEFILE);
	 gpuIdleFileName = std::string(IDLETIMEFILE);
#endif

	std::ifstream infileDatabase(gpuDatabaseFileName.c_str());
	if (!infileDatabase.is_open()) {
		DIA_TR_ERR("dia_GPUInfo::getGPUIdleTime: Failed to open file %s", gpuDatabaseFileName.c_str());
		return DIA_FAILED;
	}

	tU64 idleTime = 0;
	std::string idleTimeLineStr;
	if (std::getline(infileDatabase, idleTimeLineStr)) {
		if (idleTimeLineStr[0] == 'G') { // Note: Instead of comparing the whole string "GPU" we compare only to "G"
			sscanf(idleTimeLineStr.c_str(), "GPU Idle: %llu ns", &idleTime);
		}
	}
	infileDatabase.close();

	gpuIdleTimePercent = 99.0; // default value
	if (idleTime > 0) {
		std::ifstream infileIdle(gpuIdleFileName.c_str());
		if (!infileIdle.is_open()) {
			DIA_TR_ERR("dia_GPUInfo::getGPUIdleTime: Failed to open file %s", gpuIdleFileName.c_str());
			return DIA_FAILED;
		}

		tU64 elapsedTimeSinceLastQuery = 0;
		std::string onTimeLineStr;
		while (std::getline(infileIdle, onTimeLineStr)) {
			if (onTimeLineStr[0] == 'O') { // Note: Instead of comparing the whole string "On" we compare only to "O"
				sscanf(onTimeLineStr.c_str(), "On:      %llu ns", &elapsedTimeSinceLastQuery);
				break;
			}
		}
		infileIdle.close();

		if (elapsedTimeSinceLastQuery > 0 && idleTime < elapsedTimeSinceLastQuery) { // Sometimes idleTime > elapsedTimeSinceLastQuery which must be a bug
			gpuIdleTimePercent = (tFloat)((idleTime) / (tDouble)(elapsedTimeSinceLastQuery)) * 100;
		}
//		DIA_TR_INF("dia_GPUInfo::getGPUIdleTime: gpuIdleTimePercent: %lf  elapsedTimeSinceLastQuery: %lld", gpuIdleTimePercent, elapsedTimeSinceLastQuery);
	}

	return DIA_SUCCESS;
}
