/**
 * \file      dia_BaseStream.h
 *
 * \brief     Abstract class BaseStream,  base class for concrete stream implementation
 *
 *
 *  Created on: Oct 20, 2015
 *      Author: jas1hi
 *
 * \copyright Robert Bosch Car Multimedia 2015
 */

#ifndef DIA_BASESTREAM_H_
#define DIA_BASESTREAM_H_

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

const tS32 DIA_STREAM_ERROR = -1;
const tS32 DIA_STREAM_EOF = 0;


class dia_BaseStream
{
public:
//	dia_BaseStream();
	virtual ~dia_BaseStream() {};

protected:
	/**
	*   Writes data of specified length to the stream
	*	Note: To be overwritten in concrete implementation
	*	@return number of bytes written or -1 if error is detected
	*/
	virtual tS32 writeBytes( const tVoid* buffer, tSize length ) = 0;

	/**
	*   Reads data of from the stream
	*	Note: To be overwritten in concrete implementation
	*	@return number of bytes read or -1 if error is detected
	*/
	virtual tS32 readBytes( tVoid* buffer, tSize length )const = 0;

public:
	/**
	* Set position for read/write operation
	* @param position new read/write position
	* @return previous position or -1 if error is detected
	*/
	virtual tSize setPosition( tSize position ) = 0 ;

	/**
	*	Returns current position for read/write operation
	*	@return current position for read/write operation
	*/
	virtual tS32 getPosition()const = 0;

	/**
	* Returns length of data in stream
	* @return length of data in stream or -1 if error is detected
	*/
//	virtual size_t length()const = 0;

	/**
	* Flush stream
	* @return -1 if error, otherwise 1
	*/
	virtual tS32 flush() = 0;

	/**
	*	Close stream and frees resources ( for example file handler and etc)
	*/
	 virtual tVoid close() = 0;

	/**
	* Reads one byte
	* @param data  pointer to variable which after read completion contains read byte
	* @return -1 if error is detected, or the byte that was read
	*/
	 tS32 readByte( tU8* data )const;

	/**
	* Reads numberToRead bytes in outBuffer
	* @param outBuffer buffer which after read completion contains read data and size() returns actual read data size
	* @param numberToRead number bytes requested to read
	* @return -1 if error is detected, or number of bytes read
	*/
	 tS32 readBuffer( std::vector<tU8>* outBuffer, tSize numberToRead )const;

	/**
	* Reads 4 byte unsigned
	* @param data  pointer to variable which after read completion contains string
	* @return -1 if error is detected, or number of bytes read
	*/
	 tS32 readString( std::string* str, tSize numberToRead )const;

	/**
	* Reads line from stream
	* line is character of symbols terminated with '\n' or '\r'
	* @param str  pointer to buffer which after read completion contains read string
	* @return -1 if error is detected, or number of bytes read
	*/
	 tS32 readLine( std::string& str )const;

	/**
	* Writes  bytes array
	* @param buffer  specifies both bytes and number to be written
	* @return -1 if error is detected, or number of bytes written
	*/
	  tS32 writeBuffer( const std::vector<tU8>& buffer  );

	/**
	* Writes string
	* @param str  string to be written
	* @return -1 if error is detected, or number of bytes written
	*/
	  tS32 writeString( const std::string& str );

	/**
	* Writes line to stream and writes terminating "\n" sequence
	* @param str  string to be written
	* @return -1 if error is detected, or number of bytes written
	*/
	  tS32 writeLine( const std::string& str );
};

#endif /* DIA_BASESTREAM_H_ */
