/**
 * \file    dia_VINStateMachineHandler.cpp
 *
 * \brief   Implements VIN handling for the VIN Slave ECU
 *
 * \author  ama8kor
 * \date    11.03.2019
 *
 * \copyright Robert Bosch Car Multimedia 2019
 */

#ifndef __INCLUDED_DIA_VIN_STATE_MACHINE_HANDLER__
#include "project/framework/vin/dia_VINStateMachineHandler.h"
#endif 

#ifndef __INCLUDED_DIA_DEFS_CONFIG_PROJECT__
#include "project/framework/config/dia_defsProjectConfig.h"
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include "common/framework/config/dia_ConfigManager.h"
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG_VIN__
#include "project/framework/vin/dia_defsVinConfig.h"
#endif

#ifndef __INCLUDED_DIA_LOCK_SCOPE__
#include <common/framework/application/dia_LockScope.h>
#endif

#ifndef __INCLUDED_DIA_VINODOMETERHANDLER__
#include "project/framework/vin/dia_Odometer.h"
#endif

#ifndef __INCLUDED_DIA_CANSIGNALHANDLER__
#include "project/framework/vin/dia_CANSignalHandler.h"
#endif

using namespace dia;
  
DIA_IMPL_SINGLETON_WITH_SETUP_AND_TEARDOWN(dia_VINStateMachineHandler);

std::vector<tU8> dia_VINStateMachineHandler::vecVINDefault;
std::vector<tU8> dia_VINStateMachineHandler::vecVINOriginal;
std::vector<tU8> dia_VINStateMachineHandler::vecVINCurrent;
std::vector<tU8> dia_VINStateMachineHandler::vecVINReceived;
tU8 u8Odometer = 0;
/************************************************************************
*FUNCTION: 		dia_VINStateMachineHandler
*DESCRIPTION: 	Class Constructor : Initialize lock 
*PARAMETER:		void
*RETURNVALUE: 	None
*
************************************************************************/
dia_VINStateMachineHandler::dia_VINStateMachineHandler()
	: mSyncObj("dia_VINStateMachineHandler_LK"), 
	  u8Odometer(0)
{
	DIA_TR_INF("dia_VINStateMachineHandler::dia_VINStateMachineHandler(...)");
}
/************************************************************************
*FUNCTION: 		~dia_VINStateMachineHandler
*DESCRIPTION: 	Class Destructor
*PARAMETER:		None
*RETURNVALUE: 	None 
*
************************************************************************/
dia_VINStateMachineHandler::~dia_VINStateMachineHandler()
{
	DIA_TR_INF("dia_VINStateMachineHandler::~dia_VINStateMachineHandler(...)");
}
/************************************************************************
*FUNCTION: 		setup
*DESCRIPTION: 	Method to initialize local default, original, current, received VIN & Odometer
*PARAMETER:		None
*RETURNVALUE: 	tDiaResult -  DIA_SUCCESS/DIA_FAILED 
*
************************************************************************/
tDiaResult 
dia_VINStateMachineHandler::setup( void )
{
	dia_tclFnctTrace oTrace("dia_VINStateMachineHandler::setup(...)");
	
	vecVINDefault.reserve(VIN_LENGTH_MAX);
	//vecVINDefault.resize(VIN_LENGTH_MAX);
	vecVINOriginal.reserve(VIN_LENGTH_MAX);
	//vecVINOriginal.resize(VIN_LENGTH_MAX);
	vecVINCurrent.reserve(VIN_LENGTH_MAX);
	//vecVINCurrent.resize(VIN_LENGTH_MAX);
	vecVINReceived.reserve(VIN_LENGTH_MAX);
	//vecVINReceived.resize(VIN_LENGTH_MAX);
	
	for(tU8 i=0; i<VIN_LENGTH_MAX; ++i)
	{
		vecVINDefault.push_back(VIN_DEFAULT_VALUE);
		vecVINOriginal.push_back(VIN_DEFAULT_VALUE);
		vecVINCurrent.push_back(VIN_DEFAULT_VALUE);
		vecVINReceived.push_back(VIN_DEFAULT_VALUE);
	}

	return DIA_SUCCESS;
}
/************************************************************************
*FUNCTION: 		tearDown
*DESCRIPTION: 	Method to reset local default, original, current, received VIN & Odometer
*				Release lock
*PARAMETER:		None
*RETURNVALUE: 	tDiaResult -  DIA_SUCCESS/DIA_FAILED
*
************************************************************************/
tDiaResult 
dia_VINStateMachineHandler::tearDown( void )
{
	dia_tclFnctTrace oTrace("dia_VINStateMachineHandler::tearDown(...)");
	
	std::fill(vecVINDefault.begin(), vecVINDefault.end(), 0);
	std::fill(vecVINOriginal.begin(), vecVINOriginal.end(), 0);
	std::fill(vecVINCurrent.begin(), vecVINCurrent.end(), 0);
	std::fill(vecVINReceived.begin(), vecVINCurrent.end(), 0);
	u8Odometer = 0;
	
	LockScope lock(mSyncObj);
	
	return DIA_SUCCESS;
}
/************************************************************************
*FUNCTION: 		processVINRequest
*DESCRIPTION: 	This method implements the VIN Current handling 
*PARAMETER:		rcvdVin - received VIN ( via CAN Message OR Diag Command )
*RETURNVALUE: 	tDiaResult -  DIA_SUCCESS/DIA_FAILED
*
************************************************************************/
tDiaResult
dia_VINStateMachineHandler::processVINRequest( const std::vector<tU8>& rcvdVin )
{
	dia_tclFnctTrace oTrace("dia_VINStateMachineHandler::processVINRequest(...)");
	
	LockScope lock(mSyncObj);
	// initialize return value
	tDiaResult retCode = DIA_FAILED;
	
	// Copy received VIN into the variable
	vecVINReceived.clear();
	for(tU8 i=0; i<VIN_LENGTH_MAX; ++i)
	{
		vecVINReceived[i] = (rcvdVin[i]);
	}
	
	if(!(DIA_SUCCESS == validateRcvdVIN(vecVINReceived)))
	{
		DIA_TR_ERR("dia_VINStateMachineHandler::processVINRequest - Received VIN is not Valid !!!");
		return DIA_FAILED;
	}
	
	// Compare received VIN with default VIN 
	if(!(DIA_SUCCESS == compareVIN(DIA_EN_VIN_COMPARE_TYPE_CURRENT_RCVD_AND_DEFAULT)))
	{
		DIA_TR_INF("dia_VINStateMachineHandler::processVINRequest - received VIN & default VIN are different !!!");
		// Compare received VIN with VIN Current Stored
		if(!(DIA_SUCCESS == compareVIN(DIA_EN_VIN_COMPARE_TYPE_CURRENT_RCVD_AND_STORED)))
		{
			DIA_TR_INF("dia_VINStateMachineHandler::processVINRequest - received VIN & VIN Current Stored are different !!!");
			// Copy VIN received as VIN Current
			vecVINCurrent.clear();
			for(tU8 i=0; i<VIN_LENGTH_MAX; ++i)
			{
				vecVINCurrent.push_back(vecVINReceived[i]);
			}
			
			// Write VIN received as VIN Current into KDS
			if(DIA_SUCCESS == writeProperty(vecVINCurrent, DIA_PROP_CENTER_CMC_19_VIN_CURRENT))
			{
				//
				if(DIA_SUCCESS == compareVIN(DIA_EN_VIN_COMPARE_TYPE_ORIGINAL_STORED_AND_DEFAULT))
				{
					
					if(DIA_SUCCESS == processVINCommon())
					{
						DIA_TR_INF("dia_VINStateMachineHandler::processVINRequest - VIN REQUEST PROCESSED SUCCESSFULLY !!!");
						retCode = DIA_SUCCESS;
					}
					else
					{
						DIA_TR_ERR("dia_VINStateMachineHandler::processVINRequest - VIN REQUEST FAILED TO PROCESS !!!");
						return retCode;
					} /* if(DIA_SUCCESS == processVINCommon()) */
				}
				else
				{

#ifndef VARIANT_S_FTR_ENABLE_CMC19_EVOBUS_VIN_THREAD_CONCEPT
					DIA_TR_INF("dia_VINStateMachineHandler::processVINRequest - default VIN with VIN Original Stored are different !!!");
					if(VIN_NOT_LOCKED == isVINlocked())

#else 

					if(DIA_SUCCESS == compareVINOdometer())

#endif
					{
						if(!(DIA_SUCCESS == compareVIN(DIA_EN_VIN_COMPARE_TYPE_CURRENT_STORED_AND_ORIGINAL)))
						{
							if(DIA_SUCCESS == processVINCommon())
							{
								DIA_TR_INF("dia_VINStateMachineHandler::processVINRequest - VIN REQUEST PROCESSED SUCCESSFULLY !!!");
								retCode = DIA_SUCCESS;
							}
							else
							{
								DIA_TR_ERR("dia_VINStateMachineHandler::processVINRequest - VIN REQUEST FAILED TO PROCESS !!!");
								return retCode;
							} /* if(DIA_SUCCESS == processVINCommon()) */
						}
						else
						{
							DIA_TR_INF("dia_VINStateMachineHandler::processVINRequest - RECEIVED/CURRENT VIN IS SAME AS ORIGINAL VIN !!!");
							retCode = DIA_SUCCESS;
						} /* if(!(DIA_SUCCESS == compareVIN(DIA_EN_VIN_COMPARE_TYPE_CURRENT_STORED_AND_ORIGINAL))) */
					}
					else
					{
						DIA_TR_INF("dia_VINStateMachineHandler::processVINRequest - VIN ODOMETER >= ODOMETER LIMIT !!!");
						retCode = DIA_SUCCESS;
					}/* if(DIA_SUCCESS == compareVINOdometer()) */
				} /* if(DIA_SUCCESS == compareVIN(DIA_EN_VIN_COMPARE_TYPE_ORIGINAL_STORED_AND_DEFAULT)) */
			}
			else
			{
				DIA_TR_ERR("dia_VINStateMachineHandler::processVINRequest - VIN CURRENT WRITE FAILED !!!");
				return retCode;
			} /* if(DIA_SUCCESS == writeProperty()) */
		}
		else
		{
			DIA_TR_INF("dia_VINStateMachineHandler::processVINRequest - RECEIVED VIN IS SAME AS CURRENT VIN !!!");
			retCode = DIA_SUCCESS;
		} /* if(!(DIA_SUCCESS == compareVIN(DIA_EN_VIN_COMPARE_TYPE_CURRENT_RCVD_AND_STORED))) */
	}
	else
	{
		DIA_TR_INF("dia_VINStateMachineHandler::processVINRequest - RECEIVED VIN IS SAME AS DEFAULT VIN !!!");
		retCode = DIA_SUCCESS;
	} /* if(!(DIA_SUCCESS == compareVIN(DIA_EN_VIN_COMPARE_TYPE_CURRENT_RCVD_AND_DEFAULT))) */
	
	return retCode;
}
/************************************************************************
*FUNCTION: 		processVINCommon
*DESCRIPTION: 	This method implements the part of VIN Current handling 
*PARAMETER:		None
*RETURNVALUE: 	tDiaResult -  DIA_SUCCESS/DIA_FAILED
*
************************************************************************/
tDiaResult
dia_VINStateMachineHandler::processVINCommon()
{
	dia_tclFnctTrace oTrace("dia_VINStateMachineHandler::processVINCommon(...)");
	// initialize return value
	tDiaResult retCode = DIA_FAILED;
	// temporary variable to read VIN current
	std::vector<tU8> vecVINCurStored;
	vecVINCurStored.reserve(VIN_LENGTH_MAX);
	//vecVINCurStored.resize(VIN_LENGTH_MAX);
	
	// Read VIN Current & Write as VIN Original
	vecVINCurStored.clear();
	if(DIA_SUCCESS == readProperty(DIA_PROP_CENTER_CMC_19_VIN_CURRENT, vecVINCurStored))
	{
		// Copy read VIN into the out variable
		vecVINOriginal.clear();
		for(tU8 i=0; i<VIN_LENGTH_MAX; ++i)
		{
			vecVINOriginal.push_back(vecVINCurStored[i]);
			DIA_TR_INF("dia_VIN::processVINCommon :vecVINOriginal VIN[%d] = 0x%X", i, vecVINOriginal[i]);
		}
	}
	else
	{
		DIA_TR_ERR("dia_VINStateMachineHandler::processVINCommon - READ VIN CURRENT FAILED !!!");
		return retCode;
	}
	
	// Try to save VIN Original into KDS
	if(DIA_SUCCESS == writeProperty(vecVINOriginal, DIA_PROP_CENTER_CMC_19_VIN_ORIGINAL))
	{
		// before resetting vin odometer, save the current raw value for future reference
		std::vector<tU8> vcOdoRef;
		vcOdoRef.reserve(VIN_ODOMETER_CAN_SIGNAL_LENGTH);
		//vcOdoRaw.resize(VIN_ODOMETER_CAN_SIGNAL_LENGTH);
#ifndef VARIANT_S_FTR_ENABLE_CMC19_EVOBUS_VIN_THREAD_CONCEPT

		/* Read Odometer from the CAN Signal */
   		tU32 rawOdometer = 0;
		
		if(DIA_SUCCESS == readOdometerValue(&rawOdometer))
		{
			DIA_TR_INF("dia_VINStateMachineHandler::processVINCommon - Raw odometer Read from CAN Signal.");
			
			for(tU8 i = 0; i < VIN_ODOMETER_CAN_SIGNAL_LENGTH; i++)
			{
				vcOdoRef[i] = (tU8)(rawOdometer >> (i*8));
			}	    

			if(DIA_SUCCESS == writeProperty(vcOdoRef, DIA_PROP_CENTER_CMC_19_VIN_ODOMETER_REF_RAW))
			{
				DIA_TR_INF("dia_VINStateMachineHandler::processVINCommon - Writing Ref odometer done !!!");
			}
			else
			{
				DIA_TR_ERR("dia_VINStateMachineHandler::processVINCommon - Writing Ref odometer failed !!!");
			}
		}

#else		
		if(DIA_SUCCESS == readProperty(DIA_PROP_CENTER_CMC_19_VIN_ODOMETER_CSM_SIGNAL_RAW, vcOdoRaw))
		{
			if(DIA_SUCCESS == writeProperty(vcOdoRaw, DIA_PROP_CENTER_CMC_19_VIN_ODOMETER_REF_RAW))
			{
				DIA_TR_INF("dia_VINStateMachineHandler::processVINCommon - Writing Ref odometer done !!!");
			}
			else
			{
				DIA_TR_ERR("dia_VINStateMachineHandler::processVINCommon - Writing Ref odometer failed !!!");
			}
		}
		else
		{
			DIA_TR_ERR("dia_VINStateMachineHandler::processVINCommon - Reading Raw odometer failed !!!");
		}
#endif
		
		if(DIA_SUCCESS == resetVIN(DIA_PROP_CENTER_CMC_19_VIN_ODOMETER_COUNTER))
		{
			DIA_TR_INF("dia_VINStateMachineHandler::processVINCommon - RESET VIN ODOMETER TO 0KM SUCCESS !!!");
			retCode = DIA_SUCCESS;
		}
		else
		{
			DIA_TR_ERR("dia_VINStateMachineHandler::processVINCommon - RESET VIN ODOMETER FAILED !!!");
			return retCode;
		} /* if(DIA_SUCCESS == resetVINOdometer()) */
	}
	else
	{
		DIA_TR_ERR("dia_VINStateMachineHandler::processVINCommon - VIN ORIGINAL WRITE FAILED !!!");
		return retCode;
	} /* if(DIA_SUCCESS == writeProperty()) */
	
	return retCode;
}
/************************************************************************
*FUNCTION: 		compareVIN
*DESCRIPTION: 	This method compares two different VIN based on the type passed
*PARAMETER:		dia_enVINCompareType - 
*				DIA_EN_VIN_COMPARE_TYPE_CURRENT_RCVD_AND_DEFAULT [ received & default ]
*				DIA_EN_VIN_COMPARE_TYPE_CURRENT_RCVD_AND_STORED [ received & stored current ]
*				DIA_EN_VIN_COMPARE_TYPE_CURRENT_STORED_AND_ORIGINAL [ stored current & stored original ]
*				DIA_EN_VIN_COMPARE_TYPE_ORIGINAL_STORED_AND_DEFAULT [ stored original & default ]
*RETURNVALUE: 	tDiaResult -  DIA_SUCCESS/DIA_FAILED
*
************************************************************************/
tDiaResult
dia_VINStateMachineHandler::compareVIN(dia_enVINCompareType mType)
{
	dia_tclFnctTrace oTrace("dia_VINStateMachineHandler::compareVIN(...)");
	// initialize return value
	tDiaResult retCode = DIA_SUCCESS;
	
	switch(mType)
	{
		case DIA_EN_VIN_COMPARE_TYPE_CURRENT_RCVD_AND_DEFAULT:
		{

			// Comparing Received VIN & Default VIN	
			for(tU8 i = 0; i < VIN_LENGTH_MAX; i++)	
			{
				if(vecVINReceived[i] != vecVINDefault[i])
				{
					return DIA_FAILED;
				}
			}
			DIA_TR_INF("dia_VINStateMachineHandler::processVINRequest - received VIN & default VIN are same !!!");
			/*
			if(vecVINReceived == vecVINDefault)
			//if(std::equal(vecVINReceived.begin(), vecVINReceived.end(), vecVINDefault))
			{
				// Both are same
				DIA_TR_INF("dia_VINStateMachineHandler::processVINRequest - received VIN & default VIN are same !!!");
				retCode = DIA_SUCCESS;
			}
			else
			{
				// do nothing
			}*/
		}
		break;
		
		case DIA_EN_VIN_COMPARE_TYPE_CURRENT_RCVD_AND_STORED:
		{
			// temporary variable to read VIN current
			std::vector<tU8> vecVINCurStored;
			vecVINCurStored.reserve(VIN_LENGTH_MAX);
			//vecVINCurStored.resize(VIN_LENGTH_MAX);
			// Read VIN Current Stored
			if(DIA_SUCCESS == readProperty(DIA_PROP_CENTER_CMC_19_VIN_CURRENT, vecVINCurStored))
			{
				for(tU8 i = 0; i < VIN_LENGTH_MAX; i++)	
				{
					if(vecVINCurStored[i] != vecVINReceived[i])
					{
						return DIA_FAILED;
					}
				}
				DIA_TR_INF("dia_VINStateMachineHandler::processVINRequest - received VIN & Current VIN are same !!!");
				/*
				// Compare VIN Current Received & VIN Current Stored
				if(vecVINReceived == vecVINCurStored)
				//if(std::equal(vecVINReceived.begin(), vecVINReceived.end(), vecVINCurStored))
				{
					// Both are same
					retCode = DIA_SUCCESS;
				}
				else
				{
					// do nothing
				} // if(vecVINReceived == vecVINCurStored) 
				*/
			}
			else
			{
				// do nothing
			} /* if(DIA_SUCCESS == readProperty(DIA_PROP_CENTER_CMC_19_VIN_CURRENT, vecVINCurStored)) */
		}
		break;
		
		case DIA_EN_VIN_COMPARE_TYPE_CURRENT_STORED_AND_ORIGINAL:
		{
			// temporary variable to read VIN current
			std::vector<tU8> vecVINCurStored;
			vecVINCurStored.reserve(VIN_LENGTH_MAX);
			//vecVINCurStored.resize(VIN_LENGTH_MAX);
			// temporary variable to read VIN original
			std::vector<tU8> vecVINOrgStored;
			vecVINOrgStored.reserve(VIN_LENGTH_MAX);
			//vecVINOrgStored.resize(VIN_LENGTH_MAX);
			// Read VIN Current Stored
			if(DIA_SUCCESS == readProperty(DIA_PROP_CENTER_CMC_19_VIN_CURRENT, vecVINCurStored))
			{
				// Read VIN Original Stored
				if(DIA_SUCCESS == readProperty(DIA_PROP_CENTER_CMC_19_VIN_ORIGINAL, vecVINOrgStored))
				{
					for(tU8 i = 0; i < VIN_LENGTH_MAX; i++)	
					{
						if(vecVINOrgStored[i] != vecVINCurStored[i])
						{
							return DIA_FAILED;
						}
					}
					DIA_TR_INF("dia_VINStateMachineHandler::processVINRequest - Current Stored VIN & Original Stored VIN are same !!!");
					/*
					// Compare VIN Original Stored & VIN Current Stored
					if(vecVINOrgStored == vecVINCurStored)
					//if(std::equal(vecVINOrgStored.begin(), vecVINOrgStored.end(), vecVINCurStored))
					{
						// Both are same
						retCode = DIA_SUCCESS;
					}
					else
					{
						// do nothing
					} // if(vecVINOrgStored == vecVINCurStored) 
					*/
				}
				else
				{
					// do nothing
				} /* if(DIA_SUCCESS == readProperty(DIA_PROP_CENTER_CMC_19_VIN_ORIGINAL, vecVINOrgStored)) */
			}
			else
			{
				// do nothing
			} /* if(DIA_SUCCESS == readProperty(DIA_PROP_CENTER_CMC_19_VIN_CURRENT, vecVINCurStored)) */
			
			
		}
		break;
		
		case DIA_EN_VIN_COMPARE_TYPE_ORIGINAL_STORED_AND_DEFAULT:
		{
			// temporary variable to read VIN original
			std::vector<tU8> vecVINOrgStored;
			vecVINOrgStored.reserve(VIN_LENGTH_MAX);
			//vecVINOrgStored.resize(VIN_LENGTH_MAX);
			// Read VIN Original Stored
			if(DIA_SUCCESS == readProperty(DIA_PROP_CENTER_CMC_19_VIN_ORIGINAL, vecVINOrgStored))
			{
				for(tU8 i = 0; i < VIN_LENGTH_MAX; i++)	
				{
					if(vecVINOrgStored[i] != vecVINDefault[i])
					{
						return DIA_FAILED;
					}
				}
				DIA_TR_INF("dia_VINStateMachineHandler::processVINRequest - Default VIN & Original Stored VIN are same !!!");

				/*
				// Compare VIN Original Stored & VIN Default
				if(vecVINOrgStored == vecVINDefault)
				//if(std::equal(vecVINOrgStored.begin(), vecVINOrgStored.end(), vecVINDefault))
				{
					// Both are same
					retCode = DIA_SUCCESS;
				}
				else
				{
					// do nothing
				} /* if(vecVINOrgStored == vecVINDefault) 
				*/
			}
			else
			{
				// do nothing
			} /* if(DIA_SUCCESS == readProperty(DIA_PROP_CENTER_CMC_19_VIN_ORIGINAL, vecVINOrgStored)) */
			
		}
		break;
		
		default:
		{
			DIA_TR_ERR("dia_VINStateMachineHandler::compareVIN - Invalid  comparision type !!");
		}
		break;
	}
	
	return retCode;
}
/************************************************************************
*FUNCTION: 		writeProperty
*DESCRIPTION: 	This method saves VIN into non volatile memory based on the type passed
*PARAMETER:		u32VinPropertyType - VIN proprty Type
*				vecVINProperty - VIN Data
*RETURNVALUE: 	tDiaResult -  DIA_SUCCESS/DIA_FAILED
*
************************************************************************/
tDiaResult
dia_VINStateMachineHandler::writeProperty( const std::vector<tU8>& vecVINProperty, const tU32 u32VinPropertyType )
{
	dia_tclFnctTrace oTrace("dia_VINStateMachineHandler::writeProperty(...)");
	// initialize return value
	tDiaResult retCode = DIA_FAILED;
	// Get the property size for VIN Current
	tU16 propLength = dia_getPropertySize(u32VinPropertyType);
	// initialize the data buffer
	tU8 u8propData[propLength] = {0};
	(void) ::memset(u8propData,0,propLength);
	
	// Copy the VIN data into data buffer
	for( tU8 index = 0; index < propLength; ++index)
	{
		u8propData[index] = vecVINProperty[index];
		DIA_TR_INF("dia_VINStateMachineHandler::writeProperty : vecVINProperty[%d] = 0x%X", index, vecVINProperty[index]);
	}
	
	/*
	if(DIA_PROP_CENTER_CMC_19_VIN_CURRENT == u32VinPropertyType)
	{
		DIA_TR_INF("dia_VINStateMachineHandler::writeProperty - Writing VIN Current Now !!");
	}
	else //! (DIA_PROP_CENTER_CMC_19_VIN_ORIGINAL == u32VinPropertyType) 
	{
		DIA_TR_INF("dia_VINStateMachineHandler::writeProperty - Writing VIN Original Now !!");
	}
	*/
	
	// Try to save the VIN data into KDS
	if(DIA_SUCCESS == dia_setProperty ( u32VinPropertyType, u8propData, propLength ))
	{
		DIA_TR_INF("dia_VINStateMachineHandler::writeProperty - SUCCESS (WRITE VIN DATA TO PERSISTENT MEMORY) !!");
		retCode = DIA_SUCCESS;
	}
	else
	{
		DIA_TR_ERR("dia_VINStateMachineHandler::writeProperty - FAILED (WRITE VIN DATA TO PERSISTENT MEMORY) !!");
	}
	
	return retCode;
}
/************************************************************************
*FUNCTION: 		readProperty
*DESCRIPTION: 	This method reads VIN data based on the type passed
*PARAMETER:		u32VinPropertyType - VIN property Type
*				vecVINPropertyout - vector to fill VIN property Data
*RETURNVALUE: 	tDiaResult -  DIA_SUCCESS/DIA_FAILED
*
************************************************************************/
tDiaResult
dia_VINStateMachineHandler::readProperty( const tU32 u32VinPropertyType,  std::vector<tU8>& vecVINPropertyout )
{
	dia_tclFnctTrace oTrace("dia_VINStateMachineHandler::readProperty(...)");
	// initialize return value
	tDiaResult retCode = DIA_FAILED;
	// initialize the data vector
	std::vector<tU8> dataVecIn;
	// Get the property size for VIN Current
	tU16 propLength = dia_getPropertySize(u32VinPropertyType);
	// 
	dataVecIn.reserve(propLength);
    //dataVecIn.resize(propLength);
	
	if(DIA_SUCCESS == dia_getProperty ( u32VinPropertyType, dataVecIn ))
	{
		DIA_TR_INF("dia_VINStateMachineHandler::readProperty - SUCCESS (READ VIN DATA FROM PERSISTENT MEMORY) !!");
		retCode = DIA_SUCCESS;
	}
	else
	{
		DIA_TR_ERR("dia_VINStateMachineHandler::readProperty - FAILED (READ VIN DATA FROM PERSISTENT MEMORY) !!");
	}
	
	// Copy read VIN into the out variable
	vecVINPropertyout.clear();
	for(tU8 i=0; i<VIN_LENGTH_MAX; ++i)
	{
		vecVINPropertyout.push_back(dataVecIn[i]);
	}
	
	return retCode;
}
/************************************************************************
*FUNCTION: 		compareVINOdometer
*DESCRIPTION: 	This method compares Odometer with Odometer Limit
*PARAMETER:		None
*RETURNVALUE: 	tDiaResult -  DIA_SUCCESS/DIA_FAILED
*
************************************************************************/
tDiaResult
dia_VINStateMachineHandler::compareVINOdometer()
{
	dia_tclFnctTrace oTrace("dia_VINStateMachineHandler::compareVINOdometer(...)");
	// initialize return value
	tDiaResult retCode = DIA_FAILED;
	tU8 u8Odometer = 0;
	tU8 u8OdometerLimit = 0;
	
	u8OdometerLimit = readVINOdometer(DIA_PROP_CENTER_CMC_19_VIN_ODOMETER_COUNTER_LIMIT);
	u8Odometer = readVINOdometer(DIA_PROP_CENTER_CMC_19_VIN_ODOMETER_COUNTER);

	if(u8Odometer < u8OdometerLimit)
	{
		DIA_TR_INF("dia_VINStateMachineHandler::compareVINOdometer - VIN ODOMETER < ODOMETER LIMIT !!!");
		retCode = DIA_SUCCESS;
	}
	else
	{
		DIA_TR_INF("dia_VINStateMachineHandler::compareVINOdometer - VIN ODOMETER >= ODOMETER LIMIT !!!");
	}
	
	return retCode;
}
/************************************************************************
*FUNCTION: 		readVINOdometer
*DESCRIPTION: 	This method retrievs Odometer value based on the type passed
*PARAMETER:		u32VinPropertyType - odometer/odometer limit
*RETURNVALUE: 	tDiaResult -  DIA_SUCCESS/DIA_FAILED
*
************************************************************************/
tU8
dia_VINStateMachineHandler::readVINOdometer( const tU32  u32VinPropertyType )
{
	dia_tclFnctTrace oTrace("dia_VINStateMachineHandler::readVINOdometer(...)");
	tU8 retVal = 0;
	// initialize the data vector
	std::vector<tU8> dataVecIn;
	// Get the property size for VIN Current
	tU16 propLength = dia_getPropertySize(u32VinPropertyType);
	// 
	dataVecIn.reserve(propLength);
    //dataVecIn.resize(propLength);
	
	if(DIA_SUCCESS == dia_getProperty (u32VinPropertyType, dataVecIn ))
	{
		DIA_TR_INF("dia_VINStateMachineHandler::readVINOdometer - SUCCESS (Reading Vin Odometer from Memory) !!");
		// Get the value of Odometer
		retVal = dataVecIn[0];
	}
	else
	{
		DIA_TR_ERR("dia_VINStateMachineHandler::readVINOdometer - FAILED (Reading Vin Odometer from Memory) !!");
	}
	
	return retVal;
}
/************************************************************************
*FUNCTION: 		resetVIN
*DESCRIPTION: 	This method resets VIN patameters to their default value based on the type passed
*PARAMETER:		u32VinPropertyType - VIN Current/Original/odometer/odometer limit
*RETURNVALUE: 	tDiaResult -  DIA_SUCCESS/DIA_FAILED
*
************************************************************************/
tDiaResult
dia_VINStateMachineHandler::resetVIN(const tU32  u32VinPropertyType )
{
	dia_tclFnctTrace oTrace("dia_VINStateMachineHandler::resetVINOdometer(...)");
	// initialize return value
	tDiaResult retCode = DIA_FAILED;
	// Get the property size for VIN Odometer
	tU16 propLength = dia_getPropertySize(u32VinPropertyType);
	// initialize the data buffer
	tU8 u8propData[propLength] = {0};
	(void) ::memset(u8propData,0,propLength);
	
	switch(u32VinPropertyType)
	{
		case DIA_PROP_CENTER_CMC_19_VIN_ODOMETER_COUNTER:
		{
			// Copy the default data into data buffer
			for( tU8 index = 0; index < propLength; ++index)
			{
				u8propData[index] = (tU8)VIN_ODOMETER_DEFAULT;
			}
			DIA_TR_INF("dia_VINStateMachineHandler::resetVINOdometer - Resetting VIN Odometer Now !!");
		}
		break;
		
		case DIA_PROP_CENTER_CMC_19_VIN_ODOMETER_COUNTER_LIMIT:
		{
			// Copy the default data into data buffer
			for( tU8 index = 0; index < propLength; ++index)
			{
				u8propData[index] = (tU8)VIN_ODOMETER_LIMIT_DEFAULT;
			}
			DIA_TR_INF("dia_VINStateMachineHandler::resetVINOdometer - Resetting VIN Odometer Limit Now !!");
		}
		break;
		
		case DIA_PROP_CENTER_CMC_19_VIN_ORIGINAL:
		case DIA_PROP_CENTER_CMC_19_VIN_CURRENT:
		{
			// Copy the default data into data buffer
			for( tU8 index = 0; index < propLength; ++index)
			{
				u8propData[index] = (tU8)VIN_DEFAULT_VALUE;
			}
			
			if(DIA_PROP_CENTER_CMC_19_VIN_CURRENT == u32VinPropertyType)
			{
				DIA_TR_INF("dia_VINStateMachineHandler::resetVINOdometer - Resetting VIN Current Now !!");
			}
			else
			{
				DIA_TR_INF("dia_VINStateMachineHandler::resetVINOdometer - Resetting VIN Original Now !!");
			}
		}
		break;
		
		case DIA_PROP_CENTER_CMC_19_VIN_ODOMETER_CSM_SIGNAL_RAW:
		case DIA_PROP_CENTER_CMC_19_VIN_ODOMETER_REF_RAW:
		case DIA_PROP_CENTER_CMC_19_VIN_LOCK:
		{
			// Copy the default data into data buffer
			for( tU8 index = 0; index < propLength; ++index)
			{
				u8propData[index] = (tU8)VIN_ODOMETER_DEFAULT;
			}
		}
		break;
		
		default:
		{
			DIA_TR_ERR("dia_VINStateMachineHandler::resetVINOdometer - Unknown VIN property type !!");
		}
		break;
	}
	
	// Try to save the VIN data into KDS
	if(DIA_SUCCESS == dia_setProperty ( u32VinPropertyType, u8propData, propLength ))
	{
		DIA_TR_INF("dia_VINStateMachineHandler::resetVIN - SUCCESS !!");
		retCode = DIA_SUCCESS;
	}
	else
	{
		DIA_TR_ERR("dia_VINStateMachineHandler::resetVIN - FAILED  !!");
	}
	
	return retCode;
}
/************************************************************************
*FUNCTION: 		resetAllVINParameters
*DESCRIPTION: 	This method resets all VIN patameters to their default value
*PARAMETER:		None
*RETURNVALUE: 	tDiaResult -  DIA_SUCCESS/DIA_FAILED
*
************************************************************************/
tDiaResult
dia_VINStateMachineHandler::resetAllVINParameters()
{
	dia_tclFnctTrace oTrace("dia_VINStateMachineHandler::resetAllVINParameters(...)");
	// initialize return value
	tDiaResult retCode = DIA_SUCCESS;
	
	//! Reset VIN Original
	if(DIA_SUCCESS == resetVIN(DIA_PROP_CENTER_CMC_19_VIN_ORIGINAL))
	{
		DIA_TR_INF("dia_VINStateMachineHandler::resetAllVINParameters - RESET VIN ORIGINAL SUCCESS !!!");
	}
	else
	{
		DIA_TR_ERR("dia_VINStateMachineHandler::resetAllVINParameters - RESET VIN ORIGINAL FAILED !!!");
		return DIA_FAILED;
	}
	//! Reset VIN Current
	if(DIA_SUCCESS == resetVIN(DIA_PROP_CENTER_CMC_19_VIN_CURRENT))
	{
		DIA_TR_INF("dia_VINStateMachineHandler::resetAllVINParameters - RESET VIN CURRENT SUCCESS !!!");
	}
	else
	{
		DIA_TR_ERR("dia_VINStateMachineHandler::resetAllVINParameters - RESET VIN CURRENT FAILED !!!");
		return DIA_FAILED;
	}
	//! Reset VIN Odometer
	if(DIA_SUCCESS == resetVIN(DIA_PROP_CENTER_CMC_19_VIN_ODOMETER_COUNTER))
	{
		DIA_TR_INF("dia_VINStateMachineHandler::resetAllVINParameters - RESET VIN ODOMETER SUCCESS !!!");
	}
	else
	{
		DIA_TR_ERR("dia_VINStateMachineHandler::resetAllVINParameters - RESET VIN ODOMETER FAILED !!!");
		return DIA_FAILED;
	}
	//! Reset VIN Odometer Limit
	if(DIA_SUCCESS == resetVIN(DIA_PROP_CENTER_CMC_19_VIN_ODOMETER_COUNTER_LIMIT))
	{
		DIA_TR_INF("dia_VINStateMachineHandler::resetAllVINParameters - RESET VIN ODOMETER LIMIT SUCCESS !!!");
	}
	else
	{
		DIA_TR_ERR("dia_VINStateMachineHandler::resetAllVINParameters - RESET VIN ODOMETER LIMIT FAILED !!!");
		return DIA_FAILED;
	}
	//! Reset VIN Odometer Raw
	if(DIA_SUCCESS == resetVIN(DIA_PROP_CENTER_CMC_19_VIN_ODOMETER_CSM_SIGNAL_RAW))
	{
		DIA_TR_INF("dia_VINStateMachineHandler::resetAllVINParameters - RESET VIN ODOMETER RAW SUCCESS !!!");
	}
	else
	{
		DIA_TR_ERR("dia_VINStateMachineHandler::resetAllVINParameters - RESET VIN ODOMETER RAW FAILED !!!");
		return DIA_FAILED;
	}
	//! Reset VIN Odometer Ref
	if(DIA_SUCCESS == resetVIN(DIA_PROP_CENTER_CMC_19_VIN_ODOMETER_REF_RAW))
	{
		DIA_TR_INF("dia_VINStateMachineHandler::resetAllVINParameters - RESET VIN ODOMETER REF SUCCESS !!!");
	}
	else
	{
		DIA_TR_ERR("dia_VINStateMachineHandler::resetAllVINParameters - RESET VIN ODOMETER REF FAILED !!!");
		return DIA_FAILED;
	}
	//! Reset VIN Odometer Ref
	if(DIA_SUCCESS == resetVIN(DIA_PROP_CENTER_CMC_19_VIN_LOCK))
	{
		DIA_TR_INF("dia_VINStateMachineHandler::resetAllVINParameters - RESET VIN LOCK SUCCESS !!!");
	}
	else
	{
		DIA_TR_ERR("dia_VINStateMachineHandler::resetAllVINParameters - RESET VIN LOCK FAILED !!!");
		return DIA_FAILED;
	}
		
	return retCode;
}
/******************************************************************************
* FUNCTION:    setVINtoDefaultState
*
* DESCRIPTION: sets all the VIN parameters to the default value
*
* PARAMETER:   void
*
* RETURNVALUE: tDiaResult -  DIA_SUCCESS/DIA_FAILED
*
******************************************************************************/
tDiaResult 
dia_VINStateMachineHandler::setVINtoDefaultState(void)
{
	dia_tclFnctTrace oTrace("dia_VINStateMachineHandler::setVINtoDefaultState(...)");
	return resetAllVINParameters();
}
/************************************************************************
*FUNCTION: 		validateRcvdVIN
*
*DESCRIPTION: 	This method checks if the received VIN via CAN Msg or Diag Command is valid
*
*PARAMETER:		vecRcvdVIN - Received VIN 
*
*RETURNVALUE: 	tDiaResult -  DIA_SUCCESS/DIA_FAILED
*
************************************************************************/
tDiaResult 
dia_VINStateMachineHandler::validateRcvdVIN( const std::vector<tU8>&  vecRcvdVIN )
{
	dia_tclFnctTrace oTrace("dia_VINStateMachineHandler::validateVIN(...)");
	// initialize return value
	tDiaResult retCode = DIA_FAILED;
	
	//! Valid Hex Range - 0x30-0x39; 0x41-0x5A without 0x49, 0x4F and 0x51
	for(tU8 i=0; i<VIN_LENGTH_MAX; ++i)
	{
		//DIA_TR_INF("Magic Received VIN 1[%d] = 0x%X", i, vecRcvdVIN[i] );
		if(((tU8)VIN_VALID_RANGE_NUMBER_LOW <= vecRcvdVIN[i]) && ((tU8)VIN_VALID_RANGE_NUMBER_HIGH >= vecRcvdVIN[i]))
		{
			retCode = DIA_SUCCESS;
		} 
		else
		{
			DIA_TR_INF("dia_VINStateMachineHandler::validateRcvdVIN - Received VIN[%d] = 0x%X  not in Valid Hex Range (0x30-0x39) !!!", i, vecRcvdVIN[i] );
			if(((tU8)VIN_VALID_RANGE_CHAR_LOW <= vecRcvdVIN[i]) && ((tU8)VIN_VALID_RANGE_CHAR_HIGH) >= vecRcvdVIN[i])
			{
				//DIA_TR_INF("dia_VINStateMachineHandler::validateRcvdVIN - VIN in Valid Hex Range (0x41-0x5A) !!!");
				if(((tU8)VIN_INVALID_SPECIAL_CHAR_0x49 == vecRcvdVIN[i]) || ((tU8)VIN_INVALID_SPECIAL_CHAR_0x4F == vecRcvdVIN[i]) || ((tU8)VIN_INVALID_SPECIAL_CHAR_0x51 == vecRcvdVIN[i]))
				{
					DIA_TR_INF("dia_VINStateMachineHandler::validateRcvdVIN - Received VIN[%d] = 0x%X  is not in Valid Hex Range (0x41-0x5A without 0x49, 0x4F and 0x51)  !!!", i, vecRcvdVIN[i]);
					retCode = DIA_FAILED;
					return retCode;
				}
				else
				{
					retCode = DIA_SUCCESS;
				} /* if((VIN_INVALID_SPECIAL_CHAR_0x49 == vecRcvdVIN[i]) || (VIN_INVALID_SPECIAL_CHAR_0x4F == vecRcvdVIN[i]) || (VIN_INVALID_SPECIAL_CHAR_0x51 == vecRcvdVIN[i])) */
				
			}
			else
			{
				DIA_TR_INF("dia_VINStateMachineHandler::validateRcvdVIN - Received VIN[%d] = 0x%X  not in Valid Hex Range (0x41-0x5A) !!!" , i, vecRcvdVIN[i]);
				retCode = DIA_FAILED;
				return retCode;
			} /* if(!(VIN_VALID_RANGE_CHAR_LOW <= vecRcvdVIN[i]) && (vecRcvdVIN[i] <= VIN_VALID_RANGE_CHAR_HIGH)) */
		} /* if(!(VIN_VALID_RANGE_NUMBER_LOW <= vecRcvdVIN[i]) && (vecRcvdVIN[i] <= VIN_VALID_RANGE_NUMBER_HIGH)) */
	} /* for(tU8 i=0; i<vecRcvdVIN.size(); ++i) */
	
	return retCode;
}
/************************************************************************
*FUNCTION: 		isVINlocked
*
*DESCRIPTION: 	This method return the VIN lock status
*
*PARAMETER:		void
*
*RETURNVALUE: 	tBool
*
************************************************************************/
tBool
dia_VINStateMachineHandler::isVINlocked ( void )
{
	dia_tclFnctTrace oTrace("dia_VINStateMachineHandler::isVINlocked(...)");

	dia_Odometer* pOdometerHandler = getInstanceOfOdometerHandler();
   	if(pOdometerHandler)
   	{
   		if (ODOMETER_LOCKED == pOdometerHandler->isOdometerLocked())
   		{
   			return VIN_LOCKED;
   		}
   	}

   	return VIN_NOT_LOCKED;
}