/************************************************************************
* FILE:           dia_VIN.cpp
* PROJECT:        COACH MEDIA CENTER - DAIMLER EVOBUS
* SW-COMPONENT:   DIAGNOSIS
*----------------------------------------------------------------------
*
* DESCRIPTION:    Interface between CAN Signal handler & VIN Handling State Machine.
*
*----------------------------------------------------------------------
* HISTORY:
* Date           | Author                            | Modification
*----------------------------------------------------------------------
* 12.03.2019     | Arjun Manjunath Sanu (RBEI/ECA2)  | Initial
************************************************************************/
#ifndef __INCLUDED_DIA_VIN__
#include "project/framework/vin/dia_VIN.h"
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG_PROJECT__
#include "project/framework/config/dia_defsProjectConfig.h"
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include "common/framework/config/dia_ConfigManager.h"
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG_VIN__
#include "project/framework/vin/dia_defsVinConfig.h"
#endif

#ifndef __INCLUDED_DIA_VIN_STATE_MACHINE_HANDLER__
#include "project/framework/vin/dia_VINStateMachineHandler.h"
#endif 

using namespace std;

dia_VIN*
dia_VIN::mpInstance = 0;

/************************************************************************
*FUNCTION: 		dia_VIN
*DESCRIPTION: 	Class Constructor : Initialize thread data
*PARAMETER:		void
*RETURNVALUE: 	None
*
************************************************************************/
dia_VIN::dia_VIN()
   : dia_ActiveObject(strVINThreadName, DIA_PROP_VIN_THREAD_PRIO, DIA_PROP_VIN_THREAD_STACK_SIZE),
	  mIsSetupDone(FALSE),
     mpMsgQueue(NULL)
{
   dia_tclFnctTrace trc("dia_VIN::dia_VIN");
   dia_VIN::mpInstance = this;
}
/************************************************************************
*FUNCTION: 		getInstance
*DESCRIPTION: 	returns the class instance
*PARAMETER:		void
*RETURNVALUE: 	dia_VIN* -> mpInstance   
*
************************************************************************/
dia_VIN* 
dia_VIN::getInstance()
{ 
	dia_tclFnctTrace oTrace("dia_VIN::getInstance()");
	return mpInstance;
}
/************************************************************************
*FUNCTION: 		deleteInstance
*DESCRIPTION: 	deletes the current class instance
*PARAMETER:		None
*RETURNVALUE: 	void 
*
************************************************************************/
void
dia_VIN::deleteInstance()
 { 
	 dia_tclFnctTrace oTrace("dia_VIN::deleteInstance()");
	 if ( mpInstance )                                                                               
	 {                                                                                               
        delete mpInstance;                                                                           
        mpInstance = 0;                                                                              
     }
 }
/************************************************************************
*FUNCTION: 		~dia_VIN
*DESCRIPTION: 	Class Destructor
*PARAMETER:		None
*RETURNVALUE: 	None 
*
************************************************************************/
//lint -e1579 Warning: Pointer member might have been freed by a separate function but no '-sem(dia_VIN::tearDown,cleanup)' was seen
dia_VIN::~dia_VIN ( void )
{
   // destructors should not throw exceptions (--> lint), but functions called in the
   // could possibly throw excepections. So we catch them here and raise an assert
   _BP_TRY_BEGIN
   {
      (void) dia_VIN::tearDown();
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia_VIN::~dia_VIN !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}
//lint +e1579 Warning: Pointer member might have been freed by a separate function but no '-sem(dia_VIN::tearDown,cleanup)' was seen
/************************************************************************
*FUNCTION: 		setup
*DESCRIPTION: 	Method to create message queue & VIN thread
*PARAMETER:		None
*RETURNVALUE: 	None 
*
************************************************************************/
tDiaResult
dia_VIN::setup ( void )
{
   dia_tclFnctTrace trc("dia_VIN::setup");

   // we setup the object only once
   if ( mIsSetupDone )
   {
      DIA_TR_INF("### DIA_VIN - ENGINE WAS ALREADY SET UP. RETURNING... ###");
      return DIA_SUCCESS;
   }

   //
   // create message queue
   //
   mpMsgQueue = OSAL_NEW dia_Queue<dia_VINMessage>(strVINQueueName,VIN_QUEUE_SIZE);
   if ( !mpMsgQueue || (mpMsgQueue->open() != DIA_SUCCESS) ) return DIA_FAILED;

   DIA_TR_INF("### READING DIA_VIN ENGINE CONFIGURATION (DIA_VIN ENGINE ADDR=0x%p)... ###",this);

   if (!(DIA_SUCCESS == startThread()))
   {
      DIA_TR_INF("### FAILED TO START VIN ODOMETER HANDLER THREAD ###");
      return DIA_FAILED;
   }

   mIsSetupDone = TRUE;

   return mIsSetupDone ? DIA_SUCCESS : DIA_FAILED;
}
/************************************************************************
*FUNCTION: 		tearDown
*DESCRIPTION: 	Method to delete message queue, VIN thread & the class instance
*PARAMETER:		None
*RETURNVALUE: 	tDiaResult -  DIA_SUCCESS/DIA_FAILED
*
************************************************************************/
tDiaResult
dia_VIN::tearDown ( void )
{
   dia_tclFnctTrace oTrace("dia_VIN::tearDown()");

   if ( mpMsgQueue )
   {
      dia_VINMessage* pMsg = OSAL_NEW dia_VINMessage(DIA_EN_VIN_HANDLING_TYPE_TERMINATE); //lint !e429: custodial pointer is freed by after engine has processed the message
      if ( pMsg )
      {
         mpMsgQueue->addElement(pMsg);

         tS32 s32WaitCount = 0;
         while ( s32WaitCount < 10 )
         {
            DIA_TR_INF("Sleep for 100ms");

#ifndef VARIANT_S_FTR_ENABLE_THREAD_AS_PTHREAD_THREAD
            OSAL_s32ThreadWait(100);
#else
            usleep(100*1000); // 100ms
#endif
            ++s32WaitCount;
         }

         pMsg = 0; //lint !e423 Warning: Creation of memory leak in assignment to 'pMsg'. --> lifetime is controlled by engine
      }

      mpMsgQueue->close();
   } //lint !e438 Warning: last value assigned to variable 'pMsg' not used

   OSAL_DELETE mpMsgQueue;
   mpMsgQueue = 0;

   dia_VIN::mpInstance = 0;

   return DIA_SUCCESS;
} //lint !e1578: Pointer member dia_EngineRunIn::mCmdIter neither freed nor zeroed by cleanup function. No heap allocated for iterator
/************************************************************************
*FUNCTION: 		vThreadEntrypointObject
*DESCRIPTION: 	This method processes the events posted from same/different thread.
*				As soon as the thread is created, this is the first function to hit & 
*				will stay here until thread is terminated.
*PARAMETER:		void
*RETURNVALUE: 	void
*
************************************************************************/
void
dia_VIN::vThreadEntrypointObject ( void )
{
   dia_tclFnctTrace oTrace("dia_VIN::vThreadEntrypointObject");

   dia_VINMessage* pMessage = 0;

   bool isDone = false;

   do
   {
      pMessage = 0;
      if ( (!mpMsgQueue) || (mpMsgQueue->getElement(&pMessage) != DIA_SUCCESS) )
      {
         DIA_TR_ERR("##### INVALID QUEUE POINTER, NO MESSAGE HANDLING ERROR. ABORTING #####");
         break;
      }

      DIA_TR_INF("pMessage = %s", pMessage ? "not null" : "null");

      if ( pMessage )
      {
         DIA_TR_INF("Message Type = %d", pMessage->mType);

         switch ( pMessage->mType )
         {
         case DIA_EN_VIN_HANDLING_TYPE_TERMINATE:
            isDone = true;
            break;

         case DIA_EN_VIN_HANDLING_TYPE_EXTERNAL_REQUEST:
            {
			   dia_VINSignal* VIN = (dia_VINSignal*)pMessage->mpMsgData;
			   std::vector<tU8> vcVin;
			   vcVin.reserve(17);

			   for(tU8 i=4;i<VIN->mVINData.size();++i)
			   {
				   DIA_TR_INF("dia_VIN::vThreadEntrypointObject : VIN[%d] = 0x%X", i-3, VIN->mVINData[i]);
				   vcVin.push_back(VIN->mVINData[i]);
			   }
			   
			   dia_VINStateMachineHandler* poVINsm = dia_VINStateMachineHandler::getInstance();
			   if(poVINsm != NULL)
			   {			   
				   if(DIA_SUCCESS == poVINsm->processVINRequest(vcVin))
				   {
					   DIA_TR_INF("dia_VIN::vThreadEntrypointObject - VIN REQUEST PROCESSED SUCCESSFULLY !!!");
				   }
				   else
				   {
					   DIA_TR_ERR("dia_VIN::vThreadEntrypointObject - FAILED TO PROCESS VIN REQUEST  !!!");
				   }
			   }
			   else
			   {
				   DIA_TR_ERR("dia_VIN::vThreadEntrypointObject - FAILED TO PROCESS VIN REQUEST !!!");
			   }

               OSAL_DELETE pMessage;
               pMessage = 0;
            }
            break;

         default:
            DIA_TR_INF("##### INVALID MESSAGE TYPE -- IGNORED #####");
            break;
         }

         DIA_TR_INF("Message Processed !!");
         OSAL_DELETE pMessage;
         pMessage = 0;
      }
   }
   while ( !isDone ) ;

   DIA_TR_ERR("############################################################");
   DIA_TR_ERR("#");
   DIA_TR_ERR("#      VIN ENGINE RUN IN THREAD IS GETTING TERMINATED      #");
   DIA_TR_ERR("#");
   DIA_TR_ERR("############################################################");

#ifndef VARIANT_S_FTR_ENABLE_THREAD_AS_PTHREAD_THREAD
   // Ask OSAL what the last ERROR to occur was
   tU32 u32ErrorReason = OSAL_u32ErrorCode();
   if ( u32ErrorReason != OSAL_E_NOERROR ) DIA_TR_ERR("msg queue error: %d", u32ErrorReason);
#endif
}
/************************************************************************
*FUNCTION: 		postEvent
*DESCRIPTION: 	This method is used to post events to the thread.
*PARAMETER:		None
*RETURNVALUE: 	None
*
************************************************************************/
void
dia_VIN::postEvent (dia_enVINHandlingType type, void* msgData)
{
    dia_VINMessage* pMsg = OSAL_NEW dia_VINMessage(type, msgData);  //lint !e429: custodial pointer is freed by after engine has processed the message
	
	if ( pMsg && mpMsgQueue ) 
	{
		mpMsgQueue->addElement(pMsg);
	}
    
   delete pMsg;
	pMsg = 0; //lint !e423 Warning: Creation of memory leak in assignment to 'pMsg'. --> lifetime is controlled by engine
}

