/*!
 * \file       dia_TraceCmdClearDTCs.cpp
 *
 * \brief      project specific Trace command for ClearDTC
 *
 * \details    project specific Trace command for ClearDTC
 *
 * \component  Diagnosis
 *
 * \ingroup    diaProjectTraceCommands
 *
 * \copyright  (c) 2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef DIA_TRACECMDCLEARDTCS_H_
#include "dia_TraceCmdClearDTCs.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include <common/framework/sysadapters/dia_SystemAdapterFacade.h>
#endif

#ifndef __INCLUDED_DIA_TRACE_PLUGIN__
#include "common/framework/trace/dia_TracePluginCMD.h"
#endif

#define CMD_LENGTH					(tU8) 9
#define NEG_RESP_LENGTH			    (tU8) 3
#define POS_RESP_LENGTH			    (tU8) 1
#define SID_CLEAR_DTC				(tU8) 0x14
#define RESP_NRC_COND_NOT_CORRECT	(tU8) 0x22
#define RESP_NRC_INVALID_FORMAT		(tU8) 0x13
#define RESP_DIA_SUCCESS			(tU8) 0x00

#define RES_ACCEPTED				(tU8) 0x00
#define RES_REJECTED				(tU8) 0x01

   // p = pMessage
   //				p0 p1 p2  p3  p4  p5  p6  p7  p8  p9
   //						  |   |   |   |   |   |   |
   // (2 bytes) cmdID	    --|-+-|   |   |   |   |   |
   // (1 byte)  SID (0x14)         ---|   |   |   |   |
   // (1 byte)  GrpID			       ---|   |   |   |  
   // (3 bytes) DTCMask                    ---|-+-|-+-|


dia_TraceCmdClearDTCs::dia_TraceCmdClearDTCs( tVoid )
{
   dia_tclFnctTrace trc("dia_TraceCmdClearDTCs::dia_TraceCmdClearDTCs(void)");
}

dia_TraceCmdClearDTCs::~dia_TraceCmdClearDTCs ( tVoid )
{
    _BP_TRY_BEGIN
    {
       (tVoid) unsetSysAdapterListener<const dia_IErrorLogASFListener>(&m_Listener);
    }
    _BP_CATCH_ALL
    {
        DIA_TR_ERR("EXCEPTION CAUGHT: dia_TraceCmdClearDTCs::~dia_TraceCmdClearDTCs !!!");
        NORMAL_M_ASSERT_ALWAYS();
    }
    _BP_CATCH_END
}

void dia_TraceCmdClearDTCs::execute(const tU8* pMessage) const
{
   dia_tclFnctTrace oTrace("dia_TraceCmdClearDTCs::execute");

   tU8 retCode = RESP_DIA_SUCCESS;

   DIA_TR_INF("dia_TraceCmdClearDTCs::execute, message length = 0x%02x", pMessage[0]);

   for (tU8 i=0; i<pMessage[0]+1; i++) { // +1 because the very first byte is the size byte
	   DIA_TR_INF("dia_TraceCmdClearDTCs::execute, pMessage[%d] = 0x%02x", i, pMessage[i]);
   }

   if (pMessage[0] < CMD_LENGTH) // At least 5 bytes has to come
   {
      DIA_TR_ERR("dia_TraceCmdClearDTCs::execute Incorrect message length = 0x%02x", pMessage[0]);
	  //return;
      retCode = RESP_NRC_INVALID_FORMAT;
   }
   
   if (pMessage[5] != SID_CLEAR_DTC) // the SID for ClearDTC is 0x14
   {
      DIA_TR_ERR("dia_TraceCmdClearDTCs::execute Incorrect SID for Clear DTC 0x%02x", pMessage[5]);
	  //return;
      retCode = RESP_NRC_INVALID_FORMAT;
   }
   //extract the DTC-Grp and DTCMask from the Trace input message
   tU8  dtcGrp  = pMessage[6];
   tU32 dtcMask = ((tU32)pMessage[7]<<16)|((tU32)pMessage[8]<<8)|((tU32)pMessage[9]);

   if(RESP_DIA_SUCCESS == retCode)
   {
	   dia_IErrorLogASF* pInterface = 0;
	   if ((querySysAdapterInterface<dia_IErrorLogASF>(&pInterface) == DIA_SUCCESS) && pInterface)
	   {
		   (void) setSysAdapterListener<const dia_IErrorLogASFListener>(&m_Listener);
		   if((pInterface->sendClearDTCRequest(dtcGrp,dtcMask)) == DIA_SUCCESS)
		   {
			   DIA_TR_ERR("dia_TraceCmdClearDTCs::sendClearDTCRequest SUCCESSFUL!!!");
			   //retCode = RESP_DIA_SUCCESS;
		   }
		   else
	       {
			   DIA_TR_ERR("dia_TraceCmdClearDTCs::sendClearDTCRequest FAILED!!!");
			   (tVoid) unsetSysAdapterListener<const dia_IErrorLogASFListener>(&m_Listener);
			   retCode = RESP_NRC_COND_NOT_CORRECT;
		   }
       }  
   }

   if(retCode != RESP_DIA_SUCCESS)
   {
	   tU8 au8negRespString[NEG_RESP_LENGTH] = {0x7F, SID_CLEAR_DTC, retCode};
       dia_TracePluginCMD::vSendResponseToTrace(au8negRespString, NEG_RESP_LENGTH, 0);
   }
   
}

void dia_TraceCmdClearDTCs::Listener::vOnClearDTCResp(tU8 status)
{
   dia_tclFnctTrace oTrace("dia_TraceCmdClearDTCs::vOnClearDTCResp");

   DIA_TR_INF("dia_TraceCmdClearDTCs::vOnClearDTCResp status DTC 0x%02x", status);
   if(status == RES_ACCEPTED)
   {
	   DIA_TR_INF("dia_TraceCmdClearDTCs::vOnClearDTCResp sending positive response...");
	   tU8 au8posRespString[POS_RESP_LENGTH] = {0x54};
       dia_TracePluginCMD::vSendResponseToTrace(au8posRespString, POS_RESP_LENGTH, 0);
   }
   else
   {
       DIA_TR_ERR("dia_TraceCmdClearDTCs::vOnClearDTCResp sending negative response...");
	   tU8 au8negRespString[NEG_RESP_LENGTH] = {0x7F, SID_CLEAR_DTC, 0x22};
       dia_TracePluginCMD::vSendResponseToTrace(au8negRespString, NEG_RESP_LENGTH, 0);
   }
             
   (tVoid) unsetSysAdapterListener<const dia_IErrorLogASFListener>(this);
}
