/**
 * \file      dia_TestConfigHash_AIVI.cpp
 *
 * \brief     Test class for calculation of Configuration Hash in A-IVI project
 *
 * \details
 *           1. Read expected configuration hash of CD-DEF (DIA_PROP_AIVI_DEF_CONFIG_HASH_VALUE).
 *              If it is not available, set result to false (FAILED).
 *           2. Compare both Hash: Calculated CD and Expected CD-DEF.
 *
 *           NISSAN:
 *           3. Overwrite result.
 *              3a. If both hash values are identical, set result to false (FAILED) .
 *              3b. Otherwise, do nothing.
 *
 *           RENAULT:
 *           3. Overwrite result.
 *              3a. If both hash values are identical, set result to false (FAILED).
 *              3b. Otherwise set result to true (PASSED).
 *
 * \author    kaa1hi
 * \date      Nov 3, 2015
 *
 * \copyright Robert Bosch Car Multimedia 2015
 */


#ifndef __INCLUDED_DIA_DEFINES_UDS__
#include <common/framework/protocols/uds/dia_defsUds.h>
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG_PROJECT__
#include "project/framework/config/dia_defsProjectConfig.h"
#endif

#ifndef __INCLUDED_DIA_TEST_CONFIG_HASH_AIVI_H_
#include "dia_TestConfigHash_AIVI.h"
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include "common/framework/config/dia_ConfigManager.h"
#endif

#ifndef __INCLUDED_DIA_UTILITIES__
#include "common/framework/utils/dia_utilities.h"
#endif

#ifndef __DIA_UNIT_TESTING__
#define VD_DIAGLOG_S_IMPORT_INTERFACE_MSG
#include "vd_diaglog_if.h"
#endif

using namespace std;


#define DIA_PROP_AIVI_ECU_PART_NUMBER_SIZE 10

//-----------------------------------------------------------------------------

dia_TestConfigHash_AIVI::dia_TestConfigHash_AIVI ( tU32 propRefConfFingerprintShortFormat, tU32 propResultFingerprintLongFormat, tU16 itc )
   : dia_TestConfigHash( propRefConfFingerprintShortFormat, propResultFingerprintLongFormat, itc)
{
   dia_tclFnctTrace oTrace("dia_TestConfigHash_AIVI::dia_TestConfigHash_AIVI()");
}

//-----------------------------------------------------------------------------

dia_TestConfigHash_AIVI::~dia_TestConfigHash_AIVI ( void )
{

}

bool
dia_TestConfigHash_AIVI::checkAdditionalConditionForFPComparison ( const std::vector<tU8> calcCDHash, bool result )
{
   dia_tclFnctTrace oTrace("dia_TestConfigHash_AIVI::checkAdditionalConditionForFPComparison()");

   tU8 configurationHashCDDEF[DIA_CONFIG_HASH_BINARY_SIZE] = {0};
   tU32 sizeOfVector = calcCDHash.size();

   if (DIA_CONFIG_HASH_BINARY_SIZE!=sizeOfVector)
   {
      DIA_TR_ERR("Internal Error. Size of calculatedCDHash is %d, expected is %d", sizeOfVector, DIA_CONFIG_HASH_BINARY_SIZE);
      return false;
   }

   // STEP 1. Read expected configuration hash of CD-DEF (DIA_PROP_AIVI_DEF_CONFIG_HASH_VALUE). If it is not available, set result to false (FAILED).
   tDiaResult retCodeCDDEF = dia_getProperty(DIA_PROP_CM_CONF_INFO, configurationHashCDDEF, DIA_CONFIG_HASH_BINARY_SIZE);

   if ( DIA_SUCCESS!=retCodeCDDEF )
   {
      DIA_TR_ERR("ERROR. UNABLE TO READ ITEM (Code: 0x%08X): DIA_PROP_AIVI_DEF_CONFIG_HASH_VALUE - set result to FAILED", retCodeCDDEF);
      return false;
   }
   else
   {
      DIA_TR_INF("Read DIA_PROP_AIVI_DEF_CONFIG_HASH_VALUE was successful.");

      //trace
      tU8* p = &configurationHashCDDEF[0];
      DIA_TR_INF("AIVI_DEF_CONFIG_HASH_VALUE       Hex: %02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X", p[0], p[1], p[2], p[3], p[4], p[5], p[6], p[7], p[8], p[9], p[10], p[11], p[12], p[13], p[14], p[15], p[16], p[17], p[18], p[19] );

      //this redundant check has been added to avoid lint warning.
      if (DIA_CONFIG_HASH_BINARY_SIZE==calcCDHash.size())
      {
         DIA_TR_INF("Calculated Configuration CD Hash Hex: %02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%2X%02X%02X%02X", calcCDHash[0], calcCDHash[1], calcCDHash[2], calcCDHash[3], calcCDHash[4], calcCDHash[5], calcCDHash[6], calcCDHash[7], calcCDHash[8], calcCDHash[9], calcCDHash[10], calcCDHash[11], calcCDHash[12], calcCDHash[13], calcCDHash[14], calcCDHash[15], calcCDHash[16], calcCDHash[17], calcCDHash[18], calcCDHash[19] );
      }
   }

   // STEP 2. Compare both Hash: Calculated CD and Expected CD-DEF.
   bool retVal = (bool) std::equal( calcCDHash.begin(), calcCDHash.end(), configurationHashCDDEF);   //lint !e864:  Expression involving variable 'calculatedHash' possibly DOESN'T depend on order of evaluation

   //trace
   if (true==retVal)
   {
      DIA_TR_INF("Both hash values are identical.");
   }
   else
   {
      DIA_TR_INF("Both hash values are NOT identical.");
   }

   // STEP 3.
   switch (getItc())
   {
      case ITC_DIAG_CUSTOMER1_CONFIG_INVALID:   /* NISSAN */
      {
         DIA_TR_INF("mITC is ITC_DIAG_CUSTOMER1_CONFIG_INVALID (NISSAN)" );
         if (true==retVal)
         {
            DIA_TR_INF("NISSAN: Set result to false." );
            result = false;
         }
         else
         {
            /* Do nothing. */
         }
      }
      break;

      case ITC_DIAG_CUSTOMER2_CONFIG_INVALID:   /* RENAULT */
      {
         DIA_TR_INF("mITC is ITC_DIAG_CUSTOMER2_CONFIG_INVALID (RENAULT)" );

         if (true==retVal)
         {
            DIA_TR_INF("RENAULT: Set result to false." );
            result = false;
         }
         else
         {
            DIA_TR_INF("RENAULT: Set result to true." );
            result = true;
         }
      }
      break;

      default:
      {
         DIA_TR_ERR("ERROR. mITC is unknown. Result set to false." );
         result = false;
      }
      break;
   }

   return result;
}
