/*!
 * \file       dia_SAFeatureWifiConnman.cpp
 *
 * \brief      header file for dia_ASFComponentWifiBL
 *
 * \details    ASF component for Wifi BL ConflictManagement
 *
 * \component  Diagnosis
 *
 * \ingroup    diaASFComponents
 *
 * \copyright  (c) 2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_SAFEATURE_WIFI_CONNMAN__
#include "dia_SAFeatureWifiConnman.h"
#endif

#ifndef __INCLUDED_DIA_COMMON__
#include "common/framework/application/dia_common.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_SERVICE_PLUGIN_ASF__
#include <common/framework/platform/asf/dia_SystemAdapterServicePluginASF.h>
#endif


dia_SAFeatureWifiConnman::dia_SAFeatureWifiConnman (dia_SystemAdapterServicePluginASF<ManagerProxy>& pSrvPlugin )
	: dia_SystemAdapterFeatureASF<ManagerProxy>(pSrvPlugin)
{

}

//-----------------------------------------------------------------------------

dia_SAFeatureWifiConnman::~dia_SAFeatureWifiConnman()
{
	// TODO Auto-generated destructor stub
}

//-----------------------------------------------------------------------------


tDiaResult dia_SAFeatureWifiConnman::startMonitoring()
{
   dia_tclFnctTrace trc("dia_SAFeatureWifiConnman::startMonitoring");

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------


tDiaResult dia_SAFeatureWifiConnman::stopMonitoring()
{
   dia_tclFnctTrace trc("dia_SAFeatureWifiConnman::stopMonitoring");

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult dia_SAFeatureWifiConnman::GetWifiStations( )
{
	dia_tclFnctTrace trc("dia_SAFeatureWifiConnman::GetWifiStations");

	tDiaResult retCode = DIA_FAILED;

	if ((mpSrvPlugin) && (mpSrvPlugin->getProxy()))
	{
	   act_t token = mpSrvPlugin->getProxy()->sendGetServicesRequest(*this);
	   if (0!=token)
	   {
		   retCode = DIA_SUCCESS;
	   }
	}
	else
	{
	   DIA_TR_ERR("dia_SAFeatureWifiConnman::GetWifiStations. mpSrvPlugin or mpSrvPlugin->getProxy is NULL.");
	}

	DIA_TR_INF("dia_SAFeatureWifiConnman::GetWifiStations. returned %s", (DIA_SUCCESS==retCode? "SUCCESS": "FAILED"));

	return retCode;
}

//=============================================================================
//=============================================================================

void
dia_SAFeatureWifiConnman::onGetServicesError(const ::boost::shared_ptr< ManagerProxy >& proxy, const ::boost::shared_ptr< GetServicesError >& error)
{
	 dia_tclFnctTrace trc("dia_SAFeatureWifiConnman::onGetServicesError");

	 DIA_TR_ERR( "dia_SAFeatureWifiConnman::onGetServicesError. ErrorName     = \"%s\"", error->getName().c_str() );
	 DIA_TR_ERR( "dia_SAFeatureWifiConnman::onGetServicesError. ErrorMessage  = \"%s\"", error->getMessage().c_str() );
}

//-----------------------------------------------------------------------------

void
dia_SAFeatureWifiConnman::onGetServicesResponse(const ::boost::shared_ptr< ManagerProxy >& proxy, const ::boost::shared_ptr< GetServicesResponse >& response)
{
	dia_tclFnctTrace trc("dia_SAFeatureWifiConnman::onGetServicesResponse");

   std::vector<dia_tWifiStation> diaWifiStations;

	std::vector< GetServicesResponseServicesStruct > services = response->getServices();
	for (std::vector< GetServicesResponseServicesStruct >::const_iterator it = services.begin(); it != services.end(); it++)
	{
      std::string objPath = it->getElem1();

      DIA_TR_INF("objPath = '%s'", objPath.c_str());

      dia_tWifiStation diaWifiStation(DIA_EN_WIFI_STATE_UNKNOWN);
      std::string strType, strState;

      const std::map< std::string, asf::dbus::DBusVariant > resMap = it->getElem2();
      std::map< std::string, asf::dbus::DBusVariant >::const_iterator ele2Iter = resMap.begin();
      for ( ; ele2Iter != resMap.end(); ele2Iter++)
      {
         const char* key = ele2Iter->first.c_str();
//         DIA_TR_INF("- \"%s\"", key);

         void* vpVal = NULL;
         asf::dbus::DBusVariant valVariant = ele2Iter->second;
         DBusMessageIter* iter1 = valVariant.getReadIterator();

         if (!strcmp(key, "Name") && ((vpVal = getDbusIterValue(iter1)) != NULL))
         {
            diaWifiStation.mName = reinterpret_cast<char*>(vpVal);
            DIA_TR_INF("- \"%s\": '%s'", key, diaWifiStation.mName.c_str());
            free(vpVal); vpVal = NULL;
         }

         if (!strcmp(key, "Type") && ((vpVal = getDbusIterValue(iter1)) != NULL))
         {
            strType = reinterpret_cast<char*>(vpVal);
            DIA_TR_INF("- \"%s\": '%s'", key, strType.c_str());
            free(vpVal); vpVal = NULL;
         }

         if (!strcmp(key, "State") && ((vpVal = getDbusIterValue(iter1)) != NULL))
         {
            strState = reinterpret_cast<char*>(vpVal);
            DIA_TR_INF("- \"%s\": '%s'", key, strState.c_str());
            free(vpVal); vpVal = NULL;
         }

         if (!strcmp(key, "Strength") && ((vpVal = getDbusIterValue(iter1)) != NULL))
         {
            diaWifiStation.mStrength = (*(reinterpret_cast<int*>(vpVal))) & 0xFF;
            DIA_TR_INF("- \"%s\": %d", key, diaWifiStation.mStrength);
            free(vpVal); vpVal = NULL;
         }

         if (!strcmp(key, "IPv4"))
         {
            if (dbus_message_iter_get_arg_type(iter1) == DBUS_TYPE_ARRAY)
            {
               DBusMessageIter arrayIter;
               DBusMessageIter dictIter;
               dbus_message_iter_recurse(iter1, &arrayIter);

               while (dbus_message_iter_get_arg_type(&arrayIter) != DBUS_TYPE_INVALID)
               {
                  std::string strIPv4Key;
                  dbus_message_iter_recurse(&arrayIter, &dictIter);
                  {
                     const char* value = NULL;
                     dbus_message_iter_get_basic(&dictIter, &value);
                     strIPv4Key.assign(value);
                  }

#if 0
                  DIA_TR_INF("strIPv4Key: \"%s\"", strIPv4Key.c_str());

                  while (dbus_message_iter_next(&dictIter))
                  {
                     vpVal = getDbusIterValue(&dictIter);
                     if (!strIPv4Key.compare("Address"))
                     {
                        diaWifiStation.mAddress = reinterpret_cast<char*>(vpVal);
                        DIA_TR_INF("Address = '%s'", diaWifiStation.mAddress.c_str());
                     }
                     free(vpVal); vpVal = NULL;
                  }
#else
                  if (!strIPv4Key.compare("Address"))
                  {
                     if (dbus_message_iter_next(&dictIter) && ((vpVal = getDbusIterValue(&dictIter)) != NULL))
                     {
                        diaWifiStation.mAddress = reinterpret_cast<char*>(vpVal);
                        DIA_TR_INF("- \"IPv4.Address\": '%s'", diaWifiStation.mAddress.c_str());
                        free(vpVal); vpVal = NULL;
                     }
                  }
#endif

                  dbus_message_iter_next(&arrayIter);
               } //while (dbus_message_iter_get_arg_type(&arrayIter) != DBUS_TYPE_INVALID)
            }
            else
            {
               DIA_TR_INF("Empty IPv4 block!");
            }
         } //if (!strcmp(key, "IPv4"))
      } //for ( ; ele2Iter != resMap.end(); ele2Iter++)

      if (!strState.compare("idle"))
      {
         diaWifiStation.mState = DIA_EN_WIFI_STATE_IDLE;
      }
      else if (!strState.compare("online"))
      {
         diaWifiStation.mState = DIA_EN_WIFI_STATE_ONLINE;
      }
      else if (!strState.compare("ready"))
      {
         diaWifiStation.mState = DIA_EN_WIFI_STATE_READY;
      }

      if (!strType.compare("wifi"))
      {
         diaWifiStations.push_back(diaWifiStation);
      }
      else if (!strType.compare("ethernet"))
      {
         //Not handled (Don't add to the list of Wifi stations)
      }

	} //for (std::vector< GetServicesResponseServicesStruct >::const_iterator it = services.begin(); it != services.end(); it++)

	getInstanceOfApplication()->postMessage(
		 OSAL_NEW dia_tclDiagSession::tclEventIntMsgRxGeneric (
			   OSAL_NEW dia_FunctorOneTemplateArgNoReturnValue<dia_SAFeatureWifiConnman,dia_tWifiStation,std::vector>(this, &dia_SAFeatureWifiConnman::onWifiStationsUpdate, diaWifiStations)
		 )
	);
}

//-----------------------------------------------------------------------------

void
dia_SAFeatureWifiConnman::onWifiStationsUpdate( const std::vector<dia_tWifiStation>& stationList )
{
	dia_tclFnctTrace trc("dia_SAFeatureWifiConnman::onWifiStationsUpdate");

	dia_IWifiConnmanListener* pListener = 0;
	if ((querySysAdapterListener<dia_IWifiConnmanListener>(&pListener) == DIA_SUCCESS) && pListener)
	{
		pListener->OnWifiStations(stationList);
	}
}

//-----------------------------------------------------------------------------

void* dia_SAFeatureWifiConnman::getDbusIterValue(DBusMessageIter* iter)
{
   void* pVal = NULL;
   int type = dbus_message_iter_get_arg_type(iter);

   DIA_TR_INF("dia_SAFeatureWifiConnman::getDbusIterValue: type = %d", type);

   if (type == DBUS_TYPE_INVALID)
   {
      return NULL;
   }
   switch (type)
   {
      case DBUS_TYPE_STRING:
      case DBUS_TYPE_SIGNATURE:
      case DBUS_TYPE_OBJECT_PATH:
      {
         char* val;
         dbus_message_iter_get_basic(iter, &val);
         if (val)
         {
            char* cpVal = (char*) malloc(strlen(val) + 1);
            if (cpVal)
            {
               strcpy(cpVal, val);
               return cpVal;
            }
         }
         break;
      }
      case DBUS_TYPE_INT16:
      case DBUS_TYPE_UINT16:
      {
         dbus_uint16_t val;
         dbus_message_iter_get_basic(iter, &val);
         pVal = malloc(sizeof(val));
         if (pVal != NULL)
         {
            memcpy(pVal, &val, sizeof(val));
         }
         break;
      }
      case DBUS_TYPE_INT32:
      case DBUS_TYPE_UINT32:
      {
         dbus_uint32_t val;
         dbus_message_iter_get_basic(iter, &val);
         pVal = malloc(sizeof(val));
         if (pVal != NULL)
         {
            memcpy(pVal, &val, sizeof(val));
         }
         break;
      }
      case DBUS_TYPE_INT64:
      case DBUS_TYPE_UINT64:
      {
         dbus_uint64_t val;
         dbus_message_iter_get_basic(iter, &val);
         pVal = malloc(sizeof(val));
         if (pVal != NULL)
         {
            memcpy(pVal, &val, sizeof(val));
         }
         break;
      }
      case DBUS_TYPE_DOUBLE:
      {
         double val;
         dbus_message_iter_get_basic(iter, &val);
         pVal = malloc(sizeof(val));
         if (pVal != NULL)
         {
            memcpy(pVal, &val, sizeof(val));
         }
         break;
      }
      case DBUS_TYPE_BYTE:
      {
         unsigned char val;
         dbus_message_iter_get_basic(iter, &val);
         int iVal = static_cast< int >(val);
         pVal = malloc(sizeof(int));
         if (pVal != NULL)
         {
            memcpy(pVal, &iVal, sizeof(val));
         }
         break;
      }

      case DBUS_TYPE_BOOLEAN:
      {
         dbus_bool_t val;
         dbus_message_iter_get_basic(iter, &val);
         pVal = malloc(sizeof(bool));
         if (pVal != NULL)
         {
            memcpy(pVal, &val, sizeof(bool));
         }
         return pVal;
         break;
      }
      case DBUS_TYPE_VARIANT:
      {
         DBusMessageIter subiter;
         dbus_message_iter_recurse(iter, &subiter);
         void* vpVal = getDbusIterValue(&subiter);
         return vpVal;
      }
      case DBUS_TYPE_DICT_ENTRY:
      {
         DBusMessageIter subiter;
         dbus_message_iter_recurse(iter, &subiter);
         getDbusIterValue(&subiter);
         dbus_message_iter_next(&subiter);
         void* vpVal = getDbusIterValue(&subiter);
         return vpVal;
      }
      case DBUS_TYPE_ARRAY:
      case DBUS_TYPE_STRUCT:
      {
         int current_type;
         DBusMessageIter subiter;
         dbus_message_iter_recurse(iter, &subiter);
         void* vpVal = NULL;
         while ((current_type = dbus_message_iter_get_arg_type(&subiter))
                != DBUS_TYPE_INVALID)
         {
            vpVal = getDbusIterValue(&subiter);
            dbus_message_iter_next(&subiter);
            if (dbus_message_iter_get_arg_type(&subiter) != DBUS_TYPE_INVALID)
            {
               //TO DO:
            }
         }
         return vpVal;
         break;
      }

      default:
      {
         return pVal;
      }
   }
   return pVal;
}

//-----------------------------------------------------------------------------

