/*
 * \file        dia_SAFeatureSelftestController.cpp
 *
 * \brief       System Adapter for selftest routine control (used by any
 *              component communicated with DiagLib)
 *
 * \details
 *
 * \author      wgl2hi, kaa1hi
 *
 * \date        14.11.2014
 *
 * \copyright   Robert Bosch Car Multimedia 2014
 *
 */

#include "dia_SAFeatureSelftestController.h"

#ifndef __INCLUDED_DIA_COMMON_CORE__
#include "common/framework/application/dia_common_core.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_DIAGLIB__
#include "project/framework/sysadapters/dia_SAFeatureDiaglib.h"
#endif

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h" //lint !e451 !e537 repeatedly included header file without standard include guard

#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_TYPES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_FUNCTIONIDS
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_SERVICEINFO
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_DIAGLIBFI_TYPES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_DIAGLIBFI_FUNCTIONIDS
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#include "midw_fi_if.h" //lint !e451 !e537 repeatedly included header file without standard include guard

#include "common/depricated/SystemAdapter/dia_IdSpecEnum.h"

/// MESSAGE MAP:
// The message map maps IDs of incoming CCA messages to proper handle methods
BEGIN_MSG_MAP(dia_SAFeatureSelftestController,dia_SystemAdapterFeatureDiaglib)
   ON_MESSAGE( MIDW_DIAGLIBFI_C_U16_ROUTINECONTROL, vHandleRoutineControl )
   ON_MESSAGE( MIDW_DIAGLIBFI_C_U16_GETROUTINERESULT, vHandleRoutineControlResult )
END_MSG_MAP()

static const char* tenInternalErrorToTxt[] =
{
   "EN_ERROR_ID_NOT_SUPPORTED",                    //EN_ERROR_ID_NOT_SUPPORTED = 0,
   "EN_ERROR_INCOMPATIBLE_PARAMETER_SIGNATURE",    //EN_ERROR_INCOMPATIBLE_PARAMETER_SIGNATURE = 1,
   "EN_ERROR_ROUTINE_NOT_RUNNING",                 //EN_ERROR_ROUTINE_NOT_RUNNING = 2,
   "EN_ERROR_WRITE_NOT_SUPPORTED",                 //EN_ERROR_WRITE_NOT_SUPPORTED = 3,
   "EN_ERROR_READ_NOT_SUPPORTED",                  //EN_ERROR_READ_NOT_SUPPORTED = 4,
   "EN_ERROR_ROUTINE_NEVER_RAN",                   //EN_ERROR_ROUTINE_NEVER_RAN = 5,
   "EN_ERROR_ROUTINE_STILL_RUNNING",               //EN_ERROR_ROUTINE_STILL_RUNNING = 6,
   "EN_ERROR_DEFSET_IN_PROGRESS",                  //EN_ERROR_DEFSET_IN_PROGRESS = 7,
   "EN_ERROR_FATAL_INTERNAL",                      //EN_ERROR_FATAL_INTERNAL = 8,
   "EN_ERROR_IOCONTROL_NOT_SUPPORTED",             //EN_ERROR_IOCONTROL_NOT_SUPPORTED          = 9,
   "EN_ERROR_IOCONTROL_ACTIONID_NOT_SUPPORTED",    //EN_ERROR_IOCONTROL_ACTIONID_NOT_SUPPORTED = 10,
   "EN_ERROR_NO_DATA",                             //EN_ERROR_NO_DATA                          = 11,
   "EN_ERROR_DATA_LENGTH_INCORRECT",               //EN_ERROR_DATA_LENGTH_INCORRECT            = 12,
   "EN_ERROR_NOT_DEFINED"                          //EN_ERROR_NOT_DEFINED                      = 13
};

//------------------------------------------------------------------------------

dia_SAFeatureSelftestController::dia_SAFeatureSelftestController (
      tCString name,    // feature name
      dia_eSelftestControllerPluginType id,
      tU32 diagLibRoutineCtrlID,
      dia_SystemAdapterServicePluginDiaglib& pSrvPlugin
   )
    : dia_SystemAdapterFeatureDiaglib(name,pSrvPlugin),
      dia_SelftestControllerPlugin(name,id),
      mDiagLibRoutineCtrlID(diagLibRoutineCtrlID)
{
   dia_tclFnctTrace trc("dia_SAFeatureSelftestController::dia_SAFeatureSelftestController");
}


//-----------------------------------------------------------------------------

dia_SAFeatureSelftestController::~dia_SAFeatureSelftestController ( tVoid )
{}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureSelftestController::getDiaglibIDs ( std::list<tU32>& results ) const
{
   dia_tclFnctTrace trc("dia_SAFeatureSelftestController::getDiaglibIDs()");
   results.push_back(mDiagLibRoutineCtrlID);

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureSelftestController::startSelftest ( void )
{
   dia_tclFnctTrace trc("dia_SAFeatureSelftestController::startSelftest()");

   tDiaResult retCode = DIA_FAILED;

   /* no input parameter is necessary */
   tDiaglibParamVector vecRequest;

   if (dia_SAFeatureDiaglib::bRoutineControlStart(mDiagLibRoutineCtrlID, EN_ROUTINE_FLAG_WRITE_TO_RM, vecRequest, mpSrvPlugin /*this*/) == TRUE)
   {
      DIA_TR_INF("dia_SAFeatureSelftestController::startSelftest bRoutineControlStart SUCCESS.");
      retCode = DIA_SUCCESS;
   }
   else
   {
      DIA_TR_ERR("dia_SAFeatureSelftestController::startSelftest bRoutineControlStart FAILED.");
   }

   DIA_TR_INF("dia_SAFeatureSelftestController::startSelftest plugin name: %s", getName());

   mStartStatus = retCode;

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureSelftestController::forceSelftestResult ( void )
{
   dia_tclFnctTrace trc("dia_SAFeatureSelftestController::forceSelftestResult()");

   tDiaResult retCode = DIA_FAILED;

   if (dia_SAFeatureDiaglib::bGetRoutineResult(mDiagLibRoutineCtrlID, mpSrvPlugin /*this*/) == TRUE)
   {
      DIA_TR_INF("dia_SAFeatureSelftestController::forceSelftestResult bGetRoutineResult SUCCESS.");
      retCode = DIA_SUCCESS;
   }
   else
   {
      DIA_TR_INF("dia_SAFeatureSelftestController::forceSelftestResult bGetRoutineResult FAILED.");
   }

   return retCode;
}

//-----------------------------------------------------------------------------
// .... HANDLER ...............................................................
//-----------------------------------------------------------------------------

tVoid
dia_SAFeatureSelftestController::vHandleRoutineControl ( amt_tclServiceData* poMessage )
{
   dia_tclFnctTrace trc("dia_SAFeatureSelftestController::vHandleRoutineControl()");

   dia_SAFeatureDiaglib::u32EvalRoutineCtrlMessage( poMessage, this );
}

//-----------------------------------------------------------------------------

tVoid
dia_SAFeatureSelftestController::vHandleRoutineControlResult ( amt_tclServiceData* poMessage )
{
   dia_tclFnctTrace trc("dia_SAFeatureSelftestController::vHandleRoutineControlResult()");

   dia_SAFeatureDiaglib::u32EvalRoutineCtrlMessage( poMessage, this );
}

//-----------------------------------------------------------------------------

tVoid
dia_SAFeatureSelftestController::vOnRoutineCtrlResult ( tU32 u32RoutineId,
                                                       tenRoutineCtrlResult enRoutineStatus,
                                                       const tDiaglibParamVector& oStartResult
                                                      )
{
   dia_tclFnctTrace trc("dia_SAFeatureSelftestController::vOnRoutineCtrlResult");

   DIA_PARAMETER_INTENTIONALLY_UNUSED(u32RoutineId);

   // expected only 1 byte response for all selftest routine ctrls
   if (oStartResult.size()!=1 /*e.g. ROUTINEID_RSE_ROUTINE_CONTROL_RESULT_LEN */)
   {
      DIA_TR_ERR("dia_SAFeatureSelftestController::vOnRoutineCtrlResult - number of params=%d ERROR!!", oStartResult.size());

      tU32 i = 0;
      for (std::vector<trParameter>::const_iterator it =  oStartResult.begin(); it != oStartResult.end(); ++it)
      {
         DIA_TR_INF("vector[%d] = 0x%02X", i++, it->u8Value );
      }
   }
   else
   {
      if ( EN_ROUTINE_CONTROL_OK == enRoutineStatus )
      {
         /* Only one output parameter is expected (selftest result) */
         tenDiaglibSelfTestResult selfTestResult = static_cast<tenDiaglibSelfTestResult>(oStartResult[0].u8Value);

         DIA_TR_INF("dia_SAFeatureSelftestController::vOnRoutineCtrlResult SelftTest Result = %d", selfTestResult);

         dia_eSelftestControllerResultType eSelftestResult = DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_UNKNOWN;

         switch(selfTestResult)
         {
            case EN_SelfTestResultCOMPLETED_AND_OK:
                DIA_TR_INF("dia_SAFeatureSelftestController::vOnRoutineCtrlResult COMPLETED_OK");
                eSelftestResult = DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_COMPLETED_OK;
            break;

            case EN_SelfTestResultCOMPLETED_AND_NOK:
                DIA_TR_INF("dia_SAFeatureSelftestController::vOnRoutineCtrlResult COMPLETED_NOK");
                eSelftestResult = DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_COMPLETED_NOK;
            break;

            case EN_SelfTestResultNA:
                DIA_TR_INF("dia_SAFeatureSelftestController::vOnRoutineCtrlResult NA ==> COMPLETED_OK");
                eSelftestResult = DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_COMPLETED_OK;
            break;

            default:
                DIA_TR_INF("dia_SAFeatureSelftestController::vOnRoutineCtrlResult ABORTED");
                eSelftestResult = DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_ABORTED;
            break;
         }

         vOnSelftestControllerUpdate( eSelftestResult );
      }
      else if ( EN_ROUTINE_CONTROL_STARTED == enRoutineStatus )
      {
         DIA_TR_INF("dia_SAFeatureSelftestController::vOnRoutineCtrlResult EN_ROUTINE_CONTROL_STARTED");

         tenInternalError internalError = static_cast<tenInternalError>(oStartResult[0].u8Value);

         if (internalError < EN_ERROR_NOT_DEFINED+1)
         {
            DIA_TR_INF("dia_SAFeatureSelftestController::vOnRoutineCtrlResult internalError=%s", tenInternalErrorToTxt[internalError]);
         }
         else
         {
            DIA_TR_INF("dia_SAFeatureSelftestController::vOnRoutineCtrlResult internalError=%d", internalError);
         }

         DIA_TR_INF("dia_SAFeatureSelftestController::vOnRoutineCtrlResult IN_PROGRESS");
         vOnSelftestControllerUpdate( DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_IN_PROGRESS );
      }
      else if ( EN_ROUTINE_CONTROL_NOT_OK == enRoutineStatus )
      {
         DIA_TR_ERR("dia_SAFeatureSelftestController::vOnRoutineCtrlResult EN_ROUTINE_CONTROL_NOT_OK !!!");

         tenInternalError internalError = static_cast<tenInternalError>(oStartResult[0].u8Value);

         if (internalError < EN_ERROR_NOT_DEFINED+1)
         {
            DIA_TR_INF("dia_SAFeatureSelftestController::vOnRoutineCtrlResult internalError=%s", tenInternalErrorToTxt[internalError]);
         }
         else
         {
            DIA_TR_INF("dia_SAFeatureSelftestController::vOnRoutineCtrlResult internalError=%d", internalError);
         }

         vOnSelftestControllerUpdate( DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_ABORTED );
      }
      else
      {
         DIA_TR_INF("dia_SAFeatureSelftestController::vOnRoutineCtrlResult enRoutineStatus = %d is UNKNOWN!!!", enRoutineStatus);

         vOnSelftestControllerUpdate( DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_ABORTED );
      }
   }
}

//-----------------------------------------------------------------------------

tVoid
dia_SAFeatureSelftestController::vOnGetRoutineResult ( tU32 u32RoutineId,
                                                      tenRoutineCtrlResult enRoutineStatus,
                                                      const tDiaglibParamVector& oStartResult
                                                     )
{
   dia_tclFnctTrace trc("dia_SAFeatureSelftestController::vOnGetRoutineResult()");

   DIA_PARAMETER_INTENTIONALLY_UNUSED(u32RoutineId);

   DIA_TR_INF ("dia_SAFeatureSelftestController::vOnGetRoutineResult - Information forwarded to listener");

   // expected only 1 byte response for all selftest routine ctrls
   if (oStartResult.size()!=1)
   {
      DIA_TR_ERR("dia_SAFeatureSelftestController::vOnGetRoutineResult - number of params=%d ERROR!!", oStartResult.size());

      tU32 i = 0;
      for (std::vector<trParameter>::const_iterator it =  oStartResult.begin(); it != oStartResult.end(); ++it)
      {
         DIA_TR_INF("vector[%d] = 0x%02X", i++, it->u8Value );
      }
   }
   else
   {
      if ( EN_ROUTINE_CONTROL_OK == enRoutineStatus )
      {
         DIA_TR_INF("dia_SAFeatureSelftestController::vOnGetRoutineResult EN_ROUTINE_CONTROL_OK");

         tenDiaglibSelfTestResult selftestResult = static_cast<tenDiaglibSelfTestResult>(oStartResult[0].u8Value);
         dia_eSelftestControllerResultType eSelftestResult = DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_UNKNOWN;

         switch(selftestResult)
         {
            case EN_SelfTestResultCOMPLETED_AND_OK:
                DIA_TR_INF("dia_SAFeatureSelftestController::vOnGetRoutineResult COMPLETED_OK");
                eSelftestResult = DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_COMPLETED_OK;
            break;

            case EN_SelfTestResultCOMPLETED_AND_NOK:
                DIA_TR_INF("dia_SAFeatureSelftestController::vOnGetRoutineResult COMPLETED_NOK");
                eSelftestResult = DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_COMPLETED_NOK;
            break;

            case EN_SelfTestResultNA:
                DIA_TR_INF("dia_SAFeatureSelftestController::vOnGetRoutineResult NA ==> COMPLETED_OK");
                eSelftestResult = DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_COMPLETED_OK;
            break;

            default:
                DIA_TR_ERR("dia_SAFeatureSelftestController::vOnGetRoutineResult ABORTED");
                eSelftestResult = DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_ABORTED;
            break;
         }

         vOnSelftestControllerUpdate(eSelftestResult);
      }
      else if ( EN_ROUTINE_CONTROL_NOT_OK == enRoutineStatus )
      {
         DIA_TR_INF("dia_SAFeatureSelftestController::vOnGetRoutineResult EN_ROUTINE_CONTROL_NOT_OK");

         tenInternalError internalError = static_cast<tenInternalError>(oStartResult[0].u8Value);

         dia_eSelftestControllerResultType eSelftestResult = DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_ABORTED;

         switch(internalError)
         {
            case EN_ERROR_ROUTINE_STILL_RUNNING:
                DIA_TR_INF("dia_SAFeatureSelftestController::vOnGetRoutineResult EN_ERROR_ROUTINE_STILL_RUNNING");
                eSelftestResult = DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_IN_PROGRESS;
            break;

            default:
                DIA_TR_INF("dia_SAFeatureSelftestController::vOnGetRoutineResult ABORTED");

                if ( internalError < (EN_ERROR_NOT_DEFINED+1))
                {
                   DIA_TR_INF("vOnGetRoutineResult enRoutineStatus = %s", tenInternalErrorToTxt[internalError]);
                }
                else
                {
                   DIA_TR_INF("vOnGetRoutineResult enRoutineStatus = %d", internalError);
                }
            break;
         }

         vOnSelftestControllerUpdate( eSelftestResult );
      }
      else
      {
         DIA_TR_ERR("ERROR vOnGetRoutineResult enRoutineStatus = %d", enRoutineStatus);
      }
   }
}
