/*!
 * \file       dia_SAFeatureSelfDiagnosis.cpp
 *
 * \brief      service handler for routine control Self Diagnosis
 *
 * \details    Framework to handle sending trigger for starting Self Diagnosis of external ECUs. 
 *             Due to the fact that the number of components and their names are unknown we have this flexible framework.
 *             This class is derived from the system settings framework which already exists and is proved. 
 *             The super-class functions prepare and finalize are implemented in that way that they have no function for this usecase. 
 *             We only need the execute command where we have the chance to send the self diagnosis trigger.
 *
 * \component  Diagnosis
 *
 * \ingroup    diaSAFeatureProject
 *
 * \copyright  (c) 2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */


#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_SAFEATURE_SYSTEM_SETTINGS__
#include <common/framework/sysadapters/dia_SAFeatureSystemSettings.h>
#endif

#ifndef __INCLUDED_DIA_SAFEATURE_SELF_DIAGNOSIS__
#include "dia_SAFeatureSelfDiagnosis.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_SETTINGS_MANAGER__
#include <common/framework/sysset/dia_SystemSettingsManager.h>
#endif

static const dia_SystemSettingsModuleTypeInfo arSystemSettingsModuleTypeInfo[] = {
      { DIA_C_U32_SYSTEM_SETTING_TYPE_MASK_SELF_DIAGNOSIS,  DIA_C_STR_SYSTEM_SETTING_TYPE_SELF_DIAGNOSIS,   DIA_C_UID_SYSTEM_SETTING_TYPE_SELF_DIAGNOSIS }
};

//-----------------------------------------------------------------------------

dia_SAFeatureSelfDiagnosis::dia_SAFeatureSelfDiagnosis (
   tCString name,
   tU32 u32SystemSetId,
   tU32 u32SystemSetTypes,
   dia_SystemAdapterServicePlugin& pSrvPlugin
   )
    : dia_SystemAdapterFeature(name,pSrvPlugin),
      dia_SystemSettingsModule(name,u32SystemSetTypes),
      mSystemSettingTypeUID(0),
      mu32SystemSetId(u32SystemSetId),
      mu32SystemSetTypes(u32SystemSetTypes)
{
   this->dia_SystemSettingsModule::loadSystemSettingsModuleTypeInfo(arSystemSettingsModuleTypeInfo,(tU16) (sizeof(arSystemSettingsModuleTypeInfo)/sizeof(dia_SystemSettingsModuleTypeInfo)));
}

//-----------------------------------------------------------------------------

dia_SAFeatureSelfDiagnosis::~dia_SAFeatureSelfDiagnosis ( void )
{}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureSelfDiagnosis::prepareSystemSetting ( dia_UID typeUID, const dia_SystemSettingExtendedData& extData )
{
   dia_tclFnctTrace trc("dia_SAFeatureSelfDiagnosis::prepareSystemSetting()");

   tDiaResult retCode = DIA_FAILED;

   DIA_TR_INF("Prepare system setting (module=\"%s\", typeUID=0x%08x) -- THIS MODULE DOES NOT SUPPORT EXTENDED DATA !!",getName(),typeUID);
   mSystemSettingTypeUID = typeUID;

   if ( isSystemSettingTypeSupported(typeUID) ) {
      // if Self Diagnosis type is called do nothing in the prepare state and set state machine to execute state
      if(typeUID == DIA_C_UID_SYSTEM_SETTING_TYPE_SELF_DIAGNOSIS) {
         retCode = DIA_SUCCESS;
         getInstanceOfSystemSettingsManager()->onSystemSettingsModuleUpdate(*this,mSystemSettingTypeUID,DIA_EN_SYSTEM_SETTING_LEVEL_PREPARE,retCode);
      } else {
         retCode = this->dia_SystemSettingsModule::prepareSystemSetting(typeUID, extData);
      }
   }
   else
   {
      DIA_TR_ERR("dia_SAFeatureSelfDiagnosis::prepareSystemSetting System setting type is not supported !!!");
      getInstanceOfSystemSettingsManager()->onSystemSettingsModuleUpdate(*this,mSystemSettingTypeUID,DIA_EN_SYSTEM_SETTING_LEVEL_PREPARE,DIA_SUCCESS);
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureSelfDiagnosis::finalizeSystemSetting ( dia_UID typeUID, const dia_SystemSettingExtendedData& extData )
{
   dia_tclFnctTrace trc("dia_SAFeatureSelfDiagnosis::finalizeSystemSetting()");

   tDiaResult retCode = DIA_FAILED;

   DIA_TR_INF("Finalize system setting (module=\"%s\", typeUID=0x%08x) -- THIS MODULE DOES NOT SUPPORT EXTENDED DATA !!", getName(), typeUID);
   mSystemSettingTypeUID = typeUID;

   if (isSystemSettingTypeSupported(typeUID)) {
      // if Self Diagnosis type is called do nothing in the prepare state and set state machine to execute state
      if (typeUID == DIA_C_UID_SYSTEM_SETTING_TYPE_SELF_DIAGNOSIS) {
         retCode = DIA_SUCCESS;
         getInstanceOfSystemSettingsManager()->onSystemSettingsModuleUpdate(*this, mSystemSettingTypeUID, DIA_EN_SYSTEM_SETTING_LEVEL_FINALIZE, retCode);
      } else {
         retCode = this->dia_SystemSettingsModule::finalizeSystemSetting(typeUID, extData);
      }
   } else {
      DIA_TR_INF("dia_SAFeatureSelfDiagnosis::finalizeSystemSetting System setting type is not supported !!!");
      getInstanceOfSystemSettingsManager()->onSystemSettingsModuleUpdate(*this,mSystemSettingTypeUID,DIA_EN_SYSTEM_SETTING_LEVEL_FINALIZE,DIA_SUCCESS);
   }

   return retCode;
}

//-----------------------------------------------------------------------------

