#ifndef __INCLUDED_DIA_SIGMAVSSDRECORD__
#include "project/framework/sigma/dia_SigmaVSSDRecord.h"
#endif

#ifndef __INCLUDED_DIA_DATALOGGER__
#include <common/framework/datalogger/dia_DataLogger.h>
#endif

#ifndef __INCLUDED_DIA_SIGMA__
#include "project/framework/sigma/dia_Sigma.h"
#endif

tU32 dia_SigmaVSSDRecord::ignStatusLogHandle;
tU32 dia_SigmaVSSDRecord::mileageLogHandle;
tU32 dia_SigmaVSSDRecord::mileageUnitLogHandle;

dia_SigmaVSSDRecord::dia_SigmaVSSDRecord(tU8 sigNo, bool triggered): dia_SigmaRecord(RECORD_SIZE, 0xff) 
{
   addData(sigNo, SIGNO_OFFSET);
   addData(triggered? 0x0: 0x80, TRIGGER_BYTE_START);
}

void dia_SigmaVSSDRecord::setIgnition(dia_DataLoggerRecord const& dlr) 
{
   std::vector<tU8> ign = dlr.getItemData(ignStatusLogHandle);
   if(ign.size() == 0)
   {
      return;
   }
   tU8 src = (ign[0] & 0x01)? 0: 1;
   tU8 data = getData(IGN_BYTE_START);
   data |= static_cast<tU8>(shiftLeft(src, IGN_BIT_START));
   addData(data, IGN_BYTE_START);
}

void dia_SigmaVSSDRecord::convertCSMMileageToSigma(std::vector<tU8>& mileage, bool convertToKm) const
{
   double fMileage = static_cast<double>(mileage[3]);
   fMileage += static_cast<double>(shiftLeft(mileage[2], 8));
   fMileage += static_cast<double>(shiftLeft(mileage[1], 16));
   fMileage += static_cast<double>(shiftLeft(mileage[0], 24));
   if(convertToKm)
   { // distance is miles so we need to convert to km
      fMileage = convertMilesToKm(fMileage);
   }
   fMileage = adjustKmBasedOnCangen(fMileage);
   // abstract mileage is always in CAN Gen 6 format
   fMileage /= 10.0f;
   tU32 uMileage = static_cast<tU32>(fMileage);
   mileage.clear();
   mileage.push_back((uMileage >> 16) & 0xff);
   mileage.push_back((uMileage >> 8) & 0xff);
   mileage.push_back(uMileage & 0xff);
}

double dia_SigmaVSSDRecord::adjustKmBasedOnCangen(double distance) const
{
   eCanGen canGen = dia_Sigma::getCanGen();
   if(canGen == AIVI_CAN_GEN_3)
   {
      tU32 uDistance = static_cast<tU32>(distance /= 100.0f); 
      distance = uDistance * 100.0f; 
   }
   return distance;
}


double dia_SigmaVSSDRecord::convertMilesToKm(double mileage) const
{
   const double DIAGLOG_MILE_TO_KM_CONV_RATIO = 1.609344f;
   mileage *= DIAGLOG_MILE_TO_KM_CONV_RATIO;
   return mileage;
}

void dia_SigmaVSSDRecord::setMileage(dia_DataLoggerRecord const& dlr) 
{
   std::vector<tU8> mileage = dlr.getItemData(mileageLogHandle);
   if(mileage.size() == 0)
   {
      return;
   }
   eCanGen canGen = dia_Sigma::getCanGen();
   std::vector<tU8> mileageUnit;
   bool convertToKm = false;
   if(canGen != AIVI_CAN_GEN_6)
   {
      mileageUnit = dlr.getItemData(mileageUnitLogHandle); 
      if(mileageUnit.size() != 0)
      {
         convertToKm = mustConvertToKm(mileageUnit[0]);
      }
   }
   convertCSMMileageToSigma(mileage, convertToKm);
   for(tU32 i = 0; i < MILEAGE_BIT_SIZE/8; ++i)
   {
      addData(mileage[i], MILEAGE_BYTE_START + i);
   }
}

void dia_SigmaVSSDRecord::setInfo(std::vector<tU32> const& info) 
{
   tU8 infoByte1 = 0;
   tU8 infoByte2 = 0;
   for(tU32 i = 0; i < info.size(); ++i)
   {
      if(i == 0)
      {
         continue;
      }
      else if(i < 4)
      {
         tS32 shifter = 6 - (static_cast<tS32>(i) * 2);
         infoByte1 |= static_cast<tU8>(shiftLeft(info[i] + 1, shifter) & 0xff);
      }
      else if(i < 8)
      {
         tS32 shifter = 6 - ((static_cast<tS32>(i) - 4) * 2);
         infoByte2 |= static_cast<tU8>(shiftLeft(info[i] + 1, shifter) & 0xff);
      }
      else
      {
         break;
      }
   }
   addData(infoByte1, INFO_BYTE_START);
   addData(infoByte2, INFO_BYTE_START + 1);
} 

void dia_SigmaVSSDRecord::setFrame(std::vector<dia_SIGMATriggerFrame> const& triggerFrames) 
{
   if(triggerFrames.size() == 0)
   {
      return;
   }
   std::vector<dia_SIGMATriggerFrame>::const_iterator i = triggerFrames.begin();
   std::vector<tU8> buffer;
   tU32 byteOffset = FRAME_BYTE_START;
   for(; i != triggerFrames.end(); ++i)
   {
      i->getAsBuffer(buffer);
      std::vector<tU8>::const_iterator j = buffer.begin();
      for(; j != buffer.end(); ++j, ++byteOffset)
      {
         addData(*j, byteOffset);
      }
   }
   tU8 data = getData(TRICOUNT_BYTE_START);
   tU8 triCount = static_cast<tU8>((triggerFrames.size() - 1) & 0x07);
   data |= static_cast<tU8>(shiftLeft(triCount, TRICOUNT_BIT_START - (TRICOUNT_BIT_SIZE - 1)) & 0xff);
   addData(data, TRICOUNT_BYTE_START);
}

void dia_SigmaVSSDRecord::setData(std::vector<tU32> const& info, std::vector<dia_SIGMATriggerFrame> const& triggerFrames)
{
   dia_DataLogger* pDataLogger = dia_DataLogger::getInstance();
   if (pDataLogger) 
   {
      dia_DataLoggerRecord dlr;
      pDataLogger->getData(dlr, 0);
      setIgnition(dlr);
      setMileage(dlr);
   }
   setInfo(info);
   setFrame(triggerFrames);
}

bool dia_SigmaVSSDRecord::mustConvertToKm(tU8 mileageUnit) const
{
   bool convert = false;
   eCanGen canGen = dia_Sigma::getCanGen();
   if(
      ((canGen == AIVI_CAN_GEN_5) && (mileageUnit != 0))
      ||
      ((canGen == AIVI_CAN_GEN_3) && (mileageUnit & 0x20))
      )
   { // distance is miles so we need to convert to km
      convert = true;
   }
   return convert;
}
