/*!
 * \file       dia_CustomerDiagHandlerSpeedLimitAIVI.cpp
 *
 * \brief      Pre-service handler for all A-IVI customer services (check vehicle speed,
 *             if not zero, send NRC 0x22 for specific service handler).
 *             If speed is not available (because of inter-component communication), do not send NRC 0x22.
 *
 * \details
 *
 * \component  Diagnosis
 *
 * \ingroup    diaServicesProject
 *
 * \copyright  (c) 2017 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_FACTORY__
#include <common/framework/application/dia_Factory.h>
#endif

#ifndef __INCLUDED_DIA_MESSAGE_BUFFER_UDS__
#include <common/framework/protocols/uds/dia_MessageBufferUDS.h>
#endif


#include "dia_CustomerDiagHandlerSpeedLimitAIVI.h"

template <typename T>
class make_vector {
public:
  typedef make_vector<T> my_type;
  my_type& operator<< (const T& val) {
    data_.push_back(val);
    return *this;
  }
  operator std::vector<T>() const {
    return data_;
  }
private:
  std::vector<T> data_;
};

#define DIA_U16_SIZEOF_SUBFUNCTION_IN_ECU_RESET                ((tU16) (1))
#define DIA_U16_SIZEOF_IO_CONTROL_DATA_IDENTIFIER              ((tU16) (2))
#define DIA_U16_SIZEOF_IO_CONTROL_CONTROL_OPTION_RECORD        ((tU16) (1))
#define DIA_U16_SIZEOF_ROUTINE_ROUTINE_IDENTIFIER              ((tU16) (2))
#define DIA_U16_SIZEOF_ROUTINE_SUBFUNCTION                     ((tU16) (1))

tU8 ECUResetSubfunction[] =
{
      0x01,
      0x02
};

tU16 IOControlDataIdentifier[] =
{
      0x9001,
      0x9000,
      0x9002,
      0x9005,
      0x9006,
      0x9007,
      0x9009,
      0x900A,
      0x900B,
      0x9010
};

tU16 RoutineIdentifier[] =
{
      0x8002,
      0x8001
};

//-----------------------------------------------------------------------------

dia_CustomerDiagHandlerSpeedLimitAIVI::dia_CustomerDiagHandlerSpeedLimitAIVI ( void )
   : mIsSetupDone(false),
     mpActiveMessageBuffer(0)
{
   dia_tclFnctTrace trc("dia_MultipleReadDataByIDHandler::dia_MultipleReadDataByIDHandler");

   if ( dia_CustomerDiagHandlerSpeedLimitAIVI::setup() != DIA_SUCCESS )
   {
      DIA_TR_INF("### SETUP OF MULTIPLE DID READER FAILED ###");
   }
}

//-----------------------------------------------------------------------------

dia_CustomerDiagHandlerSpeedLimitAIVI::dia_CustomerDiagHandlerSpeedLimitAIVI ( dia_MessageHandler* nextHandler )
   : dia_MessageHandler(nextHandler),
     mIsSetupDone(false),
     mpActiveMessageBuffer(0)
{
   dia_tclFnctTrace trc("dia_MultipleReadDataByIDHandler::dia_MultipleReadDataByIDHandler(dia_MessageHandler*)");

   if ( dia_CustomerDiagHandlerSpeedLimitAIVI::setup() != DIA_SUCCESS )
   {
      DIA_TR_INF("### SETUP OF MULTIPLE DID READER FAILED ###");
   }
}

//-----------------------------------------------------------------------------

dia_CustomerDiagHandlerSpeedLimitAIVI::~dia_CustomerDiagHandlerSpeedLimitAIVI ( void )
{
   _BP_TRY_BEGIN
   {
      (void) dia_CustomerDiagHandlerSpeedLimitAIVI::tearDown();
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia_MultipleReadDataByIDHandler::~dia_MultipleReadDataByIDHandler !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-----------------------------------------------------------------------------

tDiaResult
dia_CustomerDiagHandlerSpeedLimitAIVI::setup ( void )
{
   dia_tclFnctTrace trc("dia_CustomerDiagHandlerSpeedLimitAIVI::setup");


   // we setup the engine object only once
   if ( mIsSetupDone )
   {
      DIA_TR_INF("### MULTIPLE DID READER WAS ALREADY SET UP. RETURNING... ###");
      return DIA_SUCCESS;
   }

   mIsSetupDone = true;

   // ECU Reset
   for (tU16 i=0; i<(sizeof(ECUResetSubfunction)/sizeof(tU8)); i++)
   {
      mEcuResetRepo.push_back(ECUResetSubfunction[i]);
   }

   // IO Controls
   std::vector<tU8> subFunctionIOCtrl =  make_vector<tU8>() << 0x00 << 0x03;
   for (tU16 i=0; i<(sizeof(IOControlDataIdentifier)/sizeof(tU16)); i++)
   {
      mIOControlRepo[IOControlDataIdentifier[i]] = subFunctionIOCtrl;
   }

   // Routine Controls
   std::vector<tU8> subFunctionRoutineCtrl =  make_vector<tU8>() << 0x01;
   for (tU16 i=0; i<(sizeof(RoutineIdentifier)/sizeof(tU16)); i++)
   {
      mRoutineRepo[RoutineIdentifier[i]] = subFunctionRoutineCtrl;
   }

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_CustomerDiagHandlerSpeedLimitAIVI::tearDown ( void )
{
   dia_tclFnctTrace trc("dia_MultipleReadDataByIDHandler::tearDown");

   OSAL_DELETE mpActiveMessageBuffer;
   mpActiveMessageBuffer = 0;

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_CustomerDiagHandlerSpeedLimitAIVI::handleMessage ( dia_MessageBuffer& msg )
{
   dia_tclFnctTrace trc("dia_CustomerDiagHandlerSpeedLimitAIVI::handleMessage(dia_MessageBuffer&)");

   tDiaResult retCode = DIA_FAILED;
   tU32 memoryAddress = 0;
   tU32 groupOfDtcProgramming = 0;
   bool ServiceSpeedLimit = false;

   dia_enProtocol protocol = msg.getProtocol();
   if ( DIA_EN_PROTOCOL_UDS == protocol)
   {
      const tU16 dataLength = msg.u16GetDataLength();
      tU16 offset = (msg.getFormat() == dia_MessageBuffer::format_length_and_data) ? 2 : 1; /* one length byte plus one byte SID */

      if (dataLength < offset) //Coverity Fix(CID:80635)
      {
         DIA_TR_ERR("### dataLength=%d < offset=%d. Error in the framework! ####", dataLength, offset);
         msg.vSetNegResp(getInstanceOfFactory()->makeNRC(DIA_E_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT));
         msg.vSendResponse();
         return DIA_FAILED;
      }

      const tU8 sid = msg.getDataU8(offset-1);
      if ( DIA_C_U8_UDS_SID_ECU_RESET == sid )
      {
         DIA_TR_INF("dia_CustomerDiagHandlerSpeedLimitAIVI::handleMessage DIA_C_U8_UDS_SID_ECU_RESET");
         tU16 expLen = offset + DIA_U16_SIZEOF_SUBFUNCTION_IN_ECU_RESET;
         if ( dataLength == expLen )
         {
            const tU8 subFunction = msg.getDataU8(offset);
            ServiceSpeedLimit = CheckEcuReset(subFunction);
         }
         else
         {
            DIA_TR_ERR("### dataLength %d is incorrect. expLen is %d. ####", dataLength, expLen);
         }
      }
      else if ( DIA_C_U8_UDS_SID_IOCONTROL_BY_IDENTIFIER == sid )
      {
         DIA_TR_INF("dia_CustomerDiagHandlerSpeedLimitAIVI::handleMessage DIA_C_U8_UDS_SID_IOCONTROL_BY_IDENTIFIER");
         tU16 expMinLen =  (offset + DIA_U16_SIZEOF_IO_CONTROL_DATA_IDENTIFIER + DIA_U16_SIZEOF_IO_CONTROL_CONTROL_OPTION_RECORD);
         if ( dataLength >= expMinLen )
         {
            const tU16 dataIdentifier = (msg.getDataU8(offset)<<8) | (msg.getDataU8(offset+1));
            const tU8 controlOptionRecord = msg.getDataU8(offset+2);
            ServiceSpeedLimit = CheckIOControl(dataIdentifier, controlOptionRecord);
         }
         else
         {
            DIA_TR_ERR("### dataLength %d is incorrect. expMinLen is %d. ####", dataLength, expMinLen);
         }
      }
      else if (DIA_C_U8_UDS_SID_ROUTINE_CONTROL == sid)
      {
         DIA_TR_INF("dia_CustomerDiagHandlerSpeedLimitAIVI::handleMessage DIA_C_U8_UDS_SID_ROUTINE_CONTROL");
         tU16 expMinLen =  (offset + DIA_U16_SIZEOF_ROUTINE_ROUTINE_IDENTIFIER + DIA_U16_SIZEOF_ROUTINE_SUBFUNCTION);
         if ( dataLength >= expMinLen )
         {
            const tU16 routineIdentifier = (msg.getDataU8(offset+1)<<8) | (msg.getDataU8(offset+2));
            const tU8 subfunction = msg.getDataU8(offset);
            ServiceSpeedLimit = CheckRoutineControl(routineIdentifier, subfunction);
         }
         else
         {
            DIA_TR_ERR("### dataLength %d is incorrect in DIA_C_U8_UDS_SID_IOCONTROL_BY_IDENTIFIER ####", dataLength);
         }
      }
   }
   else
   {
      DIA_TR_ERR("### current protocol (%d) is not equal to DIA_EN_PROTOCOL_UDS (%d) ####", protocol, DIA_EN_PROTOCOL_UDS);
   }

   if (true==ServiceSpeedLimit)
   {
      DIA_TR_INF("Forwarding dia_CustomerDiagHandlerSpeedLimitAIVI::getVehicleSpeed call to save thread.");

      // Wait until vehicle speed is known.
      // store the original message as it contains the callback method used to deliver the final result
      mpActiveMessageBuffer = &msg;
      retCode = DIA_SUCCESS;

      getInstanceOfApplication()->postMessage (
            OSAL_NEW dia_tclDiagSession::tclEventIntMsgRxGeneric (
                  OSAL_NEW dia_FunctorNoArgsNoReturnValue< dia_CustomerDiagHandlerSpeedLimitAIVI>(this, &dia_CustomerDiagHandlerSpeedLimitAIVI::getVehicleSpeed)
            )
      );
   }
   else
   {
      //Go on processing, do not wait for vehicle speed.
      retCode = dia_MessageHandler::handleMessage(msg);
   }

   return retCode;
}

//-----------------------------------------------------------------------------

bool
dia_CustomerDiagHandlerSpeedLimitAIVI::CheckEcuReset ( tU8 subFunction )
{
   dia_tclFnctTrace trc("dia_CustomerDiagHandlerSpeedLimitAIVI::CheckEcuReset(tU8)");

   std::vector<tU8>::iterator it = find (mEcuResetRepo.begin(), mEcuResetRepo.end(), subFunction);
   if ( it != mEcuResetRepo.end() )
   {
      DIA_TR_INF("subFunction 0x%02X found in repo. CheckEcuReset returned TRUE.", subFunction);
      return true;
   }
   else
   {
      DIA_TR_INF("subFunction 0x%02X not found in repo. CheckEcuReset returned FALSE.", subFunction);
      return false;
   }
}

//-----------------------------------------------------------------------------

bool
dia_CustomerDiagHandlerSpeedLimitAIVI::CheckIOControl ( tU16 dataIdentifier, tU8 controlOptionRecord )
{
   dia_tclFnctTrace trc("dia_CustomerDiagHandlerSpeedLimitAIVI::CheckIOControl(tU16,tU8)");

   std::map<tU16, std::vector<tU8> >::iterator it = mIOControlRepo.find(dataIdentifier);
   if ( it != mIOControlRepo.end() )
   {
      DIA_TR_INF("dataIdentifier 0x%04X found in repo.", dataIdentifier);

      std::vector<tU8>::iterator itVec = find (it->second.begin(), it->second.end(), controlOptionRecord);
      if (itVec != it->second.end())
      {
         DIA_TR_INF("controlOptionRecord 0x%02X found in repo. CheckIOControl returned TRUE.", controlOptionRecord);
         return true;
      }
      else
      {
         DIA_TR_INF("controlOptionRecord 0x%02X not found in repo. CheckIOControl returned FALSE.", dataIdentifier);
         return false;
      }
   }
   else
   {
      DIA_TR_INF("dataIdentifier 0x%04X not found in repo. CheckIOControl returned FALSE.", dataIdentifier);
      return false;
   }
}

//-----------------------------------------------------------------------------

bool
dia_CustomerDiagHandlerSpeedLimitAIVI::CheckRoutineControl ( tU16 routineIdentifier, tU8 subFunction )
{
   dia_tclFnctTrace trc("dia_CustomerDiagHandlerSpeedLimitAIVI::CheckRoutineControl(tU16,tU8)");

   std::map<tU16, std::vector<tU8> >::iterator it = mRoutineRepo.find(routineIdentifier);
   if ( it != mRoutineRepo.end() )
   {
      DIA_TR_INF("routineIdentifier 0x%04X found in repo.", routineIdentifier);

      std::vector<tU8>::iterator itVec = find (it->second.begin(), it->second.end(), subFunction);
      if (itVec != it->second.end())
      {
         DIA_TR_INF("subFunction 0x%02X found in repo. CheckRoutineControl returned TRUE.", subFunction);
         return true;
      }
      else
      {
         DIA_TR_INF("subFunction 0x%02X not found in repo. CheckRoutineControl returned FALSE.", subFunction);
         return false;
      }
   }
   else
   {
      DIA_TR_INF("routineIdentifier 0x%04X not found in repo. CheckRoutineControl returned FALSE.", routineIdentifier);
      return false;
   }
}

//-----------------------------------------------------------------------------

void
dia_CustomerDiagHandlerSpeedLimitAIVI::getVehicleSpeed()
{
   dia_tclFnctTrace trc("dia_CustomerDiagHandlerSpeedLimitAIVI::getVehicleSpeed()");

   tBool errorDetected = TRUE;

   dia_ISensorSpeed* pInterface = 0;
   if (querySysAdapterInterface<dia_ISensorSpeed>(&pInterface) == DIA_SUCCESS)
   {
      if (pInterface)
      {
         (void) setSysAdapterListener<dia_ISensorSpeedListener>(this);
         if (pInterface->getSensorSpeed() == DIA_SUCCESS)
         {
            errorDetected = FALSE;
         }
         else
         {
            DIA_TR_ERR("### dia_CustomerDiagHandlerSpeedLimitAIVI::getVehicleSpeed() failed.");
         }
      }
      else
      {
         DIA_TR_ERR("### dia_CustomerDiagHandlerSpeedLimitAIVI::getVehicleSpeed() pInterface NULL.");
      }
   }
   else
   {
      DIA_TR_ERR("### dia_CustomerDiagHandlerSpeedLimitAIVI::getVehicleSpeed() - querySysAdapterInterface NOK !");
   }

   if (errorDetected)
   {
      (void) unsetSysAdapterListener<dia_ISensorSpeedListener>(this);
      DIA_TR_ERR("### dia_CustomerDiagHandlerSpeedLimitAIVI::getVehicleSpeed() ---  SEND TO SensorPV SERVER FAILED !!!");
      NORMAL_M_ASSERT_ALWAYS();

      DIA_TR_INF("dia_CustomerDiagHandlerSpeedLimitAIVI::getVehicleSpeed forward mpActiveMessageBuffer to UDS engine.");
      getInstanceOfApplication()->postMessage(OSAL_NEW dia_tclDiagSession::tclEventReqRx(mpActiveMessageBuffer));
   }
}

//-----------------------------------------------------------------------------

void
dia_CustomerDiagHandlerSpeedLimitAIVI::vOnSensorSpeed(tU16 speed)
{
   dia_tclFnctTrace trc("dia_CustomerDiagHandlerSpeedLimitAIVI::vOnSensorSpeed");
   DIA_TR_INF("dia_CustomerDiagHandlerSpeedLimitAIVI::vOnSensorSpeed speed received: %d, 0x%04x .", speed, speed);

   bool vehicleSpeedIsZero = false;
   bool vehicleSpeedIsInvalid = false;

   (void) unsetSysAdapterListener<dia_ISensorSpeedListener>(this);

   if (SENSOR_SPEED_INVALID!=speed)
   {
      // phys = 0.01 * data
      speed = static_cast<tU16>(speed * 100);
      DIA_TR_INF("Recalculated speed acc.CDD : %d, 0x%04x .", speed, speed);

      if (0==speed)
      {
         DIA_TR_INF("dia_CustomerDiagHandlerSpeedLimitAIVI::vOnSensorSpeed vehicle speed is zero.");
         vehicleSpeedIsZero = true;
      }
   }
   else
   {
      vehicleSpeedIsInvalid = true;
      DIA_TR_ERR("### dia_CustomerDiagHandlerSpeedLimitAIVI::vOnSensorSpeed Speed Signal INVALID/UNAVAILABLE !");
   }

   if (vehicleSpeedIsZero || vehicleSpeedIsInvalid)
   {
      DIA_TR_INF("dia_CustomerDiagHandlerSpeedLimitAIVI::vOnSensorSpeed forward mpActiveMessageBuffer to UDS engine.");
      getInstanceOfApplication()->postMessage(OSAL_NEW dia_tclDiagSession::tclEventReqRx(mpActiveMessageBuffer));
   }
   else
   {
      DIA_TR_ERR("### dia_CustomerDiagHandlerSpeedLimitAIVI::vOnSensorSpeed speed not zero or invalid.");
      mpActiveMessageBuffer->vSetNegResp(getInstanceOfFactory()->makeNRC(DIA_E_CONDITIONS_NOT_CORRECT));
      mpActiveMessageBuffer->vSendResponse();
   }
}
