/*!
 * \file       dia_SubsystemDiagnosisStrategy.h
 *
 * \brief      subsystem strategy used for subsystem diagnosis
 *
 * \details    ...
 *
 * \component  Diagnostics
 *
 * \ingroup    subsystem diagnosis
 *
 * \author     Arjun Manjunath Sanu (RBEI/ECA2)
 *
 * \date       23.09.2019
 *
 * \copyright  (c) 2019 Robert Bosch Engineering & Business Solutions Ltd.
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */
 
#ifndef __INCLUDED_DIA_SUBSYSTEM_DIAGNOSIS_STRATEGY__
#define __INCLUDED_DIA_SUBSYSTEM_DIAGNOSIS_STRATEGY__

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_DEFINES_SUBSYSTEM_DIAGNOSIS__
#include <project/framework/cis/subsystem/dia_defsSubsystemDiagnosis.h>
#endif

class dia_SubsystemDiagnosisManagerPlugin;

class SubsystemDiagnosisStrategy
{
	friend class dia_SubsystemDiagnosisManager;
	
public:
	//! class constructor
	SubsystemDiagnosisStrategy ( const std::string& name, tU32 Id );
	//! class constructor
	SubsystemDiagnosisStrategy ( const char* name, tU32 Id );
	//! class destructor
	virtual ~SubsystemDiagnosisStrategy ( void );
	
	//! 2nd level constructor/destructor
	virtual tDiaResult setup ( void ) { return DIA_SUCCESS; }
	virtual tDiaResult tearDown ( void ) { return DIA_SUCCESS; }
	
	//! return strategy name
	virtual const std::string& getName ( void ) const;
	//! return the unique identifier associated with the strategy
	virtual tU32 getID ( void ) = 0;
	
	//! procaess a subsystem session activation request
	virtual tDiaResult activateSubsystemSession ( void ) = 0;
	//! procaess a subsystem session deactivation request
	virtual tDiaResult deactivateSubsystemSession ( void ) = 0;
	//! process a request to discover entities
	virtual tDiaResult identyfyAllSubsystems ( void ) = 0;
    //! process a request to discover entities request result 
	virtual tDiaResult identyfyAllSubsystemsRequestResult( void ) = 0;
    //! process a request to read subsystem node list
	virtual tDiaResult readSubsystemNodeList( void ) = 0;
	//! process a request to connect to subsystem
	virtual tDiaResult connectToSubsystem ( std::vector<tU8>& requestData ) = 0;
	//! process a route request
	virtual tDiaResult routeRequestToSubsystem ( std::vector<tU8>& requestData ) = 0;
	
	//! return the request data for the subsystem request
	virtual const std::vector<tU8>* getRequestData ( void ) const { return &mSubSysDiagReq; }
	//! return the response data for the processed request
	virtual const std::vector<tU8>* getResponseData ( void ) const { return &mSubSysDiagRes; }
	
	//! assign the plugin used for project specific NRC handling etcetera.
	void setPlugin ( const dia_SubsystemDiagnosisManagerPlugin& plugin ) { mpPlugin = &plugin; }
	
protected:
	//! depricated default constructor
	SubsystemDiagnosisStrategy ( void );
	
protected:
	//! name of the subsystem strategy
	std::string mStrategyName;
	//! id of the subsystem strategy
	tU32 mStrategyID;
	//! configuration object to customise project specific NRC handling etcetera
	const dia_SubsystemDiagnosisManagerPlugin* mpPlugin;
	//! vector to store the diagnostic request
	std::vector<tU8> mSubSysDiagReq;
	//! vector to store the diagnostic response
	std::vector<tU8> mSubSysDiagRes;
	
};

#endif /* __INCLUDED_DIA_SUBSYSTEM_DIAGNOSIS_STRATEGY__ */
