/*!
 * \file       dia_SubsystemDiagnosisManager.h
 *
 * \brief      Manager class that controls subsystem diagnosis application 
 *			   (identifying, connectiong, routing requests/responses to/from subsystems)
 *
 * \details    ...
 *
 * \component  Diagnostics
 *
 * \ingroup    subsystem diagnosis
 *
 * \author     Arjun Manjunath Sanu (RBEI/ECA2), Kirty Mayank (RBEI/ECA2)
 *
 * \date       19.09.2019
 *
 * \copyright  (c) 2019 Robert Bosch Engineering & Business Solutions Ltd.
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */
 
#ifndef __INCLUDED_DIA_SUBSYSTEM_DIAGNOSIS_MANAGER__
#include "project/framework/cis/subsystem/dia_SubsystemDiagnosisManager.h"
#endif

#ifndef __INCLUDED_DIA_SUBSYSTEM_DIAGNOSIS_STRATEGY_DEFAULT__
#include "project/framework/cis/subsystem/dia_SubsystemDiagnosisStrategyDefault.h"
#endif

#ifndef __INCLUDED_DIA_APPCONTROLLER__
#include <common/framework/application/dia_AppController.h>
#endif

#ifndef __INCLUDED_DIA_FACTORY__
#include "common/framework/application/dia_Factory.h"
#endif

DIA_IMPL_SINGLETON_WITH_SETUP_AND_TEARDOWN(dia_SubsystemDiagnosisManager)

#ifndef __DIA_UNIT_TESTING__

dia_SubsystemDiagnosisManager*
getInstanceOfSubsystemDiagnosisManager ( void )
{
   return dia_SubsystemDiagnosisManager::getInstance();
}

void
releaseInstanceOfSubsystemDiagnosisManager ( void )
{
   dia_SubsystemDiagnosisManager::deleteInstance();
}

#endif

//-----------------------------------------------------------------------------

dia_SubsystemDiagnosisManager::dia_SubsystemDiagnosisManager ( void )
   : mpPlugin(0),
     mpActiveStrategy(0)
{}

//-----------------------------------------------------------------------------

dia_SubsystemDiagnosisManager::~dia_SubsystemDiagnosisManager ( void )
{
   _BP_TRY_BEGIN
   {
      delete mpPlugin;
      mpPlugin = nullptr;

      std::map<tU32,SubsystemDiagnosisStrategy*>::iterator iter = mStrategyRep.begin();
      for ( ; iter != mStrategyRep.end(); ++iter )
      {
         if ( iter->second )
         {
            (void)(iter->second)->tearDown();
            delete iter->second;
         }
      }
      mStrategyRep.clear();

      mpActiveStrategy  = nullptr;
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia_SubsystemDiagnosisManager::~dia_SubsystemDiagnosisManager !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SubsystemDiagnosisManager::setup ( void )
{
   dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisManager::setup");

   if ( !mpPlugin )
   {
      mpPlugin = new dia_SubsystemDiagnosisManagerPlugin;
   }

   if ( !mpPlugin ) return DIA_FAILED;

   std::string strategy = "dia_SubsystemDiagnosisStrategyDefault";
   dia_SubsystemDiagnosisStrategyDefault* pStrategy = new dia_SubsystemDiagnosisStrategyDefault(strategy, (tU32)DIA_C_SUBSYSTEM_DIAGNOSIS_STRATEGY_DEFAULT);
   if (pStrategy)
   {
      if( pStrategy->setup() != DIA_SUCCESS )
      {
         DIA_TR_ERR("Failed to setup dia_SubsystemDiagnosisStrategyDefault (ADDR=0x%p)", pStrategy);
      }
   }
   tDiaResult retCode = addSubsystemDiagnosisStrategy(pStrategy);

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SubsystemDiagnosisManager::tearDown ( void ) const
{
   dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisManager::tearDown");
   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

void
dia_SubsystemDiagnosisManager::setPlugin ( const dia_SubsystemDiagnosisManagerPlugin& plugin )
{
   dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisManager::setPlugin()");

   // delete the existing plugin
   if ( mpPlugin )
   {
      DIA_TR_INF("Subsystem Plugin to be removed: %p", mpPlugin);
      delete mpPlugin;
   }

   mpPlugin = &plugin;
   DIA_TR_INF("Subsystem Plugin installed: %p", mpPlugin);

   // now inject the new plugin into every upload/download strategy
   std::map<tU32,SubsystemDiagnosisStrategy*>::iterator iter = mStrategyRep.begin();
   for ( ; iter != mStrategyRep.end(); ++iter )
   {
      if ( iter->second ) iter->second->setPlugin(*mpPlugin);
   }
}

//------------------------------------------------------------------------------

tDiaResult
dia_SubsystemDiagnosisManager::activateSubsystemSession ( void )
{
   dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisManager::activateSubsystemSession()");

   if ( !mpActiveStrategy )
   {
		// Not sure if this piece of code is required, to be checked later...
		std::map<tU32,SubsystemDiagnosisStrategy*>::iterator iter = mStrategyRep.find((tU32)DIA_C_SUBSYSTEM_DIAGNOSIS_STRATEGY_PRJ);
		if ( iter == mStrategyRep.end() )
		{
			DIA_TR_ERR("activateSubsystemSession, strategy not found" );
			return DIA_E_OUT_OF_RANGE;
		}

		mpActiveStrategy = iter->second;
   }
   /*
   if ( !mpActiveStrategy )
   {
	    DIA_TR_ERR("dia_SubsystemDiagnosisManager: FAILED (NO STRATEGY FOUND) !!");
		return DIA_E_ERROR;
   }
   */
   return mpActiveStrategy->activateSubsystemSession();
}

//------------------------------------------------------------------------------

tDiaResult
dia_SubsystemDiagnosisManager::deactivateSubsystemSession ( void )
{
   dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisManager::deactivateSubsystemSession()");

   if ( !mpActiveStrategy )
   {
		// Not sure if this piece of code is required, to be checked later...
		std::map<tU32,SubsystemDiagnosisStrategy*>::iterator iter = mStrategyRep.find((tU32)DIA_C_SUBSYSTEM_DIAGNOSIS_STRATEGY_PRJ);
		if ( iter == mStrategyRep.end() )
		{
			DIA_TR_ERR("activateSubsystemSession, strategy not found" );
			return DIA_E_OUT_OF_RANGE;
		}

		mpActiveStrategy = iter->second;
   }

   /*
   if ( !mpActiveStrategy )
   {
	    DIA_TR_ERR("dia_SubsystemDiagnosisManager: FAILED (NO STRATEGY FOUND) !!");
		return DIA_E_ERROR;
   }
   */

   return mpActiveStrategy->deactivateSubsystemSession();
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SubsystemDiagnosisManager::addSubsystemDiagnosisStrategy ( SubsystemDiagnosisStrategy* pStrategy )
{
   dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisManager::addSubsystemDiagnosisStrategy()");

   if ( !pStrategy ) return DIA_E_INVALID_POINTER;

   tDiaResult retCode = DIA_FAILED;
   DIA_TR_INF("dia_SubsystemDiagnosisManager::addSubsystemDiagnosisStrategy, pStrategy->getID() == 0x%08x ", pStrategy->getID() );

   std::map<tU32,SubsystemDiagnosisStrategy*>::iterator iter = mStrategyRep.find(pStrategy->getID());
   if ( iter == mStrategyRep.end() )
   {
      DIA_TR_INF("#######################################################");
      DIA_TR_INF("#");
      DIA_TR_INF("# ADDING SUBSYSTEM DIAGNOSIS STRATEGY \"%s\"", pStrategy->getName().c_str());
      DIA_TR_INF("#");
      DIA_TR_INF("#######################################################");
      mStrategyRep[pStrategy->getID()] = pStrategy;
      if ( mpPlugin ) pStrategy->setPlugin(*mpPlugin);
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_SubsystemDiagnosisManager::removeSubsystemDiagnosisStrategy ( tU32 id )
{
   dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisManager::removeSubsystemDiagnosisStrategy()");

   tDiaResult retCode = DIA_E_NOT_FOUND;

   std::map<tU32,SubsystemDiagnosisStrategy*>::iterator iter = mStrategyRep.find(id);
   if ( iter != mStrategyRep.end() )
   {
      (void)(iter->second)->tearDown();
      delete iter->second;
      mStrategyRep.erase(id);

      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_SubsystemDiagnosisManager::querySubsystemDiagnosisStrategy ( tU32 id, SubsystemDiagnosisStrategy** ppStrategy )
{
   dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisManager::querySubsystemDiagnosisStrategy()");

   tDiaResult retCode = DIA_E_NOT_FOUND;

   if ( !ppStrategy ) return DIA_E_INVALID_POINTER;

   (*ppStrategy) = 0;
   std::map<dia_UID,SubsystemDiagnosisStrategy*>::iterator iter = mStrategyRep.find(id);
   if ( iter != mStrategyRep.end() )
   {
      (*ppStrategy) = iter->second;
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//------------------------------------------------------------------------------

tU16
dia_SubsystemDiagnosisManager::getNumberOfSubsystemDiagnosisStrategies ( void ) const
{
   return (tU16) mStrategyRep.size();
}

//------------------------------------------------------------------------------

tDiaResult
dia_SubsystemDiagnosisManager::identyfyAllSubsystems ( void )
{
   dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisManager::identyfyAllSubsystems()");

   if ( !mpActiveStrategy )
   {
		// Not sure if this piece of code is required, to be checked later...
		std::map<tU32,SubsystemDiagnosisStrategy*>::iterator iter = mStrategyRep.find((tU32)DIA_C_SUBSYSTEM_DIAGNOSIS_STRATEGY_PRJ);
		if ( iter == mStrategyRep.end() )
		{
			DIA_TR_ERR("activateSubsystemSession, strategy not found" );
			return DIA_E_OUT_OF_RANGE;
		}

		mpActiveStrategy = iter->second;
   }
   
   /*
   if ( !mpActiveStrategy )
   {
	    DIA_TR_ERR("dia_SubsystemDiagnosisManager: FAILED (NO STRATEGY FOUND) !!");
		return DIA_E_ERROR;
   }
   */

   return mpActiveStrategy->identyfyAllSubsystems();
}

//------------------------------------------------------------------------------

tDiaResult 
dia_SubsystemDiagnosisManager::identyfyAllSubsystemsRequestResult( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisManager::identyfyAllSubsystemsRequestResult(...)");

   if ( !mpActiveStrategy )
   {
		// Not sure if this piece of code is required, to be checked later...
		std::map<tU32,SubsystemDiagnosisStrategy*>::iterator iter = mStrategyRep.find((tU32)DIA_C_SUBSYSTEM_DIAGNOSIS_STRATEGY_PRJ);
		if ( iter == mStrategyRep.end() )
		{
			DIA_TR_ERR("activateSubsystemSession, strategy not found" );
			return DIA_E_OUT_OF_RANGE;
		}

		mpActiveStrategy = iter->second;
   }
   
   /*
   if ( !mpActiveStrategy )
   {
	    DIA_TR_ERR("dia_SubsystemDiagnosisManager: FAILED (NO STRATEGY FOUND) !!");
		return DIA_E_ERROR;
   }
   */

   return mpActiveStrategy->identyfyAllSubsystemsRequestResult();
}

//------------------------------------------------------------------------------

tDiaResult
dia_SubsystemDiagnosisManager::readSubsystemNodeList( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisManager::readSubsystemNodeList(...)");

   if ( !mpActiveStrategy )
   {
		// Not sure if this piece of code is required, to be checked later...
		std::map<tU32,SubsystemDiagnosisStrategy*>::iterator iter = mStrategyRep.find((tU32)DIA_C_SUBSYSTEM_DIAGNOSIS_STRATEGY_PRJ);
		if ( iter == mStrategyRep.end() )
		{
			DIA_TR_ERR("activateSubsystemSession, strategy not found" );
			return DIA_E_OUT_OF_RANGE;
		}

		mpActiveStrategy = iter->second;
   }

   /*
   if ( !mpActiveStrategy )
   {
	    DIA_TR_ERR("dia_SubsystemDiagnosisManager: FAILED (NO STRATEGY FOUND) !!");
		return DIA_E_ERROR;
   }
   */

   return mpActiveStrategy->readSubsystemNodeList();
}

//------------------------------------------------------------------------------

tDiaResult
dia_SubsystemDiagnosisManager::connectToSubsystem ( std::vector<tU8>& IP )
{
   dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisManager::connectToSubsystem()");

   if ( !mpActiveStrategy )
   {
		// Not sure if this piece of code is required, to be checked later...
		std::map<tU32,SubsystemDiagnosisStrategy*>::iterator iter = mStrategyRep.find((tU32)DIA_C_SUBSYSTEM_DIAGNOSIS_STRATEGY_PRJ);
		if ( iter == mStrategyRep.end() )
		{
			DIA_TR_ERR("activateSubsystemSession, strategy not found" );
			return DIA_E_OUT_OF_RANGE;
		}

		mpActiveStrategy = iter->second;
   }
   
   /*
   if ( !mpActiveStrategy )
   {
	    DIA_TR_ERR("dia_SubsystemDiagnosisManager: FAILED (NO STRATEGY FOUND) !!");
		return DIA_E_ERROR;
   }
   */

   return mpActiveStrategy->connectToSubsystem(IP);
}

//------------------------------------------------------------------------------

tDiaResult
dia_SubsystemDiagnosisManager::routeRequestToSubsystem ( std::vector<tU8>& diagRequest )
{
   dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisManager::routeRequestToSubsystem()");

   if ( !mpActiveStrategy )
   {
		// Not sure if this piece of code is required, to be checked later...
		std::map<tU32,SubsystemDiagnosisStrategy*>::iterator iter = mStrategyRep.find((tU32)DIA_C_SUBSYSTEM_DIAGNOSIS_STRATEGY_PRJ);
		if ( iter == mStrategyRep.end() )
		{
			DIA_TR_ERR("activateSubsystemSession, strategy not found" );
			return DIA_E_OUT_OF_RANGE;
		}

		mpActiveStrategy = iter->second;
   }
   
   /*
   if ( !mpActiveStrategy )
   {
	    DIA_TR_ERR("dia_SubsystemDiagnosisManager: FAILED (NO STRATEGY FOUND) !!");
		return DIA_E_ERROR;
   }
   */

   return mpActiveStrategy->routeRequestToSubsystem(diagRequest);
}
