/*!
 * \file       dia_SubsystemLogoInstallationStrategyCore.cpp
 *
 * \brief      strategy used for subsystem diagnosis
 *
 * \details    ...
 *
 * \component  Diagnostics
 *
 * \ingroup    cis subsystem hmi logo installation
 *
 * \author     Arjun Manjunath Sanu (RBEI/ECA2)
 *
 * \date       17.03.2020
 *
 * \copyright  (c) 2020 Robert Bosch Engineering & Business Solutions Ltd.
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_SUBSYSTEM_LOGO_INSTALLATION_STRATEGY_CORE__
#include "project/framework/cis/logoinstallation/dia_SubsystemLogoInstallationStrategyCore.h"
#endif
 
#ifndef __INCLUDED_DIA_SUBSYSTEM_LOGO_INSTALLATION_MANAGER__
#include "project/framework/cis/logoinstallation/dia_SubsystemLogoInstallationManager.h"
#endif

#ifndef __INCLUDED_DIA_SUBSYSTEM_LOGO_INSTALLATION_MANAGER_PLUGIN__
#include "project/framework/cis/logoinstallation/dia_SubsystemLogoInstallationManagerPlugin.h"
#endif

#ifndef __INCLUDED_DIA_SUBSYSTEM_LOGO_INSTALLATION_DOIP_CLIENT__
#include "project/framework/cis/logoinstallation/dia_SubsystemLogoIstallationDoIPClient.h"
#endif

#ifndef __INCLUDED_DIA_DEFINES_SUBSYSTEM_LOGO_INSTALLATION__
#include <project/framework/cis/logoinstallation/dia_defsSubsystemLogoInstallation.h>
#endif

#ifndef __INCLUDED_DIA_DEFINES_SUBSYSTEM_DIAGNOSIS__
#include <project/framework/cis/subsystem/dia_defsSubsystemDiagnosis.h>
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#ifndef __INCLUDED_DIA_UTILITIES__
#include <common/framework/utils/dia_utilities.h>
#endif

#ifndef __INCLUDED_DIA_FILE_DIR__
#include <common/framework/application/dia_FileDir.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fstream>     
#include <stdlib.h>    
#include <sys/wait.h>

static std::string CIS_CONTAINER   = "/var/opt/bosch/persistent/avdecc/cisinstallation";

//-----------------------------------------------------------------------------

dia_SubsystemLogoInstallationStrategyCore::dia_SubsystemLogoInstallationStrategyCore ( std::string& name)
	: SubsystemLogoInstallationStrategy(name),
	  mpFSM(nullptr),
	  mpDoIPClient(nullptr),
	  mInstCondChkStat(false),
	  mDataTransferCompletionStat(false),
	  mPowerModeStat(false),
	  mRoutingActivationStat(false),
	  mDoIPClientActivStat(false),
	  mFinalResponseStat(false),
	  mTcpSocketStat(false),
	  mVehicleListEmptyStat(false),
	  mVehicleListValidity(false),
	  mVehicleValidity(false),
	  mLogoTrnsfrToAllDone(false),
	  mErrorCode(DIA_FAILED),
	  mStatusCode(DIA_E_SUBSYSTEM_LOGO_INSTALLATION_STATUS_UNKNOWN)
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::dia_SubsystemLogoInstallationStrategyCore");
	
	// create the state machine object
	if ( !(dia_CISStartupLogoInstallationFSM::Fsm::createFSM(&mpFSM,this)) )
	{
		DIA_TR_ERR( "### STATE MACHINE INITIALIZATION FAILED ###");
	}
	
	// Create Tester 
	DoIP_Cfg doipCfg;
	mpDoIPClient = new dia_SubsystemLogoInstallationDoIPClient(DIA_S_SUBSYSTEM_DOIP_CLIENT_ETH_NAME, DIA_C_SUBSYSTEM_DOIP_CLIENT_PORT_NUMBER, DIA_C_SUBSYSTEM_DOIP_CLIENT_TESTER_ADDR, doipCfg);	

}

//-----------------------------------------------------------------------------

dia_SubsystemLogoInstallationStrategyCore::dia_SubsystemLogoInstallationStrategyCore ( char* name )
	: SubsystemLogoInstallationStrategy(name),
	  mpFSM(nullptr),
	  mpDoIPClient(nullptr),
	  mInstCondChkStat(false),
	  mDataTransferCompletionStat(false),
	  mPowerModeStat(false),
	  mRoutingActivationStat(false),
	  mDoIPClientActivStat(false),
	  mFinalResponseStat(false),
	  mTcpSocketStat(false),
	  mVehicleListEmptyStat(false),
	  mVehicleListValidity(false),
	  mVehicleValidity(false),
	  mLogoTrnsfrToAllDone(false),
	  mErrorCode(DIA_FAILED),
	  mStatusCode(DIA_E_SUBSYSTEM_LOGO_INSTALLATION_STATUS_UNKNOWN)
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::dia_SubsystemLogoInstallationStrategyCore");

	// create the state machine object
	if ( !(dia_CISStartupLogoInstallationFSM::Fsm::createFSM(&mpFSM,this)) )
	{
		DIA_TR_ERR( "### STATE MACHINE INITIALIZATION FAILED ###");
	}
	
	DoIP_Cfg cfg;
	mpDoIPClient = new dia_SubsystemLogoInstallationDoIPClient(DIA_S_SUBSYSTEM_DOIP_CLIENT_ETH_NAME, DIA_C_SUBSYSTEM_DOIP_CLIENT_PORT_NUMBER, DIA_C_SUBSYSTEM_DOIP_CLIENT_TESTER_ADDR, cfg);
	
}

//-----------------------------------------------------------------------------

dia_SubsystemLogoInstallationStrategyCore::~dia_SubsystemLogoInstallationStrategyCore ( void )
{
	DIA_TR_INF("dia_SubsystemLogoInstallationStrategyCore::~dia_SubsystemLogoInstallationStrategyCore");
		
	_BP_TRY_BEGIN
	{
		delete mpFSM;
		mpFSM = 0;	
		delete mpDoIPClient;
		mpDoIPClient = NULL;

	}
	_BP_CATCH_ALL
	{
		DIA_TR_ERR("EXCEPTION CAUGHT: dia_SubsystemLogoInstallationStrategyCore::~dia_SubsystemLogoInstallationStrategyCore !!!");
		DIA_ASSERT_ALWAYS();
	}
	_BP_CATCH_END
	
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SubsystemLogoInstallationStrategyCore::start ( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::start()");
	
	mErrorCode = DIA_E_SUBSYSTEM_LOGO_INSTALLATION_RUNNING;
	
	if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evRequestInstallLogo, 0);
	
	return mErrorCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SubsystemLogoInstallationStrategyCore::requestResults ( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::requestResults()");
	
	if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evReqResultsInstallLogo, 0);
	
	return mStatusCode;
}

//-----------------------------------------------------------------------------
// FSM actions
void 
dia_SubsystemLogoInstallationStrategyCore::assembleData ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::assembleData()");
	
	if( !mpDataDownloader ) return;
	
	// assemble the next data block to transfer
	if( DIA_FAILED == mpDataDownloader->assembleData())
	{
		// Trigger next state machine event
		if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evDataError, 0);
	}
	else
	{
		// Trigger next state machine event
		if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evDataTransfer, 0);
	}
	
	
	
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::calcKey ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::calcKey()");
	
	std::vector<tU8> seed;
	tU8 EID[6], GID[6];
	
	memset(EID, 0, sizeof(EID));
	memset(GID, 0, sizeof(GID));
	
	memcpy(EID, mpDoIPClient->getCurrentNode()->getEID(), 6);
	//DIA_TR_INF("calcKey :: EID Data: %s", dia::utils::bin2str(EID,' ').c_str());
	memcpy(GID, mpDoIPClient->getCurrentNode()->getGID(), 6);
	//DIA_TR_INF("calcKey :: GID Data: %s", dia::utils::bin2str(GID,' ').c_str());
	
	
	for(int i = 0; i < (mSubSysDiagRes.size() - 2); ++i)
	{
		seed.push_back(mSubSysDiagRes[i+2]);
	}
	
	mpSecurityAlgorithm->calculateKey(seed, EID, GID);

}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::chkIfLogoTrnsfrToAllDone ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::chkIfLogoTrnsfrToAllDone()");
	
	//DIA_TR_INF("dia_SubsystemLogoInstallationStrategyCore::isVehicleListValid() : %s ", (mLogoTrnsfrToAllDone=true) ? "YES" : "NO");
	
	mLogoTrnsfrToAllDone = true;
	
	std::list<SubsystemLogoInstCfgItem>::iterator it;
	
	for(it = mGlassLogoInstCfgRepo.begin(); it != mGlassLogoInstCfgRepo.end(); it++)
	{
		DIA_TR_INF("::chkIfLogoTrnsfrToAllDone -> IP (0x%08x), installation status(%s)", it->Node.subsystemIPAddress, ((it->logoInstallationDone) ? "YES" : "NO"));
		if(false == it->logoInstallationDone)
		{
			mLogoTrnsfrToAllDone = false;
			break;
		}
	}
	
	if(mLogoTrnsfrToAllDone)
	{
		DIA_TR_INF("::chkIfLogoTrnsfrToAllDone() : DONE");
		
		if (mpDoIPClient)
		{
			// creating udp sockets in this context as this is required for PMR
			mpDoIPClient->Stop();
			// Set the callbacks from the client
			mpDoIPClient->setCallback(NULL);
		}
		
		if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evFinalise, 0);
	}
	else
	{
		DIA_TR_INF("::chkIfLogoTrnsfrToAllDone() : NOT YET DONE");
		// implementing a wait before we switch to next logo transfer
		if ( mpDoIPClient ) mpDoIPClient->startLogoTransferDelayTimer();
		
	}
	
}

void 
dia_SubsystemLogoInstallationStrategyCore::vOnLogoTransferWaitOver( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::vOnLogoTransferWaitOver()");
	
	DIA_TR_INF("::vOnLogoTransferWaitOver() : STARTING NEXT LOGO TRANSFER AFTER WAIT");
	if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evNxtLogoTrnsfr, 0);
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::chkVehicle ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::chkVehicle()");
	// Initialise the vahicle as valid
	mVehicleValidity = true;
	
	// double confirm if the selected vehicle is Glasses only
	DoIP_Node* node = mpDoIPClient->getCurrentNode();
	const tU8* cGID = node->getGID();
	tU32 IP = node->getIPAddr();
	tU16 glassCode = ((cGID[0]) << 8)|(cGID[1]);
	
	if(DIA_C_SUBSYSTEM_LOGO_INSTALLATION_GID_GLASS_CODE != glassCode)
	{
		// set vehicle as a invalid
		mVehicleValidity = false;
		
		// update the logo installation status and completion status in vehicle repo
		// installation status ->  done
		vUpdateSubsystemLogoInstallationStatus(IP, true);
		// completion status -> failed
		vUpdateSubsystemLogoInstallationCompletionStatus(IP, false);	
	}
	
	// Trigger next state machine event
	if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evVehicleChecked, 0);
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::closeCurrentConnection ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::closeCurrentConnection()");
	
	if(mpDoIPClient) 
	{
		// deactivate tester present
		mpDoIPClient->disableTesterPresent();
		
		mpDoIPClient->getCurrentNode()->setDoIPCallback(nullptr);
	
		// Close the current node connection
		(void)mpDoIPClient->closeNodeTCPSocket();
	}
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::connectToVehicle ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::connectToVehicle()");
	
	// initialise the tcp socket status to fail/false
	mTcpSocketStat = false;
	
	// Create a TCP Socket
	if(false == mpDoIPClient->createNodeTCPSocket())
	{
		// Failed to create TCP Socket, flag already set... do nothing!
		DIA_TR_INF("::connectToVehicle --> TCP socket creation failed! ");
		return;
	}
	
	// TCP Socket created, Now try to connect ...
	if(true == mpDoIPClient->connectNodeTCP())
	{
		// succeeded to Connect. Set the flag!
		DIA_TR_INF("::connectToVehicle --> Connect Done ! ");
		mTcpSocketStat = true;
	}
	
	// Trigger next state machine event
	if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evChkTCPSocket, 0);
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::evaluateRequestDownloadResponse ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::evaluateRequestDownloadResponse()");
	
	// validate the session change response
	if((mSubSysDiagRes[DIA_C_SUBSYSTEM_LOGO_INSTALLATION_POS_RESP_SRV_ID_IDX] == DIA_C_SUBSYSTEM_REQUEST_DOWNLOAD_REQUEST_POS_RESP_ID))
	{
		//This is a positive response
		DIA_TR_INF("Request Download Response : Length: %d Data: %s", mSubSysDiagRes.size(), dia::utils::bin2str(mSubSysDiagRes,' ').c_str());
		// Trigger nxt processing event
		if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evPositiveResponse, 0);
	}
	else
	{
		if(mSubSysDiagRes[DIA_C_SUBSYSTEM_LOGO_INSTALLATION_NEG_RESP_SRV_ID_IDX] == DIA_C_SUBSYSTEM_LOGO_INSTALLATION_NEG_RESP_SRV_ID)
		{
			if(mSubSysDiagRes[DIA_C_SUBSYSTEM_RESPONSE_TYPE_WAIT_NRC_IDX] == DIA_C_SUBSYSTEM_RESPONSE_TYPE_WAIT_NRC)
			{
				// This is a wait response
				DIA_TR_INF("Request Download Response : 0x%02x 0x%02x 0x%02x", mSubSysDiagRes[0], mSubSysDiagRes[1], mSubSysDiagRes[2] );
				// Trigger nxt processing event
				if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evWaitResponse, 0);
			}
			else
			{
				// This is a negative response
				DIA_TR_INF("Request Download Response : 0x%02x 0x%02x 0x%02x", mSubSysDiagRes[0], mSubSysDiagRes[1], mSubSysDiagRes[2] );
				// Trigger nxt processing event
				if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evNegativeResponse, 0);
			}
		}
		else
		{
			// unknown response ??? print first 2 bytes only
			//DIA_TR_INF(" (Unknown) Request Download Response : [size: %d] 0x%02x 0x%02x ...", mSubSysDiagRes.size(), mSubSysDiagRes[0], mSubSysDiagRes[1] );
			DIA_TR_INF(" (Unknown) Request Download Response : Length: %d Data: %s", mSubSysDiagRes.size(), dia::utils::bin2str(mSubSysDiagRes,' ').c_str());
		}
	}
	
	
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::evaluateResponseTransferData ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::evaluateResponseTransferData()");
	
	// validate the TRANSFER DATA response
	if((mSubSysDiagRes[DIA_C_SUBSYSTEM_LOGO_INSTALLATION_POS_RESP_SRV_ID_IDX] == DIA_C_SUBSYSTEM_TRANSFER_DATA_REQUEST_POS_RESP_ID))
	{
		//This is a positive response
		DIA_TR_INF("Transfer Data Response : Length: %d Data: %s", mSubSysDiagRes.size(), dia::utils::bin2str(mSubSysDiagRes,' ').c_str());
		// Trigger nxt processing event
		if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evPositiveResponse, 0);
	}
	else
	{
		if(mSubSysDiagRes[DIA_C_SUBSYSTEM_LOGO_INSTALLATION_NEG_RESP_SRV_ID_IDX] == DIA_C_SUBSYSTEM_LOGO_INSTALLATION_NEG_RESP_SRV_ID)
		{
			if(mSubSysDiagRes[DIA_C_SUBSYSTEM_RESPONSE_TYPE_WAIT_NRC_IDX] == DIA_C_SUBSYSTEM_RESPONSE_TYPE_WAIT_NRC)
			{
				// This is a wait response
				DIA_TR_INF("Transfer Data Response : 0x%02x 0x%02x 0x%02x", mSubSysDiagRes[0], mSubSysDiagRes[1], mSubSysDiagRes[2] );
				// Trigger nxt processing event
				if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evWaitResponse, 0);
			}
			else
			{
				// This is a negative response
				DIA_TR_INF("Transfer Data Response : 0x%02x 0x%02x 0x%02x", mSubSysDiagRes[0], mSubSysDiagRes[1], mSubSysDiagRes[2] );
				// Trigger nxt processing event
				if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evNegativeResponse, 0);
			}
		}
		else
		{
			// unknown response ??? print first 2 bytes only
			//DIA_TR_INF(" (Unknown) Transfer Data Response : [size: %d] 0x%02x 0x%02x ...", mSubSysDiagRes.size(), mSubSysDiagRes[0], mSubSysDiagRes[1]);
			DIA_TR_INF(" (Unknown) Transfer Data Response : Length: %d Data: %s", mSubSysDiagRes.size(), dia::utils::bin2str(mSubSysDiagRes,' ').c_str());
		}
	}
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::evaluateResponseTransferExit ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::evaluateResponseTransferExit()");
	
	// validate the Transfer Exit response
	if((mSubSysDiagRes[DIA_C_SUBSYSTEM_LOGO_INSTALLATION_POS_RESP_SRV_ID_IDX] == DIA_C_SUBSYSTEM_TRANSFER_EXIT_REQUEST_POS_RESP_ID))
	{
		//This is a positive response
		DIA_TR_INF("Transfer Exit Response : Length: %d Data: %s", mSubSysDiagRes.size(), dia::utils::bin2str(mSubSysDiagRes,' ').c_str());
		// Trigger nxt processing event
		if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evPositiveResponse, 0);
	}
	else
	{
		if(mSubSysDiagRes[DIA_C_SUBSYSTEM_LOGO_INSTALLATION_NEG_RESP_SRV_ID_IDX] == DIA_C_SUBSYSTEM_LOGO_INSTALLATION_NEG_RESP_SRV_ID)
		{
			if(mSubSysDiagRes[DIA_C_SUBSYSTEM_RESPONSE_TYPE_WAIT_NRC_IDX] == DIA_C_SUBSYSTEM_RESPONSE_TYPE_WAIT_NRC)
			{
				// This is a wait response
				DIA_TR_INF("Transfer Exit Response : 0x%02x 0x%02x 0x%02x", mSubSysDiagRes[0], mSubSysDiagRes[1], mSubSysDiagRes[2] );
				// Trigger nxt processing event
				if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evWaitResponse, 0);
			}
			else
			{
				// This is a negative response
				DIA_TR_INF("Transfer Exit Response : 0x%02x 0x%02x 0x%02x", mSubSysDiagRes[0], mSubSysDiagRes[1], mSubSysDiagRes[2] );
				// Trigger nxt processing event
				if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evNegativeResponse, 0);
			}
		}
		else
		{
			// unknown response ??? print first 2 bytes only
			//DIA_TR_INF(" (Unknown) Transfer Exit Response : [size: %d] 0x%02x 0x%02x ...", mSubSysDiagRes.size(), mSubSysDiagRes[0], mSubSysDiagRes[1]);
		    DIA_TR_INF(" (Unknown) Transfer Exit Response : Length: %d Data: %s", mSubSysDiagRes.size(), dia::utils::bin2str(mSubSysDiagRes,' ').c_str());
		}
	}
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::getNextBlock ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::getNextBlock()");
	
	if( !mpDataDownloader ) return;
	
	// set the next data block to transfer
	if( DIA_FAILED == mpDataDownloader->setNextBlock()) return;

}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::onFailedPowerMode ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::onFailedPowerMode()");
	DIA_TR_ERR("ERROR --> POWER MODE REQUEST HAS FAILED! ");
	
	vUpdateSubsystemStatusFailed();
	
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::onFailedRoutingActivation ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::onFailedRoutingActivation()");
	DIA_TR_ERR("ERROR --> ROUTING ACTIVATION HAS FAILED! ");
	
	vUpdateSubsystemStatusFailed();
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::onRoutingActivationSuccess ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::onRoutingActivationSuccess()");
	DIA_TR_ERR("INFO --> ROUTING ACTIVATION SUCCEEDED! ");
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::postprocessLogoInstallation ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::postprocessLogoInstallation()");
	
	DIA_TR_INF("");
	DIA_TR_INF("");
	DIA_TR_INF("#####################################################");
	DIA_TR_INF("#####################################################");
	DIA_TR_INF("##|-----------------------------------------------|##");
	DIA_TR_INF("##|      CIS HMI LOGO INSTALLATION COMPLETED      |##");
	DIA_TR_INF("##|-----------------------------------------------|##");
	DIA_TR_INF("#####################################################");
	DIA_TR_INF("#####################################################");
	DIA_TR_INF("");
	DIA_TR_INF("");
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::postprocessRequestDownloadResp ( void* /* pArg */ )
{
	// dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::postprocessRequestDownloadResp()");
	
	DIA_TR_INF("INFO --> POSITIVE RESPONSE RECEIVED FOR REQUEST DOWNLOAD REQUEST! ");
	
	// retrive the block size from the request download response
	// and set the value of block size in the data downloader.
	if( !mSubSysDiagRes.empty() )
	{
		tU16 bs = ((mSubSysDiagRes[2] << 8) | mSubSysDiagRes[3]);
		if( mpDataDownloader )
		{
			(void)mpDataDownloader->setBlockSize(bs);
		}
	}
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::postprocessSecurityUnlock ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::postprocessSecurityUnlock()");
	
	DIA_TR_INF("INFO --> POSITIVE RESPONSE RECEIVED FOR SECURITY KEY REQUEST! ");
	
	if ( mpSecurityAlgorithm ) mpSecurityAlgorithm->finalize();
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::postprocessSeedResponse ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::postprocessSeedResponse()");
	
	DIA_TR_INF("INFO --> POSITIVE RESPONSE RECEIVED FOR SECURITY SEED REQUEST! ");
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::postprocessSessionSwitch ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::postprocessSessionSwitch()");
	
	// Nothing to be done at the moment
	DIA_TR_INF("INFO --> POSITIVE RESPONSE RECEIVED FOR SESSION SWITCH REQUEST! ");
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::postprocessTransferDataResp ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::postprocessTransferDataResp()");
	
	DIA_TR_INF("INFO --> POSITIVE RESPONSE RECEIVED FOR TRANSFER DATA REQUEST! ");
	
	if( !mpDataDownloader ) return;
	
	mpDataDownloader->postprocessTransferData();
	
	//Transfer the next block bro ;P
	DIA_TR_INF("INFO --> PREPARING NEXT DATA BLOCK TRANSFER! ");
	if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evDataTransferred, 0);
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::postprocessTransferExitResp ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::postprocessTransferExitResp()");
	
	if( !mpDataDownloader ) return;
	
	mpDataDownloader->finaliseDataDownload();
	
	// Update the logo download status
	vUpdateSubsystemStatusSucceeded();
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::postprocessVehicleConnected ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::postprocessVehicleConnected()");
	DIA_TR_INF("INFORMATION --> CONNECTION SUCCESSFUL ! ");
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::prepareDownload ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::prepareDownload()");

	// Setting the permissions for the /cisinstallation directory
	//! Create instance of the File-Directory class
	dia_FileDir ciscontainer(CIS_CONTAINER);
	//! Check if the directory exists
	if((ciscontainer.doesExist()))
	{
		DIA_TR_INF("directory \"%s\" exists.", CIS_CONTAINER.c_str());
		// chmod 764 CIS_CONTAINER
		std::string cmd_chmod = "chmod 764 " + CIS_CONTAINER;
		int ret = system(cmd_chmod.c_str());
		// Evaluate the system command execution result
		if ( WIFSIGNALED(ret) && ((WTERMSIG(ret) == SIGINT) || (WTERMSIG(ret) == SIGQUIT)) )
		{
			DIA_TR_ERR("  +--> SYSTEM COMMAND EXECUTION FAILED FOR cmd_chmod \"%s\"!!!", CIS_CONTAINER.c_str());
		}	
	}
	else
	{
		DIA_TR_INF("directory \"%s\"  does not exists. ", CIS_CONTAINER.c_str());
	}
	
	if( !mpDataDownloader )
	{
		mpDataDownloader = getInstanceOfSubsystemLogoInstallationManager()->getDataDownloader();
	}
	
	if( !mpDataDownloader ) return;
	
	mpDataDownloader->initializeDataDownload();
	
	// Load the Logo to download
	if( DIA_FAILED == mpDataDownloader->load() )
	{
		DIA_TR_INF("::prepareDownload --> Failed To Load the Logo ! ");
	}
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::prepareLogoInstallation ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::prepareLogoInstallation()");
	
	// Set the callbacks from the client
	if (mpDoIPClient) mpDoIPClient->setCallback(this);
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::prepareNextLogoTransfer ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::prepareNextLogoTransfer()");
	// No need to prepare anything for now
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::prepareSecurityUnlock ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::prepareSecurityUnlock()");
	
	if ( mpSecurityAlgorithm )
	{
		mpSecurityAlgorithm->initialize();
	}
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::prepareTransferData ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::prepareTransferData()");
	
	if( !mpDataDownloader ) return;
	
	// Split the hex dump based on block size
	if( DIA_FAILED == mpDataDownloader->split() )
	{
		DIA_TR_INF("::prepareTransferData --> Failed To Split the Logo Data ! ");
	}
	
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::reset ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::reset()");
	
	if(mpSecurityAlgorithm)	mpSecurityAlgorithm->initialize();
	
	if(mpDataDownloader) mpDataDownloader->initializeDataDownload();
	
	mGlassLogoInstCfgRepo.clear();
	
	mKey.clear();
	
	mSubSysDiagRes.clear();
	mSubSysDiagReq.clear();
	
	mInstCondChkStat = false;
	mDataTransferCompletionStat = false;
	mPowerModeStat = false;
	mRoutingActivationStat = false;
	mDoIPClientActivStat = false;
	mFinalResponseStat = false;
	mTcpSocketStat = false;
	mVehicleListEmptyStat = false;
	mVehicleListValidity = false;
	mVehicleValidity = false;
	mLogoTrnsfrToAllDone = false;
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::selectVehicleForLogoDownload ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::selectVehicleForLogoDownload()");
	
	std::list<SubsystemLogoInstCfgItem>::iterator it;
	bool found = false;
	
	for(it = mGlassLogoInstCfgRepo.begin(); it != mGlassLogoInstCfgRepo.end(); it++)
	{
		DIA_TR_INF("::selectVehicleForLogoDownload -> Analysing Vehicle : 0x%08x", it->Node.subsystemIPAddress);
		DIA_TR_INF("::selectVehicleForLogoDownload -> Installation Status is : (%s)", ((it->logoInstallationDone) ? "Done" : "Not Done"));
		if(false == it->logoInstallationDone)
		{
			if(true == mpDoIPClient->isNodeAvailable(it->Node.subsystemIPAddress))
			{
				DIA_TR_INF("::selectVehicleForLogoDownload -> Selected Vehicle : 0x%08x", it->Node.subsystemIPAddress);
				mpDoIPClient->setCurrentNode(it->Node.subsystemIPAddress);
				found = true;
				break;
			}
		}
	}
	
	if(!found)
	{
		DIA_TR_INF("::selectVehicleForLogoDownload -> No more Vehicle available");
		mLogoTrnsfrToAllDone = true;
	}

}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::sendErr_AlreadyRequested ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::sendErr_AlreadyRequested()");
	
	mErrorCode = ( mpPlugin ) ? mpPlugin->getErrCodeAlreadyRequested() : DIA_E_SUBSYSTEM_LOGO_ALREADY_REQUESTED;
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::sendErr_ConditionsNotCorrect ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::sendErr_ConditionsNotCorrect()");
	
	// Something bad happened and we cannot continue further. send a negative response to the tester.
	
	mErrorCode = ( mpPlugin ) ? mpPlugin->getErrCodeConditionsNotCorrect() : DIA_E_SUBSYSTEM_LOGO_INSTALLATION_INCORRECT_CONDITIONS;

} 

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::sendErr_InstallationAborted ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::sendErr_InstallationAborted()");
	
	// Something bad happened and we cannot continue further. send a negative response to the tester.
	
	mErrorCode = ( mpPlugin ) ? mpPlugin->getErrCodeInstallationAborted() : DIA_E_SUBSYSTEM_LOGO_INSTALLATION_ABORTED;
	
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::sendRequestDownload ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::sendRequestDownload()");
	
	mSubSysDiagReq.clear();
	
	if(DIA_SUCCESS == mpDataDownloader->getDownloadRequest(mSubSysDiagReq))
	{
		if ( mpDoIPClient )
		{
			DIA_TR_INF("::sendRequestDownload : Length: %d Data: %s", mSubSysDiagReq.size(), dia::utils::bin2str(mSubSysDiagReq,' ').c_str());
			mpDoIPClient->diagSendRequest(mSubSysDiagReq);
		}
	}
	else
	{
		DIA_TR_ERR("::sendRequestDownload - Request Download Request Cannot Be Sent!");
	}
	
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::sendRequestExtSessionChange ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::sendRequestExtSessionChange()");
	
	mSubSysDiagReq.clear();
	mSubSysDiagReq.push_back(DIA_C_SUBSYSTEM_EXTENDED_SESSION_REQUEST_SRV_ID);
	mSubSysDiagReq.push_back(DIA_C_SUBSYSTEM_EXTENDED_SESSION_REQUEST_SES_ID);

	if ( mpDoIPClient )
	{
		DIA_TR_INF("::sendRequestExtSessionChange - DIA_REQ UDS 0x%02x 0x%02x", mSubSysDiagReq[0], mSubSysDiagReq[1]);
		mpDoIPClient->diagSendRequest(mSubSysDiagReq);
	}
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::sendRequestPowerMode ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::sendRequestPowerMode()");
	
	//  power mode status will be updated when the power mode response is received
    //  this is handled in callback fn from client - onPMRResponse(tU8 pwrMode)
	if( mpDoIPClient ) { (void) mpDoIPClient->diagSendPowerModeReq(); }
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::sendRequestRoutingActivation ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::sendRequestRoutingActivation()");
	
	if ( mpDoIPClient ) 
	{ 
		// Routing Activation Status will be updated when the activation response is received
		// this is handled in callback fn from client - onRAResponse(tU8 respCode)
		// We do not care about the return value; we handle it in timeout function for RA
		(void) mpDoIPClient->diagSendRoutingActivationReq();	
	}
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::sendRequestSecurityKey ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::sendRequestSecurityKey()");
	
	mSubSysDiagReq.clear();
	
	if(DIA_SUCCESS == mpSecurityAlgorithm->getKeyRequest(mSubSysDiagReq))
	{
		if ( mpDoIPClient )
		{
			DIA_TR_INF("::sendRequestSecurityKey - DIA_REQ UDS 0x%02x 0x%02x 0x%02x 0x%02x ..", 
			                                       mSubSysDiagReq[0], mSubSysDiagReq[1], mSubSysDiagReq[2], mSubSysDiagReq[3]);
			mpDoIPClient->diagSendRequest(mSubSysDiagReq);
		}
	}
	else
	{
		DIA_TR_ERR("::sendRequestSecuritySeed - Security Key Cannot Be Sent!");
	}
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::sendRequestSecuritySeed ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::sendRequestSecuritySeed()");
	
	mSubSysDiagReq.clear();
	
	if(DIA_SUCCESS == mpSecurityAlgorithm->getSeedRequest(mSubSysDiagReq))
	{
		if ( mpDoIPClient )
		{
			DIA_TR_INF("::sendRequestSecuritySeed - DIA_REQ UDS 0x%02x 0x%02x", mSubSysDiagReq[0], mSubSysDiagReq[1]);
			mpDoIPClient->diagSendRequest(mSubSysDiagReq);
		}
	}
	else
	{
		DIA_TR_ERR("::sendRequestSecuritySeed - Security Seed Request Cannot Be Sent!");
	}
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::sendRequestTransferData ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::sendRequestTransferData()");
	
	mSubSysDiagReq.clear();
	
	if(DIA_SUCCESS == mpDataDownloader->getNextTransferDataRequest(mSubSysDiagReq))
	{
		if ( mpDoIPClient )
		{
			DIA_TR_INF("::sendRequestTransferData - DIA_REQ UDS 0x%02x 0x%02x ..", mSubSysDiagReq[0], mSubSysDiagReq[1]);
			mpDoIPClient->diagSendRequest(mSubSysDiagReq);
		}
	}
	else
	{
		DIA_TR_ERR("::sendRequestTransferData - Transfer Data Request Cannot Be Sent!");
	}
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::sendRequestTransferExit ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::sendRequestTransferExit()");
	
	mSubSysDiagReq.clear();
	mSubSysDiagReq.push_back( DIA_C_SUBSYSTEM_TRANSFER_EXIT_REQUEST_SID );
	
	if ( mpDoIPClient )
	{
		DIA_TR_INF("::sendRequestTransferExit - DIA_REQ UDS 0x%02x ", mSubSysDiagReq[0]);
		mpDoIPClient->diagSendRequest(mSubSysDiagReq);
	}
	else
	{
		DIA_TR_ERR("::sendRequestTransferExit - Transfer Exit Request Cannot Be Sent!");
	}
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::sendReqVI ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::sendReqVI()");
	
	if( mpDoIPClient )
	{
		// clear the current node list if any
		mpDoIPClient->clearDoIPNodeList();
		// Send Vehicle Identification Request
		mpDoIPClient->discoverEntities();
	}
	else
	{
		mErrorCode = ( mpPlugin ) ? mpPlugin->getErrCodeDoIPClientNotAvailable() : DIA_E_SUBSYSTEM_LOGO_INSTALLATION_DOIP_CLIENT_NOT_AVAILABLE;
	}
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::sendRoutineFinalResponse ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::sendRoutineFinalResponse()");
	
	 // the  response will be set by the mErrorCode 
	 // Nothing to do here
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::sendRoutineRunningResponse ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::sendRoutineRunningResponse()");
	
	mErrorCode = ( mpPlugin ) ? mpPlugin->getResCodeLogoInstallationInProgress() : DIA_E_SUBSYSTEM_LOGO_INSTALLATION_RUNNING;
	//Update current status
	mStatusCode = mErrorCode;
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::setInstallationActivated ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::setInstallationActivated()");
	
	// Important Annoucement : 
	// Hello all, We have activated the CIS Startup Logo Installation .. <:))))
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::startDoIPClient ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::startDoIPClient()");
	
	mDoIPClientActivStat = false;
	
	if(!mpDoIPClient) return;
	
	//! trigger tester to set up doip communication
	mDoIPClientActivStat = (true == mpDoIPClient->startDoIPClient()) ? true : false;

	//! trigger next statemachine event
	if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evDoIPClientActivated, 0);
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::startTimerPowerMode ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::startTimerPowerMode()");
	
	if ( mpDoIPClient ) mpDoIPClient->startPowerModeRequestDelayTimer();
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::startTimerRequestDownload ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::startTimerRequestDownload()");
	
	DIA_TR_INF("INFO --> STARTING THE REQUEST DOWNLOAD TIMER! ");
	
	if ( mpDoIPClient ) mpDoIPClient->startSubsystemRoutingDelayTimer();
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::startTimerRoutingActivation ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::startTimerRoutingActivation()");
	
	if ( mpDoIPClient ) mpDoIPClient->startRoutingActivationDelayTimer();
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::startTimerSecurityKey ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::startTimerSecurityKey()");
	
	DIA_TR_INF("INFO --> STARTING THE SECURITY KEY TIMER! ");
	
	if ( mpDoIPClient ) mpDoIPClient->startSubsystemRoutingDelayTimer();
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::startTimerSecuritySeed ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::startTimerSecuritySeed()");
	
	DIA_TR_INF("INFO --> STARTING THE SECURITY SEED TIMER! ");
	
	if ( mpDoIPClient ) mpDoIPClient->startSubsystemRoutingDelayTimer();
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::startTimerSessionSwitch ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::startTimerSessionSwitch()");
	
	DIA_TR_INF("INFO --> STARTING THE SESSION CHANGE TIMER! ");
	
	if ( mpDoIPClient ) mpDoIPClient->startSubsystemRoutingDelayTimer();
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::startTimerTransferData ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::startTimerTransferData()");
	
	DIA_TR_INF("INFO --> STARTING THE TRANSFER DATA TIMER! ");
	
	if ( mpDoIPClient ) mpDoIPClient->startSubsystemRoutingDelayTimer();
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::startTimerTransferExit ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::startTimerTransferExit()");
	
	DIA_TR_INF("INFO --> STARTING THE TRANSFER EXIT TIMER! ");
	
	if ( mpDoIPClient ) mpDoIPClient->startSubsystemRoutingDelayTimer();
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::startTimerVI ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::startTimerVI()");
	
	// Start the Vehicle Identification Timer
	
	if ( mpDoIPClient ) 
	{
		mpDoIPClient->startDiscoverEntitiesDelayTimer();
	}
	else
	{
		mErrorCode = ( mpPlugin ) ? mpPlugin->getErrCodeDoIPClientNotAvailable() : DIA_E_SUBSYSTEM_LOGO_INSTALLATION_DOIP_CLIENT_NOT_AVAILABLE;
	}
	
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::stopLogoInstallation ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::stopLogoInstallation()");
	
	// It is difficult to know exactly what the statemachine is currently doing.. 
	// So lets just update the list of subsbsytems with status as completed nok
	// thus the statemachine will not continue anymore
	
	std::list<SubsystemLogoInstCfgItem>::iterator it;
	
	for(it = mGlassLogoInstCfgRepo.begin(); it != mGlassLogoInstCfgRepo.end(); it++)
	{
		DIA_TR_INF("::stopLogoInstallation -> Updating status for Vehicle : 0x%08x", it->Node.subsystemIPAddress);
		it->logoInstallationDone = true;
		it->logoInstallationSucceeded = false;
	}
	
	
	if(!mpDoIPClient) return;
	
	//! trigger tester to close up doip communication
	// We have to do a controlled communication closure
	mDoIPClientActivStat = (true == mpDoIPClient->stopDoIPClient()) ? false : true;	
	
	// after this the state machine will enter the not active state.. 
	// so we need not update the error code..

}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::stopTimerPowerMode ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::stopTimerPowerMode()");
	
	DIA_TR_INF("INFO --> POWER MODE RESPONSE RECEIVED. STOPPING THE TIMER! ");
	
	if ( mpDoIPClient ) mpDoIPClient->stopPowerModeRequestDelayTimer();
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::stopTimerRequestDownload ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::stopTimerRequestDownload()");
	
	DIA_TR_INF("INFO --> FINAL RESPONSE RECEIVED. STOPPING REQUEST DOWNLOAD TIMER! ");
	
	if ( mpDoIPClient ) mpDoIPClient->stopSubsystemRoutingDelayTimer();
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::stopTimerRoutingActivation ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::stopTimerRoutingActivation()");
	
	DIA_TR_INF("INFO --> ROUTING ACTIVATION RESPONSE RECEIVED. STOPPING THE TIMER! ");
	
	if ( mpDoIPClient ) mpDoIPClient->stopRoutingActivationDelayTimer();
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::stopTimerSecurityKey ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::stopTimerSecurityKey()");
	
	DIA_TR_INF("INFO --> FINAL RESPONSE RECEIVED. STOPPING SECURITY KEY TIMER! ");
	
	if ( mpDoIPClient ) mpDoIPClient->stopSubsystemRoutingDelayTimer();
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::stopTimerSecuritySeed ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::stopTimerSecuritySeed()");
	
	DIA_TR_INF("INFO --> FINAL RESPONSE RECEIVED. STOPPING SECURITY SEED TIMER! ");
	
	if ( mpDoIPClient ) mpDoIPClient->stopSubsystemRoutingDelayTimer();
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::stopTimerSessionSwitch ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::stopTimerSessionSwitch()");
	
	DIA_TR_INF("INFO --> FINAL RESPONSE RECEIVED. STOPPING SESSION CHANGE TIMER! ");
	
	if ( mpDoIPClient ) mpDoIPClient->stopSubsystemRoutingDelayTimer();
	
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::stopTimerTransferData ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::stopTimerTransferData()");
	
	DIA_TR_INF("INFO --> FINAL RESPONSE RECEIVED. STOPPING TRANSFER DATA TIMER! ");
	
	if ( mpDoIPClient ) mpDoIPClient->stopSubsystemRoutingDelayTimer();
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::stopTimerTransferExit ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::stopTimerTransferExit()");
	
	DIA_TR_INF("INFO --> FINAL RESPONSE RECEIVED. STOPPING TRANSFER EXIT TIMER! ");
	
	if ( mpDoIPClient ) mpDoIPClient->stopSubsystemRoutingDelayTimer();
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::validateLogoInstallationRequest ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::validateLogoInstallationRequest()");
	
	// installation conditions are all fine
	mInstCondChkStat = true;
	
	// nxt processing event
	if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evRequestInstallLogoValid, 0);
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::validateSecurityKeyReponse ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::validateSecurityKeyReponse()");
	
	tU8 curSecurityLevel = mpSecurityAlgorithm->getCurrentSecurityLevel();
	
	// if security seed sub function is 1, then key sub function is 2 ( usually )
	curSecurityLevel += 1;
	
	// validate the session change response
	if((mSubSysDiagRes[DIA_C_SUBSYSTEM_LOGO_INSTALLATION_POS_RESP_SRV_ID_IDX] == DIA_C_SUBSYSTEM_SECURITY_ACCESS_POS_RESP_ID) && 
	   (mSubSysDiagRes[DIA_C_SUBSYSTEM_SECURITY_ACCESS_POS_RESP_SUB_ID_IDX] == curSecurityLevel))
	{
		//This is a positive response
		DIA_TR_INF("Security Key Response : 0x%02x 0x%02x", mSubSysDiagRes[0], mSubSysDiagRes[1] );
		// Trigger nxt processing event
		if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evPositiveResponse, 0);
	}
	else
	{
		if(mSubSysDiagRes[DIA_C_SUBSYSTEM_LOGO_INSTALLATION_NEG_RESP_SRV_ID_IDX] == DIA_C_SUBSYSTEM_LOGO_INSTALLATION_NEG_RESP_SRV_ID)
		{
			if(mSubSysDiagRes[DIA_C_SUBSYSTEM_RESPONSE_TYPE_WAIT_NRC_IDX] == DIA_C_SUBSYSTEM_RESPONSE_TYPE_WAIT_NRC)
			{
				// This is a wait response
				DIA_TR_INF("Security Key Response : 0x%02x 0x%02x 0x%02x", mSubSysDiagRes[0], mSubSysDiagRes[1], mSubSysDiagRes[2] );
				// Trigger nxt processing event
				if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evWaitResponse, 0);
			}
			else
			{
				// This is a negative response
				DIA_TR_INF("Security Key Response : 0x%02x 0x%02x 0x%02x", mSubSysDiagRes[0], mSubSysDiagRes[1], mSubSysDiagRes[2] );
				// Trigger nxt processing event
				if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evNegativeResponse, 0);
			}
		}
		else
		{
			// unknown response ??? print first 2 bytes only
			//DIA_TR_INF(" (Unknown) Security Key Response : [size: %d] 0x%02x 0x%02x ...", mSubSysDiagRes.size(), mSubSysDiagRes[0], mSubSysDiagRes[1]);
		    DIA_TR_INF(" (Unknown) Security Key Response : Length: %d Data: %s", mSubSysDiagRes.size(), dia::utils::bin2str(mSubSysDiagRes,' ').c_str());
		}
	}
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::validateSecuritySeedReponse ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::validateSecuritySeedReponse()");
	
	tU8 curSecurityLevel = mpSecurityAlgorithm->getCurrentSecurityLevel();
	
	// validate the session change response
	if((mSubSysDiagRes[DIA_C_SUBSYSTEM_LOGO_INSTALLATION_POS_RESP_SRV_ID_IDX] == DIA_C_SUBSYSTEM_SECURITY_ACCESS_POS_RESP_ID) && 
	   (mSubSysDiagRes[DIA_C_SUBSYSTEM_SECURITY_ACCESS_POS_RESP_SUB_ID_IDX] == curSecurityLevel))
	{
		//This is a positive response
		//DIA_TR_INF("Security Seed Response : [size: %d] 0x%02x 0x%02x ...", mSubSysDiagRes.size(), mSubSysDiagRes[0], mSubSysDiagRes[1] ");
		DIA_TR_INF("Security Seed Response : Length: %d Data: %s", mSubSysDiagRes.size(), dia::utils::bin2str(mSubSysDiagRes,' ').c_str());
		// Trigger nxt processing event
		if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evPositiveResponse, 0);
	}
	else
	{
		if(mSubSysDiagRes[DIA_C_SUBSYSTEM_LOGO_INSTALLATION_NEG_RESP_SRV_ID_IDX] == DIA_C_SUBSYSTEM_LOGO_INSTALLATION_NEG_RESP_SRV_ID)
		{
			if(mSubSysDiagRes[DIA_C_SUBSYSTEM_RESPONSE_TYPE_WAIT_NRC_IDX] == DIA_C_SUBSYSTEM_RESPONSE_TYPE_WAIT_NRC)
			{
				// This is a wait response
				DIA_TR_INF("Security Seed Response : 0x%02x 0x%02x 0x%02x", mSubSysDiagRes[0], mSubSysDiagRes[1], mSubSysDiagRes[2] );
				// Trigger nxt processing event
				if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evWaitResponse, 0);
			}
			else
			{
				// This is a negative response
				DIA_TR_INF("Security Seed Response : 0x%02x 0x%02x 0x%02x", mSubSysDiagRes[0], mSubSysDiagRes[1], mSubSysDiagRes[2] );
				// Trigger nxt processing event
				if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evNegativeResponse, 0);
			}
		}
		else
		{
			// unknown response ??? print first 2 bytes only
			//DIA_TR_INF(" (Unknown) Security Seed Response : [size: %d] 0x%02x 0x%02x ...", mSubSysDiagRes.size(), mSubSysDiagRes[0], mSubSysDiagRes[1]);
			DIA_TR_INF(" (Unknown) Security Seed Response : Length: %d Data: %s", mSubSysDiagRes.size(), dia::utils::bin2str(mSubSysDiagRes,' ').c_str());
		}
	}
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::validateSessionChangeReponse ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::validateSessionChangeReponse()");
	
	// validate the session change response
	if((mSubSysDiagRes[DIA_C_SUBSYSTEM_EXTENDED_SESSION_POS_RESP_SRV_ID_IDX] == DIA_C_SUBSYSTEM_EXTENDED_SESSION_POS_RESP_SRV_ID) && 
	   (mSubSysDiagRes[DIA_C_SUBSYSTEM_EXTENDED_SESSION_POS_RESP_SES_ID_IDX] == DIA_C_SUBSYSTEM_EXTENDED_SESSION_POS_RESP_SES_ID))
	{
		// This is a positive response
		//DIA_TR_INF("Session Switch Response : 0x%02x 0x%02x", mSubSysDiagRes[0], mSubSysDiagRes[1] );
		DIA_TR_SM("Session Switch Response : Length: %d Data: %s", mSubSysDiagRes.size(), dia::utils::bin2str(mSubSysDiagRes,' ').c_str());
		// Trigger nxt processing event
		if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evPositiveResponse, 0);
		
	}
	else 
	{
		if(mSubSysDiagRes[DIA_C_SUBSYSTEM_LOGO_INSTALLATION_NEG_RESP_SRV_ID_IDX] == DIA_C_SUBSYSTEM_LOGO_INSTALLATION_NEG_RESP_SRV_ID)
		{
			if(mSubSysDiagRes[DIA_C_SUBSYSTEM_RESPONSE_TYPE_WAIT_NRC_IDX] == DIA_C_SUBSYSTEM_RESPONSE_TYPE_WAIT_NRC)
			{
				// This is a wait response
				DIA_TR_INF("Session Switch Response : 0x%02x 0x%02x 0x%02x", mSubSysDiagRes[0], mSubSysDiagRes[1], mSubSysDiagRes[2] );
				// Trigger nxt processing event
				if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evWaitResponse, 0);
			}
			else
			{
				// This is a negative response
				DIA_TR_INF("Session Switch Response : 0x%02x 0x%02x 0x%02x", mSubSysDiagRes[0], mSubSysDiagRes[1], mSubSysDiagRes[2] );
				// Trigger nxt processing event
				if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evNegativeResponse, 0);
			}
		}
		else
		{
			// unknown response ??? print first 2 bytes only
			//DIA_TR_INF(" (Unknown) Session Switch Response : [size: %d] 0x%02x 0x%02x ", mSubSysDiagRes.size(), mSubSysDiagRes[0], mSubSysDiagRes[1]);
			DIA_TR_INF(" (Unknown) Session Switch Response : Length: %d Data: %s", mSubSysDiagRes.size(), dia::utils::bin2str(mSubSysDiagRes,' ').c_str());
		}
	}
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::validateVehicleList ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::validateVehicleList()");
	
	// Initialize the vehicle list validity status to invalid
	mVehicleListValidity = false;
	
	// Here we have to filter the subsystem list with only Glasses
	std::list<dia_tSubsystemInfo> nodelist;
	mGlassLogoInstCfgRepo.clear();
	
	if( mpDoIPClient )
	{
		nodelist = mpDoIPClient->getDoIPClientNodes();
		
		// iterate through the list of nodes and filter out only Glasses
		std::list<dia_tSubsystemInfo>::iterator it;
		for(it = nodelist.begin(); it != nodelist.end(); it++)
		{
			DIA_TR_INF("#############################################");
			DIA_TR_INF("::validateVehicleList --> Selected Node Info :");
			DIA_TR_INF("::validateVehicleList --> IP ADDRESS : 0x%08x ", (it->subsystemIPAddress));
			DIA_TR_INF("::validateVehicleList --> ENTITY NAME : 0x%04x ", (it->subsystemEntityName));
			if(mpDoIPClient) 
			{
				tU16 logicalAddr = ((mpDoIPClient->vGetNodeByIP(it->subsystemIPAddress))->logicalAddr);
				DIA_TR_INF("::validateVehicleList --> LOGICAL ADDR : 0x%02x ", logicalAddr);
			}
			//DIA_TR_INF("::validateVehicleList --> Comparing against DIA_C_SUBSYSTEM_LOGO_INSTALLATION_GID_GLASS_CODE ");
			//DIA_TR_INF("::validateVehicleList --> whose value is : 0x%04x ", DIA_C_SUBSYSTEM_LOGO_INSTALLATION_GID_GLASS_CODE);
			
			if(DIA_C_SUBSYSTEM_LOGO_INSTALLATION_GID_GLASS_CODE == it->subsystemEntityName)
			{
				DIA_TR_INF("::validateVehicleList: New Glass Node Found ( IP : 0x%08x )", (it->subsystemIPAddress));
				// Add the node to Glas list
				SubsystemLogoInstCfgItem glassNodeCfg;
				glassNodeCfg.Node = *it;
				DIA_TR_INF("::validateVehicleList: Conforming new node ( IP : 0x%08x )", (glassNodeCfg.Node.subsystemIPAddress));
				mGlassLogoInstCfgRepo.push_back(glassNodeCfg);
			}
			DIA_TR_INF("#############################################");
		}
	}
	
	// Check if atleast one Glass Node is available
	if( !mGlassLogoInstCfgRepo.empty() )
	{
		DIA_TR_INF("::validateVehicleList --> Total %d Glass Nodes Found ", mGlassLogoInstCfgRepo.size());
		// The list is valid as it contains Glass Nodes
		mVehicleListValidity = true;
	}
	
	// Trigger next state machine event
	if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evVehicleListValidated, 0);
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::vHandleDataError ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::vHandleDataError()");
	
	// Should we do a retry? This is just a hypothetical situation, but we have to consider it!
	// may be in future..
	DIA_TR_ERR("ERROR --> SOMETHING BAD HAPPENED WHILE ASSEMBLING THE DATA! ");
	DIA_TR_INF("INFO  --> CLOSING CONNECTION AND CONTINUING WITH NEXT SUBSYSTEM ");
	
	vUpdateSubsystemStatusFailed();
		
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::vHandleFinalResponse ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::vHandleFinalResponse()");
	
	//Based on the completion status of all the subsystems in the repo take a decision
	
	bool failedInstanceFound = false;
	std::list<SubsystemLogoInstCfgItem>::iterator it;
	
	for(it = mGlassLogoInstCfgRepo.begin(); it != mGlassLogoInstCfgRepo.end(); it++)
	{
		DIA_TR_INF("::vHandleFinalResponse: Checking installation status for Vehicle: 0x%08x", it->Node.subsystemIPAddress);
		if(false == it->logoInstallationSucceeded)
		{
			DIA_TR_INF("::vHandleFinalResponse: Installation Failed for Vehicle: 0x%08x", it->Node.subsystemIPAddress);
			failedInstanceFound = true;
			// we can break here as we have to send completed nok even there is one failure.
			break;
		}
	}
	
	if(failedInstanceFound)
	{
		DIA_TR_INF("::vHandleFinalResponse: mStatusCode : DIA_E_SUBSYSTEM_LOGO_INSTALLATION_COMPLETED_NOK ");
		mStatusCode = ( mpPlugin ) ? mpPlugin->getErrCodeLogoInstallationCompletedNok() : DIA_E_SUBSYSTEM_LOGO_INSTALLATION_COMPLETED_NOK;
	}
	else
	{
		DIA_TR_INF("::vHandleFinalResponse: mStatusCode : DIA_E_SUBSYSTEM_LOGO_INSTALLATION_COMPLETED_OK ");
		mStatusCode = ( mpPlugin ) ? mpPlugin->getResCodeLogoInstallationCompletedOk() : DIA_E_SUBSYSTEM_LOGO_INSTALLATION_COMPLETED_OK;
	}
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::vHandleRequestDownloadError ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::vHandleRequestDownloadError()");
	
	DIA_TR_ERR("ERROR --> NEGATIVE RESPONSE RECEIVED FOR REQUEST DOWNLOAD REQUEST! ");
	DIA_TR_INF("INFO  --> CLOSING CONNECTION AND CONTINUING WITH NEXT SUBSYSTEM ");
	
	vUpdateSubsystemStatusFailed();
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::vHandleRequestDownloadTimeout ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::vHandleRequestDownloadTimeout()");
	
	DIA_TR_ERR("ERROR --> REQUEST DOWNLOAD TIMER TIMED OUT! ");
	DIA_TR_INF("INFO  --> CLOSING CONNECTION AND CONTINUING WITH NEXT SUBSYSTEM ");
	
	vUpdateSubsystemStatusFailed();
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::vHandleRequestSeedError ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::vHandleRequestSeedError()");
	
	DIA_TR_ERR("ERROR --> NEGATIVE RESPONSE RECEIVED FOR SECURITY SEED REQUEST! ");
	DIA_TR_INF("INFO  --> CLOSING CONNECTION AND CONTINUING WITH NEXT SUBSYSTEM ");
	
	vUpdateSubsystemStatusFailed();
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::vHandleSecurityKeyError ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::vHandleSecurityKeyError()");
	
	DIA_TR_ERR("ERROR --> NEGATIVE RESPONSE RECEIVED FOR SECURITY KEY REQUEST! ");
	DIA_TR_INF("INFO  --> CLOSING CONNECTION AND CONTINUING WITH NEXT SUBSYSTEM ");
	
	vUpdateSubsystemStatusFailed();
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::vHandleSecurityKeyTimeout ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::vHandleSecurityKeyTimeout()");
	
	DIA_TR_ERR("ERROR --> SECURITY KEY TIMER TIMED OUT! ");
	DIA_TR_INF("INFO  --> CLOSING CONNECTION AND CONTINUING WITH NEXT SUBSYSTEM ");
	
	vUpdateSubsystemStatusFailed();
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::vHandleSecuritySeedTimeout ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::vHandleSecuritySeedTimeout()");
	
	DIA_TR_ERR("ERROR --> SECURITY SEED TIMER TIMED OUT! ");
	DIA_TR_INF("INFO  --> CLOSING CONNECTION AND CONTINUING WITH NEXT SUBSYSTEM ");
	
	vUpdateSubsystemStatusFailed();
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::vHandleSessionChangeError ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::vHandleSessionChangeError()");
	
	DIA_TR_ERR("ERROR --> NEGATIVE RESPONSE RECEIVED FOR SESSION SWITCH REQUEST! ");
	DIA_TR_INF("INFO  --> CLOSING CONNECTION AND CONTINUING WITH NEXT SUBSYSTEM ");
	
	vUpdateSubsystemStatusFailed();
	
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::vHandleSessionChangeTimeout ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::vHandleSessionChangeTimeout()");
	
	DIA_TR_ERR("ERROR --> SESSION SWITCH TIMER TIMED OUT! ");
	DIA_TR_INF("INFO  --> CLOSING CONNECTION AND CONTINUING WITH NEXT SUBSYSTEM ");
	
	vUpdateSubsystemStatusFailed();
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::vHandleSessionRequest ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::vHandleSessionRequest()");
	
	DIA_TR_INF("INFO  --> SESSION CHANGED, NOTHING TO DO ");
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::vHandleTransferDataError ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::vHandleTransferDataError()");
	
	DIA_TR_ERR("ERROR --> NEGATIVE RESPONSE RECEIVED FOR TRANSFER DATA REQUEST! ");
	DIA_TR_INF("INFO  --> CLOSING CONNECTION AND CONTINUING WITH NEXT SUBSYSTEM ");
	
	vUpdateSubsystemStatusFailed();
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::vHandleTransferDataTimeout ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::vHandleTransferDataTimeout()");
	
	DIA_TR_ERR("ERROR --> TRANSFER DATA TIMER TIMED OUT! ");
	DIA_TR_INF("INFO  --> CLOSING CONNECTION AND CONTINUING WITH NEXT SUBSYSTEM ");
	
	vUpdateSubsystemStatusFailed();
	
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::vHandleTransferExitError ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::vHandleTransferExitError()");
	
	DIA_TR_ERR("ERROR --> NEGATIVE RESPONSE RECEIVED FOR TRANSFER EXIT REQUEST! ");
	DIA_TR_INF("INFO  --> CLOSING CONNECTION AND CONTINUING WITH NEXT SUBSYSTEM ");
	
	vUpdateSubsystemStatusFailed();
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::vHandleTransferExitTimeout ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::vHandleTransferExitTimeout()");
	
	DIA_TR_ERR("ERROR --> TRANSFER EXIT TIMER TIMED OUT! ");
	DIA_TR_INF("INFO  --> CLOSING CONNECTION AND CONTINUING WITH NEXT SUBSYSTEM ");
	
	vUpdateSubsystemStatusFailed();
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::vHandleVehicleConnectError ( void* /* pArg */ )
{
	// dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::vHandleVehicleConnectError()");
	
	DIA_TR_ERR("ERROR --> FAILED TO CONNECT TO THE SELECTED VEHICLE! ");
	DIA_TR_INF("INFO  --> CLOSING CONNECTION AND CONTINUING WITH NEXT SUBSYSTEM "); 
	
	vUpdateSubsystemStatusFailed();
}

//-----------------------------------------------------------------------------

// FSM guards
bool 
dia_SubsystemLogoInstallationStrategyCore::areInstallationConditionsOk ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::areInstallationConditionsOk()");
	
	return mInstCondChkStat;
}

//-----------------------------------------------------------------------------

bool 
dia_SubsystemLogoInstallationStrategyCore::bFsmIsDataTransferComplete ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::bFsmIsDataTransferComplete()");
	
	// if data downloader itself not there, we cannot go ahead. So just send true,
	// which means data transfer completed.
	return ((mpDataDownloader) ? mpDataDownloader->isDataTransferComplete() : true);
}

//-----------------------------------------------------------------------------

bool 
dia_SubsystemLogoInstallationStrategyCore::bIsPowerModeOk ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::bIsPowerModeOk()");
	
	return mPowerModeStat;
}

//-----------------------------------------------------------------------------

bool 
dia_SubsystemLogoInstallationStrategyCore::bIsRoutingActivationOk ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::bIsRoutingActivationOk()");
	
	return mRoutingActivationStat;
}

//-----------------------------------------------------------------------------

bool 
dia_SubsystemLogoInstallationStrategyCore::isClientActivationOk ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::isClientActivationOk()");
	
	return mDoIPClientActivStat;
}

//-----------------------------------------------------------------------------

bool 
dia_SubsystemLogoInstallationStrategyCore::isFinalResponseReady ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::isFinalResponseReady()");
	
	// if the logo transfer is not yet done for all the Glasses,
	// Status should be still not done
	mFinalResponseStat = mLogoTrnsfrToAllDone;
	
	// However, if there is a fatal error like below, we need to stop
	// So forcefully update the final status.
	switch(mErrorCode)
	{
		case DIA_E_SUBSYSTEM_LOGO_INSTALLATION_DOIP_CLIENT_NOT_AVAILABLE:
		case DIA_E_SUBSYSTEM_LOGO_INSTALLATION_ABORTED:
		{
			DIA_TR_ERR("ERROR --> FORCEFULLY UPDATING THE FINAL STATUS! ");
			mFinalResponseStat = true;
			mStatusCode = mErrorCode;
		}
		break;
	}
	
	return mFinalResponseStat;
}

//-----------------------------------------------------------------------------

bool 
dia_SubsystemLogoInstallationStrategyCore::isTCPSocketOk ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::isTCPSocketOk()");
	
	DIA_TR_INF("dia_SubsystemLogoInstallationStrategyCore::isTCPSocketOk() : %s ", (mTcpSocketStat==true) ? "YES" : "NO");
	
	return mTcpSocketStat;
}

//-----------------------------------------------------------------------------

bool 
dia_SubsystemLogoInstallationStrategyCore::isVehicleListEmpty ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::isVehicleListEmpty()");
	
	std::list<dia_tSubsystemInfo> doipNodes;
	mVehicleListEmptyStat = true;

	if( mpDoIPClient )
	{
		doipNodes = mpDoIPClient->getDoIPClientNodes();
		mVehicleListEmptyStat = (( doipNodes.empty() == true ) ? true : false );
	}
	
	DIA_TR_INF("dia_SubsystemLogoInstallationStrategyCore::isVehicleListEmpty() : %s ", (mVehicleListEmptyStat==true) ? "YES" : "NO");
	
	return mVehicleListEmptyStat;
}

//-----------------------------------------------------------------------------

bool 
dia_SubsystemLogoInstallationStrategyCore::isVehicleListValid ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::isVehicleListValid()");
	
	DIA_TR_INF("dia_SubsystemLogoInstallationStrategyCore::isVehicleListValid() : %s ", (mVehicleListValidity==true) ? "YES" : "NO");
	
	/* if vehicle list is valid, we can open the local UDP sockets */
	if(mVehicleListValidity)
	{
		if (mpDoIPClient)
		{
			// creating udp sockets in this context as this is required for PMR
			mpDoIPClient->Start();
		}
	}
	
	return mVehicleListValidity;
}

//-----------------------------------------------------------------------------

bool 
dia_SubsystemLogoInstallationStrategyCore::isVehicleValid ( void* /* pArg */ )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::isVehicleValid()");
	
	DIA_TR_INF("dia_SubsystemLogoInstallationStrategyCore::isVehicleValid() : %s ", (mVehicleValidity==true) ? "YES" : "NO");
	
	return mVehicleValidity;
}

//-----------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::onTimeout ( tU32 timerID )
{
	//dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::onTimeout(...)");
	
	if(mpDoIPClient->getTimerId_IE() == timerID)
	{
		DIA_TR_INF("::onTimeout -> Identify Entity Timer Elapsed --- ID = %d", timerID);
		if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evVITimeout, 0);
	}
	else if(mpDoIPClient->getTimerId_PMR() == timerID)
	{
		DIA_TR_INF("::onTimeout -> Power Mode Request Timer Elapsed --- ID = %d", timerID);
		if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evTimeoutPowerMode, 0);
	}
	else if(mpDoIPClient->getTimerId_RA() == timerID)
	{
		DIA_TR_INF("::onTimeout -> Routing Activation Timer Elapsed --- ID = %d", timerID);
		if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evTimeoutRoutingActivation, 0);
	}
	else if(mpDoIPClient->getTimerId_RR() == timerID)
	{
		DIA_TR_INF("::onTimeout -> Routing Request Timer Elapsed --- ID = %d", timerID);
		if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evTimeoutDiagRequest, 0);
	}
	else if(mpDoIPClient->getTimerId_LT() == timerID)
	{
		DIA_TR_INF("::onTimeout -> Logo Transfer Timer Elapsed --- ID = %d", timerID);
		vOnLogoTransferWaitOver();
	}
	else
	{
		DIA_TR_INF("::onTimeout -> Unknown Timer --- ID = %d", timerID);
	}
}

//------------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::vUpdateSubsystemLogoInstallationStatus(tU32 IP, bool status)
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::vUpdateSubsystemLogoInstallationStatus(tU32 IP)");
	
	DIA_TR_INF("::vUpdateSubsystemLogoInstallationStatus -> Update Requested for IP (0x%08x), status(%s)", IP, (status ? "true" : "false"));
	
	std::list<SubsystemLogoInstCfgItem>::iterator it;
	
	for(it = mGlassLogoInstCfgRepo.begin(); it != mGlassLogoInstCfgRepo.end(); it++)
	{
		if(IP == (it->Node.subsystemIPAddress))
		{
			it->logoInstallationDone = status;
			DIA_TR_INF("::vUpdateSubsystemLogoInstallationStatus -> Update done with status(%s)", ((it->logoInstallationDone) ? "true" : "false"));
			break;
		}
	}
}

//------------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::vUpdateSubsystemLogoInstallationCompletionStatus(tU32 IP, bool status)
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::vUpdateSubsystemLogoInstallationCompletionStatus(tU32 IP)");
	
	DIA_TR_INF("::vUpdateSubsystemLogoInstallationCompletionStatus -> Update Requested for IP (0x%08x), status(%s)", IP, (status ? "true" : "false"));
	
	std::list<SubsystemLogoInstCfgItem>::iterator it;
	
	for(it = mGlassLogoInstCfgRepo.begin(); it != mGlassLogoInstCfgRepo.end(); it++)
	{
		if(IP == ((it->Node).subsystemIPAddress))
		{
			it->logoInstallationSucceeded = status;
			DIA_TR_INF("::vUpdateSubsystemLogoInstallationCompletionStatus -> Update done with status(%s)", ((it->logoInstallationSucceeded) ? "true" : "false"));
			break;
		}
	}
}

//------------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::onPMRResponse(tU8 pwrMode)
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::onPMRResponse(...)");
	
	DIA_TR_INF("Power Mode Response Code = %d", pwrMode);
	
	mPowerModeStat = (( DIA_C_SUBSYSTEM_POWER_MODE_READY == pwrMode ) ? true : false );
	
	// trigger next state machine event
	if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evOnPowerModeResponse, 0);
}

//------------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::onRAResponse(tU8 respCode)
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::onRAResponse(...)");
	
	DIA_TR_INF("Routing Activation Response Code = %d", respCode);
	mRoutingActivationStat = (( DOIP_RA_RESPCODE_SUCCESS == respCode ) ? true : false );
	
	if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evOnResponseRoutingActivation, 0);
}

//------------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::onDiagnosticResponse(tU8 const * data, tU32 U32len)
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::onDiagnosticResponse(...)");
	
	if (U32len == 0) 
	{
		DIA_TR_ERR("::onDiagnosticResponse --> response buffer null or response length zero");
		return;
	}
	
	//! Just store the response data here
	mSubSysDiagRes.clear();
	mSubSysDiagRes.reserve(U32len);
	
	for(int i = 0; i < U32len; i++)
	{
		mSubSysDiagRes.push_back(data[i]);
	}
	
	if ( mpFSM ) mpFSM->acceptEvent(dia_CISStartupLogoInstallationFSM::evRespRcvd, 0);
	
}

//------------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::vUpdateSubsystemStatusFailed()
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::vUpdateSubsystemStatusFailed()");

	if( !mpDoIPClient ) return;
	
	tU32 IP;
	// get the active doip node ip
	IP = mpDoIPClient->getCurrentNodeIP();
	
	DIA_TR_INF("INFO  --> Updating the Logo Installation Status for 0x%08x as Done", IP);
	// installation status ->  done
	vUpdateSubsystemLogoInstallationStatus(IP, true);
	DIA_TR_INF("INFO  --> Updating the Logo Installation Completion Status for 0x%08x as Failed", IP);
	// completion status -> failed
	vUpdateSubsystemLogoInstallationCompletionStatus(IP, false);
	
}
//------------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationStrategyCore::vUpdateSubsystemStatusSucceeded()
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationStrategyCore::vUpdateSubsystemStatusSucceeded()");
	
	if( !mpDoIPClient ) return;
	
	tU32 IP;
	// get the active doip node ip
	IP = mpDoIPClient->getCurrentNodeIP();
	
	DIA_TR_INF("INFO  --> Updating the Logo Installation Status for 0x%08x as Done", IP);
	// installation status ->  done
	vUpdateSubsystemLogoInstallationStatus(IP, true);
	DIA_TR_INF("INFO  --> Updating the Logo Installation Completion Status for 0x%08x as Succeeded", IP);
	// completion status -> failed
	vUpdateSubsystemLogoInstallationCompletionStatus(IP, true);
	
}

//------------------------------------------------------------------------------

//------------------------------------------------------------------------------

//------------------------------------------------------------------------------