/*!
 * \file       dia_SubsystemLogoInstallationManager.cpp
 *
 * \brief      Manager class that controls installation of the HMI Logo
 *             to all the connected subsystems ( Glasses only )
 *
 * \details    ...
 *
 * \component  Diagnostics
 *
 * \ingroup    cis subsystem hmi logo installation
 *
 * \author     Arjun Manjunath Sanu (RBEI/ECA2)
 *
 * \date       13.03.2020
 *
 * \copyright  (c) 2019 Robert Bosch Engineering & Business Solutions Ltd.
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */
 
#ifndef __INCLUDED_DIA_SUBSYSTEM_LOGO_INSTALLATION_MANAGER__
#include "project/framework/cis/logoinstallation/dia_SubsystemLogoInstallationManager.h"
#endif

#ifndef __INCLUDED_DIA_SUBSYSTEM_LOGO_INSTALLATION_STRATEGY_DEFAULT__
#include "project/framework/cis/logoinstallation/dia_SubsystemLogoInstallationStrategyDefault.h"
#endif

#ifndef __INCLUDED_DIA_APPCONTROLLER__
#include <common/framework/application/dia_AppController.h>
#endif

#ifndef __INCLUDED_DIA_FACTORY__
#include "common/framework/application/dia_Factory.h"
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG_PROJECT__
#include "project/framework/config/dia_defsProjectConfig.h"
#endif


DIA_IMPL_SINGLETON_WITH_SETUP_AND_TEARDOWN(dia_SubsystemLogoInstallationManager)

#ifndef __DIA_UNIT_TESTING__

dia_SubsystemLogoInstallationManager*
getInstanceOfSubsystemLogoInstallationManager ( void )
{
   return dia_SubsystemLogoInstallationManager::getInstance();
}

void
releaseInstanceOfSubsystemLogoInstallationManager ( void )
{
   dia_SubsystemLogoInstallationManager::deleteInstance();
}

#endif

//-----------------------------------------------------------------------------

dia_SubsystemLogoInstallationManager::dia_SubsystemLogoInstallationManager ( void )
   : mpPlugin(0),
     mpActiveStrategy(0),
	 mpActiveSecurityAlgorithm(0),
	 mpActiveDownloader(0),
	 mCurrentLogoId(0)
{}

//-----------------------------------------------------------------------------

dia_SubsystemLogoInstallationManager::~dia_SubsystemLogoInstallationManager ( void )
{
   _BP_TRY_BEGIN
   {
      delete mpPlugin;
      mpPlugin = nullptr;

      mpActiveStrategy->tearDown();
      mpActiveStrategy  = nullptr;
	  
	  mpActiveSecurityAlgorithm->finalize();
	  mpActiveSecurityAlgorithm = nullptr;
	  
	  mDatasetcfgRep.clear();
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia_SubsystemLogoInstallationManager::~dia_SubsystemLogoInstallationManager !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SubsystemLogoInstallationManager::setup ( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationManager::setup");
	tDiaResult retCode = DIA_SUCCESS;
	
	if ( !mpPlugin )
	{
		mpPlugin = new dia_SubsystemLogoInstallationManagerPlugin;
	}
	
	if ( !mpPlugin ) return DIA_FAILED;
	
	//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
	
	std::string strategy = "dia_SubsystemLogoInstallationStrategyDefault";
	dia_SubsystemLogoInstallationStrategyDefault* pStrategy = new dia_SubsystemLogoInstallationStrategyDefault(strategy);
	
	if ( !pStrategy ) return DIA_FAILED;
	
	if( pStrategy->setup() != DIA_SUCCESS )
	{
		DIA_TR_ERR("Failed to setup dia_SubsystemLogoInstallationStrategyDefault (ADDR=0x%p)", pStrategy);
		retCode = DIA_FAILED;
	}
	else
	{
		mpActiveStrategy = pStrategy;
		DIA_TR_INF("Current Active Logo Installation Strategy : %s ", strategy.c_str());
	}
	
	//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
	
	dia_SubsystemLogoInstallationSecurityAccessDefault* pAlgorithm = new dia_SubsystemLogoInstallationSecurityAccessDefault 
																			(
																				DIA_C_SUBSYSTEM_SECURITY_ACCESS_LEVEL_CIS_LOGO_INST, 
																				DIA_C_SUBSYSTEM_SECURITY_ACCESS_LEVEL_DOIP_SEED_LEN, 
																				DIA_C_SUBSYSTEM_SECURITY_ACCESS_LEVEL_DOIP_KEY_LEN 
																			);
	
	if ( !pAlgorithm ) return DIA_FAILED;
	
	mpActiveSecurityAlgorithm = pAlgorithm;
	
	//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
	
	dia_DatasetConfigLogo cfg
	(  
		0,
		"CISHMILogo1",  
		CIS_HMI_LOGO_START,  
		CIS_HMI_LOGO_END,  
		CIS_HMI_LOGO_LEN,  
		UPDOWNLOADABLE_NO_CRC_CHECK|DIA_C_U16_DATASET_CFG_VARIABLE_LENGTH, 
		DIA_PROP_CMC_CIS_HMI_LOGO_1
	);
	
	
	mpActiveDownloader = new dia_SubsystemLogoInstallationDataDownload(cfg);
	
	if(!mpActiveDownloader) return DIA_FAILED;
	
	mpActiveStrategy->setPlugin(*mpPlugin);
	mpActiveStrategy->setSecurityAlgorithm(*mpActiveSecurityAlgorithm);
	mpActiveStrategy->setDownloader(*mpActiveDownloader);
	
	mCurrentLogoId = 0;
	
	return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SubsystemLogoInstallationManager::tearDown ( void ) const
{
   dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationManager::tearDown");
   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

void
dia_SubsystemLogoInstallationManager::setPlugin ( const dia_SubsystemLogoInstallationManagerPlugin& plugin )
{
   dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationManager::setPlugin()");

   // delete the existing plugin
   if ( mpPlugin )
   {
      DIA_TR_INF("Subsystem Plugin to be removed: %p", mpPlugin);
      delete mpPlugin;
   }

   mpPlugin = &plugin;
   
   DIA_TR_INF("Subsystem Plugin installed: %p", mpPlugin);
   
   if ( mpActiveStrategy ) mpActiveStrategy->setPlugin(*mpPlugin);

}

//-----------------------------------------------------------------------------

void
dia_SubsystemLogoInstallationManager::setStrategy ( SubsystemLogoInstallationStrategy& strategy )
{
   dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationManager::setStrategy()");

   // delete the existing strategy
   if ( mpActiveStrategy )
   {
      DIA_TR_INF("Subsystem Strategy to be removed: %p", mpActiveStrategy);
      delete mpActiveStrategy;
   }

   mpActiveStrategy = &strategy;
   
   DIA_TR_INF("Subsystem Strategy installed: %p", mpActiveStrategy);
   
   if ( mpActiveStrategy )
   {
		mpActiveStrategy->setPlugin(*mpPlugin);
		mpActiveStrategy->setSecurityAlgorithm(*mpActiveSecurityAlgorithm);
		mpActiveStrategy->setDownloader(*mpActiveDownloader);
   }

}

//-----------------------------------------------------------------------------

void
dia_SubsystemLogoInstallationManager::setSecurityAlgorithm ( SubsystemLogoInstallationSecurityAccess& algorithm )
{
   dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationManager::setSecurityAlgorithm()");

   // delete the existing strategy
   if ( mpActiveSecurityAlgorithm )
   {
      DIA_TR_INF("Security Algorithm to be removed: %p", mpActiveSecurityAlgorithm);
      delete mpActiveSecurityAlgorithm;
   }

   mpActiveSecurityAlgorithm = &algorithm;
   
   DIA_TR_INF("Security Algorithm installed: %p", mpActiveSecurityAlgorithm);
   
   if ( mpActiveStrategy )
   {
		mpActiveStrategy->setSecurityAlgorithm(*mpActiveSecurityAlgorithm);
   }

}

//------------------------------------------------------------------------------

tDiaResult 
dia_SubsystemLogoInstallationManager::setDataDownloadItem(const dia_DatasetConfigLogo cfg )
{
   dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationManager::setDataDownloadItem()");
   
   if( (cfg.mName == NULL) || (cfg.mName[0] == '\0') ) return DIA_FAILED;	   

   // delete the existing downloader
   if ( mpActiveDownloader )
   {
      DIA_TR_INF("Subsystem Logo Downloader to be removed: %p", mpActiveDownloader);
      delete mpActiveDownloader;
   }

   mpActiveDownloader = new dia_SubsystemLogoInstallationDataDownload(cfg);
   
   DIA_TR_INF("Subsystem Logo Downloader installed: %p", mpActiveDownloader);
   
   if ( mpActiveStrategy )
   {
		mpActiveStrategy->setDownloader(*mpActiveDownloader);
		mCurrentLogoId = cfg.mLogoId;	
   }
   
   return DIA_SUCCESS;

}

//-----------------------------------------------------------------------------

void
dia_SubsystemLogoInstallationManager::destroy ( void )
{
   mDatasetcfgRep.clear();
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SubsystemLogoInstallationManager::addDatasetCfg ( dia_DatasetConfigLogo* pDatasetCfg )
{
   dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationManager::addDatasetCfg()");

   if ( !pDatasetCfg ) return DIA_E_INVALID_POINTER;

   tDiaResult retCode = DIA_FAILED;

   std::map<tU8,dia_DatasetConfigLogo*>::iterator iter = mDatasetcfgRep.find(pDatasetCfg->mLogoId);
   
   if ( iter == mDatasetcfgRep.end() )
   {
      DIA_TR_INF("#######################################################");
      DIA_TR_INF("#");
      DIA_TR_INF("# ADDING DATASET CFG \"%s\"", pDatasetCfg->mName );
      DIA_TR_INF("#");
      DIA_TR_INF("#######################################################");
      mDatasetcfgRep[pDatasetCfg->mLogoId] = pDatasetCfg;
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_SubsystemLogoInstallationManager::removeDatasetCfg ( tU8 LogoId )
{
   dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationManager::removeDatasetCfg()");

   tDiaResult retCode = DIA_E_NOT_FOUND;

   std::map<tU8,dia_DatasetConfigLogo*>::iterator iter = mDatasetcfgRep.find(LogoId);
   if ( iter != mDatasetcfgRep.end() )
   {
      mDatasetcfgRep.erase(LogoId);
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_SubsystemLogoInstallationManager::queryDatasetCfg ( tU8 LogoId, dia_DatasetConfigLogo** ppDatasetCfg )
{
   dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationManager::queryDatasetCfg()");

   tDiaResult retCode = DIA_E_NOT_FOUND;

   DIA_TR_INF("dia_SubsystemLogoInstallationManager::queryDatasetCfg startAddr=0x%08X", LogoId);

   if ( !ppDatasetCfg ) return DIA_E_INVALID_POINTER;

   std::map<tU8,dia_DatasetConfigLogo*>::iterator iter = mDatasetcfgRep.find(LogoId);
   if ( iter != mDatasetcfgRep.end() )
   {
      (*ppDatasetCfg) = iter->second;
      retCode = DIA_SUCCESS;
      DIA_TR_INF("dia::UploadDownloadStrategyDatasets::Data set cfg found");
   }
   else
   {
      (*ppDatasetCfg) = 0;
      DIA_TR_ERR("dia::UploadDownloadStrategyDatasets::Data set cfg NOT found");
   }

   return retCode;
}

//------------------------------------------------------------------------------

tU16
dia_SubsystemLogoInstallationManager::numberOfDataSetCfgs ( void ) const
{
   return (tU16) mDatasetcfgRep.size();
}

//------------------------------------------------------------------------------

tDiaResult
dia_SubsystemLogoInstallationManager::loadDatasetCfgs ( dia_DatasetConfigLogo data[], tU16 numOfItems )
{
   dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationManager::loadDatasetCfgs");

   tDiaResult retCode = DIA_SUCCESS;

   if ( data )
   {
      for ( tU16 i=0; i<numOfItems; ++i )
      {
		 dia_DatasetConfigLogo* pDataSetCfg(&data[i]);
         mDatasetcfgRep[pDataSetCfg->mLogoId] = pDataSetCfg;
         DIA_TR_INF("loadDatasetCfgs LogoID = 0x%04x Name = %s", pDataSetCfg->mLogoId, pDataSetCfg->mName );
      }
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_SubsystemLogoInstallationManager::startLogoInstallation ( tU8 logoId )
{
   dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationManager::startLogoInstallation( tU8 logoId )");

   if ( !mpActiveStrategy ) return DIA_FAILED;
   
   if( logoId != mCurrentLogoId )
   {
	   dia_DatasetConfigLogo* pDataSetCfg;
	   if(DIA_SUCCESS == queryDatasetCfg(logoId, &pDataSetCfg))
	   {
		  dia_DatasetConfigLogo Cfg = *pDataSetCfg;
		  if(DIA_SUCCESS != setDataDownloadItem(Cfg))
		  {
				return DIA_FAILED;
		  }
	   }
   }
	   
   return mpActiveStrategy->startLogoInstallation();
}

//------------------------------------------------------------------------------

tDiaResult
dia_SubsystemLogoInstallationManager::requestLogoInstallationResults ( void )
{
   dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationManager::requestLogoInstallationResults()");

   if ( !mpActiveStrategy ) return DIA_FAILED;
   
   return mpActiveStrategy->requestLogoInstallationResults();
}

//-----------------------------------------------------------------------------
