/*
 * dia_SrvHandler_USBDeviceInfo.cpp
 *
 *  Created on: 24.07.2012
 *      Author: sta8hi
 */
// TTFis: DIA_REQ UDS 04 22 63 20

#include "dia_SrvHandler_USBDeviceInfo.h"

#ifndef __INCLUDED_DIA_DEFINES_UDS__
#include <common/framework/protocols/uds/dia_defsUds.h>
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include "common/framework/config/dia_ConfigManager.h"
#endif

#ifndef __INCLUDED_DIA_MESSAGE_BUFFER__
#include "common/framework/engine/dia_MessageBuffer.h"
#endif

#ifndef __INCLUDED_DIA_INTERFACE_DEVICE_MANAGER_LISTENER__
#include "common/interfaces/dia_IDeviceMGRListener.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif


#define DATA_START 3
#define STRUCT_SIZE 11


#define DIA_BIT_MASK_USB_DEVICE_INFO ((tU8)  0x01)
#define DIA_BIT_MASK_USB_PORT_STATES ((tU8)  0x02)
#define DIA_BIT_MASK_USB_ALL         ((tU8)(DIA_BIT_MASK_USB_DEVICE_INFO | DIA_BIT_MASK_USB_PORT_STATES))


//defined according to Gen3g_CMDS_1v11.cdd
static const char* connStatus2Txt[] =
{
   "undefined",
   "warning",
   "connected",
   "removed",
   "unavailable BAT low voltage",
   "HW malfunction"
};

//defined according to Gen3g_CMDS_1v11.cdd
static const char* devType2Txt[] =
{
   "unknown",  //0
   "USB",      //1
   "SD card",  //2
   "iPod",     //3
   "iPhone",   //4
   "MTP",      //5
   "MSZune"    //6
};

//-----------------------------------------------------------------------------

dia_SrvHandler_USBDeviceInfo::dia_SrvHandler_USBDeviceInfo ( void )
    : dia_ServiceHandlerUDS( "dia_SrvHandler_USBDeviceInfo", DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, DIA_C_U16_DID_RBCM_USB_DEVICES_INFO/*DID*/ ),
      mPropStatusBitMask(0)
{
   dia_tclFnctTrace trc( "dia_SrvHandler_USBDeviceInfo::dia_SrvHandler_USBDeviceInfo(void)" );

   for (tU8 i=0; i<DIA_MAX_NUM_USB_DEVICES; i++)
   {
      moUSBDeviceInfo[i] = dia_tDeviceInfo();
      mbRetCorrectDevices[i] = FALSE;
   }

   for (tU8 i=0; i<DIA_MAX_USB_PORT_NUMBER; i++)
   {
      moUSBPortStates[i] = dia_tUsbPortStates();
      mbRetCorrectPortStates[i] = FALSE;
   }
}

//-----------------------------------------------------------------------------

dia_SrvHandler_USBDeviceInfo::~dia_SrvHandler_USBDeviceInfo( void )
{
    _BP_TRY_BEGIN
    {
       (void) unsetSysAdapterListener<dia_IDeviceMGRListener>(this);
    }
    _BP_CATCH_ALL
    {
        DIA_TR_ERR( "EXCEPTION CAUGHT: dia_SrvHandler_USBDeviceInfo::~dia_SrvHandler_USBDeviceInfo !!!" );
        DIA_ASSERT_ALWAYS();
    }
    _BP_CATCH_END
}

//-------------------------------------------------------------------------------

void
dia_SrvHandler_USBDeviceInfo::vProcessRequest ( const std::vector<tArgsType>& /*vecArgs*/ )
{
   dia_tclFnctTrace trc( "dia_SrvHandler_USBDeviceInfo::vProcessRequest" );

   tBool errorDetected = TRUE;

   //reset all values for all devices
   for (tU8 i=0; i<DIA_MAX_NUM_USB_DEVICES; i++)
   {
      moUSBDeviceInfo[i] = dia_tDeviceInfo();
      mbRetCorrectDevices[i] = FALSE;
   }

   for (tU8 i=0; i<DIA_MAX_USB_PORT_NUMBER; i++)
   {
      moUSBPortStates[i] = dia_tUsbPortStates();
      mbRetCorrectPortStates[i] = FALSE;
   }

   dia_IDeviceMGR* pIDeviceMGR = NULL;
   if( querySysAdapterInterface<dia_IDeviceMGR>(&pIDeviceMGR) == DIA_SUCCESS )
   {
      DIA_TR_INF( "dia_SrvHandler_USBDeviceInfo::vProcessRequest - querySysAdapterInterface SUCCESS!" );

      if( pIDeviceMGR )
      {
         DIA_TR_INF( "dia_SrvHandler_USBDeviceInfo::vProcessRequest - pDeviceMGR NOT NULL" );

         (void) setSysAdapterListener<dia_IDeviceMGRListener>(this);
         if ( pIDeviceMGR->getDevicesValues() == DIA_SUCCESS )
         {
            errorDetected = FALSE;
            DIA_TR_INF( "dia_SrvHandler_USBDeviceInfo::vProcessRequest - getDevicesValues SUCCESS!" );
         }
         else
         {
            DIA_TR_INF( "dia_SrvHandler_USBDeviceInfo::vProcessRequest - getDevicesValues FAILED!" );
         }
      }
   }
   else
   {
      DIA_TR_INF( "dia_SrvHandler_USBDeviceInfo::vProcessRequest - querySysAdapterInterface FAILED!" );
   }

   if( errorDetected )
   {
      DIA_TR_ERR( "dia_SrvHandler_USBDeviceInfo: ---  SEND TO Media Manager Component failed!!!!" );
      (void) unsetSysAdapterListener<dia_IDeviceMGRListener>(this);
      oDiagMsgBuffer().vSetNegResp( DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT );
      vResReadyAndQuit();
   }
}

//-----------------------------------------------------------------------------

void
dia_SrvHandler_USBDeviceInfo::vOnUSBDeviceInfo( const std::vector<dia_tDeviceInfo>& DeviceInfoList )
{
   dia_tclFnctTrace trc( "dia_SrvHandler_USBDeviceInfo::vOnUSBDeviceInfo" );

   DIA_TR_INF( "dia_SrvHandler_USBDeviceInfo::vOnUSBDeviceInfo: DeviceInfoList.size()=%zu", DeviceInfoList.size() );

   if( !DeviceInfoList.empty() )
   {
      std::vector<dia_tDeviceInfo>::const_iterator  it = DeviceInfoList.begin();

      /* Only three USB devices could be printed. So take the latest three USB devices. */
      tU8 numDevToBeIgnored = static_cast<tU8>((DeviceInfoList.size() > DIA_MAX_NUM_USB_DEVICES) ? (DeviceInfoList.size() - DIA_MAX_NUM_USB_DEVICES): (0) );

      DIA_TR_INF( "dia_SrvHandler_USBDeviceInfo::vOnUSBDeviceInfo: numDevToBeIgnored=%d", numDevToBeIgnored );

      for (tU8 i=0, ii=0; it != DeviceInfoList.end(); ++it, ++i)
      {
         if (i<numDevToBeIgnored)
         {
            DIA_TR_INF( "dia_SrvHandler_USBDeviceInfo::vOnUSBDeviceInfo: Device ignored i=%d", i );
         }
         else
         {
            // Should be filtered already in dia_SAFeatureDeviceMGR::vHandleDeviceState(...)
            if(   ( (*it).DeviceType !=  DIA_EN_DEVMGR_DEVICE_TYPE_CDROM )
               && ( (*it).DeviceType !=  DIA_EN_DEVMGR_DEVICE_TYPE_SDCARD )
              )
            {
               //make a copy
               if (ii<DIA_MAX_NUM_USB_DEVICES)
               {
                  DIA_TR_INF( "dia_SrvHandler_USBDeviceInfo::vOnUSBDeviceInfo: Make a copy for index %d", ii );
                  moUSBDeviceInfo[ii] = (*it);
                  mbRetCorrectDevices[ii] = TRUE;
                  ii++;
               }
               else
               {
                  DIA_TR_INF( "dia_SrvHandler_USBDeviceInfo::vOnUSBDeviceInfo: ERROR. This code must be inactive. ii=%d", ii );
               }
            }
            else
            {
               // !!!! Not a USB device !!! -> check filter in dia_SAFeatureDeviceMGR::vHandleDeviceState(...)
               DIA_TR_INF( "dia_SrvHandler_USBDeviceInfo::vOnUSBDeviceInfo: ERROR: !!!! Device is not a USB device !!! -> check filter in dia_SAFeatureDeviceMGR::vHandleDeviceState(...)" );
               DIA_ASSERT_ALWAYS();
            }
         }
      }
   }  // if( !deviceInfo.empty() )

   mPropStatusBitMask |= DIA_BIT_MASK_USB_DEVICE_INFO;

   if ( mPropStatusBitMask == DIA_BIT_MASK_USB_ALL )
   {
      DIA_TR_INF( "dia_SrvHandler_USBDeviceInfo::vOnUSBDeviceInfo: Call assembleResponse()" );
      assembleResponse();
      mPropStatusBitMask = 0;
   }
}

void
dia_SrvHandler_USBDeviceInfo::vOnUSBPortStates( const std::vector<dia_tUsbPortStates>& PortStatesList )
{
   dia_tclFnctTrace trc( "dia_SrvHandler_USBDeviceInfo::vOnUSBPortStates" );

   DIA_TR_INF( "dia_SrvHandler_USBDeviceInfo::vOnUSBDeviceInfo: PortStatesList.size()=%zu", PortStatesList.size() );

   if( !PortStatesList.empty() )
   {
      std::vector<dia_tUsbPortStates>::const_iterator it = PortStatesList.begin();

      for(tU8 i=0; it != PortStatesList.end(); ++it, ++i )
      {
         tU32 portNo = (*it).PortNo;

         if ( (0<portNo) && (portNo<DIA_MAX_USB_PORT_NUMBER+1) )
         {
            DIA_TR_INF( "dia_SrvHandler_USBDeviceInfo::vOnUSBPortStates: Make copy for port status i=%d portNo=%d", i, portNo);

            //make a copy of USB port status
            moUSBPortStates[portNo - 1] = (*it);
            mbRetCorrectPortStates[portNo - 1] = TRUE;
         }
         else
         {
            DIA_TR_INF( "dia_SrvHandler_USBDeviceInfo::vOnUSBPortStates POSSIBLE ERROR. Port status ignored i=%d portNo=%d", i, portNo);
         }
      }
   }

   mPropStatusBitMask |= DIA_BIT_MASK_USB_PORT_STATES;

   if ( mPropStatusBitMask == DIA_BIT_MASK_USB_ALL )
   {
      DIA_TR_INF( "dia_SrvHandler_USBDeviceInfo::vOnUSBPortStates: Call assembleResponse()" );
      assembleResponse();
      mPropStatusBitMask = 0;
   }
}


//-------------------------------------------------------------------------------------------------------------------

void dia_SrvHandler_USBDeviceInfo::assembleResponse ()
{
   tU8 u8Data;

   dia_tclFnctTrace trc( "dia_SrvHandler_USBDeviceInfo::assembleResponse" );

   // Sending back the positive answer
   oDiagMsgBuffer().vSetPosResp();
   oDiagMsgBuffer().vSetDataLength( DATA_START+ (STRUCT_SIZE*DIA_MAX_NUM_USB_DEVICES) );

   for( tU16 i=0; i<(STRUCT_SIZE*DIA_MAX_NUM_USB_DEVICES); i++ )
   {
      oDiagMsgBuffer().vSetDataU8(static_cast<tU16>(DATA_START+i),0x00);
   }

   for (tU8 i=0; i<DIA_MAX_NUM_USB_DEVICES; i++)
   {
      if (0!=i)
      {
         DIA_TR_INF( "---------------------------------");
      }

      /* byte#0 to byte#9 */
      if( mbRetCorrectDevices[i] )
      {
         DIA_TR_INF( "[%d] USB device detected", i);
         tU8 deviceType;
         tU8 connectionStatus;

         moUSBDeviceInfo[i].Size >>= (10 + 10);   // convert from Byte to MB (2^10 = 1024)

         (void) oDiagMsgBuffer().vSetDataU8( static_cast<tU16>(DATA_START + i*STRUCT_SIZE + 0),(tU8) (((moUSBDeviceInfo[i].VendorID) >>8) & 0xFFU) );  //big endian
         (void) oDiagMsgBuffer().vSetDataU8( static_cast<tU16>(DATA_START + i*STRUCT_SIZE + 1),(tU8) (((moUSBDeviceInfo[i].VendorID) >>0) & 0xFFU) );
         (void) oDiagMsgBuffer().vSetDataU8( static_cast<tU16>(DATA_START + i*STRUCT_SIZE + 2),(tU8) (((moUSBDeviceInfo[i].ProductID)>>8) & 0xFFU) );  //big endian
         (void) oDiagMsgBuffer().vSetDataU8( static_cast<tU16>(DATA_START + i*STRUCT_SIZE + 3),(tU8) (((moUSBDeviceInfo[i].ProductID)>>0) & 0xFFU) );
         (void) oDiagMsgBuffer().vSetDataU8( static_cast<tU16>(DATA_START + i*STRUCT_SIZE + 4),(tU8) (((moUSBDeviceInfo[i].Size)>>24) & 0xFFU) );      //big endian
         (void) oDiagMsgBuffer().vSetDataU8( static_cast<tU16>(DATA_START + i*STRUCT_SIZE + 5),(tU8) (((moUSBDeviceInfo[i].Size)>>16) & 0xFFU) );
         (void) oDiagMsgBuffer().vSetDataU8( static_cast<tU16>(DATA_START + i*STRUCT_SIZE + 6),(tU8) (((moUSBDeviceInfo[i].Size)>>8)  & 0xFFU) );
         (void) oDiagMsgBuffer().vSetDataU8( static_cast<tU16>(DATA_START + i*STRUCT_SIZE + 7),(tU8) (((moUSBDeviceInfo[i].Size)>>0)  & 0xFFU) );

         // Byte 8: device type -> CMUSBDeviceType
         switch( moUSBDeviceInfo[i].DeviceType )
         {
            case DIA_EN_DEVMGR_DEVICE_TYPE_USB_MS:
               deviceType = 0x01;                                  // USB mass storage
               break;
            case DIA_EN_DEVMGR_DEVICE_TYPE_USB_IPOD:
               deviceType = 0x03;                                  // iPod, iPhone, iPad
               break;
            case DIA_EN_DEVMGR_DEVICE_TYPE_USB_MTP:
               deviceType = 0x05;                                  // USB MTP devices
               break;
            case DIA_EN_DEVMGR_DEVICE_TYPE_USB_MSZUNE:
               deviceType = 0x06;                                  // MicroSoft ZUNE device
               break;
            case DIA_EN_DEVMGR_DEVICE_TYPE_USB_NOT_SUPPORTED:
            case DIA_EN_DEVMGR_DEVICE_TYPE_UNKNOWN:
            default:
               deviceType = 0x00;                                  // unknown device, not supported device, default
               break;
         }
         (void) oDiagMsgBuffer().vSetDataU8( static_cast<tU16>(DATA_START + i*STRUCT_SIZE + 8), deviceType ); // Set CMUSBDeviceType for TEF (production diagnosis)

         // Byte 9: connection type -> CMUSBConnectionStatusType
         switch( moUSBDeviceInfo[i].ConnectionState )
         {
            case DIA_EN_DEVMGR_CONNECT_STATE_CONNECTED:
               connectionStatus = 0x02;                                     // USB device connected
               break;
            case DIA_EN_DEVMGR_CONNECT_STATE_REMOVED:
               connectionStatus = 0x03;                                     // USB device removed
               break;
            case DIA_EN_DEVMGR_CONNECT_STATE_UNAVAIL_BAT_LOWVOLT:
               connectionStatus = 0x04;                                     // USB port power supply in low voltage
               break;
            case DIA_EN_DEVMGR_CONNECT_STATE_UNAVAIL_HW_MALFUNC:
               connectionStatus = 0x05;                                     // e.g. short to ground
               break;
            case DIA_EN_DEVMGR_CONNECT_STATE_UNKNOWN:
            default:
               connectionStatus = 0x00;                                     // unknown connection state
               break;
         }
         (void) oDiagMsgBuffer().vSetDataU8( static_cast<tU16>(DATA_START + i*STRUCT_SIZE + 9), connectionStatus ); // Set CMUSBConnectionStatusType for TEF (production diagnosis)

         DIA_TR_INF( "[%d] VendorID            = 0x%04x",         i, moUSBDeviceInfo[i].VendorID );
         DIA_TR_INF( "[%d] ProductID           = 0x%04x",         i, moUSBDeviceInfo[i].ProductID );
         DIA_TR_INF( "[%d] DeviceConnectStatus = %d => %d (%s)",  i, (tU8)moUSBDeviceInfo[i].ConnectionState, connectionStatus, connStatus2Txt[connectionStatus] );
         DIA_TR_INF( "[%d] DeviceType          = %d => %d (%s)",  i, (tU8)moUSBDeviceInfo[i].DeviceType, deviceType, devType2Txt[deviceType] );
         DIA_TR_INF( "[%d] TotalSize [MB]      = %d",             i, (tU32)(moUSBDeviceInfo[i].Size) );
         DIA_TR_INF( "[%d] TotalSize [MB]      = %d",             i, (tU32)(moUSBDeviceInfo[i].Size) );
      }
      else // No USB device list -> no USB device connected -> fill in default values
      {
         DIA_TR_INF( "dia_SrvHandler_USBDeviceInfo::assembleResponse: No USB device in list [%d]. Fill in default values", i);

         /* bytes #0 to #8 has been zeroed before */

         // byte#9: No USB device in list. Therefore try to get the power state via PortStates
         if( (i<DIA_MAX_USB_PORT_NUMBER) && (TRUE==mbRetCorrectPortStates[i]) )
         {
            DIA_TR_INF( "dia_SrvHandler_USBDeviceInfo::assembleResponse: Fill ConnectionStatus via PowerState information [%d]", i);
            if( moUSBPortStates[i].UnderVoltage )
            {
               u8Data = 0x04;                                                 // USB port power supply in low voltage
            }
            else  // default
            {
               u8Data = 0x03;                                                 // USB device removed
            }

            (void) oDiagMsgBuffer().vSetDataU8( static_cast<tU16>(DATA_START + i*STRUCT_SIZE + 9), u8Data );    // CMUSBConnectionStatusType for TEF (production diagnosis)
         }
         else
         {
            DIA_TR_INF( "dia_SrvHandler_USBDeviceInfo::assembleResponse: No valid PowerState [%d]: Set ConnectionStatus to default.", i);
            // Also the PortState available -> set to default
            (void) oDiagMsgBuffer().vSetDataU8( static_cast<tU16>(DATA_START + i*STRUCT_SIZE + 9), 0x03 );      // default for CMUSBConnectionStatusType for TEF (production diagnosis) -> device removed
         }
      }

      /* byte#10 */
      tU8 ii = 0xFF;  //index for moUSBPortStates

      if (mbRetCorrectDevices[i])
      {
         ii = static_cast<tU8>(moUSBDeviceInfo[i].PortNo - 1);
         DIA_TR_INF( "[%d] Device available [port=%d].", i, ii );
      }
      else if (i<DIA_MAX_USB_PORT_NUMBER)
      {
         /* No USB device in list. Therefore try to get the power state via PortStates */
         ii = i;
         DIA_TR_INF( "[%d] Device not available [port=%d]", i, ii );
      }

      if( (ii < DIA_MAX_USB_PORT_NUMBER) && (TRUE==mbRetCorrectPortStates[ii]) )
      {
         DIA_TR_INF( "[%d] PortStates detected [%d]", i, ii );

         if( !moUSBPortStates[ii].UnderVoltage && !moUSBPortStates[ii].OverCurrent )
         {
            u8Data = 0x01;                                                    // on / OK
         }
         else if( moUSBPortStates[ii].UnderVoltage )                          // check for low voltage
         {
            u8Data = 0x03;                                                    // off / disabled due to low voltage
         }
         else                                                                 // no low voltage -> OverCurrent
         {
            u8Data = 0x04;                                                    // unknown / power fail
         }
         (void) oDiagMsgBuffer().vSetDataU8( static_cast<tU16>(DATA_START + i*STRUCT_SIZE + 10), u8Data );       // CMUSBPowerSupplyStatusType for TEF (production diagnosis)
         DIA_TR_INF( "[%d] PowerSupplyStatus: %d", i, u8Data );
      }
      else
      {
         // Also the PortState isn't available - zero has been set already, do nothing
         // default for CMUSBPowerSupplyStatusType for TEF (production diagnosis) -> unknown
         DIA_TR_INF( "[%d] No valid PowerState: Set PowerSupplyStatus to default.", i );
      }
   }

   (void) unsetSysAdapterListener<dia_IDeviceMGRListener>(this);
   vResReadyAndQuit();
}
