/*!
 * \file       dia_SrvHandler_ReadAnonymousVIN.cpp
 *
 * \brief      Read Anonymous part of VIN, WMI(World Manufacturer Identifier) and VDS(Vehicle Descriptor Section) only
 *
 * \component  Diagnostics
 *
 * \ingroup    diaServicesCommon
 *
 * \copyright  (c) 2018 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

// TTFis: DIA_REQ UDS 03 22 C0 09

#ifndef __INCLUDED_DIA_SRVHANDLER_READ_ANONYMOUSVIN_H__
#include "dia_SrvHandler_ReadAnonymousVIN.h"
#endif

#ifndef __INCLUDED_DIA_COMMON_CONFIG__
#include "common/framework/config/dia_common_config.h"
#endif

#define DATA_SIZE 3 //SID + DID
#define RESP_DATA_SIZE 9 //WMI (3 characters) + VDS (6 characters)
using namespace dia;

//-----------------------------------------------------------------------------

dia_SrvHandler_ReadAnonymousVIN::dia_SrvHandler_ReadAnonymousVIN ( tCString name, tU8 sid, tU16 did )
   : dia_ServiceHandlerUDS(name,sid,did)
{
   dia_tclFnctTrace trc("dia_SrvHandler_ReadAnonymousVIN::dia_SrvHandler_ReadAnonymousVIN(tCString,tU8,tU16)");
}

//-----------------------------------------------------------------------------

dia_SrvHandler_ReadAnonymousVIN::~dia_SrvHandler_ReadAnonymousVIN ( void )
{}

//-----------------------------------------------------------------------------

void
dia_SrvHandler_ReadAnonymousVIN::vProcessRequest ( const std::vector<tArgsType>& /*vecArgs*/ )
{
   dia_tclFnctTrace trc("dia_SrvHandler_ReadAnonymousVIN::vProcessRequest");

   // data length contains at least sid, ctrl byte and DID --> 4 bytes minimum
   tU16 dataLength = oDiagMsgBuffer().u16GetDataLength();
   if ( dataLength !=  (DATA_SIZE + 1) ) // one additional byte contains the message length
   {
      DIA_TR_INF("dia_SrvHandler_ReadAnonymousVIN::vProcessRequest INVALID LENGTH. EXPECTED %d B BUT RECEIVED %d B !", (DATA_SIZE+1), dataLength);
      vSendNegativeResponse(DIA_E_U8_UDS_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT);
      return;
   }

   // data memory
   std::vector<tU8> dataVec;

   // retrieve size of property
   auto length = dia_getPropertySize(DIA_PROP_VEHICLE_IDENTIFICATION_NUMBER);
   dataVec.reserve(length);
   dataVec.resize(length);

   DIA_TR_INF("dia_SrvHandler_ReadAnonymousVIN::vProcessRequest - Property length %zu, dataVec.size() %zu (cleaned).", length, dataVec.size());

   tDiaResult retCode = dia_getProperty(DIA_PROP_VEHICLE_IDENTIFICATION_NUMBER,dataVec);
   if ( retCode == DIA_SUCCESS )
   {
      DIA_TR_INF("dia_SrvHandler_ReadAnonymousVIN::vProcessRequest - Property length %zu, dataVec.size() %zu (new data).", length, dataVec.size());

      // Sending back the positive answer
      oDiagMsgBuffer().vSetPosResp();
      oDiagMsgBuffer().vSetDataLength(DATA_SIZE + RESP_DATA_SIZE);
      for (tU16 i = 0; i < RESP_DATA_SIZE; ++i)
      {
         (void) oDiagMsgBuffer().vSetDataU8((tU16) (DATA_SIZE+i), dataVec[i]);
      }

      vResReadyAndQuit();
   }
   else
   {
      DIA_TR_ERR("dia_SrvHandler_ReadAnonymousVIN::vProcessRequest UNABLE TO READ ITEM (READ OPERATION FAILED)");
      // Sending back the Negative answer
      vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
   }
}
