/*!
 * \file       dia_SrvHandler_ECUUniqueFieldDataId.cpp
 *
 * \brief      Read Unique FieldData ID or generate it if not existing
 *
 * \component  Diagnostics
 *
 * \ingroup    diaServicesCommon
 *
 * \copyright  (c) 2018 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */
#ifndef __INCLUDED_DIA_ECUUNIQUEFIELDDATAID_H__
#include "dia_SrvHandler_ECUUniqueFieldDataId.h"
#endif

#ifndef __INCLUDED_DIA_FACTORY__
#include <common/framework/application/dia_Factory.h>
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG_PROJECT__
#include <project/framework/config/dia_defsProjectConfig.h>
#endif

#ifndef __INCLUDED_DIA_LOOKUPKEY__
#include <common/framework/engine/dia_LookupKey.h>
#endif

#ifndef __INCLUDED_DIA_UTILITIES__
#include "common/framework/utils/dia_utilities.h"
#endif

#ifndef __INCLUDED_DIA_RANDOM_GENERATOR__
#include <common/framework/utils/dia_RandomGenerator.h>
#endif

#include <errno.h>
#include <sys/time.h>

static const uint16_t DID_RESPONSE_DATA_OFFSET = 3;

/**
 * @brief Process incoming diagnostic request for uniqueFieldDataId read
 * @details ServiceHandler tries to read previously generated ID from
 *          persistent memory or generates it if it has not been
 *          generated before
 *
 * @param  vecArgs Unused
 * @return void
 */
void dia_SrvHandler_ECUUniqueFieldDataId::vProcessRequest(const std::vector<tArgsType>& /*vecArgs*/)
{
    dia_tclFnctTrace trc(__PRETTY_FUNCTION__);

    tDiaResult result = DIA_FAILED;

    std::vector<uint8_t> uniqueFieldDataId(20, 0x00);
    result = readUniqueFieldDataId(uniqueFieldDataId);

    if (result != DIA_SUCCESS)
    {
        DIA_TR_INF("Reading Unique Field Data ID failed. Trying to generate.");
        result = generateUniqueFieldDataId(uniqueFieldDataId);
    }

    if (result == DIA_SUCCESS)
    {
        DIA_TR_INF("Unique Field Data ID: %s", dia::utils::bin2hexstr(uniqueFieldDataId).c_str());
        oDiagMsgBuffer().vSetPosResp();
        oDiagMsgBuffer().vSetDataLength((uint16_t)(DID_RESPONSE_DATA_OFFSET + uniqueFieldDataId.size()));
        for (size_t i = 0; i < uniqueFieldDataId.size(); ++i)
        {
            (void)oDiagMsgBuffer().vSetDataU8((uint16_t)(DID_RESPONSE_DATA_OFFSET + i), uniqueFieldDataId[i]);
        }
        vResReadyAndQuit();
    }
    else
    {
        DIA_TR_ERR("Unique Field Data ID NOT available and could not be generated. Send negative response to tester.");
        vSendNegativeResponse(getInstanceOfFactory()->makeNRC(DIA_E_CONDITIONS_NOT_CORRECT));
    }
}

/**
 * @brief Reads the Unique Fielddata ID from DIA_PROP_CM_UNIQUE_FIELD_DATA_ID
 *
 * @pre Fielddata ID needs to be available in DIA_PROP_CM_UNIQUE_FIELD_DATA_ID
 * @post In case of DIA_SUCCESS uniqueFieldDataId contains ID
 *
 * @param[out] uniqueFieldDataId - Storage vector for unique fielddata ID. Filled by function
 * @return DIA_SUCCESS in case ID could be read
 *         Errorcode in case of error
 */
tDiaResult dia_SrvHandler_ECUUniqueFieldDataId::readUniqueFieldDataId(std::vector<uint8_t>& uniqueFieldDataId)
{
    dia_tclFnctTrace trc(__PRETTY_FUNCTION__);
    return dia_getProperty(DIA_PROP_CM_UNIQUE_FIELD_DATA_ID, uniqueFieldDataId);
}

/**
 * @brief Generates the Unique Fielddata ID and stores it into persistent memory
 *
 * @pre ECU Serial Number needs to be available in DIA_PROP_CM_ECU_SERIAL_NR
 * @pre ECU SparePart Number needs to be available in DIA_PROP_CM_ECU_SPARE_PART_NR
 * @pre Timeofday must be available by system interface
 * @post In case of DIA_SUCCESS uniqueFieldDataId contains ID
 * @post In case of DIA_SUCCESS ID stored in DIA_PROP_CM_UNIQUE_FIELD_DATA_ID
 *
 * @param[out] uniqueFieldDataId - Storage vector for unique fielddata ID. Filled by function
 * @return DIA_SUCCESS in case ID could be generated and stored
 *         Errorcode in case of error
 */
tDiaResult dia_SrvHandler_ECUUniqueFieldDataId::generateUniqueFieldDataId(std::vector<uint8_t>& uniqueFieldDataId)
{
    dia_tclFnctTrace trc(__PRETTY_FUNCTION__);
    std::vector<uint8_t> serialNo;
    std::vector<uint8_t> partNo;

    if (DIA_SUCCESS != dia_getProperty(DIA_PROP_CM_ECU_SERIAL_NR, serialNo) || DIA_SUCCESS != dia_getProperty(DIA_PROP_CM_ECU_SPARE_PART_NR, partNo))
    {
        DIA_TR_ERR("Serial Number or Part Number not available. Cannot generate unique ID.");
        return DIA_E_NOT_FOUND;
    }

    dia_RandomGenerator* generator = getInstanceOfRandomGenerator();
    uint32_t randval               = generator->getRandomNumber();

    struct timeval tv;
    if (0 != gettimeofday(&tv, NULL))
    {
        DIA_TR_ERR("GetTimeOfDay failed with error %d. Cannot generate unique ID.", errno);
        return DIA_E_NOT_SUPPORTED;
    }

    std::vector<uint8_t> data;
    data.insert(data.end(), partNo.begin(), partNo.end());
    data.insert(data.end(), serialNo.begin(), serialNo.end());
    data.push_back(U32_BYTE1(randval));
    data.push_back(U32_BYTE2(randval));
    data.push_back(U32_BYTE3(randval));
    data.push_back(U32_BYTE4(randval));
    data.push_back(U32_BYTE1(tv.tv_usec));
    data.push_back(U32_BYTE2(tv.tv_usec));
    data.push_back(U32_BYTE3(tv.tv_usec));
    data.push_back(U32_BYTE4(tv.tv_usec));

    (void)dia_nsUtilities::generateHash(data, uniqueFieldDataId);

    return dia_setProperty(DIA_PROP_CM_UNIQUE_FIELD_DATA_ID, uniqueFieldDataId);
}

/**
 * @brief Generates and stores lookup keys for ServiceHandler
 *
 * @param keys Lookup-key storage to be extended by function
 * @return DIA_SUCCESS
 */
tDiaResult dia_SrvHandler_ECUUniqueFieldDataId::makeLookupKeys(std::vector<dia_LookupKey*>& keys)
{
#ifdef __DIA_UNIT_TESTING__
    dia_tclFnctTrace trc(__PRETTY_FUNCTION__);
#endif
    keys.push_back(new dia_LookupKey(DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, (tU16)DIA_C_U16_DID_RBCM_ECU_UNIQUE_FIELDDATA_ID,
                                     DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED));
    return DIA_SUCCESS;
}
