/**
 * \file      dia_RtCtrlGetWifiSNR.cpp
 *
 * \brief     Routine Control for SNR of Wifi signal for specified Access Point
 *
 * \details   {insert file description here}
 *
 * \author    TMS Plischke, kaa1hi
 * \date      Aug 4, 2014
 *
 * \copyright Robert Bosch Car Multimedia 2014
 */
 // DIA_REQ UDS 09 31 01 31 00 AA BB CC DD EE FF AA
 //                            ^^^^^^^^^^^^^^^^^^^^
 //                            MAC ADDRESS OF ACCESS POINT

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#ifndef __INCLUDED_DIA_COMMON_UDS_RTCTRL__
#include <common/framework/protocols/uds/rtctrl/dia_common_uds_rtctrl.h>
#endif

#ifndef __INCLUDED_DIA_ERROR_INFO_INTERFACE__
#include <common/framework/errmgmt/dia_ErrorInfoInterface.h>
#endif

#include "dia_RtCtrlGetWifiSNR.h"

#define DIA_SIZE_OF_MAC_ADDRESS (6U)

namespace dia
{

//------------------------------------------------------------------------------

RtCtrlGetWifiSNR::RtCtrlGetWifiSNR()
         : dia_Routine("dia_RtCtrlGetWifiSNR", DIA_C_U16_DID_RBCM_RTCTRL_GET_WIFI_SNR,
               DIA_EN_RTCTRL_ID_GET_WIFI_SNR,
               DIA_EN_RTCTRL_TYPE_SHORT_TERM)
{
   dia_tclFnctTrace oTrace("dia_RtCtrlGetWifiSNR::dia_RtCtrlGetWifiSNR()");
}

//------------------------------------------------------------------------------

RtCtrlGetWifiSNR::~RtCtrlGetWifiSNR()
{
   _BP_TRY_BEGIN
   {
      (void) unsetSysAdapterListener<dia_IWifiListener>(this);
   }
   _BP_CATCH_ALL
   {
       DIA_TR_ERR("EXCEPTION CAUGHT: dia_RtCtrlGetWifiSNR::~dia_RtCtrlGetWifiSNR !!!");
       DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//------------------------------------------------------------------------------

tDiaResult
RtCtrlGetWifiSNR::start ( std::vector<tU8>& params, tU8 /*timerValue*/ )
{
   dia_tclFnctTrace trc("dia_RtCtrlGetWifiSNR::start");

   tDiaResult retCode = DIA_FAILED;

   // prepare processing of the routine
   vInitialize();

   if (params.size() < DIA_SIZE_OF_MAC_ADDRESS)
   {
      DIA_TR_INF("dia_RtCtrlGetWifiSNR::start ERROR Size of MAC address incorrect.");

      tU8 i = 0;
      for (std::vector<tU8>::iterator it=params.begin(); it!=params.end(); ++it, ++i)
      {
         DIA_TR_INF("dia_RtCtrlGetWifiSNR::start params[%d] = 0x%02X", i, (*it));
      }
      return retCode;
   }

   dia_IWifi* pIWifi = 0;
   if ( querySysAdapterInterface<dia_IWifi>(&pIWifi) == DIA_SUCCESS )
   {
      if ( pIWifi )
      {
         (void) setSysAdapterListener<dia_IWifiListener>(this);

         tU8 au8BDAddress[DIA_SIZE_OF_MAC_ADDRESS] = { 0 };

         if (params.size() >= DIA_SIZE_OF_MAC_ADDRESS)
         {
            for (tU8 i=0; i<DIA_SIZE_OF_MAC_ADDRESS; i++)
            {
               au8BDAddress[i] = params.at(i);
            }
         }

         DIA_TR_INF("dia_RtCtrlGetWifiSNR::start => MAC=%02x:%02x:%02x:%02x:%02x:%02x",
               au8BDAddress[0],au8BDAddress[1],au8BDAddress[2],au8BDAddress[3],au8BDAddress[4],au8BDAddress[5]);

         if ( pIWifi->getWifiSNR(&au8BDAddress[0]) != DIA_SUCCESS)
         {
            DIA_TR_INF("!!! dia_RtCtrlGetWifiSNR::start => ERROR: getWifiSNR != DIA_SUCCESS");
         }
         else
         {
            DIA_TR_INF("--- dia_RtCtrlGetWifiSNR::start => Send Successfully to Server");
            eSetStatus(DIA_EN_RTCTRL_STATUS_IN_PROGRESS);
            retCode = DIA_SUCCESS;
         }
      }
      else
      {
         DIA_TR_INF("dia_RtCtrlGetWifiSNR::start pInterface is NULL");
      }
   }
   else
   {
      DIA_TR_ERR("dia_RtCtrlGetWifiSNR::start querySysAdapterInterface FAILED");
   }

   if ( retCode != DIA_SUCCESS )
   {
      (void) unsetSysAdapterListener<dia_IWifiListener>(this);
   }

   return retCode;
}

//------------------------------------------------------------------------------

void
RtCtrlGetWifiSNR::vOnWifiSNR ( tU8 u8SNR )
{
   dia_tclFnctTrace trc("dia_RtCtrlGetWifiSNR::vOnWifiSNR");

   DIA_TR_INF("dia_RtCtrlGetWifiSNR::vOnWifiSNR SNR=0x%02X",  u8SNR);

   (void) unsetSysAdapterListener<dia_IWifiListener>(this);

   eSetStatus(DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK);

   mResults.push_back(u8SNR); // S/NR

   mIsResultReady = TRUE;

#ifndef __DIA_UNIT_TESTING__
   dia_RoutineCtrlManager* pRTCtrlMgr = getInstanceOfRoutineControlManager();
   if ( pRTCtrlMgr )
   {
      pRTCtrlMgr->vOnRoutineUpdate(*this);
   }
#endif
}

//----------------------------------------------------------------------------------------

void
RtCtrlGetWifiSNR::vOnError (const dia_ErrorInfoInterface& errInfo)
{
   dia_tclFnctTrace trc("dia_RtCtrlGetWifiSNR::vOnError");

   if (errInfo.getExtendedInfoID() == DIA_EN_EXTENDED_RESULT_INFO_ID_INTERFACE)
   {
      if (errInfo.getInterfaceID() == dia_IWifi::getUID())
      {
         DIA_TR_INF("dia_RtCtrlGetWifiSNR::vOnError => dia_IWifiListener: errCode = 0x%02x", errInfo.getErrorCode());

         (void) unsetSysAdapterListener<dia_IWifiListener>(this);

         eSetStatus(DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK);

         mResults.push_back(0); // S/NR

         mIsResultReady = TRUE;

#ifndef __DIA_UNIT_TESTING__
         dia_RoutineCtrlManager* pRTCtrlMgr = getInstanceOfRoutineControlManager();
         if ( pRTCtrlMgr )
         {
            pRTCtrlMgr->vOnRoutineUpdate(*this);
         }
#endif
      }
      else
      {
         DIA_TR_ERR("dia_RtCtrlGetWifiSNR::vOnError => ERROR: Unexpected InterfaceID = %d", errInfo.getInterfaceID());
      }
   }
   else
   {
      DIA_TR_ERR("dia_RtCtrlGetWifiSNR::vOnError => ERROR: Unexpected ExtendedInfoID = %d", errInfo.getExtendedInfoID());
   }
}

//------------------------------------------------------------------------------

tDiaResult
RtCtrlGetWifiSNR::requestResult ( std::vector<tU8>& results )
{
   ScopeTrace oTrace(__PRETTY_FUNCTION__);

   results.push_back(mapStatus2UDSResult());
   results.insert(results.end(), mResults.begin(), mResults.end());

   return DIA_SUCCESS;
}

//------------------------------------------------------------------------------

void
RtCtrlGetWifiSNR::vOnServiceTimeout ( void )
{
   dia_tclFnctTrace oTrace("dia_RtCtrlGetWifiSNR::vOnServiceTimeout");

   (void) unsetSysAdapterListener<dia_IWifiListener>(this);
}

} //namespace dia
