/*!
 * \file       dia_RoutineCtrlTouchSelfTest.h
 *
 * \brief      Start Touch_controller_Selftest_Atmel_T25 and retrieve respective result.
 *
 * \details    The diagnosis tester shall be able to start the Touch_controller_Selftest_Atmel_T25 and retrieve the respective result. Type 1 Routine Control service.
 *             DID : 31 01 30 55 xx (xx: 00/01/02/03)
 *
 * \component  Diagnosis
 *
 * \ingroup    diaInputHandling
 *
 * \copyright  (c) 2015-2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef DIA_ROUTINECTRLTOUCHSELFTEST_H_
#include "dia_RoutineCtrlTouchSelfTest.h"
#endif

#ifndef __INCLUDED_DIA_COMMON_UDS_RTCTRL__
#include "common/framework/protocols/uds/rtctrl/dia_common_uds_rtctrl.h"
#endif

#ifndef __INCLUDED_DIA_DEFINES_UDS__
#include "common/depricated/dia_defsUds.h"
#endif

#ifndef __INCLUDED_DIA_COMMON_CONFIG__
#include "common/framework/config/dia_common_config.h"
#endif

#include <sys/poll.h>
#include <fcntl.h>
#include <errno.h>
#include <linux/input.h>

#define PASS_LGTH 4
#define CMD_SIZE 16
#define BUF_SIZE 32

#define VERDICT_OK      0x00
#define VERDICT_NOK     0x01
#define VERDICT_NA      0x02
#define RT_FAULT        0xFF

//! list of supported self-test commands as per diagnosis spec
#define TOUCH_RUN_ALL_TESTS                    0x00      //0xFE
#define TOUCH_ANALOG_POWER_TEST                0x01      //0x01
#define TOUCH_PIN_FAULT_TEST                   0x02      //0x12
#define TOUCH_SIGNAL_LIMIT_TEST                0x03      //0x17

//! list of supported self-test commands as to be written to file
#define WRITE_TOUCH_RUN_ALL_TESTS             "FE"   //0xFE
#define WRITE_TOUCH_ANALOG_POWER_TEST         "01"   //0x01
#define WRITE_TOUCH_PIN_FAULT_TEST            "12"   //0x12
#define WRITE_TOUCH_SIGNAL_LIMIT_TEST         "17"   //0x17

namespace dia
{

//------------------------------------------------------------------------------

RoutineCtrlTouchSelfTest::RoutineCtrlTouchSelfTest ( void )
   : dia_Routine("dia_RoutineCtrlTouchSelfTest",(tU16) DIA_C_U16_DID_RBCM_RTCTRL_TOUCH_SELFTEST, DIA_EN_RTCTRL_ID_TOUCH_SELFTEST, DIA_EN_RTCTRL_TYPE_SHORT_TERM)
{
   ScopeTrace oTrace("dia_RoutineCtrlTouchSelfTest::dia_RoutineCtrlTouchSelfTest()");

   ::memset(m_TouchDevicePath, 0, PATH_SIZE * sizeof(tChar));
}

//------------------------------------------------------------------------------

RoutineCtrlTouchSelfTest::~RoutineCtrlTouchSelfTest ( void )
{}

//------------------------------------------------------------------------------
/**
  * \brief       Get Property Details
  *            Function retrieves the data based on Property ID.
  *
  * \param[in]   tU32     - Property ID
  */
tDiaResult RoutineCtrlTouchSelfTest::getPropertyDetails(tU32 PropID)
{
   ScopeTrace trc("dia_RoutineCtrlTouchSelfTest::getPropertyDetails");
   std::vector<tU8> PropDataVec;

   // retrieve size of property
   auto Proplength = dia_getPropertySize(PropID); //DIA_PROP_LENGTH_MAX;
   DIA_TR_INF("dia_RoutineCtrlTouchSelfTest::getPropertyDetails Proplength = %zu", Proplength);
   PropDataVec.reserve(Proplength);
   PropDataVec.resize(Proplength);
   tDiaResult retCode = dia_getProperty(PropID,PropDataVec);
   if ( retCode == DIA_SUCCESS )
   {
      for (tU16 i = 0; i < PropDataVec.size(); ++i)
      {
         m_TouchDevicePath[i] = PropDataVec[i];
      }
      DIA_TR_INF("m_TouchDevicePath = %s", m_TouchDevicePath);
   }
   else
   {
      DIA_TR_ERR("dia_RoutineCtrlTouchSelfTest::getPropertyDetails Get property failed retCode=0x%08X.", retCode);
   }
   return retCode;
}
//-----------------------------------------------------------------------------------------------------------------------------

tDiaResult
RoutineCtrlTouchSelfTest::start ( std::vector<tU8>& params, tU8 /*timerValue */ )
{
   ScopeTrace oTrace("dia_RoutineCtrlTouchSelfTest::start()");

   if ( DIA_SUCCESS != getPropertyDetails(DIA_PROP_TOUCHCTRL_SELFTEST) )
   {
      DIA_TR_INF("dia_RoutineCtrlTouchSelfTest::start FAILED to read Path");
      return DIA_FAILED;
   }

   DIA_TR_INF("TouchDevicePath = %s", m_TouchDevicePath);

   tChar u8Cmd[CMD_SIZE] = {0};

   //! prepare processing of the routine
   vInitialize();

   //! Check all supported self-test types
   if ( (params.size() != 1) ||
      ((params.at(0) != TOUCH_ANALOG_POWER_TEST) &&
       (params.at(0) != TOUCH_PIN_FAULT_TEST) &&
       (params.at(0) != TOUCH_SIGNAL_LIMIT_TEST) &&
       (params.at(0) != TOUCH_RUN_ALL_TESTS)) )
   {
      DIA_TR_ERR("dia_RoutineCtrlTouchSelfTest::start params.size()=%zu", params.size());
      return DIA_FAILED;
   }

   /* The field fd contains a file descriptor for an open file */
   int fd = open(m_TouchDevicePath, O_WRONLY);
   if (fd < 0)
   {
      DIA_TR_ERR("dia_RoutineCtrlTouchSelfTest::start open touch T25 failed");
      return DIA_FAILED;
   }

   switch(params.at(0))
   {
      case TOUCH_RUN_ALL_TESTS:
         {
            sprintf(u8Cmd,WRITE_TOUCH_RUN_ALL_TESTS);
            DIA_TR_INF("ALL TESTS requested");
         }
         break;

      case TOUCH_ANALOG_POWER_TEST:
         {
            sprintf(u8Cmd,WRITE_TOUCH_ANALOG_POWER_TEST);
            DIA_TR_INF("ANALOG VDD POWER Test requested");
         }
         break;

      case TOUCH_PIN_FAULT_TEST:
         {
            sprintf(u8Cmd,WRITE_TOUCH_PIN_FAULT_TEST);
            DIA_TR_INF("PIN FAULT Test requested");
         }
         break;

      case TOUCH_SIGNAL_LIMIT_TEST:
         {
            sprintf(u8Cmd,WRITE_TOUCH_SIGNAL_LIMIT_TEST);
            DIA_TR_INF("SIGNAL LIMIT Test requested");
         }
         break;

      default:
         break;
   }
   DIA_TR_INF("dia_RoutineCtrlTouchSelfTest::start u8Cmd = %s, len =%zu", u8Cmd, strlen(u8Cmd));

   ssize_t iRet = write(fd, u8Cmd, strlen(u8Cmd));
   if (iRet < 0)
   {
      int retValClose = close(fd);
      if (0 != retValClose)
      {
         DIA_TR_ERR("dia_RoutineCtrlTouchSelfTest::start close failed with error code %d.", retValClose);
         DIA_TR_ERR("dia_RoutineCtrlTouchSelfTest::start errno %d.", errno);
      }

      DIA_TR_ERR("### dia_RoutineCtrlTouchSelfTest::start write touch T25 failed");
      return DIA_FAILED;
   }

   DIA_TR_INF("dia_RoutineCtrlTouchSelfTest::start write touch T25 successful");
   mIsResultReady = TRUE;

   /* change status to running and wait for the self-test results */
   eSetStatus(DIA_EN_RTCTRL_STATUS_IN_PROGRESS);

   int retClose = close(fd);
   if (0 != retClose)
   {
      DIA_TR_ERR("dia_RoutineCtrlTouchSelfTest::start close failed with error code %d.", retClose);
      DIA_TR_ERR("dia_RoutineCtrlTouchSelfTest::start errno %d.", errno);
   }

   DIA_TR_ERR("dia_RoutineCtrlTouchSelfTest::start retCode DIA_SUCCESS.");

   return DIA_SUCCESS;
}

//------------------------------------------------------------------------------

tDiaResult
RoutineCtrlTouchSelfTest::requestResult (std::vector<tU8>& results  )
{
   ScopeTrace oTrace("dia_RoutineCtrlTouchSelfTest::requestResult()");

   tDiaResult retCode = DIA_FAILED;
   tU8 retVerdict=VERDICT_NA;
   tChar buffer[BUF_SIZE] = {0};

   /* Result can be read from the same path, with error code and info bytes*/
   int fd = open(m_TouchDevicePath, O_RDONLY);
   DIA_TR_INF("dia_RoutineCtrlTouchSelfTest::requestResult fd=%d.", fd);
   if (fd < 0)
   {
      DIA_TR_ERR("### dia_RoutineCtrlTouchSelfTest::requestResult open touch T25 failed.");
      DIA_TR_ERR("### dia_RoutineCtrlTouchSelfTest::requestResult fd=%d errno=%d.", fd, errno );
      return DIA_FAILED;
   }

   ssize_t iRet = read(fd, buffer, BUF_SIZE);
   int iCount = 0;
   while ((iRet < 0) && (iCount < 10))
   {
      DIA_TR_INF("dia_RoutineCtrlTouchSelfTest::requestResult wait 100ms iRet=%zd errno=%d.", iRet, errno);
      iCount++;
      OSAL_s32ThreadWait(100); //wait for 0.1 sec (else testcases ALL_TESTS, PIN_FAULT_TEST give rt_running)
      iRet = read(fd, buffer, BUF_SIZE);
   }

   DIA_TR_INF("dia_RoutineCtrlTouchSelfTest::requestResult iRet=%zd.", iRet);

   if ((iRet < 0) && (errno == EAGAIN))
   {
      /*There is a possibility of reading an older result if the read happens
        before test execution completes. If user try to read the result before test completes,
        it will return EAGAIN to indicate test is running*/
      DIA_TR_ERR("dia_RoutineCtrlTouchSelfTest::requestResult running!");
      DIA_TR_INF("dia_RoutineCtrlTouchSelfTest::requestResult iRet=%zd errno=%d.", iRet, errno);

      eSetStatus(DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK);
      retVerdict = VERDICT_NA;
      retCode = DIA_SUCCESS;
   }
   else if (iRet > 0 )
   {
     DIA_TR_INF("dia_RoutineCtrlTouchSelfTest::requestResult iRet=%zd buffer=%s", iRet, buffer);
     if (strncmp(buffer, "PASS", PASS_LGTH) == 0)
     {
        DIA_TR_INF("dia_RoutineCtrlTouchSelfTest::requestResult PASS");
        eSetStatus(DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK);
        retVerdict = VERDICT_OK;
     }
     else
     {
        DIA_TR_ERR("dia_RoutineCtrlTouchSelfTest::requestResult Not PASS");
        eSetStatus(DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK);
        retVerdict = VERDICT_NOK;
     }

     retCode = DIA_SUCCESS;
   }
   else
   {
      DIA_TR_ERR("dia_RoutineCtrlTouchSelfTest::requestResult read error!");
      retCode = DIA_FAILED;
   }

   results.push_back(mapStatus2UDSResult());
   results.push_back(retVerdict);

   int retClose = close(fd);
   if (0!=retClose)
   {
      DIA_TR_ERR("dia_RoutineCtrlTouchSelfTest::start close failed with error code %d.", retClose);
      DIA_TR_ERR("dia_RoutineCtrlTouchSelfTest::start errno %d.", errno);
   }

   if (DIA_SUCCESS!=retCode)
   {
      DIA_TR_ERR("dia_RoutineCtrlTouchSelfTest::start retCode 0x%08X.", retCode);
   }

   return retCode;
}

} //namespace dia
