/************************************************************************
* FILE:         dia_RoutineCtrlStressTest.cpp
* PROJECT:      GM NexGen
* SW-COMPONENT: Diagnostic application
*----------------------------------------------------------------------
*
* DESCRIPTION: Classes for UDS service StressTest
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author                  | Modification
* 10.11.14  | TMS Plischke            | initial version
*
*************************************************************************/

//#include <iostream>
#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <stdlib.h>
#include <sys/resource.h>

#ifndef __INCLUDED_DIA_COMMON_CORE__
#include <common/framework/application/dia_common_core.h>
#endif

#ifndef __INCLUDED_DIA_COMMON_UDS_RTCTRL__
#include <common/framework/protocols/uds/rtctrl/dia_common_uds_rtctrl.h>
#endif

#include "dia_RoutineCtrlStressTest.h"

#define DIA_MAX_NUMBER_OF_ARG   20
#define DIA_MAX_CMD_LEN       1000

// Rountine Results
#define ROUTINE_OK         0
#define ROUTINE_NOK        1
#define ROUTINE_ABORT      2
#define ROUTINE_RUNNING    3
#define ROUTINE_STOPPED    4

namespace dia
{

// Command to USE
char COMMAND[] = "/usr/bin/stress";
// path to execute
char EXEC_PATH[] = "/var/opt/bosch/dynamic";

using namespace std;

RoutineCtrlStressTest::RoutineCtrlStressTest()
      : dia_Routine("dia_RoutineCtrlStressTest", DIA_C_U16_DID_DIAGNOSIS_STRESS_TEST_1, DIA_EN_RTCTRL_ID_DIAG_STRESS_TEST_1, DIA_EN_RTCTRL_TYPE_LONG_TERM),
        statusResponse(ROUTINE_STOPPED),
        child_pid(0),
        isRunning(false)
{
   // do not use standard constructor => use ADD_ROUTINE_NAME_DID_ID
    DIA_ASSERT_ALWAYS();
}

RoutineCtrlStressTest::RoutineCtrlStressTest(tCString name, tU16 udsID, dia_eRoutineID routineID)
      : dia_Routine(name, udsID, routineID, DIA_EN_RTCTRL_TYPE_SHORT_TERM),
        statusResponse(ROUTINE_STOPPED),
        child_pid(0),
        isRunning(false)
{
    dia_tclFnctTrace oTrace("dia_RoutineCtrlStressTest::dia_RoutineCtrlStressTest");
}

//------------------------------------------------------------------------------

RoutineCtrlStressTest::~RoutineCtrlStressTest()
{
    _BP_TRY_BEGIN
    {
       if(  (true == isRunning)
          &&(child_pid != 0))
       {
         // kill the running child porcess
         killChilds(child_pid);
         child_pid = 0;
         isRunning = false;
       }
    }
    _BP_CATCH_ALL
    {
       DIA_TR_ERR("dia_RoutineCtrlStressTest::~dia_RoutineCtrlStressTest - Exception caught!");
       DIA_ASSERT_ALWAYS();
    }
    _BP_CATCH_END
}

//------------------------------------------------------------------------------

tDiaResult
RoutineCtrlStressTest::start ( std::vector<tU8>& params, tU8 /*timerValue*/ )
{
   dia_tclFnctTrace oTrace("dia_RoutineCtrlStressTest::start");

   tDiaResult  retCode         = DIA_SUCCESS;
   char        paramarray[DIA_MAX_CMD_LEN];
   char        *argv[DIA_MAX_NUMBER_OF_ARG];
   char        *cmd = NULL;

   // prepare processing of the routine
   vInitialize();

   statusResponse = ROUTINE_OK;

   // check if command "stress" exist
   if(access(COMMAND, F_OK) == -1)
   {
      DIA_TR_ERR(" !!! dia_RoutineCtrlStressTest::start => File not found %s", COMMAND);
      // do not continue otherwise execv will crash
      retCode = DIA_E_GENERAL_REJECT;
      statusResponse = ROUTINE_NOK;    // do not continue
      NORMAL_M_ASSERT_ALWAYS ();       // assert here, this is a critical behaviour
   }
   else if(params.size() >= DIA_MAX_CMD_LEN)
   {
      DIA_TR_ERR(" !!! dia_RoutineCtrlStressTest::start => Data to long %zu>=%d(max)", params.size(),DIA_MAX_CMD_LEN);
      // out of range
      retCode = DIA_E_OUT_OF_RANGE;
      statusResponse = ROUTINE_NOK; // do not continue
   }
   else if(true == isRunning)
   {
      DIA_TR_ERR(" !!! dia_RoutineCtrlStressTest::start => command still running, stop first");
      // still running, do NOT start again, reject
      retCode = DIA_E_SEQUENCE_ERROR;
      statusResponse = ROUTINE_NOK; // do not continue
   }
   else
   {
      // we need to copy the params first to one char arry, because arguments must be pointer to char
      // also we do not want to overwrite incomming parameter, and parseargv will do this
      std::vector<tU8>::iterator iter = params.begin();
      for (int i=0; iter != params.end(); iter++, i++)
      {
         paramarray[i]=*iter;
      }
      DIA_TR_INF("dia_RoutineCtrlStressTest::start => Size of Data :%zu", params.size());
      paramarray[params.size()] = '\0'; // we need it NULL terminated, otherwise e.g. stroke won't work

      // get the command
      parsecmd(COMMAND,&cmd);
      DIA_TR_INF("dia_RoutineCtrlStressTest::start => Cmd:%s", cmd);
      if((cmd == NULL) || (*cmd == '\0'))
      {
         NORMAL_M_ASSERT_ALWAYS();
         statusResponse = ROUTINE_NOK;// do not continue
      }
      else
      {
         // set the command to the first argv element
         argv[0]=cmd;

         // now get the other arguments from the parameter array and put them to the argv-list
         if(parseargv(paramarray,&argv[1]) == true)
         {
            // print the arguments
            tU8 id = 0;
            while(   (id < DIA_MAX_NUMBER_OF_ARG)
                  && (argv[id] != NULL)
                  &&(*argv[id] != '\0')
                 )
            {
               DIA_TR_INF("dia_RoutineCtrlStressTest::start => argv:%s",argv[id]);
               id++;
            }
            /* now create new process */
            DIA_TR_INF("dia_RoutineCtrlStressTest::start => call fork now");
            child_pid = fork();

            if (child_pid >= 0) /* fork succeeded */
            {
               if (child_pid == 0) /* fork() returns 0 for the child process */
               {
                  // now we are running in the child process
                  // change Dir to exectution path
                  int ret = chdir(EXEC_PATH);
                  if ( ret < 0)
                  {
                     exit(EXIT_FAILURE);
                  }

                  // Create new process group with child as leader.
                  ret = setpgid(0, 0);
                  if ( ret < 0)
                  {
                     exit(EXIT_FAILURE);
                  }

                  // call shell command
                  // if exec is succesfull it will return the exit code
                  (tVoid) execv(COMMAND,argv);
                  // otherwise exit failed
                  exit(EXIT_FAILURE);
               }// if (child_pid == 0)
               else /* parent process */
               {
                  // now we are running in the diag process
                  int response = -1;
                  int status = -1;
                  // check status of child process, but do not wait (WNOHANG)!!!
                  pid_t cpid = waitpid(child_pid, &status, WNOHANG);
                  switch(cpid)
                  {
                     case 0: // FOUND
                     {
                        // we can't say at this moment we really run succesfully or not, but still trace info
                        DIA_TR_INF("dia_RoutineCtrlStressTest::start => Child Id: %d", child_pid);
                        DIA_TR_INF("dia_RoutineCtrlStressTest::start => Child WIFEXITED code: %d", WIFEXITED(status));
                        DIA_TR_INF("dia_RoutineCtrlStressTest::start => Child WEXITSTATUS code: %d", WEXITSTATUS(status));

                        DIA_TR_INF("dia_RoutineCtrlStressTest::start => Child is running");
                        isRunning = true;
                        statusResponse = ROUTINE_RUNNING;
                        break;
                     }// case 0:

                     case -1:// ERROR
                     {
                    	DIA_TR_ERRMEM("dia_RoutineCtrlStressTest::start => waitpid FAILED - Child Id:%d Errno:%x", child_pid, errno);
                        NORMAL_M_ASSERT_ALWAYS();
                        isRunning = false;
                        statusResponse = ROUTINE_NOK;
                        break;
                     }// case -1:

                     default:
                     {
                        /* from The Open Group Base Specifications Issue 7; IEEE Std 1003.1, 2013 Edition
                        WIFEXITED(stat_val)
                            Evaluates to a non-zero value if status was returned for a child process that terminated normally.
                        WEXITSTATUS(stat_val)
                            If the value of WIFEXITED(stat_val) is non-zero, this macro evaluates to the low-order 8 bits of the status argument that the child process passed to _exit() or exit(), or the value the child process returned from main().
                         */
                        // this is the first request after exit, so data is valid
                        response = WIFEXITED (status);
                        DIA_TR_INF("dia_RoutineCtrlStressTest::requestResult => Child WIFEXITED code: %d", response);
                        if(response!=0)
                        {
                           // terminated normally
                           response = WEXITSTATUS (status);
                           DIA_TR_INF("dia_RoutineCtrlStressTest::requestResult => Child WEXITSTATUS code: %d",response);
                           statusResponse = (response==0)?ROUTINE_OK:ROUTINE_NOK;
                        }
                        else
                        {
                           // terminated NOT normally
                           statusResponse = ROUTINE_NOK;
                        }
                        child_pid = 0;
                        isRunning = false;
                        break;
                     }// default:
                  }// switch(cpid)
               }// else // if (child_pid == 0)
            }// if (child_pid >= 0)
            else /* failure */
            {
               DIA_TR_ERR("dia_RoutineCtrlStressTest::start => fork failed");
               statusResponse = ROUTINE_NOK;
            }// else // if (child_pid >= 0)
         }
         else
         {
            DIA_TR_ERR("dia_RoutineCtrlStressTest::start => parseargv failed");
            // parse arguments failed
            statusResponse = ROUTINE_NOK;// do not continue
         }
      }
   }

   // fill repsonse here, so it will be ignored by requestResult, which is called directy after this by framework
   mResults.push_back(statusResponse);
   mIsResultReady = TRUE;
   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
RoutineCtrlStressTest::stop ( std::vector<tU8>& /*params*/)
{
   dia_tclFnctTrace oTrace("dia_RoutineCtrlStressTest::stop");
   tDiaResult  retCode         = DIA_SUCCESS;

   // prepare processing of the routine
   vInitialize();

   int status;
   if(true == isRunning)
   {
      pid_t cpid = waitpid(child_pid, &status, WNOHANG);
      if(cpid == 0)
      {
         DIA_TR_INF("dia_RoutineCtrlStressTest::stop => Child still running KILL Proc:%d",child_pid);
         // kill the running child process
         killChilds(child_pid);
         statusResponse = ROUTINE_STOPPED;
      }// if(cpid == 0)
      else
      {
         // child still terminated, so no kill requiered
         DIA_TR_INF("dia_RoutineCtrlStressTest::stop => Child WIFEXITED code: %d Status:%d", WIFEXITED (status),status);
         statusResponse = WEXITSTATUS(status)?ROUTINE_OK:ROUTINE_NOK;
      }// else // if(cpid == 0)
      isRunning = false;
      child_pid = 0;
   }
   else
   {
      DIA_TR_INF("dia_RoutineCtrlStressTest::stop => Child is not running");
      // use last statusresponse
   }

   // fill repsonse here, so it will be ignored by requestResult, which is called directy after this by framework
   mResults.push_back(statusResponse);
   mIsResultReady = TRUE;
   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
RoutineCtrlStressTest::requestResult ( std::vector<tU8>& results )
{
   tDiaResult retCode = DIA_E_SEQUENCE_ERROR;

   results.clear();
   if ( !(mResults.empty()) )
   {
      // data was filled before by start or stop, so use this
      DIA_TR_INF("dia_RoutineCtrlStressTest::requestResult --- 1");
      std::vector<tU8>::iterator iter = mResults.begin();
      for ( ; iter != mResults.end(); iter++ )
      {
         results.push_back(*iter);
      }
      mResults.clear();
      retCode = DIA_SUCCESS;
   }// if ( !(mResults.empty()) )
   else
   {
      if(isRunning == true)
      {
         int status;
         // check status of child process, but do not wait (WNOHANG)!!!
         DIA_TR_INF("dia_RoutineCtrlStressTest::requestResult => Child Id: %d", child_pid);
         pid_t cpid = waitpid(child_pid, &status, WNOHANG);

         switch(cpid)
         {
            case 0: // FOUND
            {
               // cpid == 0 => child_pid = cpid
               DIA_TR_INF("dia_RoutineCtrlStressTest::requestResult => Child still running");
               // this process is still running
               statusResponse = ROUTINE_RUNNING;
               break;
            }//case 0

            case -1: // FAILED
            {
               // ERROR
               DIA_TR_INF("dia_RoutineCtrlStressTest::requestResult => cpid = -1");
               statusResponse = ROUTINE_NOK;
               isRunning = false;
               child_pid = 0;
               break;
            }// case -1

            default:
            {
               // child process exit
               // this is the first request after exit, so data is valid
               int response;

               /* from The Open Group Base Specifications Issue 7; IEEE Std 1003.1, 2013 Edition
                  WIFEXITED(stat_val)
                      Evaluates to a non-zero value if status was returned for a child process that terminated normally.
                  WEXITSTATUS(stat_val)
                      If the value of WIFEXITED(stat_val) is non-zero, this macro evaluates to the low-order 8 bits of the status argument that the child process passed to _exit() or exit(), or the value the child process returned from main().
                */

               // get ExitStatus of Childprocess
               response = WIFEXITED (status);
               DIA_TR_INF("dia_RoutineCtrlStressTest::requestResult => Child WIFEXITED code: %d", response);
               if(response!=0)
               {
                  // terminated normally
                  response = WEXITSTATUS (status);
                  DIA_TR_INF("dia_RoutineCtrlStressTest::requestResult => Child WEXITSTATUS code: %d",response);
                  statusResponse = (response==0)?ROUTINE_OK:ROUTINE_NOK;
               }
               else
               {
                  // terminated NOT normally
                  statusResponse = ROUTINE_NOK;
               }
               DIA_TR_INF("dia_RoutineCtrlStressTest::requestResult => Child terminated now!!!");

               isRunning = false;
               child_pid = 0;
               break;
            }//default:
         }// switch(cpid)
      }// if(isRunning)
      else
      {
         // process is still terminate, so we have to use last data we get
         // return last statusresponse
         DIA_TR_INF("dia_RoutineCtrlStressTest::requestResult => Child is not running");
      }// else // if(isRunning)

      DIA_TR_INF("dia_RoutineCtrlStressTest::requestResult => statusResponse:%d", statusResponse);
      results.push_back(statusResponse);
      mIsResultReady = TRUE;
      retCode = DIA_SUCCESS;
   }// else // if ( !(mResults.empty()) )

   return retCode;
}

//------------------------------------------------------------------------------
// can use stroke here, because we have a copy of the string in use
bool
RoutineCtrlStressTest::parseargv(char *param, char **argv) const
{
   bool bReturn = true;
   tU8 argCounter = 0;
#if 0
   // while end of param
   while (*param != '\0')
   {
      while (*param == ' ' || *param == '\t' || *param == '\n')
      {
         // replace spaces\tabs\return with 0, so we always terminate the last argument wirh 0
         *param++ = '\0';
      }
      // now we found the first char or EndOfString
      // this is the start of an agrument, also set to next argument
      *argv++ = param;
      argCounter++;

      while ((*param != '\0') && (*param != ' ') && (*param != '\t') && (*param != '\n'))
      {
         // skip all chars
         param++;
      }
      if(argCounter > (DIA_MAX_NUMBER_OF_ARG-1))
      {
         // max args reached, last one is for 0
         NORMAL_M_ASSERT_ALWAYS();
         bReturn = false;
         break;
      }
   }// while (*param != '\0')
   // hint the last argument is always terminatet by the 0 of the param
#else
   char delimiter[] = " \t\n";
   char *ptr;
   // initalize and create first part
   ptr = strtok(param, delimiter);

   while(ptr != NULL)
   {
      *argv++ = ptr; // this is the start of an agrument, also set to next argument
      argCounter++;

      if(argCounter > (DIA_MAX_NUMBER_OF_ARG-1))
      {
         // max args reached, last one is for 0
         NORMAL_M_ASSERT_ALWAYS();
         bReturn = false;
         break;
      }
       ptr = strtok(NULL, delimiter);
   }
#endif
   *argv = NULL; // list of argument must end with 0, if there where no args it is done now twice, but never mind
   return bReturn;
}

//------------------------------------------------------------------------------
// we can't use stroke here, because we should not touch the given string
tVoid
RoutineCtrlStressTest::parsecmd(char *param, char **cmd) const
{
   *cmd = param; // set the command to the first one, maybe there is no slash

   // while end of param
   while (*param != '\0')
   {
      if (*param == '/')
      {
         // found slash, so set cmd to next param
         *cmd = param+1;
      }
      param++;

      // continue -> check if there are more slashes
   }// while (*param != '\0')
}

//------------------------------------------------------------------------------
tVoid
RoutineCtrlStressTest::killChilds(pid_t pid) const
{
   int status;

   if(pid != 0)
   {
      DIA_TR_INF("dia_RoutineCtrlStressTest::killChilds => call kill");

      // If pid is negative, then sig is sent to every process in the process group whose ID is -child_pid.
      // SIGTERM(15) => Termination request.
      (void)kill(-child_pid, SIGTERM);
      DIA_TR_INF("dia_RoutineCtrlStressTest::killChilds => waitpid");
      // finally get once waitpid, don't want any zombies
      (tVoid) waitpid(-1, &status, WUNTRACED | WCONTINUED);
      DIA_TR_INF("dia_RoutineCtrlStressTest::killChilds => waitpid end");
   }// if(pid != 0)
}

} //namespace dia
