/************************************************************************
* FILE:         dia_RoutineCtrlRunIn.cpp
* PROJECT:      GM NexGen
* SW-COMPONENT: Diagnostic application
*----------------------------------------------------------------------
*
* DESCRIPTION: Classes for UDS service StressTest
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author                  | Modification
* 10.11.14  | BSOT Plischke           | initial version
*
*************************************************************************/

#ifndef __INCLUDED_DIA_COMMON_CORE__
#include <common/framework/application/dia_common_core.h>
#endif

#ifndef __INCLUDED_DIA_COMMON_UDS_RTCTRL__
#include <common/framework/protocols/uds/rtctrl/dia_common_uds_rtctrl.h>
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include <common/framework/config/dia_ConfigManager.h>
#endif

#ifndef __INCLUDED_DIA_ROUTINE_CTRL_RUNIN__
   #include "dia_RoutineCtrlRunIn.h"
#endif

using namespace std;


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DIAGNOSTICS_PROD
#include "trcGenProj/Header/dia_RoutineCtrlRunIn.cpp.trc.h"
#endif

// general defines
#define ROUTINE_OK         0
#define ROUTINE_NOK        1
#define ROUTINE_RUNNING    3
#define ROUTINE_STOPPED    4



namespace dia
{

#define MAX_SIZE 100
static tCString strDiagRunInDir    = "/var/opt/bosch/dynamic/diagnosis/runin";


/* runin result, configure only here, no other parts necessary */
// 1(StatusFlag) + NUMBER_OF_RUNS * 110
#define NUMBER_OF_RUNS              10
#define DATA_RESULT_SIZE_PER_RUN    110

// hint: pos currently unused, only for info
const trRunInCommandList runInCommandList[] =
{
// { pos,size,{"Filename"              ,Length of , {PosResponse}            , DataPos of}}
//      ,    ,                         ,pos resp  ,                          , Response
// {   0,   1,{"status"                ,     0    , {0x00}                   ,   0      }},              // [0:0] Status (HexDump (1 Byte) => this one is reserved and handled directly in the code
   {   1,   4,{"system_operation_time" ,     3    , {0x62,0xB1,0x01,0x00,0}  ,   3      }},              // [1:0] Time stamp (system operation time) (CMTimeInS)
   {   5,   1,{"selftest_gyro"         ,     4    , {0x71,0x01,0x30,0x43,0}  ,   4      }},              // [5:0] Gyro selftest - status (CMCommonRoutineStatusType 1(1))
   {   6,   1,{"reserved"              ,     0    , {0x00,0x00,0x00,0x00,0}  ,   0      }},              // [6:0] (reserved)  - Rerserved for routine result (refer Jira ticket CMG3GB-1341).
   {   7,   1,{"selftest_acc"          ,     4    , {0x71,0x01,0x30,0x42,0}  ,   4      }},              // [7:0] ACC selftest - status (CMCommonRoutineStatusType 1(1))
   {   8,   1,{"reserved"              ,     0    , {0x00,0x00,0x00,0x00,0}  ,   0      }},              // [8:0] (reserved)  - Rerserved for routine result (refer Jira ticket CMG3GB-1341).
   {   9,   1,{"selftest_ipod"         ,     4    , {0x71,0x01,0x30,0x40,0}  ,   4      }},              // [9:0] iPod authentication chip selftest - status (CMCommonRoutineStatusType 1(1))
   {  10,   1,{"reserved"              ,     0    , {0x00,0x00,0x00,0x00,0}  ,   0      }},              // [10:0] (reserved)  - Rerserved for routine result (refer Jira ticket CMG3GB-1341).
   {  11,   1,{"stress1"               ,     4    , {0x71,0x01,0x32,0x02,0}  ,   4      }},              // [11:0] mainCpuStressTest_1 - status (CMCommonRoutineStatusType 2(1))
   {  12,   1,{"stress2"               ,     4    , {0x71,0x03,0x32,0x03,0}  ,   4      }},              // [12:0] mainCpuStressTest_2 - status (CMCommonRoutineStatusType 2(1))
   {  13,   1,{"chksum_nor_raw"        ,     4    , {0x71,0x03,0x30,0x60,0}  ,   4      }},              // [13:0] calculateAndVerifyChecksum_NOR_RAW - status (CMCommonRoutineStatusType 2(1))
   {  14,  16,{"chksum_nor_raw"        ,     4    , {0x71,0x03,0x30,0x60,0}  ,   5      }},              // [14:0] calculateAndVerifyChecksum_NOR_RAW - hash (HEX[16])
   {  30,   1,{"chksum_nor_raw"        ,     4    , {0x71,0x03,0x30,0x60,0}  ,  21      }},              // [30:0] calculateAndVerifyChecksum_NOR_RAW - result (CMVerificationResult)
   {  31,   1,{"chksum_v850_app"       ,     4    , {0x71,0x03,0x30,0x66,0}  ,   4      }},              // [31:0] calculateAndVerifyChecksum_V850_App (CMCommonRoutineStatusType 2(1))
   {  32,   8,{"chksum_v850_app"       ,     4    , {0x71,0x03,0x30,0x66,0}  ,   5      }},              // [32:0] calculateAndVerifyChecksum_V850_App - hash (HEX[8])
   {  40,   1,{"chksum_v850_app"       ,     4    , {0x71,0x03,0x30,0x66,0}  ,  13      }},              // [40:0] calculateAndVerifyChecksum_V850_App - result (CMVerificationResult)
   {  41,   1,{"chksum_v850_boot"      ,     4    , {0x71,0x03,0x30,0x65,0}  ,   4      }},              // [41:0] calculateAndVerifyChecksum_V850_Boot (CMCommonRoutineStatusType 2(1))
   {  42,   8,{"chksum_v850_boot"      ,     4    , {0x71,0x03,0x30,0x65,0}  ,   5      }},              // [42:0] calculateAndVerifyChecksum_V850_Boot - hash (HEX[8])
   {  50,   1,{"chksum_v850_boot"      ,     4    , {0x71,0x03,0x30,0x65,0}  ,  13      }},              // [50:0] calculateAndVerifyChecksum_V850_Boot - result (CMVerificationResult)
   {  51,   1,{"chksum_emmc_ffs"       ,     4    , {0x71,0x03,0x30,0x63,0}  ,   4      }},              // [51:0] calculateAndVerifyChecksum_eMMC_FFS (CMCommonRoutineStatusType 2(1))
   {  52,  16,{"chksum_emmc_ffs"       ,     4    , {0x71,0x03,0x30,0x63,0}  ,   5      }},              // [52:0] calculateAndVerifyChecksum_eMMC_FFS - hash (HEX[16])
   {  68,   1,{"chksum_emmc_ffs"       ,     4    , {0x71,0x03,0x30,0x63,0}  ,  21      }},              // [68:0] calculateAndVerifyChecksum_eMMC_FFS - result (CMVerificationResult)
   {  69,   1,{"chksum_emmc_fastboot"  ,     4    , {0x71,0x03,0x30,0x64,0}  ,   4      }},              // [69:0] calculateAndVerifyChecksum_eMMC_FastBoot (CMCommonRoutineStatusType 2(1))
   {  70,  16,{"chksum_emmc_fastboot"  ,     4    , {0x71,0x03,0x30,0x64,0}  ,   5      }},              // [70:0] calculateAndVerifyChecksum_eMMC_FastBoot - hash (HEX[16])
   {  86,   1,{"chksum_emmc_fastboot"  ,     4    , {0x71,0x03,0x30,0x64,0}  ,  21      }},              // [86:0] calculateAndVerifyChecksum_eMMC_FastBoot - result (CMVerificationResult)
   {  87,   1,{"chksum_emmc_raw"       ,     4    , {0x71,0x03,0x30,0x62,0}  ,   4      }},              // [87:0] calculateAndVerifyChecksum_eMMC_RAW (CMCommonRoutineStatusType 2(1))
   {  88,  16,{"chksum_emmc_raw"       ,     4    , {0x71,0x03,0x30,0x62,0}  ,   5      }},              // [88:0] calculateAndVerifyChecksum_eMMC_RAW - hash (HEX[16])
   { 104,   1,{"chksum_emmc_raw"       ,     4    , {0x71,0x03,0x30,0x62,0}  ,  21      }},              // [104:0] calculateAndVerifyChecksum_eMMC_RAW - result (CMVerificationResult)
   { 105,   1,{"testscreen"            ,     4    , {0x71,0x03,0x32,0x04,0}  ,   4      }},              // [105:0] Display_TestScreenSlideshow - status (CMCommonRoutineStatusType 2(1))
   { 106,   4,{"testscreen"            ,     4    , {0x71,0x03,0x32,0x04,0}  ,   5      }}               // [106:0] Display_TestScreenSlideshow - number of shown test screens (Unsigned (4 Byte))
};
/* end of configuration */

int g_iSizeRunInCommandList()
{
  return (static_cast<int>(sizeof(runInCommandList)/sizeof(trRunInCommandList)));
}


RoutineCtrlRunIn::RoutineCtrlRunIn()
      : dia_Routine("RoutineCtrlRunIn", DIA_C_U16_DID_DIAGNOSIS_RUN_IN, DIA_EN_RTCTRL_ID_DIAG_RUN_IN, DIA_EN_RTCTRL_TYPE_LONG_TERM)
{
    dia_tclFnctTrace oTrace("RoutineCtrlRunIn::RoutineCtrlRunIn");
}

//------------------------------------------------------------------------------

RoutineCtrlRunIn::~RoutineCtrlRunIn()
{
   /*
    _BP_TRY_BEGIN
    {
    }
    _BP_CATCH_ALL
    {
       DIA_TR_ERR("RoutineCtrlRunIn::~RoutineCtrlRunIn - Exception caught!");
       DIA_ASSERT_ALWAYS();
    }
    _BP_CATCH_END
    */
}

//------------------------------------------------------------------------------

tDiaResult
RoutineCtrlRunIn::start ( std::vector<tU8>& params, tU8 /*timerValue*/ )
{
   dia_tclFnctTrace oTrace("RoutineCtrlRunIn::start");
   tDiaResult  retCode         = DIA_SUCCESS;
   tDiaResult  propertyRetCode = DIA_E_ERROR;
   tU8         statusFlag      = true;
   tU8         statusResponse  = ROUTINE_OK;
   vector<tU8> vecStatus(1,0);

   switch (params.size())
   {
      case 4: // data with delay
      {
         // write flag
         vecStatus[0]      = statusFlag;

         propertyRetCode = dia_setProperty(DIA_PROP_CM_RUN_IN_ACTIVATION, vecStatus);
         //propertyRetCode = dia_setProperty(DIA_PROP_CM_RUN_IN_ACTIVATION, statusFlag);
         if( propertyRetCode == DIA_E_NOERROR )
         {
            DIA_TR_INF("DIA_PROP_CM_RUN_IN_ACTIVATION Set to %x",statusFlag);
         }
         else
         {
            DIA_TR_ERR("!!! RoutineCtrlRunIn::start => DIA_PROP_CM_RUN_IN_ACTIVATION failed Error:%x",propertyRetCode);
            retCode = DIA_E_CONDITIONS_NOT_CORRECT;
         }

         const std::vector<tU8> paramsArray = params;
         // write dealy
         tU32 delay = ((((tU32) paramsArray[0]) << 24) | (((tU32) paramsArray[1]) << 16) | (((tU32) paramsArray[2]) << 8) | ((tU32) paramsArray[3])) * 1000; //ms
         propertyRetCode = dia_setProperty(DIA_PROP_CM_RUN_IN_START_DELAY, delay);
         if( propertyRetCode == DIA_E_NOERROR )
         {
            DIA_TR_INF("DIA_PROP_CM_RUN_IN_START_DELAY Set:%d[ms]",delay);
         }
         else
         {
            DIA_TR_ERR("!!! RoutineCtrlRunIn::start => DIA_PROP_CM_RUN_IN_DELAY failed Error:%x",propertyRetCode);
            retCode = DIA_E_CONDITIONS_NOT_CORRECT;
         }
         break;
      }

      case 0: // data without delay
      {
         DIA_TR_INF("RoutineCtrlRunIn::start => default Time");
         // use default delay

         // ***********************************
         // write flag
         // ***********************************
         vecStatus[0]      = statusFlag;

         propertyRetCode = dia_setProperty(DIA_PROP_CM_RUN_IN_ACTIVATION, vecStatus);
         //tDiaResult propertyRetCode = dia_setProperty(DIA_PROP_CM_RUN_IN_ACTIVATION, statusFlag);
         if( propertyRetCode == DIA_E_NOERROR )
         {
            DIA_TR_INF("DIA_PROP_CM_RUN_IN_ACTIVATION Set to %x",statusFlag);
         }
         else
         {
            DIA_TR_ERR("!!! RoutineCtrlRunIn::start => DIA_PROP_CM_RUN_IN_ACTIVATION failed Error:%x",propertyRetCode);
            retCode = DIA_E_CONDITIONS_NOT_CORRECT;
         }
         break;
      }

      default:
      {
         // invalid length
         DIA_TR_ERR("!!! RoutineCtrlRunIn::start => invalid length");
         retCode = DIA_E_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT;
      }
   }

   // ***********************************
   // read Data from Memory
   // ***********************************

   // read flag
   propertyRetCode = dia_getProperty(DIA_PROP_CM_RUN_IN_ACTIVATION, &statusFlag, 1);
   if( propertyRetCode == DIA_E_NOERROR )
   {
      DIA_TR_INF("DIA_PROP_CM_RUN_IN_ACTIVATION Read;%x",statusFlag);
      statusResponse = (statusFlag==true)?ROUTINE_RUNNING:ROUTINE_STOPPED;
   }
   else
   {
      DIA_TR_ERR("!!! RoutineCtrlRunIn::start => DIA_PROP_CM_RUN_IN_ACTIVATION failed Error:%x",propertyRetCode);
      retCode = DIA_E_CONDITIONS_NOT_CORRECT;
   }

#if 0
   // read delay
   const tU8 Size = 4;
   tU8 getDelay[Size]={0};
   tU16 length = Size;
   propertyRetCode = dia_getProperty(DIA_PROP_CM_RUN_IN_DELAY, &getDelay[0],&length);
   if( propertyRetCode == DIA_E_NOERROR )
   {
      if(length == Size)
      {
         delay = ((tU32)getDelay[0]<<24) + ((tU32)getDelay[1]<< 16) + ((tU32)getDelay[2]<<8) + getDelay[3];
         DIA_TR_INF("DIA_PROP_CM_RUN_IN_DELAY Read:%x",delay);
      }
      else
      {
         delay = 0;
         DIA_TR_INF("DIA_PROP_CM_RUN_IN_DELAY wrong length:%d",length);
      }
   }
   else
   {
      delay = 0;
      DIA_TR_ERR("!!! RoutineCtrlRunIn::start => DIA_PROP_CM_RUN_IN_DELAY failed Error:%x",propertyRetCode);
      retCode = DIA_E_CONDITIONS_NOT_CORRECT;
   }
#endif

   // ***********************************
   // generate response
   // ***********************************
   mResults.push_back(statusResponse);
   mIsResultReady = TRUE;
   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
RoutineCtrlRunIn::stop ( std::vector<tU8>& /*params*/)
{
   dia_tclFnctTrace oTrace("RoutineCtrlRunIn::stop");
   tDiaResult  retCode         = DIA_SUCCESS;
   tU8         statusResponse  = ROUTINE_OK;

   vector<tU8> vecStatus(1,0);

   tU8 statusFlag = false;

   DIA_TR_INF("RoutineCtrlRunIn::stop");

   // ***********************************
   // write data to Memory
   // ***********************************
   vecStatus[0]      = statusFlag;

   tDiaResult propertyRetCode = dia_setProperty(DIA_PROP_CM_RUN_IN_ACTIVATION, vecStatus);
   //propertyRetCode = dia_setProperty(DIA_PROP_CM_RUN_IN_ACTIVATION, statusFlag);
   if( propertyRetCode == DIA_E_NOERROR )
   {
      DIA_TR_INF("DIA_PROP_CM_RUN_IN_ACTIVATION Set to %x",statusFlag);
   }
   else
   {
      DIA_TR_ERR("!!! RoutineCtrlRunIn::start => DIA_PROP_CM_RUN_IN_ACTIVATION failed Error:%x",propertyRetCode);
      retCode = DIA_E_CONDITIONS_NOT_CORRECT;
   }
   // ***********************************
   // read Data from Memory
   // ***********************************
   propertyRetCode = dia_getProperty(DIA_PROP_CM_RUN_IN_ACTIVATION, &statusFlag, 1);
   if( propertyRetCode == DIA_E_NOERROR )
   {
      DIA_TR_INF("DIA_PROP_CM_RUN_IN_ACTIVATION Read;%x",statusFlag);
      statusResponse = (statusFlag)?ROUTINE_RUNNING:ROUTINE_STOPPED;
   }
   else
   {
      DIA_TR_ERR("!!! RoutineCtrlRunIn::start => DIA_PROP_CM_RUN_IN_ACTIVATION failed Error:%x",propertyRetCode);
      retCode = DIA_E_CONDITIONS_NOT_CORRECT;
   }

   // ***********************************
   // generate response
   // ***********************************
   mResults.push_back(statusResponse);
   mIsResultReady = TRUE;
   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
RoutineCtrlRunIn::requestResult ( void )
{
   dia_tclFnctTrace oTrace("RoutineCtrlRunIn::requestResult()");

   mResults.clear();

   // collect result for get request. this is a synchronous operation
   // write statusFlag
   tU8 statusFlag;
   tU8 statusResponse = ROUTINE_NOK;
   tDiaResult result = getActivationFlag(statusFlag);
   if(result == DIA_SUCCESS)
   {
      statusResponse = statusFlag;
   }
   mResults.push_back(statusResponse);

   // write Data of Each Run
   for(tU16 i=1; i<=NUMBER_OF_RUNS; i++)
   {
      result = writeRunResult(i,mResults);
      if(result != DIA_SUCCESS)
      {
         DIA_TR_ERR("!!! RoutineCtrlRunIn::requestResult => Data Run:%d not succesfull",i);
      }
   }
   eSetStatus(DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK);
   mIsResultReady = TRUE;
   return DIA_SUCCESS;
}

//------------------------------------------------------------------------------

tDiaResult
RoutineCtrlRunIn::requestResult ( std::vector<tU8>& results )
{
   dia_tclFnctTrace oTrace("RoutineCtrlRunIn::requestResult");

   tDiaResult retCode   = DIA_E_SEQUENCE_ERROR;

   results.clear(); // first empty buffer

   if ( !(mResults.empty()) )
   {
      // result for start and stop genrated by RoutineCtrlRunIn:start and RoutineCtrlRunIn:stop
      DIA_TR_INF("RoutineCtrlRunIn::requestResult --- 1");
      std::vector<tU8>::iterator iter = mResults.begin();
      for ( ; iter != mResults.end(); iter++ )
      {
         results.push_back(*iter);
      }
      mResults.clear();
      mIsResultReady = FALSE;
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//------------------------------------------------------------------------------
tDiaResult RoutineCtrlRunIn::getActivationFlag (tU8& statusResponse) const
{
   dia_tclFnctTrace oTrace("RoutineCtrlRunIn::getActivationFlag");
   tDiaResult  retCode         = DIA_SUCCESS;
   tU8         statusFlag;

   // ***********************************
   // read Data from Memory
   // ***********************************
   tDiaResult propertyRetCode = dia_getProperty(DIA_PROP_CM_RUN_IN_ACTIVATION, &statusFlag, 1);
   if( propertyRetCode == DIA_E_NOERROR )
   {
      DIA_TR_INF("RoutineCtrlRunIn::getActivationFlag => Read;%x",statusFlag);
      statusResponse = (statusFlag)?ROUTINE_RUNNING:ROUTINE_STOPPED;
   }
   else
   {
      DIA_TR_ERR("!!! RoutineCtrlRunIn::getActivationFlag => DIA_PROP_CM_RUN_IN_ACTIVATION failed Error:%x",propertyRetCode);
      // return stopped in case file not exist
      statusResponse = ROUTINE_STOPPED;
   }
   return retCode;
}

//------------------------------------------------------------------------------


tDiaResult RoutineCtrlRunIn::readFileToBuffer (const char* const fileName, char Buffer[], size_t& size) const
{
   dia_tclFnctTrace oTrace("RoutineCtrlRunIn::readFileToBuffer");
   tDiaResult  retCode         = DIA_FAILED;

   if(  (fileName != NULL)
      &&(  Buffer != NULL))
   {
      DIA_TR_INF(" --- open %s",fileName);
      FILE* pFile = fopen ( fileName,"r" );
      if(pFile != NULL)
      {
         size_t usedSize = fread(Buffer,1, size,pFile); // binary read from file
         if(usedSize >= size)
         {
            // data to big
            DIA_TR_ERR("!!! readFileToBuffer => data to big");
         }
         else if(usedSize > 0)
         {
            // we have data -> OK
            size = usedSize;           // set the received size
            Buffer[usedSize]= '\0';    // terminate it with 0
            retCode = DIA_SUCCESS;

            //ETG_TRACE_USR2(( "Raw Data of the message (hex):%*x", ETG_LIST((unsigned int)size, ETG_T8, Buffer) ));
         }
         else
         {
            DIA_TR_ERR("!!! readFileToBuffer => no Data");
         }

         fclose (pFile);
      }
      else
      {
         DIA_TR_ERR("!!! readFileToBuffer => fopen FAILED");
      }
   }//  if(  (fileName != NULL)...
   else
   {
      DIA_TR_ERR("!!! readFileToBuffer => BUFFER = NULL");
   }
   return retCode;
}
//------------------------------------------------------------------------------
tDiaResult RoutineCtrlRunIn::getRequestFromBuffer (const char Buffer[], size_t bufferSize, std::vector<tU8>& requestBuffer) const
{
   dia_tclFnctTrace oTrace("RoutineCtrlRunIn::getRequestFromBuffer");
   tDiaResult  retCode         = DIA_FAILED;

   if(Buffer != NULL)
   {
      // the File include request and response so split first
      //tU32 length = ((((tU32) Buffer[3]) << 24) | (((tU32) Buffer[2]) << 16) | (((tU32) Buffer[1]) << 8) | ((tU32) Buffer[0]));
      tU16 length = static_cast<tU16>(((Buffer[1]) << 8) | Buffer[0]);
      tU32 offset = sizeof(length);

      if(length > MAX_SIZE)
      {
         DIA_TR_ERR(" !!! getRequestFromBuffer => Size to big: %d",length);
         // out of range
         return DIA_FAILED;
      }
      else if (length > (bufferSize - offset))
      {
         DIA_TR_ERR(" !!! getRequestFromBuffer => to less data: %u < %lu",length,(bufferSize - offset));
         // out of range
         return DIA_FAILED;
      }

      requestBuffer.resize(length); // set new length to vector

      for (tU32 i=0; i<length; ++i)
      {
         // copy data to vector
         requestBuffer[i] = Buffer[i+offset];
      }
      //ETG_TRACE_USR2(( "getRequestFromBuffer :%*x", ETG_LIST(length, ETG_T8, &Buffer[offset]) ));

      retCode = DIA_SUCCESS;
   }//  if(  (fileName != NULL)...
   else
   {
      DIA_TR_ERR("!!! getRequestFromBuffer => BUFFER = NULL");
   }
   return retCode;
}
//------------------------------------------------------------------------------
tDiaResult RoutineCtrlRunIn::getResponseFromBuffer (const char Buffer[], size_t bufferSize, std::vector<tU8>& responseBuffer) const
{
   dia_tclFnctTrace oTrace("RoutineCtrlRunIn::getResponseFromBuffer");
   tDiaResult  retCode         = DIA_FAILED;

   if((Buffer != NULL) && (bufferSize > 1))
   {
      // the File include request and response so split first
      tU16 requestLength = static_cast<tU16>((Buffer[1] << 8) | Buffer[0]);
      tU32 offset = static_cast<tU32>(requestLength + sizeof(requestLength)); // startposition of Response length
      if(bufferSize < offset+2)
      {
         DIA_TR_ERR(" !!! getResponseFromBuffer => Size to big: %d",offset);
         // out of range
         return DIA_FAILED;
      }
      tU16 responseLength = static_cast<tU16>((Buffer[1 + offset] << 8) | Buffer[0 + offset]);
      if(responseLength > MAX_SIZE)
      {
         DIA_TR_ERR(" !!! getResponseFromBuffer => Size to big: %d",responseLength);
         // out of range
         return DIA_FAILED;
      }
      else if (responseLength > (bufferSize - offset))
      {
         DIA_TR_ERR(" !!! getResponseFromBuffer => to less data: %u < %zu",requestLength,(bufferSize-offset));
         // out of range
         return DIA_FAILED;
      }
      responseBuffer.resize(responseLength); // set new length to vector
      offset = static_cast<tU32>(sizeof(requestLength) + requestLength + sizeof(responseLength)); // startposition of Response Data
      for (tU16 i=0; i<responseLength && (i+offset)<bufferSize; ++i)
      {
         // copy data to vector
         responseBuffer[i] = Buffer[i+offset];
      }
      //ETG_TRACE_USR2(( " --- getResponseFromBuffer :%*x", ETG_LIST(responseLength, ETG_T8, &Buffer[offset]) ));
      retCode = DIA_SUCCESS;
   }//  if(  (Buffer != NULL)...
   else
   {
      DIA_TR_ERR(" !!! getResponseFromBuffer => BUFFER = NULL");
   }
   return retCode;
}

//------------------------------------------------------------------------------
tDiaResult RoutineCtrlRunIn::compareResponse (const tU8 compareBuffer[], std::vector<tU8>& responseBuffer, tU8 size) const
{
   dia_tclFnctTrace oTrace("RoutineCtrlRunIn::compareResponse");
   tDiaResult  retCode         = DIA_FAILED;

   if(compareBuffer != NULL)
   {
      // over all bytes's of compare buffer
      for (tU8 i=0; i<size; ++i)
      {
         // is byte equal
         if(compareBuffer[i] != responseBuffer[i])
         {
            // not equal => FAIL

            // trace wrong compare
            tU8 temp[responseBuffer.size()];
            memset(temp, 0, responseBuffer.size());
            //ETG_TRACE_USR2(( " --- compareBuffer :%*x", ETG_LIST(size, ETG_T8, compareBuffer) ));
            for (unsigned int j=0; j<responseBuffer.size(); j++)
            {
               temp[j] = responseBuffer[j];
            }
            //ETG_TRACE_USR2(( " --- responseBuffer:%*x", ETG_LIST(size, ETG_T8, temp) ));
            return DIA_FAILED;
         }
      }
      DIA_TR_INF(" --- compareResponse => OK");
      retCode = DIA_SUCCESS;
   }//  if(  (fileName != NULL)...
   else
   {
      DIA_TR_ERR(" !!! compareResponse => BUFFER = NULL");
   }
   return retCode;
}


//------------------------------------------------------------------------------

tDiaResult RoutineCtrlRunIn::checkFile (tU16 run, const trFileInfo* const pFileCheckData, std::vector<tU8>& responseData ) const
{
   dia_tclFnctTrace oTrace("RoutineCtrlRunIn::checkFile");
   tDiaResult  retCode; //         = DIA_SUCCESS;
   size_t size = 2*MAX_SIZE + 2*sizeof(tU16);// response + request + 2*Length(tU32)
   char data[size];
   char fileName[300];

   sprintf(fileName,"%s/runin_%04d/%s",strDiagRunInDir,run,pFileCheckData->fileName);

   DIA_TR_INF("--- checkFile => File=%s",fileName);

   // first open the file and copy the Data to the Buffer
   retCode = readFileToBuffer(fileName, data, size);
   if(retCode != DIA_SUCCESS)
   {
      DIA_TR_ERR("!!! checkFile => readFileToBuffer Failed");
      return retCode;
   }
   // file read -> continue

   // now get the request and response from the Data
   std::vector<tU8> requestData;

   // only for validation and tracing, we don't need this
   retCode = getRequestFromBuffer(data, size, requestData);
   if(retCode != DIA_SUCCESS)
   {
      DIA_TR_ERR("!!! checkFile => getResponseFromBuffer Failed");
      return retCode;
   }
   // Data get -> continue

   retCode = getResponseFromBuffer(data, size, responseData);
   if(retCode != DIA_SUCCESS)
   {
      DIA_TR_ERR("!!! checkFile => getResponseFromBuffer Failed");
      return retCode;
   }
   // Data get -> continue

   // now check if Data is valid
   retCode = compareResponse( pFileCheckData->dataPosResponse,  // compare Buffer
                              responseData,                     // File Buffer
                              pFileCheckData->sizePosResponse); // size

   if(retCode != DIA_SUCCESS)
   {
      DIA_TR_ERR("!!! checkFile => compareResponse Failed");
      return retCode;
   }
   // positive response -> continue

   return retCode;
}
//------------------------------------------------------------------------------
tDiaResult RoutineCtrlRunIn::writeDatoToResponseBuffer (tU16 run, const trRunInCommandList* const pList, std::vector<tU8>& data ) const
{
   dia_tclFnctTrace oTrace("RoutineCtrlRunIn::RoutineCtrlRunIn");
   tDiaResult  retCode         = DIA_FAILED;

   if(0 == strcmp(pList->sDataInfo.fileName, "reserved"))
   {
      // we have to write dummy data (00)
      for(unsigned int i=0; i<pList->size;++i)
      {
         data.push_back(0);
      }
      retCode = DIA_SUCCESS;
      ETG_TRACE_COMP(( " --- responseBuffer => copy to Buffer Run:%d Cmd:%s",run,pList->sDataInfo.fileName));
   }
   else
   {
      std::vector<tU8> response;
      retCode = checkFile(run,&(pList->sDataInfo),response);
      if(retCode == DIA_SUCCESS)
      {
         // now verify that we have enough Data
         size_t expectedDataSize =   pList->sDataInfo.startPosDataToGet /* startposition of Data */
                                 + pList->size; /* size to write */
         size_t receivedDataSize = response.size();
         if(receivedDataSize < expectedDataSize)
         {
            // that should never happen, because it is not the received buffer it is the created buffer in this File
            DIA_TR_ERR("!!! RoutineCtrlRunIn => To Less Data %zu<>%u+%u",receivedDataSize,pList->sDataInfo.startPosDataToGet,pList->size);
            NORMAL_M_ASSERT_ALWAYS();
         }// if(receivedDataSize < expectedDataSize)
         else
         {
            // now write Data to destination buffer
            tU8 tempData[pList->size]; // this is need for trace, can't trace a vector
            memset(tempData, 0, pList->size);
            tU16 offset = pList->sDataInfo.startPosDataToGet;
            for(unsigned int i=0; i<pList->size;++i)
            {
               data.push_back(response[i+offset]);
               tempData[i] = response[i+offset];
            }
            ETG_TRACE_COMP(( " --- responseBuffer => copy to Buffer Run:%d Cmd:%s",run,pList->sDataInfo.fileName));
            //ETG_TRACE_COMP(( " --- responseBuffer => copy to Buffer Data:%*x",ETG_LIST(pList->size, ETG_T8, tempData) ));
            retCode = DIA_SUCCESS;
         }// else // if(receivedDataSize < expectedDataSize)
      }// if(retCode == DIA_SUCCESS)

      if(retCode != DIA_SUCCESS)
      {
         //tU16 offset = pList->sDataInfo.startPosDataToGet;
         for(unsigned int i=0; i<pList->size;++i)
         {
            data.push_back(0xFF);  // todo // what to send in case data is wrong? -> wait for response T. Mehring
         }
         ETG_TRACE_ERR(( " !!! responseBuffer => FAILED Run:%d Cmd:%s",run,pList->sDataInfo.fileName));
      }
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult RoutineCtrlRunIn::writeRunResult (tU16 run, std::vector<tU8>& data ) const
{
   dia_tclFnctTrace oTrace("RoutineCtrlRunIn::writeRunResult");

   tDiaResult  retCode         = DIA_SUCCESS;
   // get size at beginning
   size_t startSize = data.size();
   DIA_TR_INF(" --- writeRunResult => startPos:%zu",startSize);

   // [0:0] Status (HexDump (1 Byte)
   data.push_back(0); // write the statusflasg as ok, will be overwritten later if failed
   //tU32 runStatusPos = startSize + 1;

   for(int i=0; i<g_iSizeRunInCommandList();++i)
   {
      retCode = writeDatoToResponseBuffer(run, &runInCommandList[i],data);
      if(retCode != DIA_SUCCESS)
      {
         DIA_TR_ERR(" !!! writeRunResult => neg response -> set status Run:%d FAILED ",run);
         // Data incorrect set status to false
         data[startSize] = 0x01; // flase
      }
   }// for(int i=0; i<g_iSizeRunInCommandList();++i)

   // verification check
   size_t finalSize = data.size();       // get size at beginning
   size_t size = finalSize - startSize;  // calculate written size

   if(DATA_RESULT_SIZE_PER_RUN != size)
   {
      DIA_TR_ERR(" !!! writeRunResult => Size Mismatch !!! Run:%u Size:%d<>%zu",run, DATA_RESULT_SIZE_PER_RUN,size);
      retCode = DIA_FAILED;
      // should not happen assert, check configuration
      NORMAL_M_ASSERT_ALWAYS();
   }
   return retCode;
}

} //namespace dia
