/*
 * dia_SrvHandlerGenericTemperatureRead.cpp
 *
 *  Created on: 09.07.2012
 *      Author: shw2abt
 */

// TTFis: DIA_REQ UDS 04 22 63 50 (for optical drive)
// TTFis: DIA_REQ UDS 04 22 63 70 (GYRO)
// TTFis: DIA_REQ UDS 04 22 63 80 (eMMC)
// TTFis: DIA_REQ UDS 04 22 63 82 (AMP)
// TTFis: DIA_REQ UDS 04 22 63 87 (IMX)
// TTFis: DIA_REQ UDS 04 22 62 04 (DISPLAY)

#include "common/services/uds/generic/dia_SrvHandlerGenericTemperatureRead.h"

#ifndef __INCLUDED_DIA_MESSAGE_BUFFER__
#include "common/framework/engine/dia_MessageBuffer.h"
#endif

#ifndef __INCLUDED_DIA_DEFINES_UDS__
#include <common/framework/protocols/uds/dia_defsUds.h>
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#define REQUEST_LENGTH          4
#define DATA_START              3
#define DATA_TEMP_CUST_LENGTH   3
#define DATA_TEMP_PROD_LENGTH   3
#define DATA_FUNC_LENGTH        1
#define TEMP_LOW_LIMIT          -500
#define TEMP_HIGH_LIMIT         2040

// collect all DIDs used for CM temperature
tU16 aDidProduction[] =
{
   DIA_C_U16_DID_RBCM_OPTICAL_DRIVE_TEMPERATURE,
   DIA_C_U16_DID_RBCM_GYRO_TEMPERATURE,
   DIA_C_U16_DID_RBCM_EMMC_TEMPERATURE,
   DIA_C_U16_DID_RBCM_HEATSINK_TEMPERATURE,
   DIA_C_U16_DID_RBCM_AMP_TEMPERATURE,
   DIA_C_U16_DID_RBCM_V850_TEMPERATURE,
   DIA_C_U16_DID_RBCM_PANEL_TEMPERATURE,
   DIA_C_U16_DID_RBCM_DVD_DRIVER_TEMPERATURE,
   DIA_C_U16_DID_RBCM_GPS_TEMPERATURE,
   DIA_C_U16_DID_RBCM_IMX_TEMPERATURE,
   DIA_C_U16_DID_RBCM_DISPLAY_TEMPERATURE,
   DIA_C_U16_DID_RBCM_XMMODULE_TEMPERATURE,
   DIA_C_U16_DID_RBCM_TVMODULE_TEMPERATURE,
   DIA_C_U16_DID_RBCM_TVMODULESLAVE_TEMPERATURE,
   DIA_C_U16_DID_RBCM_PANEL2_TEMPERATURE
};

//-----------------------------------------------------------------------------
dia_SrvHandlerGenericTemperatureRead::dia_SrvHandlerGenericTemperatureRead ( tCString name, tU8 sid, tU16 did, tU8 temperatureID )
   : dia_ServiceHandlerUDS(name,sid,(tU16) did),
     mTemperatureID(temperatureID)
{
#ifdef __DIA_UNIT_TESTING__
   dia_tclFnctTrace trc("dia_SrvHandlerGenericTemperatureRead::dia_SrvHandlerGenericTemperatureRead(void)");
#endif

   tU16 u16Size = sizeof(aDidProduction) / sizeof(tU16);
   mProductionRequest = FALSE;

   for (tU16 i = 0; i < u16Size; i++)
   {
      if (aDidProduction[i] == did)
      {
         mProductionRequest = TRUE;
       return;
     }
   }
}

//-----------------------------------------------------------------------------
dia_SrvHandlerGenericTemperatureRead::dia_SrvHandlerGenericTemperatureRead ( tCString name, tU8 sid, tU16 did )
   : dia_ServiceHandlerUDS(name,sid,(tU16) did),
     mTemperatureID(DIA_EN_HEATCTRL_TEMPERATURE_ID_UNKNOWN)
{
#ifdef __DIA_UNIT_TESTING__
   dia_tclFnctTrace trc("dia_SrvHandlerGenericTemperatureRead::dia_SrvHandlerGenericTemperatureRead(tCString,tU8,tU16)");
#endif

   tU16 u16Size = sizeof(aDidProduction) / sizeof(tU16);
   mProductionRequest = FALSE;

   for (tU16 i = 0; i < u16Size; i++)
   {
      if (aDidProduction[i] == did)
      {
         mProductionRequest = TRUE;
       return;
     }
   }
}

//-----------------------------------------------------------------------------
dia_SrvHandlerGenericTemperatureRead::~dia_SrvHandlerGenericTemperatureRead( void )
{
    _BP_TRY_BEGIN
    {
       (void) unsetSysAdapterListener<dia_IHeatControlListener>(this);
    }
    _BP_CATCH_ALL
    {
        DIA_TR_ERR("EXCEPTION CAUGHT: dia_SrvHandlerGenericTemperatureRead::~dia_SrvHandlerGenericTemperatureRead !!!");
        DIA_ASSERT_ALWAYS();
    }
    _BP_CATCH_END
}

//-------------------------------------------------------------------------------
void
dia_SrvHandlerGenericTemperatureRead::vProcessRequest ( const std::vector<tArgsType>& /*vecArgs*/ )
{
   dia_tclFnctTrace trc("dia_SrvHandlerGenericTemperatureRead::vProcessRequest");

   tDiaResult requestErrorState = DIA_FAILED;
   tU8 nrc = DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT;

   dia_IHeatControl* pHeatControl = 0;

   if( oDiagMsgBuffer().u16GetDataLength() == REQUEST_LENGTH )
   {
      if ( querySysAdapterInterface<dia_IHeatControl>(&pHeatControl) == DIA_SUCCESS )
      {
         if ( pHeatControl )
         {
            (void) setSysAdapterListener<dia_IHeatControlListener>(this);

#if defined (DIA_C_U16_DID_VW_ECU_FUNC_LEVEL_SOUND)
            if (getDID() == DIA_C_U16_DID_VW_ECU_FUNC_LEVEL_SOUND)
            {
               requestErrorState = pHeatControl->getFuncState();
            }
            else
#endif /* DIA_C_U16_DID_VW_ECU_FUNC_LEVEL_SOUND */
            {
               dia_eHeatCtrlTemperatureClass tempratureClass = getTemperatureClass ();
               if ( tempratureClass != DIA_EN_HEATCTRL_TEMPERATURE_CLASS_ALL)
               {
                  requestErrorState = pHeatControl->getTemperatures(tempratureClass);
               }
               else
               {
                  DIA_TR_INF("dia_SrvHandlerGenericTemperatureRead::vProcessRequest unsupported sensor ID.");
                  requestErrorState = DIA_E_NOT_SUPPORTED;
               }
            }
         }
      }
   }
   else
   {
      nrc = DIA_E_U8_UDS_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT;
   }

   if ( requestErrorState != DIA_SUCCESS )
   {
      if ( pHeatControl )
      {
         (void) unsetSysAdapterListener<dia_IHeatControlListener>(this);
      }
      DIA_TR_ERR(" ---  SEND TO HEAT CONTROL SERVER FAILED!!!!");
      oDiagMsgBuffer().vSetNegResp(nrc);
      vResReadyAndQuit();
   }
}

//-----------------------------------------------------------------------------
void
dia_SrvHandlerGenericTemperatureRead::vOnTemperatureInfo ( std::list<dia_tHeatCtrlTemperature>& tempList )
{
   dia_tclFnctTrace trc("dia_SrvHandlerGenericTemperatureRead::vOnTemperatureInfo");

   (void) unsetSysAdapterListener<dia_IHeatControlListener>(this);

   if ( tempList.empty() )
   {
      oDiagMsgBuffer().vSetNegResp(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
      DIA_TR_INF("dia_SrvHandlerGenericTemperatureRead::vOnTemperatureInfo list<dia_tHeatCtrlTemperature> is empty!");

      vResReadyAndQuit();
   }
   else
   {
      if ( mProductionRequest )
      {
         vResponseOnProductionReq ( tempList );
      }
      else
      {
         vResponseOnCustomerReq ( tempList );
      }
   }
}

//-----------------------------------------------------------------------------
void
dia_SrvHandlerGenericTemperatureRead::vOnEmergencyFunc ( std::list<dia_tHeatCtrlEmergencyFunc>& funcList )
{
   dia_tclFnctTrace trc("dia_SrvHandlerGenericTemperatureRead::vOnEmergencyFunc");

   if(funcList.empty())
   {
      oDiagMsgBuffer().vSetNegResp(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
      DIA_TR_INF("dia_SrvHandlerGenericTemperatureRead::vOnEmergencyFunc list<dia_tHeatCtrlEmergencyFunc> is empty!");
   }

   dia_eHeatCtrlEmergencyFuncID funcID = DIA_EN_HEATCTRL_EMERGENCY_FUNC_ID_UNKNOWN;

   std::list<dia_tHeatCtrlEmergencyFunc>::iterator iter = funcList.begin();
   for ( ; iter != funcList.end(); iter++ )
   {
      if((*iter).mState == DIA_EN_HEATCTRL_EMERGENCY_FUNC_STATE_ACTIVE) // imply: heatCtrl muss set at least one active state
      {
        DIA_TR_INF("dia_SrvHandlerGenericTemperatureRead::vOnEmergencyFunc --- found active emergency function ID: %d", iter->mID );

        if (iter->mID > funcID) // replace with more critical level
        {
             funcID = iter->mID;
        }
      }
   }

   // assemble response
   DIA_TR_INF("dia_SrvHandlerGenericTemperatureRead::vOnEmergencyFunc - Assemble positive response with funcID: %d", funcID);
   DIA_TR_INF("dia_SrvHandlerGenericTemperatureRead::vOnEmergencyFunc - convert funcID to func level: %d", getFuncLevelPerEmergencyFunc(funcID));

   oDiagMsgBuffer().vSetPosResp();
   oDiagMsgBuffer().vSetDataLength(DATA_START + DATA_FUNC_LENGTH);
   tU8 u8DataPos = DATA_START;
   (void) oDiagMsgBuffer().vSetDataU8(u8DataPos, (tU8) getFuncLevelPerEmergencyFunc(funcID) );

   (void) unsetSysAdapterListener<dia_IHeatControlListener>(this);
   vResReadyAndQuit();
}

//-----------------------------------------------------------------------------
void
dia_SrvHandlerGenericTemperatureRead::vResponseOnCustomerReq ( std::list<dia_tHeatCtrlTemperature>& tempList ) const
{
   dia_tclFnctTrace trc("dia_SrvHandlerGenericTemperatureRead::vResponseOnCustomerReq");

   dia_eHeatCtrlTemperatureQuality eDiaQuality = DIA_EN_HEATCTRL_TEMPERATURE_QUALITY_UNKNOWN;
   tS16 temperature = 0;
   tS16 temperatureMin = 0;
   tS16 temperatureMax = 0;

   std::list<dia_tHeatCtrlTemperature>::iterator iter = tempList.begin();
   for ( ; iter != tempList.end(); iter++ )
   {
      if((*iter).mID == mTemperatureID)
      {
         temperature = (tS16)(*iter).mTemperature;
         eDiaQuality = (*iter).mDiaQuality;

         temperatureMin = (tS16)(*iter).mTemperatureMin;
         temperatureMax = (tS16)(*iter).mTemperatureMax;
      }
   }

   // assemble response
   DIA_TR_INF("dia_SrvHandlerGenericTemperatureRead::vResponseOnCustomerReq - Assemble positive response with temp: %d", temperature);
   DIA_TR_INF("dia_SrvHandlerGenericTemperatureRead::vResponseOnCustomerReq - min temp: %d", temperatureMin);
   DIA_TR_INF("dia_SrvHandlerGenericTemperatureRead::vResponseOnCustomerReq - max temp: %d", temperatureMax);
   DIA_TR_INF("dia_SrvHandlerGenericTemperatureRead::vResponseOnCustomerReq - dia quality: %d", eDiaQuality);

   oDiagMsgBuffer().vSetPosResp();
   oDiagMsgBuffer().vSetDataLength(DATA_START+DATA_TEMP_CUST_LENGTH);
   tU8 u8DataPos = DATA_START;

   if(eDiaQuality != DIA_EN_HEATCTRL_TEMPERATURE_QUALITY_VALID)
   {
       DIA_TR_ERR("dia_SrvHandlerGenericTemperatureRead::vResponseOnCustomerReq - Unknown or invalid temperature received !!!!");
       (void) oDiagMsgBuffer().vSetDataU8(u8DataPos++, 0xFF);
       (void) oDiagMsgBuffer().vSetDataU8(u8DataPos++, 0xFF);
       (void) oDiagMsgBuffer().vSetDataU8(u8DataPos, 0xFF);
   }
   else
   {
       if ((temperature < TEMP_LOW_LIMIT) || (temperature > TEMP_HIGH_LIMIT))
       {
           DIA_TR_ERR("dia_SrvHandlerGenericTemperatureRead::vResponseOnCustomerReq - the value of temperature is out of limit !!!!");
           (void) oDiagMsgBuffer().vSetDataU8(u8DataPos++, 0xFF);
           (void) oDiagMsgBuffer().vSetDataU8(u8DataPos++, 0xFF);
           (void) oDiagMsgBuffer().vSetDataU8(u8DataPos, 0xFF);
       }
       else
       {
           (void) oDiagMsgBuffer().vSetDataU8(u8DataPos++, (tU8) (50 + temperature/10) );
           (void) oDiagMsgBuffer().vSetDataU8(u8DataPos++, (tU8) (50 + temperatureMin/10) );
           (void) oDiagMsgBuffer().vSetDataU8(u8DataPos, (tU8) (50 + temperatureMax/10) );
       }
   }

   vResReadyAndQuit();
}

//-----------------------------------------------------------------------------
void
dia_SrvHandlerGenericTemperatureRead::vResponseOnProductionReq ( std::list<dia_tHeatCtrlTemperature>& tempList )
{
   dia_tclFnctTrace trc("dia_SrvHandlerGenericTemperatureRead::vResponseOnProductionReq");

   dia_eHeatCtrlTemperatureQuality eDiaQuality = DIA_EN_HEATCTRL_TEMPERATURE_QUALITY_UNKNOWN;
   tS16 s16Temp = 0;
   tU8 tempQuality = 0;

   std::list<dia_tHeatCtrlTemperature>::iterator iter = tempList.begin();
   DIA_TR_INF("dia_SrvHandlerGenericTemperatureRead::vResponseOnProductionReq: search in the list for temperature ID: %d", mTemperatureID);
   for ( ; iter != tempList.end(); iter++ )
   {
      if((*iter).mID == mTemperatureID)
      {
         s16Temp = (tS16)(*iter).mTemperature;
         tempQuality = (*iter).mQuality;
         eDiaQuality = (*iter).mDiaQuality;
       DIA_TR_INF("dia_SrvHandlerGenericTemperatureRead::vResponseOnProductionReq: found temperature = %d DIA quality = %d", s16Temp, eDiaQuality);
      }
   }

   // assemble response
   oDiagMsgBuffer().vSetPosResp();
   oDiagMsgBuffer().vSetDataLength(DATA_START+DATA_TEMP_PROD_LENGTH);
   tU8 u8DataPos = DATA_START;

   if(eDiaQuality != DIA_EN_HEATCTRL_TEMPERATURE_QUALITY_VALID)
   {
      DIA_TR_ERR("dia_SrvHandlerGenericTemperatureRead::vResponseOnProductionReq: Unknown or invalid temperature received !!!!");
      (void) oDiagMsgBuffer().vSetDataU8(u8DataPos++, 0x80); // 0x8000 -> -3276.8 C degree
      (void) oDiagMsgBuffer().vSetDataU8(u8DataPos++, 0x00);
      (void) oDiagMsgBuffer().vSetDataU8(u8DataPos, (tU8) tempQuality);
   }
   else
   {
      (void) oDiagMsgBuffer().vSetDataU8( u8DataPos++, (tU8) (((tU16) s16Temp)>>8) );
      (void) oDiagMsgBuffer().vSetDataU8( u8DataPos++, (tU8) s16Temp );
      (void) oDiagMsgBuffer().vSetDataU8( u8DataPos, (tU8) tempQuality );
   }

   vResReadyAndQuit();
}

//-----------------------------------------------------------------------------
dia_eHeatCtrlFuncLevel
dia_SrvHandlerGenericTemperatureRead::getFuncLevelPerEmergencyFunc ( dia_eHeatCtrlEmergencyFuncID emergencyFuncID ) const
{
   dia_eHeatCtrlFuncLevel funcLevel = DIA_EN_HEATCTRL_FUNC_LEVEL_NORMAL;

   switch (emergencyFuncID)
   {
   case DIA_EN_HEATCTRL_EMERGENCY_FUNC_ID_SYSTEMSHUTDOWN:
      funcLevel = DIA_EN_HEATCTRL_FUNC_LEVEL_SHOTDOWN;
     break;

   case DIA_EN_HEATCTRL_EMERGENCY_FUNC_ID_BACKLIGHTOFF:
   case DIA_EN_HEATCTRL_EMERGENCY_FUNC_ID_CDDRIVEROFF:
      funcLevel = DIA_EN_HEATCTRL_FUNC_LEVEL_LIMITED;
     break;

   case DIA_EN_HEATCTRL_EMERGENCY_FUNC_ID_ATTENUATION:
      funcLevel = DIA_EN_HEATCTRL_FUNC_LEVEL_NOTLAUF;
     break;

   default:
      break;
   }

   return funcLevel;
}

//-----------------------------------------------------------------------------
dia_eHeatCtrlTemperatureClass
dia_SrvHandlerGenericTemperatureRead::getTemperatureClass ( void ) const
{
   dia_eHeatCtrlTemperatureClass tempClass = DIA_EN_HEATCTRL_TEMPERATURE_CLASS_ALL;

   switch(mTemperatureID)
   {
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_V850:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_PANEL:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_CDDRIVE:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_DVDDRIVE:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_GPS:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_GYRO:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_EMMC:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_AMP:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_IMX:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_HEATSINK:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_DISPLAY:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_XMMODULE:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_TVMODULE:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_TVMODULESLAVE:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_PANEL2:
      {
         tempClass = DIA_EN_HEATCTRL_TEMPERATURE_CLASS_REAL;
         break;
     }

   case DIA_EN_HEATCTRL_TEMPERATURE_ID_AMPLIFIER:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_CPU:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_EXPBOARD:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_BOARD:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_DRIVE:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_SYSTEM:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_MAIN:
      {
         tempClass = DIA_EN_HEATCTRL_TEMPERATURE_CLASS_ESTIMATED;
         break;
     }

   default:
      {
         break;
     }
   }

   return tempClass;
}
