/*
 * dia_SrvHandlerGenericRoutineCtrl.cpp
 *
 *  Created on: 21.08.2012
 *      Author: gib2hi
 */

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_ROUTINE_CONTROL__
#include <common/framework/protocols/uds/rtctrl/dia_RoutineCtrl.h>
#endif

#ifndef __INCLUDED_DIA_ROUTINE_CONTROL_MANAGER__
#include <common/framework/protocols/uds/rtctrl/dia_RoutineCtrlManager.h>
#endif

#ifndef __INCLUDED_DIA_ROUTINE__
#include <common/framework/protocols/uds/rtctrl/dia_Routine.h>
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include <common/framework/sysadapters/dia_SystemAdapterFacade.h>
#endif

#ifndef __INCLUDED_DIA_LOOKUPKEY__
#include <common/framework/engine/dia_LookupKey.h>
#endif

#ifndef __INCLUDED_DIA_DEFINES_UDS__
#include <common/framework/protocols/uds/dia_defsUds.h>
#endif

#ifndef __INCLUDED_DIA_FACTORY__
#include "common/framework/application/dia_Factory.h"
#endif
// include own class
#include "dia_SrvHandlerGenericRoutineCtrl.h"

using namespace std;

#define DIA_U16_ROUTINE_CTRL_DEFAULT_INDEX_START_DATA    ((tU16)     5)  // index for payload start in the message (for standard UDS messages == 5)
#define DIA_U16_ROUTINE_CTRL_MIN_LENGTH                  ((tU16)     5)  // one leading length byte + sid + 2 byte did + ctrl byte
#define DIA_U16_ROUTINE_CTRL_DEFAULT_TIMEOUT_VALUE       ((tU16) 10000)

//-----------------------------------------------------------------------------

dia_SrvHandlerGenericRoutineCtrl::dia_SrvHandlerGenericRoutineCtrl ( void )
   : dia_ServiceHandlerUDS("dia_SrvHandlerGenericRoutineCtrl",DIA_C_U8_UDS_SID_ROUTINE_CONTROL),
     mpRtCtrlInterface(getInstanceOfRoutineControlManager()),
     mUID(0),
     mEntryOption(DIA_EN_RTCTRL_ENTRY_OPTION_UNKNOWN),
     mPayloadIndex(DIA_U16_ROUTINE_CTRL_DEFAULT_INDEX_START_DATA)
{
   dia_tclFnctTrace oTrace("dia_SrvHandlerGenericRoutineCtrl::dia_SrvHandlerGenericRoutineCtrl()");
}

//------------------------------------------------------------------------------

dia_SrvHandlerGenericRoutineCtrl::~dia_SrvHandlerGenericRoutineCtrl ( void )
{
   mpRtCtrlInterface = 0;
}

//------------------------------------------------------------------------------

void
dia_SrvHandlerGenericRoutineCtrl::vReset ( void )
{
   mUID = 0; // hash code not calculated
   mEntryOption = DIA_EN_RTCTRL_ENTRY_OPTION_UNKNOWN;
   mResults.clear();
}

//------------------------------------------------------------------------------

tU32
dia_SrvHandlerGenericRoutineCtrl::getTimeoutValue ( void ) const
{
   dia_tclFnctTrace oTrace("dia_SrvHandlerGenericRoutineCtrl::getTimeoutValue()");

   tU32 maxTime = DIA_U16_ROUTINE_CTRL_DEFAULT_TIMEOUT_VALUE;

   dia_IRoutineControl* pRCtrlMgr = 0;
   if ( querySysAdapterInterface<dia_IRoutineControl>(&pRCtrlMgr) == DIA_SUCCESS )
   {
      // extract the identifier from the message buffer and convert to the corresponding signal type
      //dia_eRoutineID id = eGetRoutineID(extractRoutineID());
     dia_UID uid = getRoutineID(extractRoutineID());

      DIA_TR_INF("Routine ID: %d", uid);

      if ( uid != DIA_EN_RTCTRL_ID_UNKNOWN )
      {
         dia_Routine* pRoutine = 0;
         if ( mpRtCtrlInterface && (mpRtCtrlInterface->queryRoutine(uid,&pRoutine) == DIA_SUCCESS) && pRoutine )
         {
            maxTime = pRoutine->getMaxProcessingTime();
            DIA_TR_INF("Maximum Processing Time for Routine: %d", maxTime);
         }
      }
   }

   DIA_TR_INF("##### TIMEOUT VALUE: %d ms", maxTime);

   return maxTime;
}

//------------------------------------------------------------------------------

void
dia_SrvHandlerGenericRoutineCtrl::vProcessRequest ( const std::vector<tArgsType>& /*vecArgs*/ )
{
   dia_tclFnctTrace oTrace("dia_SrvHandlerGenericRoutineCtrl::vProcessRequest()");

   // data length contains at least sid, ctrl byte and DID --> 4 bytes minimum
   tU16 dataLength = oDiagMsgBuffer().u16GetDataLength();
   if ( dataLength <  DIA_U16_ROUTINE_CTRL_MIN_LENGTH )
   {
      DIA_TR_INF("dataLength is shorter than %d", DIA_U16_ROUTINE_CTRL_MIN_LENGTH);
      vSendNegativeResponse(getInstanceOfFactory()->makeNRC(DIA_E_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT));
      return;
   }

   // check that we have access to the routine control manager
   if ( querySysAdapterInterface<dia_IRoutineControl>(&mpRtCtrlInterface) != DIA_SUCCESS )
   {
      DIA_TR_INF("querySysAdapterInterface failed");
      vSendNegativeResponse(getInstanceOfFactory()->makeNRC(DIA_E_GENERAL_PROGRAMMING_FAILURE));
      return;
   }

   // initialize members
   vReset();

   // query for the given routine control
   dia_Routine* pRoutine = 0;
   if ( (mpRtCtrlInterface->queryRoutineByDID(extractRoutineID(),&pRoutine) == DIA_SUCCESS) && pRoutine )
   {
      mUID = pRoutine->getUID();
      DIA_TR_INF("mUID was set to %d", mUID);
   }

   if ( !mUID )
   {
       DIA_TR_INF("UNKNOWN ROUTINE ID !!");
       vSendNegativeResponse(getInstanceOfFactory()->makeNRC(DIA_E_OUT_OF_RANGE));
       return;
   }

   // extract control state 1
   mEntryOption = extractEntryOption();

   DIA_TR_INF("RoutineControlEntryOption: 0x%02x", (tU32) mEntryOption);

   if ( pRoutine )
   {
      (tVoid) setSysAdapterListener<dia_IRoutineControlListener>(this);

      tDiaResult retCode = DIA_SUCCESS;

      switch ( mEntryOption )
      {
      case DIA_EN_RTCTRL_ENTRY_OPTION_START:
         {
            retCode = startRoutine(*pRoutine);
            if ( retCode != DIA_SUCCESS ) DIA_TR_ERR("dia_SrvHandlerGenericRoutineCtrl: FAILED (START ROUTINE REQUEST) !!");
         }
         break;

      case DIA_EN_RTCTRL_ENTRY_OPTION_STOP:
         {
            retCode = stopRoutine(*pRoutine);
            if ( retCode != DIA_SUCCESS ) DIA_TR_ERR("dia_SrvHandlerGenericRoutineCtrl: FAILED (STOP ROUTINE REQUEST) !!");
         }
         break;

      case DIA_EN_RTCTRL_ENTRY_OPTION_RESULT_GET:
         {
            retCode = getRoutineResult(*pRoutine);
            if ( retCode != DIA_SUCCESS ) DIA_TR_ERR("dia_SrvHandlerGenericRoutineCtrl: FAILED (GET ROUTINE RESULT) !!");
         }
         break;

      default:
         {
            DIA_TR_INF("UNKNOWN ROUTINE CONTROL ENTRY OPTION !!");
            (void) unsetSysAdapterListener<dia_IRoutineControlListener>(this);
            retCode = DIA_E_SUBFUNCTION_NOT_SUPPORTED;
         }
         break;
      }

      if ( retCode != DIA_SUCCESS )
      {
         vSendNegativeResponse(getInstanceOfFactory()->makeNRC(retCode));
      }
   }
}

//------------------------------------------------------------------------------

tDiaResult
dia_SrvHandlerGenericRoutineCtrl::startRoutine ( dia_Routine& routine )
{
   dia_tclFnctTrace oTrace("dia_SrvHandlerGenericRoutineCtrl::startRoutine()");

   // fill the parameter vector with data extracted from the UDS message
   std::vector<tU8> params;
   for ( tU16 i=getPayloadIndex(); i<oDiagMsgBuffer().u16GetDataLength(); i++ )
   {
      params.push_back(oDiagMsgBuffer().u8GetData(i));
   }

   // start routine and fill result vector if routine was completed
   tDiaResult retCode = routine.start(params,DIA_C_U8_UDS_RTCTRL_TIMER_VALUE_INFINITE);
   if ( retCode == DIA_SUCCESS )
   {
      if ( routine.bIsResultReady() )
      {
         (void) unsetSysAdapterListener<dia_IRoutineControlListener>(this);

         DIA_TR_INF("dia_SrvHandlerGenericRoutineCtrl::startRoutine bIsResultReady returned TRUE");

         // routine was completed, so fill the result vector and return positive response if result vector was filled properly
         retCode = routine.requestResult( mResults );
         if ( retCode == DIA_SUCCESS )
         {
            DIA_TR_INF("vSendRoutineCtrlPositiveResponse(%u,mResults.size()=%zu)", mUID, mResults.size());
            vSendRoutineCtrlPositiveResponse(mUID,mResults);
         }
         else
         {
             //vSendNegativeResponse(getInstanceOfFactory()->makeNRC(retCode));
            DIA_TR_INF("requestResult returned !=DIA_SUCCESS retCode=0x%08X", retCode);
         }
      }
      else
      {
         // routine was started successfully, but is not yet complete. We have to wait asynchronously.
         DIA_TR_INF("bIsResultReady returned FALSE. dia_SrvHandlerGenericRoutineCtrl::startRoutine returned DIA_NOT_DONE.");
      }
   }
   else
   {
      (void) unsetSysAdapterListener<dia_IRoutineControlListener>(this);
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_SrvHandlerGenericRoutineCtrl::stopRoutine ( dia_Routine& routine )
{
   dia_tclFnctTrace oTrace("dia_SrvHandlerGenericRoutineCtrl::stopRoutine()");

   tDiaResult retCode=DIA_SUCCESS;

   if(!routine.isSubfunctionStopSupported())
   {
      retCode = DIA_E_ROUTINE_SUBFUNCTION_NOT_SUPPORTED;
      DIA_TR_INF("stop() is not supported by this routine");
   }
   else
   {
      // fill the parameter vector with data extracted from the UDS message
      std::vector<tU8> params;
      for ( tU16 i=getPayloadIndex(); i<oDiagMsgBuffer().u16GetDataLength(); i++ )
      {
         params.push_back(oDiagMsgBuffer().u8GetData(i));
      }

      // stop routine and fill result vector if routine was completed
      retCode = routine.stop(params);
      if ( retCode == DIA_SUCCESS )
      {
         if ( routine.bIsResultReady() )
         {
            (void) unsetSysAdapterListener<dia_IRoutineControlListener>(this);

            DIA_TR_INF("dia_SrvHandlerGenericRoutineCtrl::stopRoutine bIsResultReady returned TRUE");

            // routine was completed, so fill the result vector and return positive response if result vector was filled properly
            retCode = routine.requestResult( mResults );
            if ( retCode == DIA_SUCCESS )
            {
               DIA_TR_INF("vSendRoutineCtrlPositiveResponse(%u,mResults.size()=%zu)", mUID, mResults.size());
               vSendRoutineCtrlPositiveResponse(mUID,mResults);
            }
            else
            {
               DIA_TR_INF("requestResult returned !=DIA_SUCCESS retCode=0x%08X", retCode);
            }
         }
         else
         {
            // routine was started successfully, but is not yet complete. We have to wait asynchronously.
            DIA_TR_INF("bIsResultReady returned FALSE. dia_SrvHandlerGenericRoutineCtrl::startRoutine returned DIA_NOT_DONE.");
         }
      }
      else
      {
         (void) unsetSysAdapterListener<dia_IRoutineControlListener>(this);
      }
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_SrvHandlerGenericRoutineCtrl::getRoutineResult ( dia_Routine& routine )
{
   dia_tclFnctTrace oTrace("dia_SrvHandlerGenericRoutineCtrl::getRoutineResult()");

   tDiaResult retCode = DIA_SUCCESS;

   if(!routine.isSubfunctionRequestResultSupported())
   {
      retCode = DIA_E_ROUTINE_SUBFUNCTION_NOT_SUPPORTED;
      DIA_TR_INF("requestResult() is not supported by this routine");
   }
   else if ( routine.isSubfunctionRequestResultSupported() && routine.bIsResultReady() )
   {
      (void) unsetSysAdapterListener<dia_IRoutineControlListener>(this);

      retCode = routine.requestResult( mResults );
      if ( retCode == DIA_SUCCESS )
      {
         DIA_TR_INF("vSendRoutineCtrlPositiveResponse(%u,mResults.size()=%zu)", mUID, mResults.size());
         vSendRoutineCtrlPositiveResponse(mUID,mResults);
      }
      else
      {
         DIA_TR_INF("requestResult(results) returned != DIA_SUCCESS retCode=0x%08X", retCode);
      }
   }
   else
   {
      // routine was either not started at all or it was started successfully, but is not yet complete. We have to wait asynchronously.
      DIA_TR_INF("bIsResultReady returned FALSE (Routine not ready yet!!");

      retCode = routine.requestResult(); // requestResult without any params !!
      if ( retCode == DIA_SUCCESS )
      {
         if ( routine.bIsResultReady() )
         {
            (void) unsetSysAdapterListener<dia_IRoutineControlListener>(this);
            retCode = routine.requestResult(mResults);
            if ( retCode == DIA_SUCCESS )
            {
               vSendRoutineCtrlPositiveResponse(mUID,mResults);
            }
            else
            {
               DIA_TR_INF("requestResult(results) returned !=DIA_SUCCESS retCode=0x%08X", retCode);
            }
         }
      }
      else if ( retCode == DIA_E_ROUTINE_RESULT_PENDING )
      {
         // result requested but not yet available. we have to wait, but that is ok
         DIA_TR_INF("Waiting for requested result !!");
         retCode = DIA_SUCCESS;
      }
      else
      {
         DIA_TR_INF("requestResult() failed. retCode=0x%08X", retCode);
      }
   }

   return retCode;
}

//------------------------------------------------------------------------------

void
dia_SrvHandlerGenericRoutineCtrl::vOnRoutineUpdate ( dia_Routine& routine )
{
   dia_tclFnctTrace oTrace("dia_SrvHandlerGenericRoutineCtrl::vOnRoutineUpdate()");

   if ( (routine.getUID() == mUID) && routine.bIsResultReady() )
   {
      (void) unsetSysAdapterListener<dia_IRoutineControlListener>(this);

      tDiaResult retCode = routine.requestResult(mResults);
      if ( retCode == DIA_SUCCESS )
      {
         vSendRoutineCtrlPositiveResponse(mUID,mResults);
      }
      else
      {
         vSendNegativeResponse(getInstanceOfFactory()->makeNRC(retCode));
      }
   }
}

//------------------------------------------------------------------------------

tU16
dia_SrvHandlerGenericRoutineCtrl::getPayloadIndex ( void )
{
   return mPayloadIndex;
}

//------------------------------------------------------------------------------

void
dia_SrvHandlerGenericRoutineCtrl::vSendRoutineCtrlPositiveResponse ( dia_UID uID, std::vector<tU8>& data ) const
{
   dia_tclFnctTrace oTrace("dia_SrvHandlerGenericRoutineCtrl::vSendRoutineCtrlPositiveResponse()");

   tU16 subID = getRoutineUdsID(uID);

   DIA_TR_INF("vSendRoutineCtrlPositiveResponse: id=0x%02x data.size=%zu subID=0x%04x", uID, data.size(), subID);

   if ( subID != 0xFFFF )
   {
      // results for this request
      vector<tU8> results;
      results.push_back(static_cast<tU8>(mEntryOption));
      results.push_back(static_cast<tU8>(subID >> 8));
      results.push_back(static_cast<tU8>(subID));

      DIA_TR_INF("Added hex: %02x %02x %02x", mEntryOption, (subID >> 8), subID);

      for ( std::vector<tU8>::iterator iter=data.begin(); iter != data.end(); ++iter )
      {
         DIA_TR_INF("Adding 0x%02x to the response !", (*iter));
         results.push_back(*iter);
      }

      vSendPositiveResponse(static_cast<tU16>(results.size() + 1), &results);
   }
}

//------------------------------------------------------------------------------

void
dia_SrvHandlerGenericRoutineCtrl::vOnTimeout ( void )
{
   dia_tclFnctTrace oTrace("dia_SrvHandlerGenericRoutineCtrl::vOnTimeout()");

   (void) unsetSysAdapterListener<dia_IRoutineControlListener>(this);

   if ( mpRtCtrlInterface && (mUID != DIA_EN_RTCTRL_ID_UNKNOWN) )
   {
      DIA_TR_INF("vOnTimeout: Forwarding Timeout... !!");
      mpRtCtrlInterface->onRoutineTimeout(mUID);
   }

   oDiagMsgBuffer().vSetNegResp(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
   vResReadyAndQuit();
}

//------------------------------------------------------------------------------

dia_UID
dia_SrvHandlerGenericRoutineCtrl::getRoutineID ( tU16 subID ) const
{
   //dia_eRoutineID id = DIA_EN_RTCTRL_ID_UNKNOWN;
   dia_UID uid = 0;

   if ( mpRtCtrlInterface )
   {
      dia_Routine* pRoutine = 0;
      if ( (mpRtCtrlInterface->queryRoutineByDID(subID,&pRoutine) == DIA_SUCCESS) && pRoutine )
      {
         uid = pRoutine->getUID();
         DIA_TR_INF("getRoutineID: 0x%04x (1)", uid);
      }
   }

   DIA_TR_INF("getRoutineID: 0x%04x (2)", uid);

   return uid;
}

//------------------------------------------------------------------------------

tU16
dia_SrvHandlerGenericRoutineCtrl::getRoutineUdsID ( dia_UID uID ) const
{
   tU16 udsID = 0xFFFF;

   if ( mpRtCtrlInterface )
   {
      dia_Routine* pRoutine = 0;
      if ( (mpRtCtrlInterface->queryRoutine(uID,&pRoutine) == DIA_SUCCESS) && pRoutine )
      {
         udsID = pRoutine->getDID();
      }
   }

   return udsID;
}

//------------------------------------------------------------------------------

dia_eRoutineEntryOption
dia_SrvHandlerGenericRoutineCtrl::extractEntryOption ( void )
{
   dia_tclFnctTrace oTrace("dia_SrvHandlerGenericRoutineCtrl::extractEntryOption()");

   dia_eRoutineEntryOption entryOption = DIA_EN_RTCTRL_ENTRY_OPTION_UNKNOWN;

   tU8 option = oDiagMsgBuffer().u8GetData(2);

   DIA_TR_INF("ROUTINE ENTRY OPTION: 0x%02x", option);

   switch ( option )
   {
   case 0x01:
      entryOption = DIA_EN_RTCTRL_ENTRY_OPTION_START;
      break;

   case 0x02:
      entryOption = DIA_EN_RTCTRL_ENTRY_OPTION_STOP;
      break;

   case 0x03:
      entryOption = DIA_EN_RTCTRL_ENTRY_OPTION_RESULT_GET;
      break;

   default:
      break;

   }

   return entryOption;
}

//------------------------------------------------------------------------------

tU16
dia_SrvHandlerGenericRoutineCtrl::extractRoutineID ( void ) const
{
   tU16 id = (tU16) ((((tU32) oDiagMsgBuffer().u8GetData(3)) << 8) + oDiagMsgBuffer().u8GetData(4));

   DIA_TR_INF("ROUTINE-DID: 0x%04x", id);

   return id;
}

//------------------------------------------------------------------------------

tDiaResult
dia_SrvHandlerGenericRoutineCtrl::makeLookupKeys ( std::vector<dia_LookupKey*>& keys )
{
#ifdef __DIA_UNIT_TESTING__
   dia_tclFnctTrace trc("dia_SrvHandlerGenericRoutineCtrl::makeLookupKeys");
#endif

   if ( mLookupKeys.empty() )
   {
      const std::map<dia_UID,dia_Routine*>& routineRep = dia_RoutineCtrlManager::getInstance()->getRoutines();
      std::map<dia_UID,dia_Routine*>::const_iterator cIter = routineRep.begin();
      for ( ; cIter != routineRep.end(); cIter++ )
      {
         tU16 did = cIter->second->getDID();
         mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_ROUTINE_CONTROL, (tU16) did, DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      }
   }

   keys = mLookupKeys;

   return DIA_SUCCESS;
}


