/*!
 * \file       dia_RoutineCtrlExecuteSystemCommand.cpp
 *
 * \brief      {enter brief description here}
 *
 * \details    {enter detailed description here}
 *
 * \component  {enter component name}
 *
 * \ingroup    {enter group name}
 *
 * \copyright  (c) 2016 Robert Bosch Car Multimedia GmbH
 *
 */

#ifndef __INCLUDED_DIA_ROUTINE_CTRL_EXECUTE_SYSTEM_COMMAND__
#include <common/services/uds/generic/dia_RoutineCtrlExecuteSystemCommand.h>
#endif

#ifndef __INCLUDED_DIA_UTILITIES__
#include <common/framework/utils/dia_utilities.h>
#endif

#include <errno.h>    //lint !e451 !e537 repeatedly included header file without standard include guard
#include <unistd.h>   //lint !e451 !e537 repeatedly included header file without standard include guard
#include <sys/wait.h> //lint !e451 !e537 repeatedly included header file without standard include guard

static const char*
convertString2ConstCharPtr ( const std::string& str )
{
   return str.c_str();
}

namespace dia
{

//---------------------------------------------------------------------------------------------------------------------

RoutineCtrlExecuteSystemCommand::RoutineCtrlExecuteSystemCommand ( tCString name, tU16 udsID, const std::string& cmdName, dia_eRoutineType routineType )
   : dia_Routine(name, udsID, routineType),
     mFullCommandName(cmdName),
     mResponseStatus(ROUTINE_STOPPED),
     mChildProcessID(0),
     mIsRunning(false),
     mErrorCode(DIA_E_NO_ERROR),
	 mResultStatus(0)
{
   dia_tclFnctTrace oTrace("dia::RoutineCtrlExecuteSystemCommand::RoutineCtrlExecuteSystemCommand");
   (void) RoutineCtrlExecuteSystemCommand::setup();
}

//---------------------------------------------------------------------------------------------------------------------

RoutineCtrlExecuteSystemCommand::RoutineCtrlExecuteSystemCommand ( const std::string& name, tU16 udsID, const std::string& cmdName, dia_eRoutineType routineType )
   : dia_Routine(name, udsID, routineType),
     mFullCommandName(cmdName),
     mResponseStatus(ROUTINE_STOPPED),
     mChildProcessID(0),
     mIsRunning(false),
     mErrorCode(DIA_E_NO_ERROR),
	 mResultStatus(0)
{
   dia_tclFnctTrace oTrace("dia::RoutineCtrlExecuteSystemCommand::RoutineCtrlExecuteSystemCommand");
   (void) RoutineCtrlExecuteSystemCommand::setup();
}

//---------------------------------------------------------------------------------------------------------------------

RoutineCtrlExecuteSystemCommand::~RoutineCtrlExecuteSystemCommand ( void )
{
   _BP_TRY_BEGIN
   {
      (void) RoutineCtrlExecuteSystemCommand::tearDown();
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("dia::RoutineCtrlExecuteSystemCommand::~RoutineCtrlExecuteSystemCommand - Exception caught!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-----------------------------------------------------------------------------

tDiaResult
RoutineCtrlExecuteSystemCommand::setup ( void )
{
   dia_tclFnctTrace trc("RoutineCtrlExecuteSystemCommand::setup");
   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
RoutineCtrlExecuteSystemCommand::tearDown ( void )
{
   if ( mIsRunning && mChildProcessID )
   {
      // kill the running child process
      RoutineCtrlExecuteSystemCommand::killChildProcess(mChildProcessID);
      mChildProcessID = 0;
      mIsRunning = false;
    }

   return DIA_SUCCESS;
}


//---------------------------------------------------------------------------------------------------------------------

void
RoutineCtrlExecuteSystemCommand::killChildProcess ( pid_t pid ) const
{
   dia_tclFnctTrace oTrace("dia::RoutineCtrlExecuteSystemCommand::killChildProcess(pid_t)");

   if ( pid )
   {
      DIA_TR_INF("dia::RoutineCtrlExecuteSystemCommand::killChildProcess => Going to call 'kill' on child process");

      // If pid is negative, then signal is sent to every process in the process group whose ID is -mChildPID. SIGTERM(15) => Termination request.
      (void)::kill(-mChildProcessID, SIGTERM);
      DIA_TR_INF("dia::RoutineCtrlExecuteSystemCommand::killChildProcess => Sent SIGTERM to child process");

      // finally get once waitpid, don't want any zombies
      int status = 0;
      (void) ::waitpid(-1, &status, WUNTRACED | WCONTINUED);
      DIA_TR_INF("dia::RoutineCtrlExecuteSystemCommand::killChildProcess => Done");
   }
}

//---------------------------------------------------------------------------------------------------------------------

void
RoutineCtrlExecuteSystemCommand::vInitialize ( void )
{
   dia_tclFnctTrace oTrace("dia::RoutineCtrlExecuteSystemCommand::vInitialize");

   dia_Routine::vInitialize();
//   mIsResultReady = FALSE;
   mResponseStatus = ROUTINE_OK;
}

//---------------------------------------------------------------------------------------------------------------------

bool
RoutineCtrlExecuteSystemCommand::isExisting ( void )
{
   dia_tclFnctTrace oTrace("dia::RoutineCtrlExecuteSystemCommand::isExisting");

   bool retCode = true;

   if ( ::access(mFullCommandName.c_str(), F_OK) == -1 )
   {
      // command is not existing
      DIA_TR_ERR("##### COMMAND NOT FOUND (\"%s\") #####", mFullCommandName.c_str());
      // do not continue otherwise execv will crash
      mErrorCode = DIA_E_GENERAL_REJECT;
      mResponseStatus = ROUTINE_NOK;
//    DIA_ASSERT_ALWAYS (); // assert here, as this is a critical behaviour
      retCode = false;
   }

   DIA_TR_ERR("##### dia::RoutineCtrlExecuteSystemCommand::isExisting => RETURNS \"%s\" #####", (retCode) ? "TRUE" : "FALSE");

   return retCode;
}

//---------------------------------------------------------------------------------------------------------------------

bool
RoutineCtrlExecuteSystemCommand::isRunning ( void )
{
   dia_tclFnctTrace oTrace("dia::RoutineCtrlExecuteSystemCommand::isRunning");

   if ( mIsRunning )
   {
      DIA_TR_ERR("##### isRunning(): COMMAND IS RUNNING. NEED TO BE STOPPED FIRST #####");

      mErrorCode = DIA_E_SEQUENCE_ERROR;
      mResponseStatus = ROUTINE_NOK; // do not continue
   }

   DIA_TR_ERR("##### dia::RoutineCtrlExecuteSystemCommand::isRunning => RETURNS \"%s\" #####", (mIsRunning) ? "TRUE" : "FALSE");

   return mIsRunning;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
RoutineCtrlExecuteSystemCommand::start ( std::vector<tU8>& params, tU8 /*timerValue*/ )
{
   dia_tclFnctTrace oTrace("dia::RoutineCtrlExecuteSystemCommand::start");

   mErrorCode = DIA_E_NO_ERROR;

   // prepare processing of the routine
//   mResults.clear();
   vInitialize();

   mErrorCode = evaluateStartParameters(params);
   if ( mErrorCode != DIA_SUCCESS )
   {
      DIA_TR_INF("RoutineCtrlExecuteSystemCommand::start INVALID PARAMETERS (ERR = 0x%08x)", mErrorCode);
      return DIA_FAILED;
   }

   executeRequestSystemCommandResults();

   if ( isExisting() && (!isRunning()) )
   {
      DIA_TR_INF("RoutineCtrlExecuteSystemCommand::start STARTING SCRIPT \"%s\"", this->mFullCommandName.c_str());

      // setup the command line to start the system command
      setSystemCommandArguments();
      // execute the system command. subclasses may overload this method to implement specific execution logic
      executeStartSystemCommand();
   }
   else
   {
      DIA_TR_ERR("dia::RoutineCtrlExecuteSystemCommand::start => command not existing or already running");
      mResponseStatus = ROUTINE_NOK;
      mIsResultReady  = TRUE;
   }

//   // fill response here, will be ignored by requestResult, which is called directy after this by framework
//   if ( mIsResultReady == TRUE )
//   {
//      mResults.push_back(mResponseStatus);
//   }

   return mErrorCode;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
RoutineCtrlExecuteSystemCommand::stop ( std::vector<tU8>& params )
{
   dia_tclFnctTrace oTrace("dia::RoutineCtrlExecuteSystemCommand::stop");

   mErrorCode = DIA_E_NO_ERROR;

   // prepare processing of the routine
   vInitialize();

   mErrorCode = evaluateStopParameters(params);
   if ( mErrorCode != DIA_SUCCESS )
   {
      DIA_TR_INF("RoutineCtrlExecuteSystemCommand::stop INVALID PARAMETERS (ERR = 0x%08x)", mErrorCode);
      return DIA_FAILED;
   }

   if ( mIsRunning )
   {
      executeStopSystemCommand();
   }
   else
   {
      DIA_TR_INF("dia::RoutineCtrlExecuteSystemCommand::stop => Child not running");
      // use last mResponseStatus
   }

   // fill repsonse here, so it will be ignored by requestResult, which is called directy after this by framework
//   mResults.push_back(mResponseStatus);
   mIsResultReady = TRUE;
   return mErrorCode;
}

//------------------------------------------------------------------------------

tDiaResult
RoutineCtrlExecuteSystemCommand::requestResult ( std::vector<tU8>& results )
{
   dia_tclFnctTrace oTrace("dia::RoutineCtrlExecuteSystemCommand::requestResult");

   results.clear();

   DIA_TR_INF("mErrorCode is 0x%08X, mResponseStatus is 0x%08X", mErrorCode, mResponseStatus);

   //bug fix for double start
   if ( (DIA_E_SEQUENCE_ERROR==mErrorCode) && (ROUTINE_NOK==mResponseStatus) )
   {
      if (mIsRunning)
      {
         executeRequestSystemCommandResults();

         if (!mIsRunning || (ROUTINE_RUNNING==mResponseStatus))
         {
            DIA_TR_INF("mErrorCode is 0x%08X, mResponseStatus is 0x%08X. Set mErrorCode to DIA_SUCCESS.", mErrorCode, mResponseStatus);
            mErrorCode = DIA_SUCCESS;
         }
      }
   }

   if (DIA_SUCCESS==mErrorCode)
   {
      DIA_TR_INF("mErrorCode is DIA_SUCCESS");

      if ( mIsRunning )
      {
         executeRequestSystemCommandResults();
      }
      else
      {
         // process is still terminate, so we have to use last data we get return last mResponseStatus
         DIA_TR_INF("RoutineCtrlExecuteSystemCommand::requestResult => Child is not running");
         //need status result
      }

      DIA_TR_INF("RoutineCtrlExecuteSystemCommand::requestResult => mResponseStatus:%d", mResponseStatus);
      results.push_back(mResponseStatus);

      //push back result status
      // data was filled before by start or stop, so use this
      results.insert(results.end(),mResults.begin(),mResults.end());

      for ( tU16 i=0; i<mResults.size(); i++ )
      {
         DIA_TR_INF("mResults[%02d] = 0x%02x",i,mResults[i]);
      }
   }

   mIsResultReady = TRUE;

   return mErrorCode;
}

//---------------------------------------------------------------------------------------------------------------------

void
RoutineCtrlExecuteSystemCommand::setSystemCommandArguments ( void )
{
   dia_tclFnctTrace oTrace("dia::RoutineCtrlExecuteSystemCommand::setSystemCommandArguments");

   mSystemCommandArgs.clear();

   // first argument is the command name itself
   mSystemCommandArgs.push_back(mFullCommandName);
   // load other command line parameters
   (void) getCommandArguments(mSystemCommandArgs);

   for ( tU16 i=0; i<mSystemCommandArgs.size(); i++ )
   {
      DIA_TR_INF("mSystemCommandArgs[%02d] = %s",i,mSystemCommandArgs[i].c_str());
   }
}

//-----------------------------------------------------------------------------

void
RoutineCtrlExecuteSystemCommand::executeStartSystemCommand ( void )
{
   dia_tclFnctTrace oTrace("RoutineCtrlExecuteSystemCommand::executeStartSystemCommand()");
   startSystemCommand();
}

//-----------------------------------------------------------------------------

void
RoutineCtrlExecuteSystemCommand::startSystemCommand ( void )
{
   dia_tclFnctTrace oTrace("RoutineCtrlExecuteSystemCommand::startSystemCommand()");

   DIA_TR_INF("dia::RoutineCtrlExecuteSystemCommand:: fork now !!");
   mChildProcessID = ::fork();

   if ( mChildProcessID >= 0 )
   {
      // child process forked successfully

      if ( mChildProcessID == 0 )
      {
         // at this point we are running in the child process ( pid == 0 )

         std::vector<const char*> cargv(mSystemCommandArgs.size()+1);
         std::transform(mSystemCommandArgs.begin(), mSystemCommandArgs.end(), cargv.begin(), convertString2ConstCharPtr);  //lint !e864 Info : Expression involving variable possibly depends on order of evaluation
         cargv[mSystemCommandArgs.size()] = 0;
         // Create new process group with child as leader.
         if ( ::setpgid(0,0) < 0 ) exit(EXIT_FAILURE);
         (void) ::execv(mFullCommandName.c_str(),const_cast<char**>(cargv.data()));

         // otherwise exit failed
         ::exit(EXIT_FAILURE);
      }
      else
      {
         // at this point we are running in the parent process ( pid > 0 )

         // check status of child process, but do not wait (WNOHANG)!!!
         int status = -1;
         pid_t cpid = ::waitpid(mChildProcessID, &status, WNOHANG);

         switch ( cpid )
         {
         case 0: // FOUND
            {
               // we can't say at this moment we really run succesfully or not, but still trace info
               DIA_TR_INF("dia::RoutineCtrlExecuteSystemCommand::start => Child Running Id: %d", mChildProcessID);
               DIA_TR_INF("dia::RoutineCtrlExecuteSystemCommand::start => Child WIFEXITED code: %d", WIFEXITED(status));
               DIA_TR_INF("dia::RoutineCtrlExecuteSystemCommand::start => Child WEXITSTATUS code: %d", WEXITSTATUS(status));
               mIsRunning = true;
               mResponseStatus = ROUTINE_RUNNING;
               mIsResultReady  = TRUE;

               onSystemCommandStarted();
            }
            break;

         case -1:// ERROR
            {
               DIA_TR_ERR("dia::RoutineCtrlExecuteSystemCommand::start => waitpid FAILED - Child Id:%d Errno:%x", mChildProcessID,errno);
               DIA_ASSERT_ALWAYS();
               mIsRunning = false;
               mResponseStatus = ROUTINE_NOK;
               mIsResultReady  = TRUE;

               onSystemCommandStartupFailed();
            }
            break;

         default:
            {
               // from The Open Group Base Specifications Issue 7; IEEE Std 1003.1, 2013 Edition
               //
               // WIFEXITED(stat_val)
               //   Evaluates to a non-zero value if status was returned for a child process that terminated normally.
               //
               // WEXITSTATUS(stat_val)
               //   If value of WIFEXITED(stat_val) is non-zero, this macro evaluates to the low-order 8 bits
               //   of the status argument that the child process passed to _exit() or exit(), or the value
               //   the child process returned from main().

               mResponseStatus = ROUTINE_NOK;

               // this is the first request after exit, so data is valid
               int response = WIFEXITED (status);
               DIA_TR_INF("dia::RoutineCtrlExecuteSystemCommand::start::requestResult => Child WIFEXITED code: %d", response);
               if ( response != 0 )
               {
                  // terminated normally
                  response = WEXITSTATUS (status);
                  DIA_TR_INF("dia::RoutineCtrlExecuteSystemCommand::start::requestResult => Child WEXITSTATUS code: %d",response);
                  if ( response == 0 )
                  {
                     mResponseStatus = ROUTINE_OK;
                  }
                  (void) evaluateCommandResponse(response);
               }

               mChildProcessID = 0;
               mIsRunning = false;
               mIsResultReady  = TRUE;
            }
            break;
         }
      }
   }
   else
   {
      DIA_TR_ERR("dia::RoutineCtrlExecuteSystemCommand::start => fork failed");
      mResponseStatus = ROUTINE_NOK;
   }
}

//---------------------------------------------------------------------------------------------------------------------

void
RoutineCtrlExecuteSystemCommand::onSystemCommandStarted ( void )
{
   dia_tclFnctTrace oTrace("dia::RoutineCtrlExecuteSystemCommand::onSystemCommandStarted()");
}

//---------------------------------------------------------------------------------------------------------------------

void
RoutineCtrlExecuteSystemCommand::onSystemCommandStartupFailed ( void )
{
   dia_tclFnctTrace oTrace("dia::RoutineCtrlExecuteSystemCommand::onSystemCommandStartupFailed()");
}

//-----------------------------------------------------------------------------

void
RoutineCtrlExecuteSystemCommand::executeStopSystemCommand ( void )
{
   dia_tclFnctTrace oTrace("RoutineCtrlExecuteSystemCommand::executeStopSystemCommand()");
   stopSystemCommand();
}

//-----------------------------------------------------------------------------

void
RoutineCtrlExecuteSystemCommand::stopSystemCommand ( void )
{
   dia_tclFnctTrace oTrace("RoutineCtrlExecuteSystemCommand::stopSystemCommand()");

   int status;
   pid_t cpid = ::waitpid(mChildProcessID, &status, WNOHANG);

   if ( cpid == 0 )
   {
      DIA_TR_INF("dia::RoutineCtrlExecuteSystemCommand::stopSystemCommand => Child still running KILL Proc:%d",mChildProcessID);
      // kill the running child process
      killChildProcess(mChildProcessID);
      mResponseStatus = ROUTINE_STOPPED;
   }
   else
   {
      // child already terminated, thus no kill is required
      DIA_TR_INF("dia::RoutineCtrlExecuteSystemCommand::stopSystemCommand => Child WIFEXITED code: %d Status:%d", WIFEXITED(status),status);
      mResponseStatus = WEXITSTATUS(status) ? ROUTINE_OK : ROUTINE_NOK;
   }

   mIsRunning = false;
   mChildProcessID = 0;
}

//-----------------------------------------------------------------------------

void
RoutineCtrlExecuteSystemCommand::executeRequestSystemCommandResults ( void )
{
   dia_tclFnctTrace oTrace("RoutineCtrlExecuteSystemCommand::executeRequestSystemCommandResults()");
   requestSystemCommandResults();
}

//-----------------------------------------------------------------------------

void
RoutineCtrlExecuteSystemCommand::requestSystemCommandResults ( void )
{
   dia_tclFnctTrace oTrace("RoutineCtrlExecuteSystemCommand::requestSystemCommandResults()");

   int status;
   // check status of child process, but do not wait (WNOHANG)!!!
   DIA_TR_INF("RoutineCtrlExecuteSystemCommand::requestResult => Child Id: %d", mChildProcessID);
   pid_t cpid = ::waitpid(mChildProcessID, &status, WNOHANG);

   switch ( cpid )
   {
   case 0: // FOUND
      {
         // cpid == 0 => mChildPID = cpid
         DIA_TR_INF("RoutineCtrlExecuteSystemCommand::requestResult => Child still running");
         // this process is still running
         mResponseStatus = ROUTINE_RUNNING;
      }
      break;

   case -1: // FAILED
      {
         // ERROR
         DIA_TR_INF("RoutineCtrlExecuteSystemCommand::requestResult => cpid = -1");
         mResponseStatus = ROUTINE_NOK;
         mIsRunning = false;
         mChildProcessID = 0;
      }
      break;

   default:
      {
         // child process exit. this is the first request after exit, so data is valid
         //
         // from The Open Group Base Specifications Issue 7; IEEE Std 1003.1, 2013 Edition
         //   WIFEXITED(stat_val)
         //       Evaluates to a non-zero value if status was returned for a child process that terminated normally.
         //   WEXITSTATUS(stat_val)
         //       If the value of WIFEXITED(stat_val) is non-zero, this macro evaluates to the low-order 8 bits of
         //       the status argument that the child process passed to _exit() or exit(), or the value the child
         //       process returned from main().

         mResponseStatus = ROUTINE_NOK;

         // get ExitStatus of Childprocess
         tU8 response = WIFEXITED (status);
         DIA_TR_INF("RoutineCtrlExecuteSystemCommand::requestResult => Child WIFEXITED code: %d", response);
         if ( response )
         {
            // terminated normally
            response = WEXITSTATUS (status);
            DIA_TR_INF("dia::RoutineCtrlExecuteSystemCommand::requestResult => Child WEXITSTATUS code: %d",response);
            if ( response == 0 )
            {
               mResponseStatus = ROUTINE_OK;
               mResultStatus = response;
            }
            else
            {
            	mResultStatus = response;
            	DIA_TR_INF("dia::RoutineCtrlExecuteSystemCommand::requestResult => Child WEXITSTATUS code : %d",mResultStatus);
            }

            tDiaResult retCode = evaluateCommandResponse(response);

            if ( retCode != DIA_SUCCESS )
            {
               DIA_TR_INF("dia::RoutineCtrlExecuteSystemCommand::requestResult => evaluateCommandResponse returned: 0x%08x",retCode);
//             mErrorCode = DIA_E_CONDITIONS_NOT_CORRECT;
            }
         }
         DIA_TR_INF("RoutineCtrlExecuteSystemCommand::requestResult => Child terminated now!!!");

         mIsRunning = false;
         mChildProcessID = 0;
      }
      break;
   }
}

}

