/**
 * \file      dia_ProcInfo.cpp
 *
 * \brief     {insert brief description here}
 *
 * \details   {insert file description here}
 *
 * \author    fan4hi
 * \date      Feb 07, 2018
 *
 * \copyright Robert Bosch Car Multimedia 2018
 */

#include <dirent.h>

#ifndef __INCLUDED_DIA_UTILITIES__
#include "dia_utilities.h"
#endif

#ifndef __INCLUDED_DIA_PROC_INFO_H_
#include "dia_ProcInfo.h"
#endif // __INCLUDED_DIA_PROC_INFO_H_

namespace dia
{

ProcInfo::ProcInfo() : _pid_cnt(0)
{
}



ProcInfo::~ProcInfo()
{
}



int ProcInfo::GetPID(const char *ProcName)
{
   ScopeTrace oTrace(__PRETTY_FUNCTION__);

   int            n;
   DIR            *dir;
   struct dirent  *entry;
   FILE           *file;
   char           fname[PROCINFO_FILE_NAME_MAX];
   size_t         linelen=0;
   char           *line=NULL, *namestr;

   if(!ProcName)
   {
      if(_pid_cnt>=_pid.size()) {
         DIA_TR_ERR("GetPID() -> No processname provided and no PID cached");
         return -1;
      }
      return _pid[_pid_cnt++];
   }

#ifdef __DIA_UNIT_TESTING__
   if(!strcmp(ProcName, "automounter"))
      return 1234;
   return -1;
#endif // __DIA_UNIT_TESTING__

   _pid.clear();
   _pid_cnt=0;
   dir=opendir("/proc");
   if(dir)
   {
      for(entry=readdir(dir); entry; entry=readdir(dir))
      {
         if(!strcmp(entry->d_name, ".") || !strcmp(entry->d_name, ".."))
            continue;

         if(sscanf(entry->d_name, "%d", &n)==1)
         {
            snprintf(fname, sizeof(fname), "/proc/%i/cmdline", n);
            if((file=fopen(fname, "r"))!=NULL)
            {
               if(getline(&line, &linelen, file)>0)
               {
                  namestr=line;
                  while((namestr=strstr(namestr, ProcName))!=NULL)
                  {
                     if(!strcmp(namestr, ProcName))
                     {
                        _pid.push_back(n);
                        break;
                     }
                     else
                        namestr++;
                  }
               }
               fclose(file);
            }
         }
      }
      closedir(dir);
   }

   if(!_pid.size())
      return -1;
   DIA_TR_INF("GetPID() -> Found PID %d for processname %s", _pid[_pid_cnt], ProcName);
   return _pid[_pid_cnt++];
}



bool ProcInfo::ParseStatValues(std::vector<tU8> &Buffer, int PID, const char *Format)
{
   ScopeTrace oTrace(__PRETTY_FUNCTION__);

   bool     success=false;
   char     *line=NULL;

   if(PID<0)
      return false;

#ifdef __DIA_UNIT_TESTING__
   line=strdup("601 (automounter) S 1 601 601 0 -1 4194368 145 0 0 0 7 6 0 0 20 0 1 0 3398 4186112 161 4294967295 65536 129424 2126806608 0 0 0 16387 4096 0 2148870772 0 0 17 0 0 0 0 0 0 194960 198879 4911104 2126806825 21268060");
   success=ParseStatString(Buffer, Format, line);
   free(line);
#else // #ifdef __DIA_UNIT_TESTING__
   FILE     *file;
   size_t   linelen=0;
   char     fname[PROCINFO_FILE_NAME_MAX];

   snprintf(fname, sizeof(fname), "/proc/%i/stat", PID);
   if(!(file=fopen(fname, "r"))) {
      DIA_TR_ERR("Could not open stat file for PID %d, error: %d", PID, errno);
      return false;
   }

   if(getline(&line, &linelen, file)>0)
      success=ParseStatString(Buffer, Format, line);

   fclose(file);
   free(line);
#endif // #ifdef __DIA_UNIT_TESTING__

   return success;
}



bool ProcInfo::ParseStatMValues(std::vector<tU8> &Buffer, int PID, const char *Format)
{
   ScopeTrace oTrace(__PRETTY_FUNCTION__);

   bool     success=false;
   char     *line=NULL;

   if(PID<0)
      return false;

#ifdef __DIA_UNIT_TESTING__
   line=strdup("1022 161 116 16 0 86 0");
   success=ParseStatString(Buffer, Format, line);
   free(line);
#else // #ifdef __DIA_UNIT_TESTING__
   FILE     *file;
   size_t   linelen=0;
   char     fname[PROCINFO_FILE_NAME_MAX];

   snprintf(fname, sizeof(fname), "/proc/%i/statm", PID);
   if(!(file=fopen(fname, "r")))
      return false;

   if(getline(&line, &linelen, file)>0)
      success=ParseStatString(Buffer, Format, line);

   fclose(file);
   free(line);
#endif // #ifdef __DIA_UNIT_TESTING__

   return success;
}



bool ProcInfo::ParseStatString(std::vector<tU8> &Buffer, const char *Format, char *String)
{
   ScopeTrace oTrace(__PRETTY_FUNCTION__);

   size_t         i, j, frmtlen;
   unsigned int   frmtsize;
   long int       clktck, pagesize;
   int64_t        int64val;
   uint64_t       uint64val;
   std::string    hexstr;
   bool           success=true;
   char           *token;

#ifdef __DIA_UNIT_TESTING__
   clktck=100;
   pagesize=4096;
#else // #ifdef __DIA_UNIT_TESTING__
   clktck=sysconf(_SC_CLK_TCK);
   pagesize=sysconf(_SC_PAGESIZE);
#endif // #ifdef __DIA_UNIT_TESTING__
   frmtlen=strlen(Format);
   for(token=strtok(String, " "), i=0; success && token && i<frmtlen-1; token=strtok(NULL, " "), i+=2)
   {
      if(Format[i]=='(' || Format[i+1]==')')
      {
         if(Format[i]!='(' || Format[i+1]!=')')
         {
            success=false;
            break;
         }
      }
      else if(Format[i]=='X')
      {
         if(Format[i+1]!=' ')
         {
            success=false;
            break;
         }
      }
      else
      {
         frmtsize=Format[i+1]-'0';
         if(frmtsize<1 || frmtsize>8)
         {
            success=false;
            break;
         }
      }

      switch(Format[i])
      {
      case '(':   // The filename in parentheses - not used
         if(token[strlen(token)-1]!=')')
         // If the filename contains spaces, token doesn't contain the whole name and the closing parenthesis is missing.
         // So skip everything up to the closing one.
         // CHECKME: What does Linux do if the filename contains spaces and a closing parenthesis?
         strtok(NULL, ")");
         break;
      case 'c':   // ASCII characters
         if(strlen(token)<frmtsize)
            success=false;
         else
            for(j=0; j<frmtsize; j++)
               Buffer.push_back(token[j]);
         break;
      case 'i':   // A signed integer in Motorola format (high byte first)
         if(!ParseInt64(&int64val, token))
            success=false;
         else
            BufferMotorolaFormat(Buffer, (uint64_t)int64val, frmtsize);
         break;
      case 't':   // A time in clock ticks. Needs conversion to ms
         if(!ParseUInt64(&uint64val, token))
            success=false;
         else
         {
            uint64val=(uint64val*1000/clktck);
            BufferMotorolaFormat(Buffer, uint64val, frmtsize);
         }
         break;
      case 'k':   // A size in kB in Motorola format (high byte first), calculated from bytes
         if(!ParseUInt64(&uint64val, token))
            success=false;
         else
         {
            uint64val=(uint64val/1024);   // CHECKME: 1000 or 1024?
            BufferMotorolaFormat(Buffer, uint64val, frmtsize);
         }
         break;
      case 'p':   // A size in kB in Motorola format (high byte first), calculated from pages
         if(!ParseUInt64(&uint64val, token))
            success=false;
         else
         {
            uint64val=(uint64val*pagesize/1024);
            BufferMotorolaFormat(Buffer, uint64val, frmtsize);
         }
         break;
      case 'h':   // An unsigned integer in Motorola format (high byte first)
      case 'u':   // An unsigned integer in Motorola format (high byte first)
         if(!ParseUInt64(&uint64val, token))
            success=false;
         else
            BufferMotorolaFormat(Buffer, uint64val, frmtsize);
         break;
      case 'X':   // An unused entry
         break;
      default:
         success=false;
         break;
      }
   }

   return success;
}



bool ProcInfo::ParseInt64(int64_t *Out, const char *Str)
{
   const char  *c;
   bool        neg=false;

   *Out=0;

   for(c=Str; isspace(*c); c++)
      if(!*c)
         return false;

   while(*c=='+' || *c=='-')
   {
      if(*c++=='-')
         neg=!neg;
      if(!*c)
         return false;
   }

   for(; *c>='0' && *c<='9'; c++)
      *Out=*Out*10+*c-'0';

   if(neg)
      *Out=-*Out;

   return true;
}



bool ProcInfo::ParseUInt64(uint64_t *Out, const char *Str)
{
   const char  *c;

   *Out=0;

   for(c=Str; isspace(*c); c++)
      if(!*c)
         return false;

   for(; *c>='0' && *c<='9'; c++)
      *Out=*Out*10+*c-'0';

   return true;
}



void ProcInfo::BufferMotorolaFormat(std::vector<tU8> &Buffer, uint64_t Value, unsigned int Bytes)
{
   unsigned int   i;

   for(i=0; i<Bytes; i++)
      Buffer.push_back((tU8)GET_BYTE_N(Value, Bytes-i-1));
}

} // namespace dia
