/**
 * \file      dia_MsgVisitor.h
 *
 * \brief     implement class MsgVisitor and MsgPrinter
 *
 * \details   MsgVisitor: utility-class to walk nested structures and act on the contained base-types (int, uint, string)
 *            MsgPrinter: utility-class to print a nested structure via TTFIS
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCommon
 *
 * \copyright  (c) 2018 Robert Bosch Car Multimedia
 */
#ifndef __INCLUDED_DIA_MSG_VISITOR__
#define __INCLUDED_DIA_MSG_VISITOR__

static inline char const *dia_uintToStr(uint32_t val, char *buf) {
   sprintf(buf, "%u", val);
   return buf;
}

#define DIA_VISIT_MSG_LIST(BASE, FIELD, VISITOR)                       \
   {                                                                   \
      uint32_t numElem=(uint32_t)BASE.FIELD.size();                    \
      dia::FiList list(#FIELD, numElem, VISITOR);                      \
      char buf[33];                                                    \
      for (uint32_t i=0;i<numElem;++i) {                               \
         dia::FiSection section(dia_uintToStr(i, buf), VISITOR);       \
         visitMsg(BASE.FIELD.at(i), VISITOR);                          \
      }                                                                \
   }


#define DIA_VISIT_MSG_SECTION(BASE, FIELD, VISITOR)      \
   {                                                     \
      dia::FiSection section(#FIELD, VISITOR);           \
      visitMsg(BASE.FIELD, VISITOR);                     \
   }


#define DIA_VISIT_MSG_FIELD(BASE, FIELD, VISITOR)    \
   {                                                 \
      VISITOR.handleField(#FIELD, BASE.FIELD);       \
   }

#define DIA_VISIT_MSG_ENUM(BASE, FIELD, VISITOR)    \
   {                                                \
      VISITOR.handleField(#FIELD, (int32_t)BASE.FIELD);      \
   }

#define DIA_VISIT_MSG_FIENUM(BASE, FIELD, VISITOR)     \
   {                                                   \
      VISITOR.handleField(#FIELD, (uint32_t)BASE.FIELD.enType);  \
   }



namespace dia {
class MsgVisitor {
public:
   
   MsgVisitor() {}
   virtual ~MsgVisitor() {}
   virtual void sectionStart(char const * /* name */) {}
   virtual void sectionEnd() {}
   virtual void listStart(char const * /* name */, uint32_t /* numElem */) {}
   virtual void listEnd() {}


   virtual void handleField(char const * /* name */, char const * /* val */) {};
   virtual void handleField(char const * /* name */, std::string const & /* val */) {};

   virtual void handleField(char const * /* name */, uint32_t /* val */) {};
   virtual void handleField(char const * /* name */, int32_t /* val */) {};

};

class FiSection {
public:
   FiSection(char const *name, MsgVisitor &visitor):
      _visitor(visitor){
      _visitor.sectionStart(name);
   }

   ~FiSection() {
      _visitor.sectionEnd();
   }

private:
   MsgVisitor &_visitor;
};



class FiList {
public:
   FiList(char const *name, uint32_t numElem, MsgVisitor &visitor):
      _visitor(visitor){
      _visitor.listStart(name, numElem);
   }

   ~FiList() {
      _visitor.listEnd();
   }

private:
   MsgVisitor &_visitor;
};





class MsgPrinter:public MsgVisitor {
public:
   
   MsgPrinter(char const *name):
      _indent(0) {
      sectionStart(name);
   }
   virtual ~MsgPrinter() {
      sectionEnd();
   }

   void flush() {
      if (_stream.tellp()>0) {
         DIA_TR_INF("%s%s",_indentString.c_str(), _stream.str().c_str());
         _stream.str("");
         _stream.clear();
      }
   }
   virtual void sectionStart(char const *name) {
      flush();
      _stream << name << " = {";
      flush();
      _indent++;
      _indentString=std::string(_indent * 3, ' ');
   }

   virtual void sectionEnd() {
      flush();
      if (_indent>0) {
         _indent--;
         _indentString=std::string(_indent * 3, ' ');
      }
      _stream<<"}";
      flush();
   }

   virtual void listStart(char const *name, uint32_t numElem) {
      flush();
      _stream<<name<< " = [ /* numElem=" << numElem << " */";
      flush();
      _indent++;
      _indentString=std::string(_indent * 3, ' ');
   }

   virtual void listEnd() {
      flush();
      if (_indent>0) {
         _indent--;
         _indentString=std::string(_indent * 3, ' ');
      }
      _stream<<"]";
      flush();
   }

   template<typename VALTYPE>
   void printField(char const *name, VALTYPE val) {
      _stream<< name << " = " << val;
      flush();      
   }

   virtual void handleField(char const *name, char const *val) {
      printField(name, val);
   }
   virtual void handleField(char const *name, std::string const &val) {
      printField(name, val);
   }
   virtual void handleField(char const *name, uint32_t val) {
      printField(name, val);
   };
   virtual void handleField(char const *name, int32_t val) {
      printField(name, val);
   }

private:
   std::stringstream _stream;
   std::string _indentString;
   uint32_t _indent;
};



}

#endif
