#ifndef __INCLUDED_DIA_DOIP__
#define __INCLUDED_DIA_DOIP__

#include <string>
#include <fstream>
#include <streambuf>
#include <sstream>



#include "common/framework/application/dia_Lock.h"

#ifndef __INCLUDED_DIA_UTILITIES__
#include <common/framework/utils/dia_utilities.h>
#endif

#ifndef __INCLUDED_DIA_DOIP_DEFINES__
#include "doipcommon/dia_DoIPDefines.h"
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include "common/framework/config/dia_ConfigManager.h"
#endif

#ifndef __INCLUDED_DIA_UDD_TYPES__
#include "common/framework/udd/udd_types.h"
#endif

#ifndef __INCLUDE_DIA_ACTIVE_OBJECT__
#include <common/framework/application/dia_ActiveObject.h>
#endif


#include "doipserver/DoIP_ServerCfg.h"
#include "doipserver/DoIP_Protocoll.h"
#include "doipserver/DoIP_Server.h"
#include "doipserver/DoIP_Channel.h"
#include "doipserver/DoIP_Connection.h"
#include "doipserver/DoIP_Factory.h"

#include "common/framework/udd/udd_CommChannelConfig.h"


namespace dia {

class dia_DoIP_Factory: public DoIP_Factory {
   virtual DoIP_Channel *createChannel(DoIP_Connection* poConn, DoIP_ChannelType type, tU8 busNumber, tU16 testerSA, tU16 targetSA, tU32 baseaddr, tU8 addrext) override;
};


class dia_IDoIPSender {
public:
virtual void onDiagnosisResponse(const tU8 au8MsgBuffer[], tU16 u16Length)=0;
};

/*
  Interface that has to be implemented by user of DoIP_Channnel

*/
class  dia_IDoIPListener {
public:
   virtual tU32 u32RxStartOfReception(tU32 /*u32Len*/) { return 0; };
   virtual void vTxConfirmation(tBool /*bOk*/) {};
   virtual void vRxError(tU32 /*u32Error*/) {};
   virtual void vRxIndication(tU8 au8Data[], tU32 u32Len) = 0;
   virtual void onDoipChnDestroy() = 0;
};


/*
  We have to derive our eth-implementation from tclEthernetBus, since methods for locking are pure virtual.
  We use standard locking of fc_diagnosis
*/
class dia_EthernetDoIPBus : public tclEthernetBus,
                            public dia_ActiveObject

{
public:
   dia_EthernetDoIPBus(tU8 u8Number, char const *device);

   dia_EthernetDoIPBus(tU8 u8Number, sockaddr_in rAddr);

   bool bStart();

   bool bStop();


protected:


   virtual bool bEnterCriticalSection() override;

   virtual bool bLeaveCriticalSection() override;



private:
   static std::string getLockName(tU8 u8Number);

    //! thread function
   void vThreadEntrypointObject ( void );

   dia::Lock _lock;
   bool _started;
};


class dia_IDoIPChannelCreator;
/*
  Configuration of our server.
  parts of the configuration cannot be hardcoded (e.g. vin)
*/
struct dia_DoIPLocalCfg {
   std::string ethInterfaceName;
   uint8_t busNumber;
   uint16_t portNumber;
   uint16_t localTargetSA;
   std::string eid;
   std::string gid;
   std::string vin;
};


/*
  Routing-configuration of our server.
  This configuration is expected to be hard-coded
*/
struct dia_DoIPCfg {
   struct TargetSA_to_RoutingActivationTarget {
      uint16_t targetSA;
      DoIP_RoutingActivationTarget routingActivationTarget;

   };


   struct RouteNumber_to_RouteCfg {
      uint8_t routeNumber;
      uint16_t bAuthRequ;
   };

   struct RouteNumber_to_RouteTarget {
      uint8_t routeNumber;
      uint16_t targetSA;
   };

   struct TesterSA_to_TesterCfg {
      uint16_t testerSA;
      uint16_t numByteDiagAckNack;
   };

   struct TesterSA_to_RouteNumber {
      uint16_t testerSA;
      uint8_t routeNumber;
   };


   dia_DoIPLocalCfg const *localCfg;
   // list of target that can be part of a route key:targetSA

   uint32_t numTargets;
   TargetSA_to_RoutingActivationTarget const *targets;

   // list of routes, key:routeNumber val:bAuthRequ
   uint32_t numRoutes;
   RouteNumber_to_RouteCfg const *routes;

   // list of targets key:routeNumber
   uint32_t numRouteTargets;
   RouteNumber_to_RouteTarget const *routeTargets;

   // list of testers. each tester can have one or more routes key:testerSA val:numByteDiagAckNack
   uint32_t numTesters;
   TesterSA_to_TesterCfg const *testers;

   // all routes, key:testerSA val:routeNumber
   uint32_t numTesterRoutes;
   TesterSA_to_RouteNumber const *testerRoutes;

   // creators for supported DoIP-channels
   uint32_t numDoIPChannelCreators;
   dia_IDoIPChannelCreator **doIPChannelCreators;


};


class dia_DoIP;
class dia_DoIPLocalCfg;
class dia_DoIPServer : public DoIP_Server {
public:
   dia_DoIPServer(dia_DoIP *diagDoIP, tclEthernetBus* poBus, dia_DoIPLocalCfg const *localCfg);
   ~dia_DoIPServer();


   void vAddDoIPChannel(DoIP_Channel* /*poChan*/) // called from DoIP stack during routing activation
   {
      // do nothing
   }


   void vRemoveDoIPChannel(DoIP_Channel* /*poChan*/) // called from DoIP stack when channels are closed
   {
      // do nothing
   }

   void vPowerModeRequest(void) // called from DoIP stack during during connection setup
   {
      vPowerModeResponse(0x01); // return PowerMode ok
   }

   void vRoutingActivationAuthenticationReq(tU16 sourceAddr, tU8 activationType, tU8* AuthenticationReqData)
   {
      (void) sourceAddr; (void) activationType;
      vRoutingActivationAuthenticationResp(TRUE, AuthenticationReqData); // TODO: implement real authentication
   }
private:
   dia_DoIP *_diagDoIP;
};


struct dia_DoIPChannelCreationParams {
   DoIP_Connection *poConn;
   DoIP_ChannelType type;
   tU8 busNumber;
   tU16 testerSA;
   tU16 targetSA;
   tU32 baseaddr;
   tU8 addrext;
};


/*
  We need to derive from DoIP_Channel to have a channel in the sense of the DoIP-stack.
  Additionaly include interface dia_IDoIPSender to allow user (CommChannel) to send
  messages via dia_DoIPChannel
 */
class dia_DoIP;
class dia_DoIPChannel : public DoIP_Channel,
                           public dia_IDoIPSender
{
public:
   dia_DoIPChannel(DoIP_Connection* poConn, tU16 u16TargetAddress, dia_IDoIPListener *listener):
      DoIP_Channel(poConn, u16TargetAddress),
      _listener(listener)
   {

   }


   virtual ~dia_DoIPChannel() {
      _listener->onDoipChnDestroy();
      _listener=0;
   }

   virtual void vTraceStackDump() override {}

   // interface dia_IDoIPSender
   virtual void onDiagnosisResponse(const tU8 au8MsgBuffer[], tU16 u16Length) override {
      vvDataReq(au8MsgBuffer, u16Length);
   }
   virtual void vvDataReq(tU8 const *au8Data, tU16 u16Len)
   {
      poConn->bChannel_Transmit(this, au8Data, u16Len);
   }

   virtual tU32 u32RxStartOfReception(tU32 u32Len) override {
      return _listener->u32RxStartOfReception(u32Len);
   }
   virtual void vRxIndication(tU8 au8Data[], tU32 u32Len) {
      _listener->vRxIndication(au8Data, u32Len);
   }
   virtual void vTxConfirmation(tBool bOk) override {
      _listener->vTxConfirmation(bOk);

   }
   virtual void vRxError(tU32 u32Error) override {
      _listener->vRxError(u32Error);

   }
private:
   dia_IDoIPListener *_listener;
};


/*
  Provider of comm-channels have to provide a ChannelCreator object.
  When DoIP-stack indicates a new tester (connection),
  dia_DoIP will search a DoIPChannelCreator that fits and ask it to create a channel.
*/
class dia_IDoIPChannelCreator {
public:
   virtual bool fits(DoIP_ChannelType type, tU16 testerSA, tU16 targetSA)=0;
   virtual DoIP_Channel *createChannel(dia_DoIPChannelCreationParams const &chnCreationParams)=0;
};


/*
  Factory to create DoIP-channel on request form DoIP-stack.
*/
class dia_DoIPChannelFactory {
   /*
     method called indirectly by DoIP-stack.
     try to find a creator and create DoIP-channel
   */

public:
   void registerCreator(dia_IDoIPChannelCreator *creator);

   DoIP_Channel* createChannel(dia_DoIPChannelCreationParams const &chnCreationParams);


private:
   std::list<dia_IDoIPChannelCreator *> _creators;
};













class dia_DoIP {
    DECL_SINGLETON_CONSTRUCTOR_AND_DESTRUCTOR(dia_DoIP);
public:
   /*
     Config
     Target: key=targetSA
     targetSA
     channelType
     targetBaseAddr
     targetAddrExt
     busNumer
     Route: key=routeNumber
     routeNumber
     bAuthRequest
     list<Target>
     Tester: key=testerSA
     testerSA
     numByteDiagAckNack
     list<Route>
     Channel: keys:pair<Target, Tester>
     Target
     Tester
     poCon
   */

   void removeServer(dia_DoIPServer *server);


   DoIP_RoutingActivationTarget const *getTarget(uint16_t targetSA);


   //  ethInterfaceName=eth0
   // localUdsAdress=0x0815
   dia_DoIP();
   bool configure(dia_DoIPCfg const &cfg);


   bool startCommunication();


   DoIP_RoutingActivationCfg *getRoute(uint16_t routeNumber);

   DoIP_TesterCfg *getTester(uint16_t testerSA);



   DoIP_Channel *createDoIPChannel(dia_DoIPChannelCreationParams const &chnCreationParams);



private:


   /* configuration */
   dia_DoIPCfg const *_cfg;
   dia_DoIPLocalCfg const *_localCfg;

   /* data and created objects accordign to configuration */
   std::map<uint16_t, DoIP_TesterCfg *> _testers;
   std::map<uint16_t, DoIP_RoutingActivationCfg> _routes;
   dia_DoIPChannelFactory _doIPChannelFactory;

   /* instances of classes that are derived from doip-lib */
   dia_EthernetDoIPBus *_ethBus;
   DoIP_Protocoll* _poDoIPStack;
   dia_DoIPServer *_server;
   dia_DoIP_Factory *_doipFactory;



};
} // namespace dia




#endif
