/*!
 * \file       dia_CommChannelINC.h
 *
 * \brief      Communication channel used for communication between APP processor and UDD on SCC
 *
 * \details    Communication channel between APP processor and UDD on SCC (system companion chip) using INC (Inter-Node-
 *             Communication) communication. This channel implements the UDD-INC message catalogue.
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreUDD
 *
 * \copyright  (c) 2011-2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_COMMCHANNEL_INC__
#define __INCLUDED_DIA_COMMCHANNEL_INC__

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDE_DIA_ACTIVE_OBJECT__
#include <common/framework/application/dia_ActiveObject.h>
#endif

#ifndef __INCLUDED_DIA_INTERFACE_RUNLEVEL_LISTENER__
#include <common/interfaces/dia_IRunLevelListener.h>
#endif

#ifndef __INCLUDED_DIA_INTERFACE_SESSION_LISTENER__
#include <common/interfaces/dia_ISessionListener.h>
#endif

#ifndef __INCLUDED_DIA_INTERFACE_SECURITYLEVEL_LISTENER__
#include <common/interfaces/dia_ISecurityListener.h>
#endif

#ifndef __INCLUDED_DIA_UDD__
#include <common/framework/udd/udd.h>
#endif

#ifndef __INCLUDED_DIA_MESSAGEHANDLER__
#include <common/framework/engine/dia_MessageHandler.h>
#endif

#ifndef __INCLUDED_DIA_INTERFACE_SCC__
#include <common/interfaces/dia_ISCC.h>
#endif

#ifndef __INCLUDED_DIA_INTERFACE_OPMODE_LISTENER__
#include <common/interfaces/dia_IOpModeListener.h>
#endif

namespace dia
{
class SCCManager;
class INCSocket;
class SocketConfiguration;
}

class dia_CommChannelINC
    : public dia_ActiveObject,
      public dia_IRunLevelListener,
      public dia_ISessionListenerImpl,
      public dia_ISecurityListenerImpl,
      public udd_CommChannel,
      public dia_MessageHandler,
      public dia::ISCC,
      public dia_IOpModeListenerImpl
{
   DECL_SINGLETON_CONSTRUCTOR_AND_DESTRUCTOR(dia_CommChannelINC);
   DECL_DEPRECATED_COPYCONSTRUCTOR_AND_ASSIGNMENTOPERATOR(dia_CommChannelINC);

public:
    //! connect to the driver
    virtual tDiaResult connect ( udd_UID uid );
    //! disconnect from the driver
    virtual tDiaResult disconnect ( udd_UID uid );

    //! asynchronous send of a diagnostic message
    virtual tDiaResult sendAsync ( udd_UID /*uid*/, const tU8 /*data*/[], tU16 /*length*/ ) { return DIA_E_NOT_IMPLEMENTED; }

    //! overloaded method from dia_IRunLevelListener
    virtual void vOnRunLevelChanged ( dia_enRunlevel newLevel, dia_enRunlevel oldLevel );
    //! overloaded method from dia_ISessionListener
    virtual void vOnSessionChanged ( tU8 newSession, tU8 oldSession );
    //! overloaded method from dia_IServiceListener
    virtual void vOnSecurityLevelChange ( const dia_SecurityLevel& secLevel, dia_eSecurityLevelStatus newStatus, dia_eSecurityLevelStatus oldStatus );

    // overloaded methods from dia::ISCC

    //! received notification about the current component status
    virtual tDiaResult sendComponentStatusToSCC ( dia::enSccComponentStatus status );
    virtual tDiaResult sendDiagnosisRequestAckToSCC ( tU8 channelID );
    virtual tDiaResult sendStateUpdateAckToSCC ( void );

protected:
    //! protected because we are using the singleton pattern
    dia_CommChannelINC ( void );
    //! protected because we are using the singleton pattern
    virtual ~dia_CommChannelINC ( void );

    //! setup (create objects,...)
    tDiaResult setup ( void );
    //! tear down (destroy objects,...)
    tDiaResult tearDown ( void ); //lint -sem(dia_CommChannelINC::tearDown, cleanup )

    // return the INC channels unique channel ID
    virtual tU8 getChannelID ( void );

    //! thread function
    virtual void vThreadEntrypointObject ( void );

    //! Copy data, create a message and send it to message processor
    void OnDiagRequest (tU8 u8Channel, const unsigned char* in_pTxBuffer, tU16 dataLength);

    // callback method attached to UDS messages used for delivery of message responses
    static void onDiagnosisResponse ( const tU8 au8MsgBuffer[], tU16 u16Length, tCookieType cookie );

//    static tVoid SendDiagResponse ( const tU8 au8MsgBuffer[], tU16 u16Length );
//    //! Send Post processing response in case of Session Handling request
    static void OnPostProcessingResponse(const tU8 data[], tU16 length, void* cookie);

    void onComponentStatus ( void ) const;
    void onDiagnosisRequest ( void );
    void onDiagnosisResponseACK ( void ) const;
    void onStateUpdate ( void );
    void onForcedStateUpdateACK ( void ) const;
    void onTesterDisconnected ( void ) /*const*/;
	void OnPostProcessingRequest (void);

    virtual tDiaResult sendForceStateChangeToSCC ( tU8 channelID, tU8 session, tU32 stateBitMask ) const;
    virtual tDiaResult sendMessageToSCC ( const tU8 data[], tU32 numBytesToSend ) const;
    virtual tDiaResult sendMessageToSCC ( const std::vector<tU8>& data ) const;

public:
    tU8 ReadOpModeState ();

protected:
    //! stateBitmask got in the message SCC_DIAGNOSIS_C_STATE_UPDATE, should be sent back in the message SCC_DIAGNOSIS_C_FORCE_STATE_CHANGE
    tU32 mSccStateBitmask;
    //! tester present indication, true if tester is present
    bool mIsTesterPresent;
    //! message received from SCC
    std::vector<tU8> mSccRxMessage;
    //! message to be sent from iMX to SCC
    std::vector<tU8> mSccTxMessage;
    //! manager for the SCC
    dia::SCCManager* mpSccManager;
    //! socket used for INC communication to V850
    dia::INCSocket* mpSocketINC;
    //! configuration  object for the INC socket
    dia::SocketConfiguration* mpSocketConfig;
};

dia_CommChannelINC* getInstanceOfCommChannelINC ( void );
void releaseInstanceOfCommChannelINC ( void );

#endif /* __INCLUDED_DIA_COMMCHANNEL_INC__ */
