/*!
 * \file       dia_CommChannelINC.cpp
 *
 * \brief      Communication channel used for communication between APP processor and UDD on SCC
 *
 * \details    Communication channel between APP processor and UDD on SCC (system companion chip) using INC (Inter-Node-
 *             Communication) communication. This channel implements the UDD-INC message catalogue.
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreUDD
 *
 * \copyright  (c) 2011-2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_COMMCHANNEL_INC__
#include "dia_CommChannelINC.h"
#endif

#ifndef __INCLUDED_DIA_APPCONTROLLER__
#include "common/framework/application/dia_AppController.h"
#endif

#ifndef __INCLUDED_DIA_ENGINE_MANAGER__
#include "common/framework/engine/dia_EngineManager.h"
#endif

#ifndef __INCLUDED_DIA_ENGINE_SERVER__
#include "common/framework/engine/dia_EngineServer.h"
#endif

#ifndef __INCLUDED_DIA_SESSION_CONTROLLER__
#include "common/framework/application/dia_SessionController.h"
#endif

#ifndef __INCLUDED_DIA_SESSION__
#include "common/framework/engine/dia_Session.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#ifndef __INCLUDED_DIA_COMMON_SECURITY__
#include "common/framework/security/dia_common_security.h"
#endif

#ifndef __INCLUDED_DIA_UTILITIES__
#include <common/framework/utils/dia_utilities.h>
#endif

#ifndef __INCLUDED_DIA_MESSAGE_BUFFER__
#include "common/framework//engine/dia_MessageBuffer.h"
#endif

#ifndef __INCLUDED_DIA_MESSAGE_BUFFER_UDS__
#include <common/framework/protocols/uds/dia_MessageBufferUDS.h>
#endif

#ifndef __INCLUDED_DIA_DIAGSESSION_UDS__
#include "common/depricated/dia_tclDiagSessionUds.h"
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG__
#include "common/framework/config/dia_defsConfig.h"
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include "common/framework/config/dia_ConfigManager.h"
#endif

#ifndef __INCLUDED_DIA_SCC_MANAGER__
#include <common/framework/application/dia_SCCManager.h>
#endif

#ifndef __INCLUDED_DIA_SOCKET_CONFIGURATION__
#include <common/framework/application/dia_SocketConfiguration.h>
#endif

#ifndef __INCLUDED_DIA_SOCKET__
#include <common/framework/application/dia_Socket.h>
#endif

#ifndef __INCLUDED_DIA_DEFS_INC__
#include <common/framework/udd/dia_defsINC.h>
#endif

#include <sys/socket.h>

#ifndef __INCLUDED_DIA_INC__
#include <common/framework/platform/cmd/dia_inc.h>
#endif

#define DIA_CUSTOM_THREAD_NAME  "DRV_DIAG_CUSTOM_THREAD"

DIA_IMPL_SINGLETON_WITH_SETUP_AND_TEARDOWN(dia_CommChannelINC)
using namespace dia;

//-----------------------------------------------------------------------------

#ifndef __DIA_UNIT_TESTING__

dia_CommChannelINC*
getInstanceOfCommChannelINC ( void )
{
   return dia_CommChannelINC::getInstance();
}

void
releaseInstanceOfCommChannelINC ( void )
{
   dia_CommChannelINC::deleteInstance();
}

#endif

//-----------------------------------------------------------------------------

dia_CommChannelINC::dia_CommChannelINC ( void ):
     dia_ActiveObject(DIA_CUSTOM_THREAD_NAME, DIA_PROP_INC_THREAD_PRIO, DIA_PROP_INC_THREAD_STACKSIZE),
     udd_CommChannel("dia_CommChannelINC",UDD_CHANNEL_UID_UDS_CAN),
     mSccStateBitmask(SCC_DIAGNOSIS_C_STATE_UPDATE_INITIAL_BITMASK),
     mIsTesterPresent(false),
     mpSccManager(0),
     mpSocketINC(0),
     mpSocketConfig(0)
{}

//-----------------------------------------------------------------------------

dia_CommChannelINC::~dia_CommChannelINC ( void )
{
   _BP_TRY_BEGIN
   {
      (void) dia_CommChannelINC::tearDown();
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia_CommChannelINC::~dia_CommChannelINC !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-----------------------------------------------------------------------------

tDiaResult
dia_CommChannelINC::setup ( void )
{
   dia_tclFnctTrace trc("dia_CommChannelINC::setup");

   mpSocketConfig = DIA_NEW dia::SocketConfiguration(AF_BOSCH_INC_AUTOSAR, SOCK_STREAM, 0);
   if ( mpSocketConfig )
   {
      mpSocketConfig->setHostNameLocal(std::string("scc-local"));
      mpSocketConfig->setHostNameServer(std::string("scc"));
      mpSocketConfig->setPortLocal(DIA_UDD_PORT);
      mpSocketConfig->setPortServer(DIA_UDD_PORT);

      mpSocketINC = dia::createObjectINCSocket(*mpSocketConfig);
      if ( !mpSocketINC )
      {
         DIA_TR_ERR("### CREATION AND CONFIGURATION OF INC SOCKET FAILED ###");
         DIA_DELETE mpSocketConfig;
         return DIA_FAILED;
      }
   }

   mpSccManager = getInstanceOfSCCManager();
   if ( !mpSccManager )
   {
      DIA_TR_ERR("### CREATION OF SCCMANAGER FAILED ###");
      return DIA_FAILED;
   }

   return mpSccManager->connectProxy(this);
}

//-----------------------------------------------------------------------------

tDiaResult
dia_CommChannelINC::tearDown ( void )
{
   releaseInstanceOfSCCManager();
   mpSccManager = 0;
   DIA_DELETE mpSocketINC;
   mpSocketINC = 0;
   DIA_DELETE mpSocketConfig;
   mpSocketConfig = 0;

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_CommChannelINC::connect ( udd_UID /*uid*/ )
{
   dia_tclFnctTrace oTrace("dia_CommChannelINC::connect");

   if ( !mpSocketINC ) return DIA_E_SOCKET_NOT_AVAILABLE;

   getInstanceOfSCCManager()->connectProxy(this);

   tDiaResult retCode = mpSocketINC->create();
   if ( retCode != DIA_SUCCESS )
   {
      DIA_TR_ERR("##### INC SOCKET CREATION FAILED (ERR=0x%08x) #####",retCode);
      return retCode;
   }

   retCode = mpSocketINC->connect();
   if ( retCode != DIA_SUCCESS )
   {
      DIA_TR_ERR("##### INC SOCKET CONNECTION FAILED (ERR=0x%08x) #####",retCode);
      return retCode;
   }

   retCode = startThread();
   if ( retCode != DIA_SUCCESS)
   {
      DIA_TR_ERR("##### FAILED TO START INC MONITORING THREAD #####");
      return retCode;
   }

   getInstanceOfSCCManager()->onSccConnected();

   dia_AppController* pAppCtrl = getInstanceOfAppController();
   if ( pAppCtrl )
   {
      pAppCtrl->addRunlevelListener(this);
   }
   else
   {
      DIA_TR_ERR("!!! dia_CommChannelINC::connect => ERROR: Unable to register for RunLevel changes");
   }

   dia_EngineServer* pEngine = 0;
   if (( getInstanceOfEngineManager()->queryEngineServer(DIA_UID_ENGINE_CUSTOMER_UDS,&pEngine) == DIA_SUCCESS ) && pEngine)
   {
      pEngine->getSessionController()->addListener(this);
   }
   else
   {
      DIA_TR_INF("!!! dia_CommChannelINC::connect => ERROR: Unable to register for Session changes");
   }

   dia_SecurityManager* pSecMgr = getInstanceOfSecurityManager();
   if ( pSecMgr )
   {
      pSecMgr->addListener(this);
   }
   else
   {
      DIA_TR_INF("!!! dia_CommChannelINC::connect => ERROR: Security manager is not available");
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_CommChannelINC::disconnect ( udd_UID /*uid*/ )
{
   dia_tclFnctTrace oTrace("dia_CommChannelINC::disconnect");

   tDiaResult retCode = DIA_SUCCESS;

   dia_EngineServer* pEngine = 0;
   if (( getInstanceOfEngineManager()->queryEngineServer(DIA_UID_ENGINE_CUSTOMER_UDS,&pEngine) == DIA_SUCCESS ) && pEngine)
   {
      pEngine->getSessionController()->removeListener(this);
   }
   else
   {
      DIA_TR_INF("!!! dia_CommChannelINC::connect => ERROR: Unable to unregister session listener");
   }

   if ( mpSocketINC && mpSocketINC->isConnected() )
   {
      retCode = mpSocketINC->disconnect();
      if ( retCode != DIA_SUCCESS )
      {
         DIA_TR_ERR("dia_CommChannelINC::disconnect - FAILED to disconnect INC channel !!!");
      }
   }

   dia_SecurityManager* pSecMgr = getInstanceOfSecurityManager();
   if ( pSecMgr )
   {
      pSecMgr->removeListener(this);
   }
   else
   {
      DIA_TR_INF("!!! dia_CommChannelINC::disconnect => ERROR: Security manager is not available");
   }

   getInstanceOfSCCManager()->onSccDisconnected();

   if ( mpSocketINC )
   {
      (void) mpSocketINC->close();
   }

   return retCode;
}

//------------------------------------------------------------------------------

tU8
dia_CommChannelINC::getChannelID ( void )
{
   tU8 channelID = DIA_C_U8_INVALID_CHANNEL_ID;
   tU32 retCode = dia_getProperty(DIA_PROP_INC_UDD_CHANNEL, channelID);
   if ( DIA_SUCCESS != retCode )
   {
      DIA_TR_ERR("dia_CommChannelINC::getChannelID: dia_getProperty(DIA_PROP_INC_UDD_CHANNEL) FAILED retCode=0x%08X", retCode);
   }

   return channelID;
}

//------------------------------------------------------------------------------
void
dia_CommChannelINC::onDiagnosisResponse ( const tU8 data[], tU16 length, tCookieType cookie )
{
   dia_tclFnctTrace oTrace("dia_CommChannelINC::onDiagnosisResponse");

   dia_CommChannelINC* pCookiePayload = UnsafeVoidCast<dia_CommChannelINC>(cookie);

   if ( (!mpInstance) || (mpInstance != pCookiePayload) || (!data) )
   {
      DIA_TR_ERR("!!! dia_CommChannelINC::onDiagnosisResponse => ERROR (obj=0x%p, cookie=0x%p, msg=0x%p) !!",mpInstance, pCookiePayload, data);
      return;
   }

   if ( (!(mpInstance->mpSocketINC)) || (!(mpInstance->mpSocketINC->isReady())) )
   {
      DIA_TR_ERR("!!! dia_CommChannelINC::onDiagnosisResponse => ERROR (SOCKET NOT AVAILABLE OR NOT READY) !!");
      return;
   }

   tU8 channelID = mpInstance->getChannelID();

   DIA_TR_SM("INC[%d] UDS msg TX: Length: %d Data: %s", channelID, length, dia::utils::bin2str(data,length,' ').c_str());

   //copy message data to the buffer
   mpInstance->mSccTxMessage.clear();
   mpInstance->mSccTxMessage.reserve(mpInstance->mpSocketINC->getConfiguration()->getMaxMsgSize());
#ifndef VARIANT_S_FTR_DISABLE_INC_TP_MARKER
   mpInstance->mSccTxMessage.push_back(SCC_DIAGNOSIS_DIAG_RESPONSE_TP_MARKER);
#endif
   mpInstance->mSccTxMessage.push_back(SCC_DIAGNOSIS_C_DIAG_RESPONSE); // PDU ID
#if 0 //TODO: UDD does not support multiple channels yet
   mpInstance->mSccTxMessage.push_back(channelID);
#endif
   mpInstance->mSccTxMessage.insert(mpInstance->mSccTxMessage.end(), &data[0], &data[length]);

   (void) mpInstance->mpSocketINC->writeData(mpInstance->mSccTxMessage);
}

//------------------------------------------------------------------------------

void
dia_CommChannelINC::vOnSessionChanged ( tU8 newSession, tU8 oldSession )
{
   dia_tclFnctTrace oTrace("dia_CommChannelINC::vOnSessionChanged");

   if ( (!mpSocketINC) || (!(mpSocketINC->isReady())) ) return;
   if ( newSession == oldSession ) return;

   DIA_TR_INF("dia_CommChannelINC::vOnSessionChanged newSession=0x%02x oldSession=0x%02x", newSession, oldSession);

   tDiaResult retCode = sendForceStateChangeToSCC(getChannelID(),newSession,mSccStateBitmask);
   if ( retCode != DIA_SUCCESS )
   {
      DIA_TR_ERR("dia_CommChannelINC::vOnSessionChanged FAILED (ERROR: 0x%08x)", retCode);
   }
}

//-----------------------------------------------------------------------------

void
dia_CommChannelINC::vOnRunLevelChanged ( dia_enRunlevel newLevel, dia_enRunlevel oldLevel )
{
   dia_tclFnctTrace oTrace("dia_CommChannelINC::vOnRunLevelChanged");

   dia_AppController* pAppCtrl = getInstanceOfAppController();
   if ( !pAppCtrl ) return;

   tU32 appState = (tU32) pAppCtrl->getAppState();

   DIA_TR_INF("dia_CommChannelINC::vOnRunLevelChanged => newLevel = %d, oldLevel = %d, appState = %d", newLevel, oldLevel, appState);

   if ((oldLevel == DIA_EN_RUNLEVEL_UNDEFINED) && (newLevel== DIA_EN_RUNLEVEL_1))
   {
      // Cold start
      getInstanceOfSCCManager()->onSccStatusRequest(dia::DIA_EN_SCC_COMP_STATUS_ACTIVE);
   }

   if ((oldLevel == DIA_EN_RUNLEVEL_2) && (newLevel== DIA_EN_RUNLEVEL_1))
   {
      if ( appState == DIA_C_U32_APP_STATE_OFF )
      {
         // Shutdown
         getInstanceOfSCCManager()->onSccStatusRequest(dia::DIA_EN_SCC_COMP_STATUS_INACTIVE);
      }
   }

   if ((oldLevel == DIA_EN_RUNLEVEL_1) && (newLevel== DIA_EN_RUNLEVEL_2))
   {
      if ( appState == DIA_C_U32_APP_STATE_NORMAL )
      {
         //Warm start
         getInstanceOfSCCManager()->onSccStatusRequest(dia::DIA_EN_SCC_COMP_STATUS_ACTIVE);
      }
   }
}

//-----------------------------------------------------------------------------

void
dia_CommChannelINC::vOnSecurityLevelChange ( const dia_SecurityLevel& secLevel, dia_eSecurityLevelStatus newStatus, dia_eSecurityLevelStatus oldStatus )
{
   dia_tclFnctTrace oTrace("dia_CommChannelINC::vOnSecurityLevelChange");

   if ( newStatus == oldStatus ) return;

   dia_EngineServer* pEngine = 0;
   if (( getInstanceOfEngineManager()->queryEngineServer(DIA_UID_ENGINE_CUSTOMER_UDS,&pEngine) != DIA_SUCCESS ) || (!pEngine))
   {
      DIA_TR_ERR("dia_CommChannelINC::vOnSecurityLevelChange: ERROR: FAILED TO GET ENGINE !!");
      return;
   }

   DIA_TR_INF("dia_CommChannelINC::vOnSecurityLevelChange => UID = 0x%08x, newStatus = %d, oldStatus = %d", secLevel.getUID(), newStatus, oldStatus);

   // create copy of complete current UDD state bitmask
   tU32 stateBitmask = mSccStateBitmask;

   DIA_TR_INF("dia_CommChannelINC::vOnSecurityLevelChange => mSccStateBitmask = 0x%08x", mSccStateBitmask);

   // merge security level bit mask of active security level into current security level bit mask on SCC
   tU8 currentSecurityByte  = (tU8) ((mSccStateBitmask & (~(DIA_REMOVE_SECURITY_LEVEL_BITMASK))) >> DIA_SECURITY_LEVEL_NUMBER_LEFT_SHIFT);
   tU8 secLevelSecurityByte = (tU8) secLevel.getConfiguration().getSecurityLevelBitmaskSCC();

   if ( !secLevelSecurityByte )
   {
      DIA_TR_ERR("ERR: UNKNOWN SECURITY LEVEL. LEVEL IS NOT CHANGED !!");
      return;
   }

   // if level is active set the corresponding bit, otherwise the bit is cleared
   secLevelSecurityByte = ( secLevel.isActive() ) ? tU8(currentSecurityByte | secLevelSecurityByte) : tU8(currentSecurityByte & (~secLevelSecurityByte));

   // insert updated security byte into the state bit mask
   stateBitmask |= (((tU32) secLevelSecurityByte) << DIA_SECURITY_LEVEL_NUMBER_LEFT_SHIFT);

   tDiaResult retCode = sendForceStateChangeToSCC(getChannelID(),pEngine->getActiveSession()->getID(),stateBitmask);
   if ( retCode != DIA_SUCCESS )
   {
      DIA_TR_ERR("dia_CommChannelINC::vOnSecurityLevelChange FAILED (ERROR: 0x%08x)", retCode);
   }
}

//-----------------------------------------------------------------------------

void
dia_CommChannelINC::onTesterDisconnected ( void ) //const
{
   dia_tclFnctTrace oTrace("dia_CommChannelINC::onTesterDisconnected");

   // retrieve the project specific default session value
   tU8 defSession = 0;
   tU32 retCode = dia_getProperty(DIA_PROP_DEFAULT_SESSION, defSession);
   if ( retCode != DIA_SUCCESS )
   {
      DIA_TR_INF("!!!  dia_CommChannelINC::onTesterDisconnected => ERROR: dia_getProperty FAILED retCode=0x%08X", retCode);
      return;
   }

   // we are simulating a session control UDS message that switches to the default session
   static tU8 tempBuffer[] = { 3, DIA_C_U8_UDS_SID_SESSION_CONTROL, 0x00 };
   tempBuffer[2] = defSession;

   DIA_TR_INF("dia_CommChannelINC::onTesterDisconnected: Switching to default session 0x%02x", defSession);

   /*lint -save -e429 Lifetime is controlled by FSM of diagnostic engine (it is deleted by dia_tclDiagSessionUds)*/
   dia_MessageBufferUDS* pMsgBuffer = DIA_NEW dia_MessageBufferUDS (
         &tempBuffer[0],
         tU16(sizeof tempBuffer),
         dia_tclDiagSessionUds::vNullResponse
         );

   if ( !pMsgBuffer )
   {
      DIA_TR_INF("dia_CommChannelINC::onTesterDisconnected ERROR (OUT OF MEMORY) !!");
      return;
   }

   // create and send a ReqRx event to uds engine
   getInstanceOfApplication()->postMessage(DIA_NEW dia_tclDiagSession::tclEventReqRx(pMsgBuffer));
   // trigger IDLE event to inform LCM that diagnosis activity has been terminated
   getInstanceOfApplication()->postMessage(DIA_NEW dia_tclDiagSession::tclEventIdle());

   (void) sendStateUpdateAckToSCC();

   /*lint -restore */
}

//-----------------------------------------------------------------------------

void
dia_CommChannelINC::vThreadEntrypointObject ( void )
{
   dia_tclFnctTrace oTrace("dia_CommChannelINC::vThreadEntrypointObject");

   if ( (!mpSocketINC) || (!(mpSocketINC->getConfiguration())) || (!(mpSocketINC->isReady())) )
   {
      DIA_TR_ERR("--- dia_CommChannelINC::vThreadEntrypointObject => INC SOCKET NOT SETUP PROPERLY. ABORTING ...");
      return;
   }

   do
   {
//      mSccRxMessage.clear();
//      mSccRxMessage.reserve(mpSocketINC->getConfiguration()->getMaxMsgSize());
      if ( mpSocketINC->readData(mSccRxMessage) != DIA_SUCCESS )
      {
         DIA_TR_INF("--- dia_CommChannelINC::vThreadEntrypointObject => Reading INC data failed");
         usleep(50000); // DiagOmitSleepWarning: Wait for the given time and then try to read INC data
         continue;
      }

      DIA_TR_INF("--- dia_CommChannelINC::vThreadEntrypointObject => INC Rx %zu bytes", mSccRxMessage.size());

      if ( mSccRxMessage.size() > 0 )
      {
         DIA_TR_SM("INC msg RX: Length: %zu Data: %s", mSccRxMessage.size(), dia::utils::bin2str(mSccRxMessage,' ').c_str());

         switch ( mSccRxMessage.at(SCC_DIAGMSG_IDX_MSG_ID) )
         {
            case SCC_DIAGNOSIS_R_COMPONENT_STATUS:
               onComponentStatus();
               break;

            case SCC_DIAGNOSIS_C_REQUEST:
               onDiagnosisRequest();
               break;

            case SCC_DIAGNOSIS_R_DIAG_RESPONSE:
               onDiagnosisResponseACK();
               break;

            case SCC_DIAGNOSIS_C_STATE_UPDATE:
               onStateUpdate();
               break;

            case SCC_DIAGNOSIS_R_FORCE_STATE_CHANGE:
               onForcedStateUpdateACK();
               break;

			case SCC_DIAGNOSIS_C_POST_PROCESSING_REQUEST:
				OnPostProcessingRequest();					
				break;

			default:
               DIA_TR_ERR("!!! dia_CommChannelINC::vThreadEntrypointObject => RECEIVED UNKNOWN SCC MESSAGE (PDU-ID=0x%02x)", mSccRxMessage.at(0));
               break;
         }
      }
   }
   while (1);  //lint !e506 Put constant value "1" intentionally
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_CommChannelINC::onComponentStatus ( void ) const
{
   dia_tclFnctTrace oTrace("dia_CommChannelINC::onComponentStatus");

   if ( mSccRxMessage.size() == SCC_DIAGNOSIS_COMPONENT_STATUS_MSGLEN )
   {
      tU8 status       = mSccRxMessage[1];
      tU8 majorVersion = mSccRxMessage[2];
      tU8 minorVersion = mSccRxMessage[3];

      DIA_TR_INF("Received SCC_DIAGNOSIS_R_COMPONENT_STATUS(status=0x%02x,major=%d,minor=%d)",status,majorVersion,minorVersion);

      if ( status == SCC_DIAGNOSIS_COMPONENT_STATUS_ACTIVE )
      {
         getInstanceOfSCCManager()->onSccStatusUpdate(dia::DIA_EN_SCC_COMP_STATUS_ACTIVE);
      }
      else if ( status == SCC_DIAGNOSIS_COMPONENT_STATUS_INACTIVE )
      {
         getInstanceOfSCCManager()->onSccStatusUpdate(dia::DIA_EN_SCC_COMP_STATUS_INACTIVE);
      }
      else
      {
         DIA_TR_INF("UNKNOWN COMPONENT STATUS RECEIVED (0x%02x)",status);
      }
   }
   else
   {
      DIA_TR_ERR("!!! dia_CommChannelINC::onComponentStatus => SCC_DIAGNOSIS_R_COMPONENT_STATUS too short (%zu)", mSccRxMessage.size());
   }
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_CommChannelINC::onDiagnosisRequest ( void )
{
   dia_tclFnctTrace oTrace("dia_CommChannelINC::onDiagnosisRequest");

   if ( (mSccRxMessage.size() > SCC_DIAGNOSIS_C_REQUEST_MSGLEN_MIN) && (mSccRxMessage.size() < DIA_C_U16_INC_DEFAULT_MSG_SIZE) )
   {
      tU8 channelID = mSccRxMessage[1];

      DIA_TR_INF("--- dia_CommChannelINC::onDiagnosisRequest => SCC_DIAGNOSIS_C_REQUEST[%d]", channelID);

      const tU8* pMsgBuffer = (const tU8*) &mSccRxMessage[2];
      tU16 length = (tU16)(mSccRxMessage.size() - SCC_DIAGNOSIS_C_REQUEST_MSGLEN_MIN);

      DIA_TR_SM("INC[%d] UDS msg RX: Length: %d Data: %s", channelID, length, dia::utils::bin2str(pMsgBuffer,length,' ').c_str());

      (void) sendDiagnosisRequestAckToSCC(channelID);

      switch (channelID)
      {
         /*lint -save -e429 Lifetime is controlled by FSM of diagnostic engine (it is deleted by dia_tclDiagSessionUds)*/
         case 0:
            {
               mIsTesterPresent = true;

               // Create an object containing the GMLAN msg.
               dia_MessageBufferUDS* pMsgBufferUDS = DIA_NEW dia_MessageBufferUDS (
                     pMsgBuffer, length,
                     dia_CommChannelINC::onDiagnosisResponse,
                     dia_MessageBuffer::holds_request,
                     dia_MessageBuffer::format_raw,
                     tCookieType(this) // cookie
               );

               if ( pMsgBufferUDS && (handleMessage(*pMsgBufferUDS) == DIA_E_NOT_AVAILABLE) )
               {
                  getInstanceOfApplication()->postMessage(DIA_NEW dia_tclDiagSession::tclEventReqRx(pMsgBufferUDS));
                  DIA_TR_INF("--- dia_CommChannelINC::onDiagnosisRequest => Posted as UDS message");
            }
         }
         break;
         /*lint -restore */

         default:
         {
            DIA_TR_INF("!!! dia_CommChannelINC::onDiagnosisRequest => ERROR: Unknown channel %d", channelID);
            break;
         }
      }
   }
   else
   {
      DIA_TR_INF("!!! dia_CommChannelINC::onDiagnosisRequest bytesRec=%zu", mSccRxMessage.size());
   }
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_CommChannelINC::onDiagnosisResponseACK ( void ) const
{
   dia_tclFnctTrace oTrace("dia_CommChannelINC::onDiagnosisResponseACK");

   if ( mSccRxMessage.size() == SCC_DIAGNOSIS_R_DIAG_RESPONSE_MSGLEN )
   {
      tU8 u8Channel = mSccRxMessage[SCC_DIAGMSG_IDX_CHANNEL_ID];

      DIA_TR_INF("--- dia_CommChannelINC::onDiagnosisResponseACK => SCC_DIAGNOSIS_R_DIAG_RESPONSE[%d]", u8Channel);

      getInstanceOfApplication()->postMessage(DIA_NEW dia_tclDiagSession::tclEventConfTxOk()); //lint !e429: custodial pointer is freed by after engine has processed the message
   }
   else
   {
      DIA_TR_ERR("!!! dia_CommChannelINC::onDiagnosisResponseACK => SCC_DIAGNOSIS_R_DIAG_RESPONSE too short");
   }
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_CommChannelINC::onStateUpdate ( void )
{
   dia_tclFnctTrace oTrace("dia_CommChannelINC::onStateUpdate");

   if ( SCC_DIAGNOSIS_C_STATE_UPDATE_MSGLEN == mSccRxMessage.size() )
   {
      DIA_TR_INF("dia_CommChannelINC::onStateUpdate => SCC_DIAGNOSIS_C_STATE_UPDATE[%d]", mSccRxMessage[SCC_DIAGMSG_IDX_CHANNEL_ID]);

      tU8 sessionId = mSccRxMessage[SCC_DIAGNOSIS_C_STATE_UPDATE_IDX_SESSION_ID];

      tU32 stateBitmask =
            ((tU32)mSccRxMessage[SCC_DIAGNOSIS_C_STATE_UPDATE_IDX_BITMASK_1] <<  0U) |
            ((tU32)mSccRxMessage[SCC_DIAGNOSIS_C_STATE_UPDATE_IDX_BITMASK_2] <<  8U) |
            ((tU32)mSccRxMessage[SCC_DIAGNOSIS_C_STATE_UPDATE_IDX_BITMASK_3] << 16U) |
            ((tU32)mSccRxMessage[SCC_DIAGNOSIS_C_STATE_UPDATE_IDX_BITMASK_4] << 24U) ;

      //store current bitmask value for a message IMX-->V850
      mSccStateBitmask = stateBitmask;

      DIA_TR_INF("dia_CommChannelINC::onStateUpdate session=0x%02X stateBitmask=0x%08X mIsTesterPresent=%s", sessionId, stateBitmask, ((mIsTesterPresent) ? "TRUE" : "FALSE"));

      DIA_TR_INF("dia_CommChannelINC::onStateUpdate OpMode=0x%02X", ReadOpModeState());
      //To DO : Notify if there has been a change in Opmode

     #ifdef GEN4_DCM_HANDLE_IMPLICIT_REQUEST
     {
        dia_EngineServer* pEngine = nullptr;
        if (( getInstanceOfEngineManager()->queryEngineServer(DIA_UID_ENGINE_CUSTOMER_UDS,&pEngine) != DIA_SUCCESS ) || (!pEngine))
        {
           DIA_TR_ERR("dia_CommChannelINC::onStateUpdate: ERROR: FAILED TO GET ENGINE !!");
           return;
        }
        tU8 activesessionId = pEngine->getActiveSession()->getID();
        //Check if session changes request needs to be sent implicitly
        if(activesessionId != sessionId)
        {
           DIA_TR_INF("dia_CommChannelINC::onStateUpdate Sending Implicit Session change request.");
           tU8 data[] = {0x03, 0x10, 0x01};//Session Change request
           data[2] = sessionId;
           dia_MessageBufferUDS* pMsgBuffer = DIA_NEW dia_MessageBufferUDS (
             &data[0],
             tU16(sizeof(data)),
             dia_tclDiagSessionUds::vNullResponse
           );
           if ( !pMsgBuffer )
           {
              DIA_TR_ERR("dia_CommChannelINC::onStateUpdate ERROR Formulating Internal Session Change message !!");
           }
           else
           {
              // create and send a ReqRx event to uds engine
              if (handleMessage(*pMsgBuffer) == DIA_E_NOT_AVAILABLE)
                 getInstanceOfApplication()->postMessage(DIA_NEW dia_tclDiagSession::tclEventReqRx(pMsgBuffer));
           }
        }
      }
      #endif

      if ( (SCC_DIAGNOSIS_C_STATE_UPDATE_TESTER_PRESENT_BITMASK & stateBitmask) && (!mIsTesterPresent) )
      {
         DIA_TR_INF("--- dia_CommChannelINC::onStateUpdate => SCC_DIAGNOSIS_C_STATE_UPDATE Tester Present");
         mIsTesterPresent = true;
         (void) sendStateUpdateAckToSCC();
      }
      else if ( (0 == (SCC_DIAGNOSIS_C_STATE_UPDATE_TESTER_PRESENT_BITMASK & stateBitmask)) && mIsTesterPresent)
      {
         DIA_TR_INF("--- dia_CommChannelINC::onStateUpdate => SCC_DIAGNOSIS_C_STATE_UPDATE Tester NOT Present");
         mIsTesterPresent = false;
         onTesterDisconnected();
      }
      else
      {
         #ifndef GEN4_DCM_HANDLE_IMPLICIT_REQUEST
         DIA_TR_INF("--- dia_CommChannelINC::onStateUpdate => SCC_DIAGNOSIS_C_STATE_UPDATE Ignore command.");
         #endif
      }
   }
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_CommChannelINC::onForcedStateUpdateACK ( void ) const
{
   DIA_TR_INF("dia_CommChannelINC::onForcedStateUpdateACK SCC_DIAGNOSIS_R_FORCE_STATE_CHANGE Length=%zu Ignored.", mSccRxMessage.size());
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
dia_CommChannelINC::sendComponentStatusToSCC ( dia::enSccComponentStatus status )
{
   dia_tclFnctTrace oTrace("dia_CommChannelINC::sendComponentStatusToSCC");

   if ( (!mpSocketINC) || (!(mpSocketINC->isReady())) ) return DIA_E_SOCKET_NOT_AVAILABLE;

   static unsigned char sccMessage[SCC_DIAGNOSIS_COMPONENT_STATUS_MSGLEN] = {
      SCC_DIAGNOSIS_C_COMPONENT_STATUS,
      0,
      SCC_DIAGNOSIS_COMPONENT_STATUS_MAJOR_VERSION,
      SCC_DIAGNOSIS_COMPONENT_STATUS_MINOR_VERSION
   };

   tDiaResult retCode = DIA_SUCCESS;

   if ( (status == dia::DIA_EN_SCC_COMP_STATUS_ACTIVE) || (status == dia::DIA_EN_SCC_COMP_STATUS_INACTIVE) )
   {
      sccMessage[1] = (tU8) ((status == dia::DIA_EN_SCC_COMP_STATUS_ACTIVE) ? SCC_DIAGNOSIS_COMPONENT_STATUS_ACTIVE : SCC_DIAGNOSIS_COMPONENT_STATUS_INACTIVE);
      retCode = sendMessageToSCC(sccMessage,sizeof(sccMessage));
   }

   return retCode;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
dia_CommChannelINC::sendDiagnosisRequestAckToSCC ( tU8 channelID )
{
   dia_tclFnctTrace oTrace("dia_CommChannelINC::sendDiagnosisRequestAckToSCC");

   static unsigned char sccMessage[SCC_DIAGNOSIS_CONFIRMATION_MSGLEN] = {
         SCC_DIAGNOSIS_R_REQUEST,
         0
   };
   sccMessage[SCC_DIAGMSG_IDX_CHANNEL_ID] = channelID;

   return sendMessageToSCC(sccMessage,sizeof(sccMessage));
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
dia_CommChannelINC::sendStateUpdateAckToSCC ( void )
{
   dia_tclFnctTrace oTrace("dia_CommChannelINC::sendStateUpdateAckToSCC");

   static unsigned char sccMessage[SCC_DIAGNOSIS_CONFIRMATION_MSGLEN] = {
         SCC_DIAGNOSIS_R_STATE_UPDATE,
         0
   };
   sccMessage[SCC_DIAGMSG_IDX_CHANNEL_ID] = mSccRxMessage[SCC_DIAGMSG_IDX_CHANNEL_ID];

   return sendMessageToSCC(sccMessage,sizeof(sccMessage));
}

//------------------------------------------------------------------------------

tDiaResult
dia_CommChannelINC::sendForceStateChangeToSCC ( tU8 channelID, tU8 session, tU32 stateBitMask ) const
{
   dia_tclFnctTrace oTrace("dia_CommChannelINC::sendForceStateChangeToSCC");

   if ( (!mpSocketINC) || (!(mpSocketINC->isReady())) ) return DIA_E_SOCKET_NOT_AVAILABLE;

   unsigned char sccMessage[SCC_DIAGNOSIS_C_FORCE_STATE_CHANGE_MSGLEN] = {
         SCC_DIAGNOSIS_C_FORCE_STATE_CHANGE,
         channelID,
         session,
         U32_BYTE4(stateBitMask), /* little-big endian change */
         U32_BYTE3(stateBitMask), /* little-big endian change */
         U32_BYTE2(stateBitMask), /* little-big endian change */
         U32_BYTE1(stateBitMask)  /* little-big endian change */
   };

   DIA_TR_INF("dia_CommChannelINC::sendForceStateChangeToSCC => channelID=0x%02x, session=0x%02x, stateMask=0x%08x", channelID, session, stateBitMask);

   return sendMessageToSCC(sccMessage,sizeof(sccMessage));
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
dia_CommChannelINC::sendMessageToSCC ( const tU8 data[], tU32 numBytesToSend ) const
{
   dia_tclFnctTrace oTrace("dia_CommChannelINC::sendMessageToSCC");

   if ( (!mpSocketINC) || (!(mpSocketINC->isReady())) ) return DIA_FAILED;

   DIA_TR_SM("INC msg TX: Length: %d Data: %s", (int) numBytesToSend, dia::utils::bin2str(data,(int) numBytesToSend,' ').c_str());

   return mpSocketINC->writeData(data,(tU16) numBytesToSend);
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
dia_CommChannelINC::sendMessageToSCC ( const std::vector<tU8>& data ) const
{
   dia_tclFnctTrace oTrace("dia_CommChannelINC::sendMessageToSCC(const std::vector<tU8>&)");

   if ( (!mpSocketINC) || (!(mpSocketINC->isReady())) ) return DIA_FAILED;

   DIA_TR_SM("INC msg TX: Length: %zu Data: %s", data.size(), dia::utils::bin2str(data,' ').c_str());
   return mpSocketINC->writeData(data);
}

//---------------------------------------------------------------------------------------------------------------------

tU8
dia_CommChannelINC::ReadOpModeState ()
{
   tU8 currOpMode = (tU8)((mSccStateBitmask & SCC_DIAGNOSIS_C_STATE_UPDATE_OPMODE_STATUSMASK) >> 19U);
   return currOpMode;
}

//---------------------------------------------------------------------------------------------------------------------

void dia_CommChannelINC::OnPostProcessingRequest ()
{
	dia_tclFnctTrace oTrace("dia_CommChannelINC::OnPostProcessingRequest");	
	
	tU8 channelID = mSccRxMessage[1];
	tU8 msgLength = mSccRxMessage[2];
	const tU8* pMsgBuffer = (const tU8*) &mSccRxMessage[3];	

	DIA_TR_INF("Channel id = %d , Msg length = %d",channelID,msgLength);

	if (SCC_DIAGNOSIS_C_POST_PROCESSING_REQUEST_MSGLEN != mSccRxMessage.size())
	{
		DIA_TR_INF("Post processing request incorrect length = %d",msgLength);
	}
	else
	{

		switch (channelID)
		{
		case 0:
			{
				// Create an object containing the GMLAN msg. It is deleted by dia_tclDiagSessionUds
				dia_MessageBufferUDS* pMsgBuff = DIA_NEW dia_MessageBufferUDS (
					pMsgBuffer, msgLength,
					dia_CommChannelINC::OnPostProcessingResponse,
					dia_MessageBuffer::holds_request,
					dia_MessageBuffer::format_raw,
					this //cookie
					);

				if ( pMsgBuff )
				{           
					getInstanceOfApplication()->postMessage(DIA_NEW dia_tclDiagSession::tclEventReqRx(pMsgBuff));
				}
				else
				{
					DIA_TR_INF("pMsgBuff = NULL");
					//FATAL_M_ASSERT_ALWAYS();
				}
				// make lint happy
				pMsgBuff = NULL; //lint !e438 !e423 Warning: Creation of memory leak in assignment to 'pMsgBuff'. --> lifetime is controlled by FSM of diagnostic engine

				break;
			}

		default:
			{
				DIA_TR_INF("!!! dia_CommChannelINC::OnPostProcessingRequest => ERROR: Unknown channel %d", channelID);
				break;
			}
		} //switch (channelID)
	} //else	
}

void
dia_CommChannelINC::OnPostProcessingResponse (const tU8 data[], tU16 length, void* /*cookie*/)
{
   dia_tclFnctTrace oTrace("--> dia_CommChannelINC::OnPostProcessingResponse");

   //*****************Only for trace********************//
   for(tU8 i=0; i<length;++i)
   {
	   DIA_TR_INF("Response byte[%d]= 0x%08x",i,data[i]);
   }
   //*****************Only for trace********************//

   //No response back to the UDD   
   getInstanceOfApplication()->postMessage(DIA_NEW dia_tclDiagSession::tclEventConfTxOk());
   
}
