/*!
 * \file       dia_trace.h
 *
 * \brief      Diagnostic trace macros, functions, ...
 *
 * \details    Diagnostic trace macros, functions, ...
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreTrace
 *
 * \copyright  (c) 2012-2017 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_TRACE__
#define __INCLUDED_DIA_TRACE__

#ifndef __INCLUDED_DIA_MACROS__
#include "common/framework/application/dia_macros.h"
#endif

//---------------------------------------------------------------------------------------------------------------------

#define DIA_C_TRACE_MAX       ((tU16) 239)

//---------------------------------------------------------------------------------------------------------------------

#define DIA_M_FILL_TRACE_BUFFER(format)                                                                              \
                                                                                                                     \
   tChar pBuffer[DIA_C_TRACE_MAX];                                                                                   \
                                                                                                                     \
   int index = 0;                                                                                                    \
   dia::createTraceHeader(pBuffer,DIA_C_TRACE_MAX,index);                                                            \
                                                                                                                     \
   ::va_list argList;                                                                                                \
   ::va_start(argList, format);                                                                                      \
                                                                                                                     \
   index += ::vsnprintf(&pBuffer[index], DIA_C_TRACE_MAX-index, format, argList);                                    \
   pBuffer[(index < DIA_C_TRACE_MAX) ? index : (DIA_C_TRACE_MAX-1)] = '\0';                                          \
                                                                                                                     \
   ::va_end(argList)

//---------------------------------------------------------------------------------------------------------------------

#define DIA_M_FILL_TRACE_BUFFER_WITH_DATA(format,data)                                                               \
                                                                                                                     \
   tChar pBuffer[DIA_C_TRACE_MAX];                                                                                   \
                                                                                                                     \
   int index = 0;                                                                                                    \
   dia::createTraceHeader(pBuffer,DIA_C_TRACE_MAX,index);                                                            \
                                                                                                                     \
   index += ::snprintf(&pBuffer[index], DIA_C_TRACE_MAX-index, format, data);                                        \
   pBuffer[(index < DIA_C_TRACE_MAX) ? index : (DIA_C_TRACE_MAX-1)] = '\0'


namespace dia {

class ScopeTrace
{
   DECL_DEPRECATED_COPYCONSTRUCTOR_AND_ASSIGNMENTOPERATOR(ScopeTrace);
public:
   ScopeTrace ( tCString str );
   ScopeTrace ( const std::string& str );
   ScopeTrace ( tU16 classID, tCString str );
   ~ScopeTrace ( void );

protected:
   // Do not use!!!
   ScopeTrace();

private:
   tCString mTraceData;
   tU16     mClassID;
   bool     mIsTracingActivated;
};

class dia_ScopeTraceVarg {
  DECL_DEPRECATED_COPYCONSTRUCTOR_AND_ASSIGNMENTOPERATOR(dia_ScopeTraceVarg);
public:
   dia_ScopeTraceVarg(const char* format, ...);
   ~dia_ScopeTraceVarg();


private:
   std::string traceString;
   ScopeTrace *scopeTrace;
};


}

#define DIA_TR_ERR      dia_TraceError
#define DIA_TR_WRN      dia_TraceWarning
#define DIA_TR_INF      dia_TraceInfo
#define DIA_TR_SM       dia_TraceSmall
#define DIA_TR_SYS      dia_TraceInfo
#define DIA_TR_CUS      dia_TraceCustomer
#define DIA_TR_USR1     dia_TraceInfo
#define DIA_TR_ERRMEM   dia_TraceErrorMemory
#define DIA_TR          dia_TraceGeneric
#define DIA_TR_BAS      dia_TraceBase
#define DIA_DBG_TR_INF  dia_DebugTraceInfo
#define DIA_DBG_TR_ERR  dia_DebugTraceError

#ifdef __cplusplus
extern "C"
{
#endif

void  dia_TraceInfo ( const char* format, ...);
void  dia_TraceWarning ( const char* format, ...);
void  dia_TraceError ( const char* format, ...);
void  dia_TraceErrorMemory ( const char* format, ...);
void  dia_TraceSmall ( const char* format, ...);
void  dia_TraceBase ( const char* format, ...);
void  dia_TraceCustomer ( const char* format, ...);
void  dia_TraceGeneric ( tU16 classID, tU16 level, const char* format, ...);
bool  dia_isTraceActive ( tU16 u16CompID, tU16 u16LevelId );
void  dia_DebugTraceError ( const char* format, ...);
void  dia_DebugTraceInfo ( const char* format, ...);

#ifdef __cplusplus
}
#endif

#ifndef __DIA_UNIT_TESTING__

#include "ostrace.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h" //lint !e451 !e537 repeatedly included header file without standard include guard

#else

#define TR_COMP_DIAGNOSIS            (256 * 33)     /* componentname = "DIAGNOSIS" */

#endif

// etrace has a macro definition for "_" that might cause trouble in boost headers files, therefore we undefine it here.
// Usage of "_" is deprecated. Please use the diagnostic macros with ","
#ifdef _
#undef _
#endif

#define DIA_TRACE_CLASS_DIAGNOSTICS_BASE     (tU16) (TR_COMP_DIAGNOSIS+1)
#define DIA_TRACE_CLASS_DIAGNOSTICS_SYSADAPT (tU16) (TR_COMP_DIAGNOSIS+2)
#define DIA_TRACE_CLASS_DIAGNOSTICS_CUST     (tU16) (TR_COMP_DIAGNOSIS+3)
#define DIA_TRACE_CLASS_DIAGNOSTICS_PROD     (tU16) (TR_COMP_DIAGNOSIS+4)
#define DIA_TRACE_CLASS_DIAGNOSTICS_FUNC     (tU16) (TR_COMP_DIAGNOSIS+5)
#define DIA_TRACE_CLASS_DIAGNOSTICS_STRING   (tU16) (TR_COMP_DIAGNOSIS+6)
#define DIA_TRACE_CLASS_DIAGNOSTICS_HMI      (tU16) (TR_COMP_DIAGNOSIS+7)
#define DIA_TRACE_CLASS_DIAGNOSTICS_DEFSET   (tU16) (TR_COMP_DIAGNOSIS+8)
#define DIA_TRACE_CLASS_DIAGNOSTICS_SYSSET   (tU16) (TR_COMP_DIAGNOSIS+8)

#define DIA_TRACE_CLASS_DIAGNOSTICS_SM       (tU16) (TR_COMP_DIAGNOSIS+30)
#define DIA_TRACE_CLASS_DIAGNOSTICS_SECURITY (tU16) (TR_COMP_DIAGNOSIS+31)
#define DIA_TRACE_CLASS_DIAGNOSTICS_CORE     (tU16) (TR_COMP_DIAGNOSIS+32)

// ========================================================================
// ========================= Needed for ETG Traces ========================
// vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv

enum tenTrcTraceClass
{
    TR_CLASS_DIAGNOSTICS_BASE       = DIA_TRACE_CLASS_DIAGNOSTICS_BASE,
    TR_CLASS_DIAGNOSTICS_SYSADAPT   = DIA_TRACE_CLASS_DIAGNOSTICS_SYSADAPT,
    TR_CLASS_DIAGNOSTICS_CUST       = DIA_TRACE_CLASS_DIAGNOSTICS_CUST,
    TR_CLASS_DIAGNOSTICS_PROD       = DIA_TRACE_CLASS_DIAGNOSTICS_PROD,
    TR_CLASS_DIAGNOSTICS_FUNC       = DIA_TRACE_CLASS_DIAGNOSTICS_FUNC,
    TR_CLASS_DIAGNOSTICS_STRING     = DIA_TRACE_CLASS_DIAGNOSTICS_STRING,
    TR_CLASS_DIAGNOSTICS_HMI        = DIA_TRACE_CLASS_DIAGNOSTICS_HMI,
    TR_CLASS_DIAGNOSTICS_DEFSET     = DIA_TRACE_CLASS_DIAGNOSTICS_DEFSET,
    TR_CLASS_DIAGNOSTICS_SM         = DIA_TRACE_CLASS_DIAGNOSTICS_SM,
    TR_CLASS_DIAGNOSTICS_SECURITY   = DIA_TRACE_CLASS_DIAGNOSTICS_SECURITY,
    TR_CLASS_DIAGNOSTICS_CORE       = DIA_TRACE_CLASS_DIAGNOSTICS_CORE
};

// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
// ========================= Needed for ETG Traces ========================
// ========================================================================

#define DIA_TRC_BASE                         (tU16) (TR_COMP_DIAGNOSIS+1)
#define DIA_TRC_CH                           (tU16) (TR_COMP_DIAGNOSIS+2) // client handler
#define DIA_TRC_SYSADAPT                     (tU16) (TR_COMP_DIAGNOSIS+2)
#define DIA_TRC_UDS                          (tU16) (TR_COMP_DIAGNOSIS+3)
#define DIA_TRC_MCNET                        (tU16) (TR_COMP_DIAGNOSIS+4)
#define DIA_TRC_FUNC                         (tU16) (TR_COMP_DIAGNOSIS+5)
#define DIA_TRC_STR                          (tU16) (TR_COMP_DIAGNOSIS+6)
#define DIA_TRC_HMI                          (tU16) (TR_COMP_DIAGNOSIS+7)
#define DIA_TRC_DEFSET                       (tU16) (TR_COMP_DIAGNOSIS+8)
#define DIA_TRC_SYSSET                       (tU16) (TR_COMP_DIAGNOSIS+8)

#define DIA_TRC_SM                           (tU16) (TR_COMP_DIAGNOSIS+30)
#define DIA_TRC_SECURITY                     (tU16) (TR_COMP_DIAGNOSIS+31)
#define DIA_TRC_CORE                         (tU16) (TR_COMP_DIAGNOSIS+32)

// TTFis msg IDs and parameters
#define PAR_DIA_MCNET            0x00
#define PAR_DIA_UDS              0x01
#define PAR_DIA_IPN_SEND         0x02
#define PAR_DIA_CHSC             0x03
#define PAR_DIA_CMD              0x04
#define PAR_DIA_GMLAN            0x05
#define PAR_DIA_NOT_AVAILABLE    0xFF

#define ID_DIA_REPORT_ITC        0x05
#define DIA_ITC_STATUS_PASSED    0x00
#define DIA_ITC_STATUS_FAILED    0x01
#define DIA_ITC_STATUS_ENABLE    0x02
#define DIA_ITC_STATUS_DISABLE   0x03

#define ID_DIA_REQ               0x00
#define ID_DIA_TESTMSG           0x01
#define ID_DIA_SECURITY          0x02
#define ID_DIA_CMD               0x03

///////////////////////// Helpers

#define dia_tclFnctTrace dia::ScopeTrace

#endif /* __INCLUDED_DIA_TRACE__ */
