/**
 * \file      dia_TestConfigHash.cpp
 *
 * \brief     Base class for unique ID generation of device configuration
 *
 * \details   Calculate checksum for configuration items and write it to registry mPropertyResultFingerprintLongFormat
 *
 * \author    kaa1hi
 * \date      Jul 21, 2014
 *
 * \copyright Robert Bosch Car Multimedia 2014
 */
// DIA_REQ UDS 04 FD FD 0D


#ifndef __INCLUDED_DIA_DEFINES_UDS__
#include <common/framework/protocols/uds/dia_defsUds.h>
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG_PROJECT__
#include "project/framework/config/dia_defsProjectConfig.h"
#endif

#ifndef __INCLUDED_DIA_TEST_CONFIG_HASH_H_
#include "dia_TestConfigHash.h"
#endif

#ifndef __INCLUDED_DIA_UTILITIES__
#include "common/framework/utils/dia_utilities.h"
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include "common/framework/config/dia_ConfigManager.h"
#endif

#ifndef __INCLUDED_EXT_DIAGLOG_IF__
#include "project/framework/application/ext_diaglog_if.h"      //label for ITC
#endif

#ifndef __INCLUDED_DIA_INTERFACE_ERRORLOG__
#include "common/interfaces/dia_IErrorLog.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#define DATA_START (1/*sizeof(SID)*/ + mDIDLen)

enum dia_ITC_value
{
   DIA_ITC_PASSED = 0,     /* =0 */
   DIA_ITC_FAILED          /* =1 */
};

using namespace std;

//-----------------------------------------------------------------------------

dia_TestConfigHash::dia_TestConfigHash ( tU32 propRefConfFingerprintShortFormat, tU32 propResultFingerprintLongFormat, tU16 itc )
   : dia_TestService("dia_TestConfigHash", DIA_C_U8_UDS_SID_DIAG_COMMAND,
                     DIA_C_U16_DID_RBCM_TEST_CONFIG_CD_HASH),
                     mPropertyReferenceConfFingerprintShortFormat(propRefConfFingerprintShortFormat),
                     mPropertyResultFingerprintLongFormat(propResultFingerprintLongFormat),
                     mITC(itc)
{
   dia_tclFnctTrace oTrace("dia_TestConfigHash::dia_TestConfigHash()");
}

//-----------------------------------------------------------------------------

dia_TestConfigHash::~dia_TestConfigHash ( void )
{
	_BP_TRY_BEGIN
	{
		(void) unsetSysAdapterListener<dia_IDisplayPatternAIVIListener>(this);
	}
	_BP_CATCH_ALL
	{
		DIA_TR_ERR("EXCEPTION CAUGHT: dia_IOCtrlDisplayTestPatternAIVI::~dia_IOCtrlDisplayTestPatternAIVI !!!");
		DIA_ASSERT_ALWAYS();
	}
	_BP_CATCH_END


}

void
dia_TestConfigHash::setPropData(const dia_FingerprintInfo* pPropData, const tU16 sizeOfArray)
{
   DIA_TR_INF("dia_TestConfigHash::setPropData() sizeOfArray=%d", sizeOfArray);

   mVectPropData.clear();
   for (tU16 i=0; i<sizeOfArray; i++)
   {
      mVectPropData.push_back(pPropData[i].mProperty);
   }
}

void
dia_TestConfigHash::setPropData(std::vector<tU32>& scrVect)
{
   mVectPropData.clear();

   for (std::vector<tU32>::iterator it = scrVect.begin(); it!=scrVect.end(); ++it)
   {
      mVectPropData.push_back(*it);
   }
}

bool
dia_TestConfigHash::compareBothFingerprints(std::vector<tU8>& calculatedHash)
{
   dia_tclFnctTrace oTrace("dia_TestConfigHash::compareBothFingerprints()");
   dia_IDisplayPatternAIVI* pInterface = 0;

   size_t size = 0;
   bool retVal = false;

   //get reference fingerprint from memory
   if ( DIA_CONFIG_HASH_BINARY_SIZE==(size = dia_getPropertySize(DIA_PROP_CM_CONF_INFO)) )
   {
      tDiaResult getResult;
      DIA_TR_INF("dia_TestConfigHash::compareBothFingerprints dia_getPropertySize OK");
      tU8 referenceHash[DIA_CONFIG_HASH_BINARY_SIZE] = { 0 };

      if ( DIA_SUCCESS==(getResult = dia_getProperty(DIA_PROP_CM_CONF_INFO, referenceHash, DIA_CONFIG_HASH_BINARY_SIZE)) )
      {
    	  DIA_TR_INF("dia_TestConfigHash::compareBothFingerprints dia_getProperty OK");

    	  retVal = (bool) std::equal( calculatedHash.begin(), calculatedHash.end(), referenceHash);   //lint !e864:  Expression involving variable 'calculatedHash' possibly DOESN'T depend on order of evaluation
    	  if ( querySysAdapterInterface<dia_IDisplayPatternAIVI>(&pInterface) == DIA_SUCCESS )
    	  {
    		  if ( pInterface )
    		  {
    			  (void) setSysAdapterListener<dia_IDisplayPatternAIVIListener>(this);
    			   pInterface->setHashComparisonStatus(retVal);
    		  }
    	  }
    	  std::vector<tU8> vectTmp(referenceHash, referenceHash+(sizeof(referenceHash)/sizeof(tU8)) );
    	  DIA_TR_INF("compareBothFingerprints calculatedHash=%s", dia_nsUtilities::convertArray2HexString(calculatedHash).c_str());
    	  DIA_TR_INF("compareBothFingerprints reference_Hash=%s", dia_nsUtilities::convertArray2HexString(vectTmp).c_str());
      }
      else
      {
         DIA_TR_ERR("### dia_TestConfigHash::compareBothFingerprints dia_getProperty NOK getResult=0x%08X ###", getResult);
      }
   }
   else
   {
      DIA_TR_ERR("### dia_TestConfigHash::compareBothFingerprints dia_getPropertySize NOK size=%zu ###", size);
   }

   DIA_TR_INF("dia_TestConfigHash::compareBothFingerprints returned %s", (retVal ? "TRUE": "FALSE"));

   return retVal;
}

void
dia_TestConfigHash::setITC(bool result)
{
   dia_tclFnctTrace oTrace("dia_TestConfigHash::setITC()");

   tU8 message[4] = {0};

   DIA_TR_INF("dia_TestConfigHash::setITC - result %s", ( result? "DIA_ITC_PASSED": "DIA_ITC_FAILED"));

   // currently only 2 Bytes ITC supported
   message[0] = 3; // the first byte of the message is length of data container
   message[1] = (tU8)(mITC >> 8);
   message[2] = (tU8)(mITC & 0xFF);
   message[3] = result ? DIA_ITC_PASSED : DIA_ITC_FAILED;

   dia_IErrorLog* pErrorLog = 0;
   tDiaResult retVal;
   if ((DIA_SUCCESS==(retVal=querySysAdapterInterface<dia_IErrorLog>(&pErrorLog))) && pErrorLog)
   {
      if ( pErrorLog->u32TestDTCStart(&message[0]) != DIA_SUCCESS )
      {
         DIA_TR_ERR("dia_TestConfigHash::setITC UNABLE TO SET ITC label=0x%04X", mITC);
      }
      else
      {
         DIA_TR_INF("dia_TestConfigHash::setITC itc=0x%04X u32TestDTCStart successful", mITC);
      }
   }
   else
   {
      DIA_TR_ERR("dia_TestConfigHash::setITC - UNABLE TO SEND TEST RESULT!!!");
      DIA_TR_ERR("dia_TestConfigHash::setITC - ERROR pErrorLog=0x%p", pErrorLog);
      DIA_TR_ERR("dia_TestConfigHash::setITC - ERROR retVal=0x%08X", retVal);
   }
}

void
dia_TestConfigHash::vProcessRequest ( const std::vector<tArgsType>& /*vecArgs*/ )
{
   dia_tclFnctTrace oTrace("dia_TestConfigHash::vProcessRequest()");

   std::vector<tU8> configData;
   std::vector<tU8> newBinHash;

   // read all configuration items and append to output buffer
   if (DIA_SUCCESS!=dia_nsUtilities::readConfiguration(mVectPropData, configData))
   {
      DIA_TR_ERR("### dia_TestConfigHash::vProcessRequest() NOT ALL CONFGURATION ITEMS WAS READ ###");
      DIA_TR_ERR("### DESPITE THE READING ERROR PLEASE CONTINUE ###");
   }

   if (!configData.empty())
   {
	   if (false==dia_nsUtilities::generateHash(configData, newBinHash))
	   {
		   DIA_TR_ERR("dia_TestConfigHash::vProcessRequest() generateHash FAILED");
		   testTerminate();
		   return;
	   }
	   else
	   {
		   DIA_TR_ERR("dia_TestConfigHash::vProcessRequest() generateHash SUCCESS");
	   }
   }
   else
   {
      DIA_TR_ERR("dia_TestConfigHash::vProcessRequest() ERROR. Size of new hash is 0.");
      testTerminate();
      return;
   }

   if ( DIA_CONFIG_HASH_BINARY_SIZE != newBinHash.size())
   {
      DIA_TR_ERR("dia_TestConfigHash::vProcessRequest() ERROR: newBinHash.size()=%zu", newBinHash.size());
      testTerminate();
      return;
   }

   //compare actual fingerprint to reference fingerprint from KDS
   bool fingerprintMatch = compareBothFingerprints(newBinHash);

   fingerprintMatch = checkAdditionalConditionForFPComparison(newBinHash, fingerprintMatch);

   //set ITC
   setITC(fingerprintMatch);

   //if fingerprints don't match, write zeros to registry.
   if (false==fingerprintMatch)
   {
      DIA_TR_INF("dia_TestConfigHash::vProcessRequest() Write zeros to registry. FPs don't match");

      if (DIA_SUCCESS!=dia_nsUtilities::writeHashLongFormatToRegistry(NULL, mPropertyResultFingerprintLongFormat))
      {
         DIA_TR_ERR("### dia_TestConfigHash::vProcessRequest() REGISTER NOT UPDATED ###");
      }
      testTerminate();
      return;
   }

   // conversion to format "{00112233-00112233-00112233-00112233-00112233}"
   string strHash;
   dia_nsUtilities::convertToHashSpecificFormat( newBinHash, strHash);

   DIA_TR_INF("dia_TestConfigHash::vProcessRequest() newBinHash=%s", dia_nsUtilities::convertArray2HexString(newBinHash).c_str() );
   DIA_TR_INF("dia_TestConfigHash::vProcessRequest() strHash=%s", strHash.c_str() );

   tU16 actualSize = static_cast<tU16>(strHash.size() + 1 /* one byte null termination */);
   tU16 expectedSize = static_cast<tU16>(dia_getPropertySize(mPropertyResultFingerprintLongFormat));

   DIA_TR_INF("dia_TestConfigHash::vProcessRequest() hash size: exp=%d, actual=%d", expectedSize, actualSize);

   if (actualSize != expectedSize)
   {
      DIA_TR_INF("dia_TestConfigHash::vProcessRequest() ERROR! WRONG SIZE OF NEW HASH SPECIFIC FORMAT");
      if (DIA_SUCCESS!=dia_nsUtilities::writeHashLongFormatToRegistry(NULL, mPropertyResultFingerprintLongFormat))
      {
         DIA_TR_ERR("### dia_TestConfigHash::vProcessRequest() REGISTER NOT UPDATED ###");
      }
      testTerminate();
      return;
   }

   DIA_TR_INF(" mPropertyResultFingerprintLongFormat=0x%08X expectedSize=%d", mPropertyResultFingerprintLongFormat, expectedSize);

   // read current/old value for trace purposes only
   tU8 buff[DIA_CONFIG_HASH_STRING_LONG_FORMAT_SIZE] = {0};

   if (DIA_SUCCESS != dia_getProperty(mPropertyResultFingerprintLongFormat, buff, expectedSize))
   {
      DIA_TR_ERR("dia_TestConfigHash::vProcessRequest() dia_getProperty for currHash FAILED");
   }
   else
   {
      DIA_TR_INF("dia_TestConfigHash::vProcessRequest() currHash=%s", buff );

      if (strHash==string((const char*)(buff))) //lint !e429: custodial pointer is freed at the end of function
      {
         DIA_TR_INF("dia_TestConfigHash::vProcessRequest() Two config hashes are identical.");
      }
      else
      {
         DIA_TR_INF("dia_TestConfigHash::vProcessRequest() Two config hashes are NOT identical.");
      }
   }

   //copy string content to tU8 buffer
   tU8 buffHashLongFormat[DIA_CONFIG_HASH_STRING_LONG_FORMAT_SIZE] = {0};
   if (expectedSize<=DIA_CONFIG_HASH_STRING_LONG_FORMAT_SIZE)
   {
      size_t length = strHash.copy( reinterpret_cast<char*>(buffHashLongFormat), expectedSize);
      buffHashLongFormat[length]='\0';

      DIA_TR_INF("dia_TestConfigHash::vProcessRequest() length=%zu", length);
   }

   if (DIA_SUCCESS!=dia_nsUtilities::writeHashLongFormatToRegistry(buffHashLongFormat, mPropertyResultFingerprintLongFormat))
   {
      DIA_TR_ERR("### dia_TestConfigHash::vProcessRequest() REGISTER NOT UPDATED ###");
   }

   testTerminate();
}

//-----------------------------------------------------------------------------

void
dia_TestConfigHash::vOnTimeout ( void )
{
   dia_tclFnctTrace oTrace("dia_TestConfigHash::vOnTimeout()");
   testTerminate();
}

bool
dia_TestConfigHash::checkAdditionalConditionForFPComparison ( const std::vector<tU8> /*calculatedCDHash*/, bool result )
{
   dia_tclFnctTrace oTrace("dia_TestConfigHash::checkAdditionalConditionForFPComparison()");

   /* Generic method */
   DIA_TR_INF("dia_TestConfigHash::checkAdditionalConditionForFPComparison DO NOTHING HERE.");

   return result;
}
