/**
 * \file       dia_SystemSettingsModule.h
 *
 * \brief      Abstract base class for all kinds of modules need to be triggered by a system setting operation
 *
 * \details    Abstract base class for all kinds of modules need to be triggered by a system setting operation
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreSystemSettings
 *
 * \copyright  (c) 2015 Robert Bosch Car Multimedia
 *
 */

#ifndef __INCLUDED_DIA_SYSTEM_SETTINGS_MODULE__
#define __INCLUDED_DIA_SYSTEM_SETTINGS_MODULE__

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_COMMON_SYSSET__
#include <common/framework/sysset/dia_common_sysset.h>
#endif

struct dia_SystemSettingsModuleTypeInfo;

struct dia_SystemSettingsModuleTypeInfo
{
   tU32 mMaskValue;
   tCString mTypeName;
   dia_UID mUID;
};

class dia_SystemSettingsModule
{
public:
   //! class constructor
   dia_SystemSettingsModule ( tCString name );
   //! class constructor
   dia_SystemSettingsModule ( tCString name, tU32 mask );
   //! class destructor
   virtual ~dia_SystemSettingsModule ( void );

   //! return the module's name
   virtual tCString getName ( void ) const;
   //! return the module's UID which the hash code retrieved from the name
   dia_UID getUID ( void ) const;

   //! return the module's name
   virtual void setTypeMask ( tU32 mask );

   //! prepare the module for the system setting
   virtual tDiaResult prepareSystemSetting ( dia_UID sysSetTypeUID ) {
      dia_SystemSettingExtendedData extData;
      return prepareSystemSetting(sysSetTypeUID,extData);
   }
   //! execute the system setting
   virtual tDiaResult executeSystemSetting ( dia_UID sysSetTypeUID  ) {
      dia_SystemSettingExtendedData extData;
      return executeSystemSetting(sysSetTypeUID,extData);
   }
   //! finalize the system setting
   virtual tDiaResult finalizeSystemSetting ( dia_UID sysSetTypeUID  ) {
      dia_SystemSettingExtendedData extData;
      return finalizeSystemSetting(sysSetTypeUID,extData);
   }

   //! prepare the module for the system setting
   virtual tDiaResult prepareSystemSetting ( dia_UID /*sysSetTypeUID*/, const dia_SystemSettingExtendedData& /*extData*/ ) { return DIA_E_NOT_IMPLEMENTED; }
   //! execute the system setting
   virtual tDiaResult executeSystemSetting ( dia_UID /*sysSetTypeUID*/, const dia_SystemSettingExtendedData& /*extData*/  ) { return DIA_E_NOT_IMPLEMENTED; }
   //! finalize the system setting
   virtual tDiaResult finalizeSystemSetting ( dia_UID /*sysSetTypeUID*/, const dia_SystemSettingExtendedData& /*extData*/  ) { return DIA_E_NOT_IMPLEMENTED; }

   virtual bool isSystemSettingTypeSupported ( dia_UID typeUID );
   virtual tDiaResult getSystemSettingTypeName ( dia_UID typeUID, std::string& name );

protected:
    //! default class constructor without implementation
   dia_SystemSettingsModule ( void );

   //! evaluates mTypeMask and populates mTypeLookupRep
   virtual void evaluateTypeMask ( void );
   //! add further entries in SystemSettingsModuleTypeInfo map or replace existing ones (based on map-key)
   virtual tDiaResult loadSystemSettingsModuleTypeInfo ( const dia_SystemSettingsModuleTypeInfo typeInfoData[], tU16 numOfTypeInfos );

protected:
   //! name of the module
   tCString mName;
   //! unique system setting module identifier
   dia_UID mUID;
   //! mask where every bit denotes a certain system setting type
   tU32 mTypeMask;
   //! this flag indicates if a mask for this modules was specified. otherwise we assume that this module supports every system setting type.
   bool isTypeMaskUsed;
   //! type info lookup table
   std::map<dia_UID,dia_SystemSettingsModuleTypeInfo> mTypeLookupRep;
};

#endif /* __INCLUDED_DIA_SYSTEM_SETTINGS_MODULE__ */
